/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : main
 *
 * Copyright 2005,2007-2008 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include <QList>
#include <QtSql>
#include <QPair>
#include <QDebug>
#include <QPixmap>
#include <QVariant>
#include <QFileInfo>
#include <QSqlRecord>
#include <QModelIndex>
#include <QStringList>
#include <QStringListModel>

#include <KUrl>
#include <KAction>
#include <KStatusBar>
#include <KFileDialog>
#include <KMessageBox>
#include <KMainWindow>
#include <KFileDialog>
#include <KConfigGroup>
#include <KApplication>
#include <KIO/NetAccess>
#include <KStandardDirs>
#include <KStandardAction>
#include <KActionCollection>

#include <linux/limits.h>

#include "uploaddlg.h"
#include "flickrcomm.h"
#include "mainwindow.h"
#include "previewmgr.h"
#include "batchwidget.h"
#include "editorwidget.h"
#include "photodelegate.h"
#include "sqlphototablemodel.h"

#include "ui_mainwidget.h"
#include "ui_removeuser.h"
#include "ui_authcomplete.h"
#include "ui_authquestiondlg.h"

class MainWindow::PrivateData
{
public:
  PrivateData()
    : uploadInProgress(false)
    , activeUser(-1,"")
    , progressDlg(0L)
  {
  }

  ~PrivateData()
  {
  }

  bool uploadInProgress;
  QPair <int,QString> activeUser; /// The currently active user (index and name)

  Ui::mainwidget ui;              /// The widget UI

  QStringListModel users;
  QStringList userNSIDs;          /// List of all the authorized user NSIDs
  QStringList userTokens;         /// List of all the authorized user tokens

  KAction* addPhotoAction;
  KAction* removePhotoAction;
  KAction* uploadAction;

  FlickrComm comm;                /// Instance of the communications class used to talk to flickr

  SqlPhotoTableModel* photoTable;
  QStringListModel tagsModel;
  QStringListModel licensesModel;
  QStringListModel photosetsModel;

  PhotoDelegate delegate;
  EditorWidget* photoEditor;
  BatchWidget* batchEditor;
  UploadDlg* progressDlg;
};

namespace
{
  enum
  {
    Status_Bandwidth,
    Status_PhotoCount
  };

  bool createDBConnection()
  {
    QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE");

    db.setDatabaseName(KStandardDirs::locateLocal("appdata", "photos.db"));
    if( !db.open() )
      return false;

    QSqlQuery query;
    query.exec(QString("create table photographs (id int primary key"
		       ", number int"
		       ", filename varchar(%1)"
		       ", title nvarchar(100)"
		       ", description nvarchar(500)"
		       ", size varchar(50)"
		       ", width int"
		       ", height int"
		       ", license varchar(100)"
		       ", photoset nvarchar(100)"
		       ", exposed boolean"
		       ", family boolean"
		       ", friends boolean"
		       ", rotation int"
		       ", tags nvarchar(500)"
		       ")").arg(PATH_MAX));

    return true;
  }

  int getBiggestRowCount(const SqlPhotoTableModel* model)
  {
    int value = -1;
    int biggest = -1;

    for( int row = model->rowCount() - 1; row >= 0; --row )
    {
      if( model->data(model->index(row, model->fieldIndex("count"))).toInt() > value )
      {
	value = model->data(model->index(row, model->fieldIndex("count"))).toInt();
	biggest = row;
      }
    }
    return biggest;
  }

  QVariant getRecordValue(const QSqlRecord& rec, const QString& field)
  {
    return rec.value(rec.indexOf(field));
  }
};
  
MainWindow::MainWindow(QWidget *parent)
  : KXmlGuiWindow(parent)
  , d(new  MainWindow::PrivateData)
{
  // setup to retrieve saved settings
  KConfigGroup config = KGlobal::config()->group("KFlickr");

  // Stored user data
  d->users.setStringList(config.readEntry("users", QStringList()));
  d->userNSIDs = config.readEntry("user_nsids", QStringList());
  d->userTokens = config.readEntry("user_tokens", QStringList());
  QString user = config.readEntry("user", QString());

  // A bunch of widget creation stuff.
  QWidget *widget = new QWidget(this);
  d->ui.setupUi(widget);
  d->ui.userCombo->setModel(&d->users);
  QMetaObject::connectSlotsByName(this);
  setCentralWidget(widget);
  setupActions();

  // Activate the stored user, if there is one.
  if( user != "" )
    setActiveUser(user);

  // Setup the photograph DB
  if(!createDBConnection())
  {
    KMessageBox::error(this,i18n("Cannot open database."));
    quit();
  }
  d->photoTable = new SqlPhotoTableModel(this);
  d->photoTable->setTable("photographs");
  d->photoTable->setHeaderData(0, Qt::Horizontal, i18n("Title"));
  d->photoTable->setSort(1, Qt::AscendingOrder);
  d->photoTable->select();

  // Associate the DB/Model with the listview, we associate the listview with the
  // rotation index since this is the only attribute the should cause a redraw
  d->ui.photoList->setModel(d->photoTable);
  d->ui.photoList->setModelColumn(d->photoTable->fieldIndex("rotation"));

  // Set in the designer but there seems to be a but, these are required
  // for drag and drop so to avoid head banging I am setting them
  // explicitely here.
  d->ui.photoList->setSelectionMode(QAbstractItemView::ExtendedSelection);
  d->ui.photoList->setDragEnabled(true);
  d->ui.photoList->setAcceptDrops(true);
  d->ui.photoList->setDropIndicatorShown(true);
  d->ui.photoList->setDragDropMode(QAbstractItemView::DragDrop);
  d->ui.photoList->setItemDelegate(&d->delegate);
  d->ui.photoList->setAutoScroll(true);

  // Setup the photo editor
  d->photoEditor = new EditorWidget;
  d->photoEditor->setEnabled(false);
  d->photoEditor->setModel(d->photoTable);
  d->photoEditor->setTagsModel(d->tagsModel);
  d->photoEditor->setLicensesModel(d->licensesModel);
  d->photoEditor->setPhotosetsModel(d->photosetsModel);
  QWidget* oldTab = d->ui.editors->widget(0);
  d->ui.editors->addTab(d->photoEditor, i18n("Single"));
  d->ui.editors->removeTab(0);
  delete oldTab;

  // Setup the batch photo editor
  d->batchEditor = new BatchWidget;
  d->batchEditor->setEnabled(false);
  d->batchEditor->setModel(d->photoTable);
  d->batchEditor->setTagsModel(d->tagsModel);
  d->batchEditor->setLicensesModel(d->licensesModel);
  d->batchEditor->setPhotosetsModel(d->photosetsModel);
  d->ui.editors->addTab(d->batchEditor, i18n("Batch"));
  d->ui.editors->setTabEnabled(1, false);

  // Setup the status bar
  statusBar()->insertPermanentItem(i18n("Unused Upload Bandwidth:%1",0), Status_Bandwidth);
  statusBar()->insertPermanentItem(i18n("# of Photos:%1",0), Status_PhotoCount);

  // Create the necessary connections
  connect(&d->comm, SIGNAL(commError(const QString&)), SLOT(commError(const QString&)));
  connect(&(d->comm), SIGNAL(returnedFrob(const QString&)),
	  SLOT(doUserAuthentication(const QString&)));
  connect(&d->comm, SIGNAL(returnedToken(const QString&, const QString &, const QString&)),
	  SLOT(addUser(const QString&, const QString&, const QString&)));
  connect(&d->comm, SIGNAL(returnedTags(const QStringList&)),
	  SLOT(setTags(const QStringList &)));
  connect(&d->comm, SIGNAL(returnedUploadStatus(const QString&)),
	  SLOT(setBandwidth(const QString&)));
  connect(&d->comm, SIGNAL(returnedLicenses(const QStringList&)),
	  SLOT(setLicenseTypes(const QStringList&)));
  connect(&d->comm, SIGNAL(returnedPhotosets(const QStringList&, const QString&)),
	  SLOT(setPhotosets(const QStringList&, const QString&)));
  connect(&d->comm, SIGNAL(returnedUploadedOK(const QString&)),
	  SLOT(photoUploaded(const QString&)));

  connect(PreviewMgr::instance(), SIGNAL(updatedPreview(int)), SLOT(updatePreview(int)));
  connect(d->ui.photoList->selectionModel(),
	  SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)),
	  SLOT(selectionChanged(const QItemSelection&, const QItemSelection&)));
  connect(d->photoTable, SIGNAL(rowsInserted(const QModelIndex&, int, int)),
	  SLOT(tableChanged(const QModelIndex&, int, int)));
  connect(d->photoTable, SIGNAL(rowsRemoved(const QModelIndex&, int, int)),
	  SLOT(tableChanged(const QModelIndex&, int, int)));

  // Get the supported licenses
  d->comm.sendLicensesRequest();

  // configure to save/restore window settings
  setAutoSaveSettings();

  // Determine if an old database exists and if it should be used
  if( d->photoTable->rowCount() )
  {
    if( askAboutOldSession() )
    {
      QSqlRecord rec;
      for( int row = 0; row < d->photoTable->rowCount(); ++row )
      {
	rec = d->photoTable->record(row);
	PreviewMgr::instance()->previewRequest(
			       rec.value(d->photoTable->fieldIndex("id")).toInt(),
			       KUrl(rec.value(d->photoTable->fieldIndex("filename")).toString()));
      }
      d->ui.uploadPhotos->setEnabled(true);
    }
    else
    {
      d->photoTable->removeRows(0, d->photoTable->rowCount());
      d->photoTable->submitAll();
    }
  }
}

MainWindow::~MainWindow()
{
  KConfigGroup config = KGlobal::config()->group("KFlickr");

  // backup database to disk
  d->photoTable->submitAll();

  // save user(s) details
  config.writeEntry("users", d->users.stringList());
  config.writeEntry("user_nsids", d->userNSIDs);
  config.writeEntry("user_tokens", d->userTokens);
  config.writeEntry("user", d->activeUser.second);

  delete d;
}

void MainWindow::quit()
{
  // Save all settings and quit
  close();
}

void MainWindow::setupActions()
{
  KStandardAction::quit(this, SLOT(quit()), actionCollection());

  d->addPhotoAction = new KAction(this);
  d->addPhotoAction->setText(i18n("Add"));
  d->addPhotoAction->setIcon(KIcon("list-add"));
  d->addPhotoAction->setShortcut(Qt::CTRL + Qt::Key_Plus);
  actionCollection()->addAction("add_photos", d->addPhotoAction);
  connect(d->addPhotoAction, SIGNAL(triggered()), SLOT(addPhotographs()));
  d->addPhotoAction->setEnabled(false);

  d->removePhotoAction = new KAction(this);
  d->removePhotoAction->setText(i18n("Remove"));
  d->removePhotoAction->setIcon(KIcon("list-remove"));
  d->removePhotoAction->setShortcut(Qt::CTRL + Qt::Key_Minus);
  actionCollection()->addAction("remove_photos", d->removePhotoAction);
  connect(d->removePhotoAction, SIGNAL(triggered()), SLOT(addPhotographs()));
  d->removePhotoAction->setEnabled(false);

  d->uploadAction = new KAction(this);
  d->uploadAction->setText(i18n("Upload"));
  d->uploadAction->setIcon(KIcon("arrow-up"));
  d->uploadAction->setShortcut(Qt::CTRL + Qt::Key_U);
  actionCollection()->addAction("upload_photos", d->uploadAction);
  connect(d->uploadAction, SIGNAL(triggered()), SLOT(addPhotographs()));
  d->uploadAction->setEnabled(false);

  setupGUI();
}

bool MainWindow::askAboutOldSession()
{
  if( KMessageBox::questionYesNo(this,
		 i18n("Previously KFlickr either closed unexpectedly or you quit before uploading"
		      " your photographs. You can now choose to restore this session or not?"),
		 i18n("Restore previous session?"),
		 KGuiItem(i18n("&Restore Session"), "dialog-ok", i18n("Restore Session")),
		 KGuiItem(i18n("&Start Fresh"), "process-stop", i18n("Start Fresh")) )
      == KMessageBox::Yes )
  {
    return true;
  }
  return false;
}

void MainWindow::setActiveUser(const QString &user)
{
  if( d->users.stringList().contains(user) )
  {
    int index = d->users.stringList().indexOf(user);
    d->ui.userCombo->setCurrentIndex(index);
    d->activeUser = QPair<int,QString>(index,user);

    // Get the user's tags, bandwidth and photosets
    d->comm.sendUpStatusRequest(d->userTokens[index]);
    d->comm.sendTagsRequest(d->userTokens[index], d->userNSIDs[index]);
    d->comm.sendPhotosetsRequest(d->userTokens[index], d->userNSIDs[index]);

    // Update the GUI buttons etc..
    d->ui.addPhotos->setEnabled(true);
    d->ui.removeUser->setEnabled(true);
    d->ui.userCombo->setEnabled(true);
    d->ui.photoList->setEnabled(true);
    d->addPhotoAction->setEnabled(true);
  }
  else
  {
    // No active user
    d->activeUser = QPair<int,QString>(-1,"");

    // Clear things out
    d->tagsModel.setStringList(QStringList());
    d->photosetsModel.setStringList(QStringList());
    setBandwidth("0.0");

    // Remove any photos
    if( d->photoTable->rowCount() > 0 )
    {
      d->photoTable->removeRows(0, d->photoTable->rowCount());
      d->photoTable->submitAll();
    }

    // Update the GUI buttons etc..
    d->ui.addPhotos->setEnabled(false);
    d->ui.removeUser->setEnabled(false);
    d->ui.userCombo->setEnabled(false);
    d->ui.photoList->setEnabled(false);
    d->addPhotoAction->setEnabled(false);     
  }
}

void MainWindow::addPhotographs()
{
  QStringList files = KFileDialog::getOpenFileNames(KUrl("kfiledialog:///:OpenPhoto"),
						    "*.jpg *.png *.gif|Photo Files");

  for( QStringList::Iterator it = files.begin(); it != files.end(); ++it )
  {
    addPhotograph(*it);
  }
}

void MainWindow::addPhotograph(const QString &filename)
{
  int id;

  // Add the photo to the DB table
  id = d->photoTable->addPhotograph(filename);
  if( id < 0 )
  {
    KMessageBox::error(this, i18n("File does not exist. (%1)", filename));
    return;
  }

  // Start the process of fetching a preview
  PreviewMgr::instance()->previewRequest(id, KUrl(filename));
}

void MainWindow::commError(const QString &error)
{
  QMessageBox::critical(this, i18n("Error"), error);
}

void MainWindow::doUserAuthentication(const QString &frob)
{
    QDialog dlg;
    Ui::AuthCompleteDlg ui;
    ui.setupUi(&dlg);

    // Open browser etc... for web authentication
    d->comm.doWebAuthentication(frob);

    // Wait for user to login and such at Flickr.com
    if( dlg.exec() != QDialog::Accepted )
    {
        return;
    }

    // Request our TOKEN from flickr.com
    d->comm.sendTokenRequest(frob);
}

void MainWindow::addUser(const QString &user, const QString &token, const QString &nsid)
{
  // Determine if this user already exists, if so update the token and
  // nsid. If the user does not exist add the user.
  if( !d->users.stringList().contains(user) )
  {
    d->users.insertRows(d->users.rowCount(),1);
    d->users.setData(d->users.index(d->users.rowCount()-1), user);
    d->userNSIDs << "";
    d->userTokens << "";
  }

  int i =  d->users.stringList().indexOf(user);   
  d->userNSIDs[i] = nsid;
  d->userTokens[i] = token;

  // make the active user
  setActiveUser(user);

  // Notify user
  QMessageBox::information(this, i18n("New User"),
			   i18n("A new user '%1' has been added successfully.", user));
}

void MainWindow::setBandwidth(const QString &value)
{
  statusBar()->changeItem(i18n("Unused Upload Bandwidth: %1",value), Status_Bandwidth);
}

void MainWindow::setPhotosets(const QStringList& list, const QString& selection)
{
  Q_UNUSED(selection);
  QStringList updatedList = list;
  updatedList.prepend(i18n("<photostream only>"));
  d->photosetsModel.setStringList(updatedList);
}

void MainWindow::setLicenseTypes(const QStringList& list)
{
  d->licensesModel.setStringList(list);
}

void MainWindow::setTags(const QStringList &list)
{
  d->tagsModel.setStringList(list);
}

void MainWindow::updatePreview(int id)
{
  int row = 0;
  int numrows = d->photoTable->rowCount();
  int col = d->photoTable->fieldIndex("id");

  // Get the item row
  while( row < numrows )
  {
    if( d->photoTable->data(d->photoTable->index(row,col)).toInt() == id )
    {
      // Use 'rotation' index as this is the index the list is set to
      d->ui.photoList->update(d->photoTable->index(row,d->photoTable->fieldIndex("rotation")));
      return;
    }
    ++row;
  }
}

void MainWindow::selectionChanged(const QItemSelection& selected, const QItemSelection& deselected)
{
  Q_UNUSED(selected);
  Q_UNUSED(deselected);

  QItemSelection selection = d->ui.photoList->selectionModel()->selection();

  // Update the availability of the remove button and menu item. Start the editor if
  // there is a valid selection of one or more photographs.
  if( selection.size() > 0 )
  {
    if( selection.size() == 1 )
    {
      d->photoEditor->setEnabled(true);
      d->batchEditor->setEnabled(false);
      d->ui.editors->setTabEnabled(d->ui.editors->indexOf(d->photoEditor), true);
      d->ui.editors->setTabEnabled(d->ui.editors->indexOf(d->batchEditor), false);
      d->ui.editors->setCurrentWidget(d->photoEditor);
      d->photoEditor->editRow((selection.indexes())[0].row());
    }
    else
    {
      d->photoEditor->setEnabled(false);
      d->batchEditor->setEnabled(true);
      d->ui.editors->setTabEnabled(d->ui.editors->indexOf(d->photoEditor), false);
      d->ui.editors->setTabEnabled(d->ui.editors->indexOf(d->batchEditor), true);
      d->ui.editors->setCurrentWidget(d->batchEditor);

      QList<int> rows;
      foreach(QModelIndex index, selection.indexes())
      {
	rows.append(index.row());
      }
      d->batchEditor->editRows(rows);
    }
    
    // As long as there is "a" selection these should be enabled.
    d->ui.removePhotos->setEnabled(true);
    d->removePhotoAction->setEnabled(true);

    // Previous and next buttons
    d->ui.next->setEnabled(true);
    d->ui.previous->setEnabled(true);
    if( selection.last().bottom() + 1 >= d->photoTable->rowCount() )
    {
      d->ui.next->setEnabled(false);
    }
    if( selection.last().top() - 1 < 0 )
    {
      d->ui.previous->setEnabled(false);
    }
  }
  else
  {
    // Since the QTabWidget requires at least one tab be enabled there is no use disabling
    // the tabs in the event that there is no valid selection. Just a heads up if you were thinking
    // that this might of been missed, and as a reminder as to why :).
    d->batchEditor->setEnabled(false);
    d->photoEditor->setEnabled(false);
    d->ui.removePhotos->setEnabled(false);
    d->removePhotoAction->setEnabled(false);
    d->ui.previous->setEnabled(false);
    d->ui.next->setEnabled(false);
  }
}

void MainWindow::tableChanged(const QModelIndex& parent, int start, int end)
{
  Q_UNUSED(parent);
  Q_UNUSED(start);
  Q_UNUSED(end);

  if( d->photoTable->rowCount() <= 0 )
  {
    d->ui.removePhotos->setEnabled(false);
    d->removePhotoAction->setEnabled(false);
    d->ui.uploadPhotos->setEnabled(false);
    d->uploadAction->setEnabled(false);
  }
  else
  {
    d->ui.uploadPhotos->setEnabled(true);
    d->uploadAction->setEnabled(true);
  }
}

void MainWindow::photoUploaded(const QString &photoID)
{
  // update unused bandwidth
  d->comm.sendUpStatusRequest(d->userTokens[d->activeUser.first]);

  // We know that the last picture uploaded was the last photograph in the list.
  /// @todo Not actually true if the previous file failed to upload this may not be the case
  int row = getBiggestRowCount(d->photoTable);
  Q_ASSERT(row >= 0);
  QSqlRecord rec = d->photoTable->record(row);

  // Add photo to a photoset
  QString photoset = getRecordValue(rec, "photoset").toString();
  if( photoset != i18n("<photostream only>") )
  {
    d->comm.addPhoto2Photoset(d->userTokens[d->activeUser.first], photoset, photoID);
  }

  // Set the license for the photograph
  d->comm.setPhotoLicense(d->userTokens[d->activeUser.first],
			  getRecordValue(rec, "license").toString(), photoID);

  // Remove the photograph as it is now uploaded
  d->photoTable->removeRow(row);

  // ensure the db is in sync.
  d->photoTable->submitAll();

  // upload the next one
  uploadNextPhoto();
}

void MainWindow::cancelUpload()
{
  d->uploadInProgress = false;
  d->comm.abortCurrentRequest();
  hideUploadDlg();
}

void MainWindow::uploadNextPhoto()
{
  if( d->photoTable->rowCount() > 0 )
  {
    // get last photo in list
    int row = getBiggestRowCount(d->photoTable);
    QSqlRecord rec = d->photoTable->record(row);

    Q_ASSERT( row >= 0 && !rec.isEmpty() );

    KIO::TransferJob* job;
    job = d->comm.sendPhoto(d->userTokens[d->activeUser.first], rec);
    updateUploadDlg(*(PreviewMgr::instance()->preview(rec.value(rec.indexOf("id")).toInt())), job);
  }
  else
  {
    // The progress dialog will sometime pop back up if
    // it is left with steps to do, so update to make %100
    updateUploadDlg(QPixmap());
    d->uploadInProgress = false;
    hideUploadDlg();
  }
}

void MainWindow::showUploadDlg(int numBeingUploaded)
{
  // create the progress dialog if it does not yet exist
  if( d->progressDlg == 0L )
  {
    d->progressDlg = new UploadDlg(this);

    // Progress dialog connections
    connect(d->progressDlg, SIGNAL(rejected()), this, SLOT(cancelUpload()));
  }

  // tune the progress dialog accordingly
  d->progressDlg->setNumPhotos(numBeingUploaded);
  d->progressDlg->setNumCompleted(-1);
  d->progressDlg->show();
}

void MainWindow::updateUploadDlg(const QPixmap &pixmap, KIO::TransferJob* job)
{
  Q_UNUSED(job);

  // Update the pixmap indicating which photo is being uploaded.
  d->progressDlg->setPixmap(pixmap);

  // move progress bar
  d->progressDlg->advance(1);
}

void MainWindow::hideUploadDlg()
{
  // hide the progress dialog
  if( d->progressDlg != 0L && d->progressDlg->isVisible() )
  {
    d->progressDlg->hide();
  }
}

void MainWindow::on_addUser_clicked()
{
  QDialog dlg;
  Ui::AuthQuestionDlg ui;
  ui.setupUi(&dlg);

  // inform the user and check if we should continue
  if( dlg.exec() != QDialog::Accepted )
  {
      return;
  }

  // Get our FROB from flickr.com
  d->comm.sendFROBRequest();
}

void MainWindow::on_removeUser_clicked()
{
  QDialog dlg;
  Ui::RemoveUserDlg ui;
  ui.setupUi(&dlg);

  // Put the active username on the button
  ui.ok->setText(i18n("YES, remove '%1' now.", d->activeUser.second));
  
  if( dlg.exec() != QDialog::Accepted )
  {
      return;
  }

  // Go ahead and remove the user
  d->users.removeRow(d->activeUser.first);
  d->userNSIDs.removeAt(d->activeUser.first);
  d->userTokens.removeAt(d->activeUser.first);

  // Set the first user in the list to the active user, otherwise
  // set the active user to nothing.
  if( d->users.rowCount() > 0 )
  {
    setActiveUser(d->users.data(d->users.index(0,0), Qt::DisplayRole).toString());
  }
  else
  {
    setActiveUser("");
  }
}

void MainWindow::on_addPhotos_clicked()
{
  addPhotographs();
}

void MainWindow::on_removePhotos_clicked()
{
  // Determine the rows of the selected photos (ie. the photos being removed)
  QModelIndexList selection = d->ui.photoList->selectionModel()->selectedRows(3);

  QList<int> rows;
  foreach(QModelIndex idx, selection)
  {
    rows << idx.row();
  }

  // Remove the selected photos
  d->photoTable->removePhotographs(rows);

  // Sync DB, this will also clear the selection
  d->photoTable->submitAll();
}

void MainWindow::on_userCombo_activated(const QString &user)
{
  setActiveUser(user);
}

void MainWindow::on_previous_clicked()
{
  QModelIndex index(d->photoTable->index(0,3));

  // Get the last photo in the current selection (if there is a selection)
  QItemSelection selection = d->ui.photoList->selectionModel()->selection();

  if( !selection.isEmpty() )
  {
    // We must use the rotation field as this is what the listview is indexed to
    index = d->photoTable->index(selection.last().top()-1, d->photoTable->fieldIndex("rotation"));
  }

  if( index.isValid() )
  {
    d->ui.photoList->selectionModel()->select(index, QItemSelectionModel::ClearAndSelect);
    d->ui.photoList->scrollTo(index);
  }
}

void MainWindow::on_next_clicked()
{
  QModelIndex index(d->photoTable->index(0, 3));

  // Get the last photo in the current selection (if there is a selection)
  QItemSelection selection = d->ui.photoList->selectionModel()->selection();

  if( !selection.isEmpty() )
  {
    // We must use the rotation field as this is what the listview is indexed to
    index =d->photoTable->index(selection.last().bottom()+1, d->photoTable->fieldIndex("rotation"));
  }

  if( index.isValid() )
  {
    d->ui.photoList->selectionModel()->select(index, QItemSelectionModel::ClearAndSelect);
    d->ui.photoList->scrollTo(index);
  }
}

void MainWindow::on_uploadPhotos_clicked()
{
  // Ensure DB is backed up and in sync and clears selection.
  d->photoTable->submitAll();

  if( d->photoTable->rowCount() > 0 )
  {
    d->uploadInProgress = true;

    // display the progress dialog
    showUploadDlg(d->photoTable->rowCount());

    // upload the first photos one at a time
    uploadNextPhoto();
  }
}

#include "mainwindow.moc"
