/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004, 2005  by Michael Buesch                           *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "masterkey/keydata.h"
#include "randomizer.h"
#include "pwmexception.h"

#include <qdatastream.h>


#define PWMKEY_MAGIC		"PWMANAGERKEY"
#define PWMKEY_MAGIC_LENGTH	12 /* byte */
#define PWMKEY_VER_MAJOR	0x00
#define PWMKEY_VER_MINOR	0x00
#define PWMKEY_DEFAULT_LEN	(512 / 8)
#define PWMKEY_MIN_LEN		20
#define PWMKEY_ID_LENGTH	16
#define PWMKEY_RESERVED_LENGTH	32

#if PWMCONF_KEYCARD_COMPAT != 0
# define KEYCARD_COMPAT_MAGIC		"PWMKEYCARD"
# define KEYCARD_COMPAT_MAGIC_LENGTH	10 /* byte */
# define KEYCARD_COMPAT_VERSION		0x01
# include <klocale.h>
# include <kmessagebox.h>
#endif


KeyData::KeyData()
 : verMajor (PWMKEY_VER_MAJOR)
 , verMinor (PWMKEY_VER_MINOR)
 , type (type_undef)
{
}

KeyData::~KeyData()
{
}

KeyData::Type KeyData::getType() const
{
	return (static_cast<Type>(type));
}

void KeyData::getId(QByteArray *k) const
{
	k->duplicate(id);
}

void KeyData::getKey(QByteArray *k) const
{
	k->duplicate(key);
}

void KeyData::generate(Type t, uint16_t len)
{
	Randomizer rnd;
	verMajor = PWMKEY_VER_MAJOR;
	verMinor = PWMKEY_VER_MINOR;
	type = t;
	id.resize(PWMKEY_ID_LENGTH);
	rnd >> id;
	if (!len)
		len = PWMKEY_DEFAULT_LEN;
	key.resize(len);
	rnd >> key;
}

bool KeyData::getStream(QByteArray *s) const
{
	if (!s->resize(estimateStreamSize()))
		return false;
	unsigned int offset = 0;
	writeToStream(s, &offset, PWMKEY_MAGIC, PWMKEY_MAGIC_LENGTH);
	writeToStream(s, &offset, &verMajor, sizeof(verMajor));
	writeToStream(s, &offset, &verMinor, sizeof(verMinor));
	writeToStream(s, &offset, &type, sizeof(type));
	PWM_ASSERT(id.size() == PWMKEY_ID_LENGTH);
	writeToStream(s, &offset, id.data(), PWMKEY_ID_LENGTH);
	uint16_t keylen = key.size();
	keylen = toBigEndian(keylen);
	writeToStream(s, &offset, reinterpret_cast<const char *>(&keylen), sizeof(keylen));
	char reserved[PWMKEY_RESERVED_LENGTH];
	memset(reserved, 0, array_size(reserved));
	writeToStream(s, &offset, reserved, array_size(reserved));
	writeToStream(s, &offset, key.data(), key.size());
	return true;
}

unsigned int KeyData::estimateStreamSize() const
{
	unsigned int ret;
	ret = PWMKEY_MAGIC_LENGTH
	    + sizeof (verMajor)		// version major
	    + sizeof (verMinor)		// version minor
	    + sizeof (type)		// key type
	    + PWMKEY_ID_LENGTH		// key ID
	    + sizeof (uint16_t)		// key length
	    + PWMKEY_RESERVED_LENGTH	// reserved
	    + key.size();		// data body size (key size)
	return ret;
}

void KeyData::writeToStream(QByteArray *s, unsigned int *offset,
			    const char *d, unsigned int size) const
{
	unsigned int i;
	for (i = 0; i < size; ++i)
		(*s)[*offset + i] = d[i];
	*offset += size;
}

bool KeyData::readFromStream(const QByteArray &s, unsigned int *offset,
			     QByteArray *d, unsigned int size)
{
	if (*offset + size > s.size()) {
		printDebug("EOF :(");
		return false;
	}
	d->resize(size, QGArray::SpeedOptim);
	unsigned int i;
	for (i = 0; i < size; ++i)
		(*d)[i] = s[*offset + i];
	*offset += size;
	return true;
}

#ifndef BIG_ENDIAN_HOST
uint16_t KeyData::toBigEndian(uint16_t i) const
{
	PWM_ASSERT(sizeof(i) == 2);
	uint16_t ret;
	(reinterpret_cast<char *>(&ret))[0] = (reinterpret_cast<char *>(&i))[1];
	(reinterpret_cast<char *>(&ret))[1] = (reinterpret_cast<char *>(&i))[0];
	return ret;
}
#endif // BIG_ENDIAN_HOST

#if PWMCONF_KEYCARD_COMPAT != 0

# ifndef BIG_ENDIAN_HOST
uint32_t KeyData::toBigEndian(uint32_t i) const
{
	PWM_ASSERT(sizeof(i) == 4);
	uint32_t ret;
	(reinterpret_cast<char *>(&ret))[0] = (reinterpret_cast<char *>(&i))[3];
	(reinterpret_cast<char *>(&ret))[1] = (reinterpret_cast<char *>(&i))[2];
	(reinterpret_cast<char *>(&ret))[2] = (reinterpret_cast<char *>(&i))[1];
	(reinterpret_cast<char *>(&ret))[3] = (reinterpret_cast<char *>(&i))[0];
	return ret;
}
# endif // BIG_ENDIAN_HOST

bool KeyData::setStream_10compat(const QByteArray &s)
{
	QByteArray d;
	unsigned int offset = 0;
	if (!readFromStream(s, &offset, &d, KEYCARD_COMPAT_MAGIC_LENGTH))
		return false;
	if (memcmp(d.data(), KEYCARD_COMPAT_MAGIC, KEYCARD_COMPAT_MAGIC_LENGTH)) {
		printDebug("KeyData::setStream_10compat(): wrong magic");
		return false;
	}
	char ver;
	if (!readFromStream(s, &offset, &d, sizeof(ver)))
		return false;
	ver = d[0];
	if (ver != KEYCARD_COMPAT_VERSION)
		return false;
	uint32_t id_tmp;
	if (!readFromStream(s, &offset, &d, sizeof(id_tmp)))
		return false;
	id_tmp = toBigEndian(*(reinterpret_cast<uint32_t *>(d.data())));
	id.fill(0, PWMKEY_ID_LENGTH);
	unsigned int i;
	for (i = 0; i < sizeof(id_tmp); ++i)
		id[i + (PWMKEY_ID_LENGTH - sizeof(id_tmp))] = *(reinterpret_cast<char *>(&id_tmp) + i);
	PWM_ASSERT(id.size() == PWMKEY_ID_LENGTH);
	uint16_t keylen;
	if (!readFromStream(s, &offset, &d, sizeof(keylen)))
		return false;
	keylen = toBigEndian(*(reinterpret_cast<uint16_t *>(d.data())));
	if (!readFromStream(s, &offset, &key, keylen))
		return false;
	applyTruncationBug(&key);
	type = type_smartcard;
	verMajor = PWMKEY_VER_MAJOR;
	verMinor = PWMKEY_VER_MINOR;
	return true;
}

void KeyData::applyTruncationBug(QByteArray *s)
{
	/* We have an ugly bug in PwManager-1.0.x, which
	 * truncates the key at the first '\0' char.
	 * Indeed, this can be critical to security.
	 * A security warning is issued in KeyData::setStream(),
	 * if the key is too short.
	 */
	s->truncate(s->find('\0'));
}

#endif // PWMCONF_KEYCARD_COMPAT

bool KeyData::setStream(const QByteArray &s)
{
	QByteArray d;
	unsigned int offset = 0;
	uint16_t keylen;
	if (!readFromStream(s, &offset, &d, PWMKEY_MAGIC_LENGTH))
		return false;
	if (memcmp(d.data(), PWMKEY_MAGIC, PWMKEY_MAGIC_LENGTH)) {
		printDebug("KeyData::setStream(): wrong magic");
		// check if we have an old keycard here.
		if (setStream_10compat(s))
			goto out_success;
		else
			return false;
	}

	if (!readFromStream(s, &offset, &d, sizeof(verMajor)))
		return false;
	verMajor = d[0];
	if (verMajor != PWMKEY_VER_MAJOR) {
		printDebug("wrong major version");
		return false;
	}
	if (!readFromStream(s, &offset, &d, sizeof(verMinor)))
		return false;
	verMinor = d[0];
	if (verMinor != PWMKEY_VER_MINOR) {
		printDebug("wrong minor version");
		return false;
	}
	if (!readFromStream(s, &offset, &d, sizeof(type)))
		return false;
	type = d[0];
	switch (type) {
	case type_undef:
	case type_smartcard:
	case type_keyfile:
		break;
	default:
		return false;
	}
	if (!readFromStream(s, &offset, &d, PWMKEY_ID_LENGTH))
		return false;
	id.duplicate(d);
	PWM_ASSERT(id.size() == PWMKEY_ID_LENGTH);
	if (!readFromStream(s, &offset, &d, sizeof(uint16_t)))
		return false;
	PWM_ASSERT(d.size() == sizeof(uint16_t));
	keylen = toBigEndian(*(reinterpret_cast<uint16_t *>(d.data())));
	if (!readFromStream(s, &offset, &d, PWMKEY_RESERVED_LENGTH))
		return false;
	if (!readFromStream(s, &offset, &key, keylen))
		return false;
	PWM_ASSERT(key.size() == keylen);

out_success:
#if PWMCONF_KEYCARD_COMPAT != 0
	if (key.size() < PWMKEY_MIN_LEN) {
		KMessageBox::information(0,
					 i18n("WARNING: Your SmartCard key is insecure!\n"
					      "You generated the key with an older version "
					      "of PwManager, which contained a major security "
					      "bug in the SmartCard code.\n\n"
					      "You should do these steps in order to fix the issue:\n"
					      "1) Backup your .pwm password datafile and your chipcard key.\n"
					      "2) Open the .pwm file in PwManager.\n"
					      "3) Click on \"Manage / Change Master Key\".\n"
					      "4) Now generate a new chipcard key in the \"New "
					      "Master Key\" dialog box.\n"
					      "5) Save the .pwm datafile by clicking \"File / Save\"."),
					 i18n("WARNING: Insecure SmartCard Key"));
	}
#endif
	return true;
}
