from DeviceManagerWidgetUI import DeviceManagerWidgetUI
from qt import *
import sys
import dbus
from Representation import Representation
from Device import Device
import Const
from kdecore import KIconLoader, KIcon, KGlobalSettings
from kdeui import KListViewItem
	
class DeviceManagerWidget(DeviceManagerWidgetUI):
	
	def __init__(self, parent):
		"""run parent constructor and connect slots"""
		DeviceManagerWidgetUI.__init__(self, parent)
		ver = getattr(dbus, 'version', (0, 0, 0))
		if ver < (0, 40, 0):
			QMessageBox.critical(self, "Error", "The DBus Python Bindings you are using are too old. Make sure you have the latest version.", QMessageBox.Ok, QMessageBox.NoButton)
			sys.exit(1)
		self.bus = dbus.SystemBus()
		self.hal_manager_obj = self.bus.get_object("org.freedesktop.Hal", "/org/freedesktop/Hal/Manager")
		self.hal_manager = dbus.Interface(self.hal_manager_obj,	"org.freedesktop.Hal.Manager")
	
		# Add listeners for all devices
		try:
			device_names = self.hal_manager.GetAllDevices()
		except:
			QMessageBox.critical(self, "Error", "Could not get device list. Make sure hald is running.", QMessageBox.Ok, QMessageBox.NoButton)
			sys.exit(1)
		
		# here we would connect HAL update signals to refresh the list
		# however python dbus bindings seem to need the gobject mainloop and we don't
		# use that so we can't listen for signals from dbus, fortunatly we can still 
		# ask dbus for stuff
		
		self.connect(self.listview, SIGNAL("selectionChanged(QListViewItem*)"), self.on_device_tree_selection_changed)
		self.dont_show_virtual = 1
		self.representation = Representation()
		self.usbPage = self.device_notebook.page(1)
		self.pciPage = self.device_notebook.page(2)
		self.listview.setColumnWidthMode(1, QListView.Manual)
		self.listview.hideColumn(1)
		self.ns_adv_properties.setColumnWidthMode(0, QListView.Manual)
		self.ns_adv_properties.setColumnWidthMode(1, QListView.Manual)
		self.ns_adv_properties.setColumnWidthMode(2, QListView.Manual)
		self.ns_adv_properties.setColumnWidth(0, 150)
		self.ns_adv_properties.setColumnWidth(1, 50)
		self.ns_adv_properties.setColumnWidth(2, 150)
		self.ns_adv_properties.setAlternateBackground(KGlobalSettings.alternateBackgroundColor())
		self.ns_adv_properties.setFullWidth()
		self.searchLine.setListView(self.listview)
		iconLoader = KIconLoader()
		if QApplication.reverseLayout():
			iconName = "clear_left"
		else:
			iconName = "locationbar_erase"
		iconSet = iconLoader.loadIconSet(iconName, KIcon.Small)
		#icon = KIconLoader.SmallIconSet("locationbar_erase");
		self.searchClearButton.setIconSet(iconSet);
		self.connect(self.searchClearButton, SIGNAL("clicked()"), self.searchLine.clear)
		self.listview.setAlternateBackground(KGlobalSettings.alternateBackgroundColor())

		self.update_device_list()

	def setVirtual(self, virtual):
		dont_show_virtual = virtual
		self.update_device_list()

	def update_device_list(self):
		"""Builds, or rebuilds, the device tree"""
		# We use a virtual root device so we have a single tree
		self.virtual_root = self.build_device_tree()
		
		self.listview.clear()
		self.virtual_root.populate_qt_tree(self.listview,
						self.dont_show_virtual,
						self.representation)
		self.listview.setSelected(self.listview.firstChild(), True)
		
	def build_device_tree(self):
		"""Retrieves the device list from the HAL daemon and builds
		a tree of Device (Python) objects. The root is a virtual
		device"""
		device_names = self.hal_manager.GetAllDevices()
		device_names.sort()
	
		virtual_root = Device("virtual_root", None, {})
		self.device_list = [virtual_root]
		
		# first build list of Device objects
		for name in device_names:
			device_dbus_obj = self.bus.get_object("org.freedesktop.Hal" ,name)
			properties = device_dbus_obj.GetAllProperties(dbus_interface="org.freedesktop.Hal.Device")
			try:
				parent_name = properties["info.parent"]
			except KeyError:
				# no parent, must be parent of virtual_root
				parent_name = "/"
			except TypeError:
				print "Error: no properties for device %s"%name
				continue
			device = Device(name, parent_name, properties)
			self.device_list.append(device)
	
		# set parent_device and children for each Device object
		for device in self.device_list:
			parent_name = device.parent_name
			device.parent_device = virtual_root
			if parent_name!="/":
				for p in self.device_list:
					if p.device_name==parent_name:
						device.parent_device = p
						p.children.append(device)
			if device!=virtual_root and device.parent_device==virtual_root:
				virtual_root.children.append(device)
			if device==virtual_root:
				device.parent_device=None
		self.listview.hideColumn(1)
		return virtual_root

	def get_current_focus_udi(self):
		"""Get the UDI of the currently focused device"""
		currentListViewItem = self.listview.currentItem()
		if currentListViewItem:
			device_udi = currentListViewItem.text(Const.UDI_COLUMN)
			return device_udi
		return None
	
	def on_device_tree_selection_changed(self, currentListViewItem):
		"""This method is called when the selection has changed in the
		device tree"""
		#device_udi = self.get_current_focus_udi()
		device_udi = currentListViewItem.text(Const.UDI_COLUMN)
		if device_udi != None:
			device = self.udi_to_device(device_udi)
			self.update_device_notebook(device)
	
	def udi_to_device(self, device_udi):
		"""Given a HAL UDI (Unique Device Identifier) this method returns
		the corresponding HAL device"""
		return self.virtual_root.find_by_udi(device_udi)
	
	def update_tab_device(self, device):
		"""Updates the 'Device' tab given a Device object"""
		
		bus = self.ns_device_bus
		state = self.ns_device_status
		vendor = self.ns_device_vendor
		product = self.ns_device_name
		category = self.ns_device_category
		capabilities = self.ns_device_capabilities
	
		if not device.properties.has_key("info.bus"):
			product.setText("Unknown")
			vendor.setText("Unknown")
		else:
			bus.setText(Const.BUS_NAMES[device.properties["info.bus"]])	    
		#state.setText(Const.STATE_NAMES[device.properties["State"]])
		if device.properties.has_key("button.has_state"):
			if device.properties["button.has_state"]:
				state.setText("true")
			else:
				state.setText("false")
		else:
			state.setText("None")
			
		# guestimate product and vendor if we have no device information file
		if device.properties.has_key("info.bus") and device.properties["info.bus"]=="usb":
			if device.properties.has_key("info.product"):
				product.setText("%s"%device.properties["info.product"])
			elif device.properties.has_key("usb.product"):
				product.setText("%s"%device.properties["usb.product"])
			elif device.properties.has_key("usb.product_id"):
				product.setText("Unknown (0x%x)"%device.properties["usb.product_id"])
			else:
				product.setText("Unknown")
	
			if device.properties.has_key("info.vendor"):
				vendor.setText("%s"%device.properties["info.vendor"])
			elif device.properties.has_key("usb.vendor"):
				vendor.setText("%s"%device.properties["usb.vendor"])
			elif device.properties.has_key("usb.vendor_id"):
				vendor.setText("Unknown (0x%x)"%device.properties["usb.vendor_id"])
			else:
				vendor.setText("Unknown")
		elif device.properties.has_key("info.bus") and device.properties["info.bus"]=="pci":
			if device.properties.has_key("info.product"):
				product.setText("%s"%device.properties["info.product"])
			elif device.properties.has_key("pci.product"):
				product.setText("%s"%device.properties["pci.product"])
			elif device.properties.has_key("pci.product_id"):
				product.setText("Unknown (0x%x)"%device.properties["pci.product_id"])
			else:
				product.setText("Unknown")
	
			if device.properties.has_key("info.vendor"):
				vendor.setText("%s"%device.properties["info.vendor"])
			elif device.properties.has_key("pci.vendor"):
				vendor.setText("%s"%device.properties["pci.vendor"])
			elif device.properties.has_key("pci.vendor_id"):
				vendor.setText("Unknown (0x%x)"%device.properties["pci.vendor_id"])
			else:
				vendor.setText("Unknown")
		elif device.properties.has_key("info.bus") and device.properties["info.bus"]=="block":
			if device.properties.has_key("info.product"):
				product.setText("%s"%device.properties["info.product"])
			else:
				product.setText("Unknown")
	
			if device.properties.has_key("info.vendor"):
				vendor.setText("%s"%device.properties["info.vendor"])
			else:
				vendor.setText("Unknown")
		else:
			product.setText("Unknown")
			vendor.setText("Unknown")
	
		# clear category, capabilities
		# set category, capabilities
		if device.properties.has_key("info.category"):
			category.setText("%s"%device.properties["info.category"])
		else:
			category.setText("Unknown")
	
		if device.properties.has_key("info.capabilities"):
			capabilities.setText("%s"%device.properties["info.capabilities"])
		else:
			capabilities.setText("Unknown")
		
		
	def update_tab_usb(self, device):
		"""Updates the 'USB' tab given a Device object; may hide it"""
		if not device.properties.has_key("info.bus") or device.properties["info.bus"]!="usb":
			#self.device_notebook.removePage(page)
			self.device_notebook.setTabEnabled(self.usbPage, False)
			return
		
		self.device_notebook.setTabEnabled(self.usbPage, True)
	
		version = self.ns_usb_version
		bandwidth = self.ns_usb_bandwidth
		maxpower = self.ns_usb_maxpower
		man_id = self.ns_usb_man_id
		prod_id = self.ns_usb_prod_id
		revision = self.ns_usb_rev
	
		bcdVersion = device.properties["usb.version_bcd"]
		version.setText("%x.%x"%(bcdVersion>>8, bcdVersion&0xff))
	
		bcdSpeed = device.properties["usb.speed_bcd"]
		bandwidth.setText("%x.%x Mbit/s"%(bcdSpeed>>8, bcdSpeed&0xff))
		#maxpower.setText("%d mA"%(device.properties["usb.max_power"]))
		if not device.properties.has_key("usb.max_power"):
			man_id.setText("Unknown")
		else:
			man_id.setText("%s"%(device.properties["usb.max_power"]))
		if not device.properties.has_key("usb.vendor"):
			man_id.setText("0x%04x"%(device.properties["usb.vendor_id"]))
		else:
			man_id.setText("%s"%(device.properties["usb.vendor"]))
		if not device.properties.has_key("usb.product"):
			prod_id.setText("0x%04x"%(device.properties["usb.product_id"]))
		else:
			prod_id.setText("%s"%(device.properties["usb.product"]))
		bcdDevice = device.properties["usb.device_revision_bcd"]
		revision.setText("%x.%x"%((bcdDevice>>8), bcdDevice&0xff))
		
	
	def update_tab_pci(self, device):
		"""Updates the 'PCI' tab given a Device object; may hide it"""
		if not device.properties.has_key("info.bus") or device.properties["info.bus"]!="pci":
			self.device_notebook.setTabEnabled(self.pciPage, False)
			return
	
		self.device_notebook.setTabEnabled(self.pciPage, True)
	
		man_id = self.ns_pci_man_id
		prod_id = self.ns_pci_prod_id
		subsys_man_id = self.ns_pci_subsys_man_id
		subsys_prod_id = self.ns_pci_subsys_prod_id
	
		if not device.properties.has_key("pci.vendor"):
			man_id.setText("Unknown (0x%04x)"%(device.properties["pci.vendor_id"]))
		else:
			man_id.setText("%s"%(device.properties["pci.vendor"]))
		if not device.properties.has_key("pci.product"):
			prod_id.setText("Unknown (0x%04x)"%(device.properties["pci.product_id"]))
		else:
			prod_id.setText("%s"%(device.properties["pci.product"]))
	
		if not device.properties.has_key("pci.subsys_vendor"):
			subsys_man_id.setText("Unknown (0x%04x)"%(device.properties["pci.subsys_vendor_id"]))
		else:
			subsys_man_id.setText("%s"%(device.properties["pci.subsys_vendor"]))
		if not device.properties.has_key("pci.subsys_product"):
			subsys_prod_id.setText("Unknown (0x%04x)"%(device.properties["pci.subsys_product_id"]))
		else:
			subsys_prod_id.setText("%s"%(device.properties["pci.subsys_product"]))
		
	
	def update_tab_advanced(self, device):
		"""Updates the 'Advanced' tab given a Device object"""
		self.ns_adv_properties.clear()
		keys = device.properties.keys()
		keys.sort()
		for p in keys:
			#iter = store.append()
			listViewItem = KListViewItem(self.ns_adv_properties)
			val = device.properties[p]
			ptype = type(val)
			listViewItem.setText(0, p)
			if ptype==str:
				listViewItem.setText(1, "string")
				listViewItem.setText(2, "%s"%val)
			elif ptype==int:
				listViewItem.setText(1, "int")
				listViewItem.setText(2, "%d (0x%x)"%(val, val))
			elif ptype==long:
				listViewItem.setText(1, "long")
				listViewItem.setText(2, "%d (0x%x)"%(val, val))
			elif ptype==bool:
				listViewItem.setText(1, "bool")
				if val:
					listViewItem.setText(2, "true")
				else:
					listViewItem.setText(2, "false")
			elif ptype==float:
				listViewItem.setText(1, "float")
				listViewItem.setText(2, "%f"%val)
			else:
				# assume strlist
				listViewItem.setText(1, "strlist")
				listViewItem.setText(2, str(val))
	
	def update_device_notebook(self, device):
		"""Updates the entire notebook of tabs given a Device object"""
		self.update_tab_device(device)
		self.update_tab_advanced(device)
		self.update_tab_usb(device)
		self.update_tab_pci(device)
