/***************************************************************************
 *   Copyright (C) 2006 by Rohan McGovern                                  *
 *   rohan.pm@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <klistview.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <kdebug.h>
#include <qtimer.h>

#include "dbusserviceenumerator.h"
#include "dbus/qdbusmessage.h"
#include "dbus/qdbuserror.h"
#include "dbus/qdbusconnection.h"

#include "dbustreewidget.h"
#include "dbusservice.h"
#include "dbusutil.h"

class DBusServiceEnumerator::Private {
public:
    /// widget which created this DBusServiceEnumerator
    DBusTreeWidget * app;

    QDBusProxy * proxy;
};

DBusServiceEnumerator::DBusServiceEnumerator(
  DBusTreeWidget * app,
  QDBusProxy * proxy
)
  : d( new DBusServiceEnumerator::Private() )
{
    d->app = app;
    d->proxy = proxy;
}

void DBusServiceEnumerator::run() {

    QString self;
    QStringList services;

    try {
        d->proxy->setService( "org.freedesktop.DBus" );
        d->proxy->setPath( "/org/freedesktop/DBus" );
        d->proxy->setInterface( "org.freedesktop.DBus" );

        DBusUtil::mutex.lock();
        QDBusMessage reply = d->proxy->sendWithReply(
            "ListNames",
            QValueList<QVariant>()
        );
        DBusUtil::mutex.unlock();
        if ( reply.type() == QDBusMessage::InvalidMessage )
            throw QDBusSendError(
                QString("%1: %2")
                .arg( d->proxy->lastError().name() )
                .arg( d->proxy->lastError().message() )
            );

        services = reply[0].toStringList();
        self = d->proxy->connection().uniqueName();
    }
    catch ( std::exception const & e ) {
        DBusUtil::mutex.unlock();
        d->app->statusMessage( i18n( "Error: %1" ).arg( e.what() ) );
        return;
    }


    for (
      QStringList::iterator svciter = services.begin();
      svciter != services.end();
      svciter++ )
    {
        d->app->statusMessage(
          i18n( "Inspecting service %1" ).arg( *svciter )
        );

        // Give Qt event loop some time to update, since dbus messages block the
        // whole application, not just this thread.  What an ugly hack.
        QThread::usleep( 50 );

        DBusService * svc = new DBusService(
          d->app->listView(),
          *svciter,
          d->proxy
        );

        try {
            // Don't try to populate ourselves, since we know we won't reply.
            if ( (*svciter).compare( self ) != 0 ) {
                DBusUtil::mutex.lock();
                try {
                    svc->populate();
                }
                catch ( ... ) {
                    DBusUtil::mutex.unlock();
                    throw;
                }
                DBusUtil::mutex.unlock();
            }
        }
        catch ( std::exception const & e ) {
            QString message =
              i18n(
                "While probing service %1: %2"
              ).arg( *svciter ).arg( e.what() );

            d->app->statusMessage( message );
        }

    }
    d->app->statusMessage(
      i18n( "Finished discovering D-BUS services." )
    );

    d->app->listView()->setEnabled( true );
    QTimer::singleShot( 0, d->app, SLOT(resizeSplitter()) );
}

DBusServiceEnumerator::~DBusServiceEnumerator() {
    delete d;
}
