/***************************************************************************
 *   Copyright (C) 2006 by Rohan McGovern                                  *
 *   rohan.pm@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "dbusproperty.h"
#include "dbusutil.h"
#include "dbusinterface.h"

#include <kdebug.h>
#include <klocale.h>
#include <qvbox.h>
#include <qlabel.h>

class DBusProperty::Private {
public:
    QString name;
    QString access;
    QString type;
    QString niceType;
};

DBusProperty::DBusProperty(
  QListViewItem * parent,
  QDomElement const & elem
) throw( QDBusXmlError )
  : DBusAnnotatable( parent, "(unknown)", "Property" ),
    d( new DBusProperty::Private() )
{
    ensureElementIsNamed( elem, "property" );
    ensureElementHasAttribute( elem, "name" );
    ensureElementHasAttribute( elem, "type" );
    ensureElementHasAttribute( elem, "access" );

    d->name = elem.attribute( "name" );
    d->type = elem.attribute( "type" );
    d->access = elem.attribute( "access" );
    m_interface = discoverInterface();
    m_object = discoverObject();
    m_service = discoverService();

    d->niceType = DBusUtil::niceType( d->type );

    QString niceAccess;
    if ( d->access == "read" )
        niceAccess = i18n("access modifier", "read only");
    else if ( d->access == "write" )
        niceAccess = i18n("access modifier", "write only");
    else if ( d->access == "readwrite" )
        niceAccess = i18n("access modifier", "read/write");
    else
        throw QDBusXmlError(
          i18n(
            "Property %1 has access modifier %2 "
            "instead of one of '%3', '%4', or '%5'"
          )
          .arg( m_name )
          .arg( m_access )
          .arg( "read" )
          .arg( "write" )
          .arg( "readwrite" )
        );

    setText( 0, d->niceType + " " + d->name + " [" + niceAccess + "]" );

    // Look for annotations
    if ( !elem.hasChildNodes() )
        return;

    QDomElement child = elem.firstChild().toElement();

    do {
        if ( child.tagName() == "annotation" )
            addAnnotation( child );
        else
            kdWarning() << "Unhandled node in introspect data: "
                        << child.tagName() << endl;
        child = child.nextSibling().toElement();
    } while ( !child.isNull() );
}

QString DBusProperty::name() const {
    return d->name;
}

bool DBusProperty::canRead() const {
    return ( d->access == "read" || d->access == "readwrite" );
}

bool DBusProperty::canWrite() const {
    return ( d->access == "write" || d->access == "readwrite" );
}

QString DBusProperty::access() const {
    return d->access;
}

QString DBusProperty::type() const {
    return d->type;
}

QString DBusProperty::niceType() const {
    return d->niceType;
}


QWidget * DBusProperty::widget( QWidget * parent ) const {
    QScrollView * scrollview = new QScrollView( parent );
    QVBox * vbox = new QVBox( scrollview->viewport() );
    scrollview->addChild( vbox );
    scrollview->setMidLineWidth( 0 );
    scrollview->setLineWidth( 0 );
    scrollview->setResizePolicy( QScrollView::AutoOneFit );
    
    new QLabel(
      i18n( "Property: %1" ).arg( d->name ),
      vbox
    );

    new QLabel(
      i18n( "Interface: %1" ).arg( m_interface ),
      vbox
    );

    new QLabel(
      i18n( "Object: %1" ).arg( m_object ),
      vbox
    );

    new QLabel(
      i18n( "Service: %1" ).arg( m_service ),
      vbox
    );

    if ( canRead() )
        new QLabel( i18n( "You may read this property." ), vbox );
    else
        new QLabel( i18n( "You may not read this property." ), vbox );

    if ( canWrite() )
        new QLabel( i18n( "You may write to this property." ), vbox );
    else
        new QLabel( i18n( "You may not write to this property." ), vbox );

    /// @todo actually get the value of the property and display here

    addAnnotationInfo( vbox );
    addErrorInfo( vbox );

    return scrollview;
}

DBusProperty::~DBusProperty() {
    delete d;
}
