/***************************************************************************
 *   Copyright (C) 2006 by Rohan McGovern                                  *
 *   rohan.pm@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "dbusmethodargument.h"
#include "dbusutil.h"
#include "dbusitem.h"

#include <klocale.h>
#include <qdom.h>

class DBusMethodArgument::Private {
public:
    /// Name of this argument (can be null)
    QString name;

    /// Type string of this argument
    QString type;

    /// Human-readable type of this argument
    QString niceType;

    /// Direction of this argument ('in' or 'out')
    QString direction;
};

DBusMethodArgument::DBusMethodArgument( QDomElement const & elem )
throw ( QDBusXmlError )
 : d( new DBusMethodArgument::Private() )
{
    DBusItem::ensureElementIsNamed( elem, "arg" );
    DBusItem::ensureElementHasAttribute( elem, "type" );

    if ( elem.hasAttribute( "name" ) )
        d->name = elem.attribute( "name" );

    d->type = elem.attribute( "type" );

    /* Direction is a bit complicated.  The spec says,
     * for a method it may be in or out; for a signal, it's
     * always out; and it doesn't have to be specified for
     * either, in which case it defaults to in for a method,
     * out for a signal.
     */
    QDomElement parent = elem.parentNode().toElement();
    if ( parent.isNull() )
        throw std::logic_error( "Argument created without parent" );

    if ( parent.tagName() != "method" && parent.tagName() != "signal" )
        throw std::logic_error(
          "Argument created with parent neither method nor signal"
        );

    if ( parent.tagName() == "signal" )
        d->direction = "out";
    else {
        d->direction = elem.attribute( "direction" );
        if ( !d->direction )
            d->direction = "in";
    }

    d->niceType = DBusUtil::niceType( d->type );
}

QString DBusMethodArgument::name() const {
    return d->name;
}

QString DBusMethodArgument::type() const {
    return d->type;
}

QString DBusMethodArgument::niceType() const {
    return d->niceType;
}

QString DBusMethodArgument::direction() const {
    return d->direction;
}

DBusMethodArgument::~DBusMethodArgument() {
    delete d;
}
