/***************************************************************************

   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "kbloggermedia.h"

#include <QFile>
#include <QDataStream>
#include <QByteArray>

#include <kdebug.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <kfileitem.h>
#include <kicon.h>

#define CACHE_DIR "media-cache/"
#define KBLOGGERMEDIANAME "kblogger-media-"

namespace KBlogger
{

KBloggerMedia::KBloggerMedia(QByteArray& data,
                             const QString& name,
                             const QString& mimetype,
                             const QString& blogname) :
                                       mCachedFileItem(0),
                                       mFullSizeImage(0),
                                       mThumbnail(0)
{
    kDebug()  << name << mimetype << blogname;
    Q_ASSERT( data.count() );
    Q_ASSERT( !name.isEmpty() );
    Q_ASSERT( !mimetype.isEmpty() );
    Q_ASSERT( !blogname.isEmpty() );

    //readData(localFileUrl);
    setData(data);

    //Initialize
    initialize(blogname, name, mimetype, checksum());

    saveDataInACacheFile();
}



KBloggerMedia::KBloggerMedia(const QString& name,
                             const QString& mimetype,
                             const QString& blogname,
                             quint16 checksum) :
                                       mCachedFileItem(0),
                                       mFullSizeImage(0),
                                       mThumbnail(0)
{
    kDebug();

    Q_ASSERT( !name.isEmpty() );
    Q_ASSERT( !mimetype.isEmpty() );
    Q_ASSERT( !blogname.isEmpty() );

    KUrl localFileUrl;

    mChecksum = checksum;
    localFileUrl = cachedFileUrl();

    readData(localFileUrl);

    if ( checksum != mChecksum) {
        kError() << "Data have a different checksum"
        << checksum <<  QString("%1").arg(checksum, 4, 16)
        << mChecksum << QString("%1").arg(mChecksum, 4, 16);
        setChecksum(0); //It's not valid!
        return;
    }

    //Initialize
    initialize(blogname, name, mimetype, mChecksum);
}

void KBloggerMedia::readData(const KUrl& localFileUrl)
{
    kDebug() << localFileUrl.path();

    //Read Data from file
    QByteArray data;
    QFile file(localFileUrl.path());

    if( !file.open(QIODevice::ReadOnly) ) {
        kError() << "cannot open file" << localFileUrl;
        return;
    }

    data = file.readAll();

    Q_ASSERT( data.count() );

    //Set Data and setChecksum from data
    setData(data);

    if (!mChecksum) {
        kError() << "mChecksum is zero";
        return;
    }
}

void KBloggerMedia::initialize(const QString& blogname,
                               const QString& name,
                               const QString& mimetypeString,
                               const quint16 checksum)
{
    kDebug();

    // should not be necessary
    delete mFullSizeImage;
    delete mThumbnail;

    //TEST
    Q_ASSERT( !name.isEmpty() );
    Q_ASSERT( !mimetypeString.isEmpty() );
    Q_ASSERT( !blogname.isEmpty() );

    mBlog = blogname;
    setName(name);
    setMimetype( mimetypeString );
    setStatus( KBlog::BlogMedia::New );
    mChecksum = checksum;
    mCachedFileUrl = cachedFileUrl();
    kDebug() << mCachedFileUrl;
    mCachedFileItem = new KFileItem (mCachedFileUrl, mimetype(), KFileItem::Unknown);
    Q_ASSERT(mCachedFileItem);
}

void KBloggerMedia::saveDataInACacheFile()
{
    kDebug();
    QByteArray mData = data();
    if ( !mData.count() ) {
        kError() << "Data is EMPTY";
        return;
    }
    QFile file( cachedFileUrl() );
    Q_ASSERT( file.open(QIODevice::WriteOnly) );
    if ( file.write(mData) == -1 ) {
        kError() << "Impossible create the file:" << cachedFileUrl();
    }

}

KBloggerMedia::~KBloggerMedia()
{
    kDebug();
    // they are only pointers to other images, which must not be deleted here
    mFullSizeImage = 0;
    mThumbnail = 0;
}

void KBloggerMedia::setData(const QByteArray& data)
{
    kDebug() << data.count();

    KBlog::BlogMedia::setData(data);
    setChecksum(data);
}

void KBloggerMedia::setChecksum(const QByteArray& data)
{
    kDebug();
    mChecksum = qChecksum(data.data(), data.count());
}

//Blogname
QString KBloggerMedia::getBlogname()
{
    return mBlog;
}

KIcon KBloggerMedia::icon()
{
    QString iconName;
    iconName = mimetype();
    iconName.replace(QChar('/'), QChar('-'));
    return KIcon(iconName);
}

void KBloggerMedia::setBlogname(const QString &blogname)
{
    mBlog = blogname;
    if ( mCachedFileItem ) {
        delete mCachedFileItem;
        mCachedFileItem = 0;
    }
    mCachedFileItem = new KFileItem(mCachedFileUrl, mimetype(), KFileItem::Unknown);
    Q_ASSERT(mCachedFileItem);
}

//Creation Data
KDateTime KBloggerMedia::creationDateTime() const
{
//     kDebug() << "CREATOR"<< mCachedFileItem->timeString(KFileItem::CreationTime);
//     kDebug() << "MODIF"<< mCachedFileItem->timeString(KFileItem::ModificationTime);
//     kDebug() << "ACCESS"<< mCachedFileItem->timeString(KFileItem::AccessTime);
    return mCachedFileItem->time( KFileItem::ModificationTime );
}

//SIZE
QString KBloggerMedia::sizeString() const
{
    return KIO::convertSize( mCachedFileItem->size() );
}

int KBloggerMedia::sizeInt() const
{
    return KIO::number( mCachedFileItem->size() ).toInt();
}

void KBloggerMedia::run()
{
    mCachedFileItem->run();
}

QString KBloggerMedia::cachedFilename()
{
    QString filename;
    QString hexChecksum;
    hexChecksum = QString("%1").arg(mChecksum, 4, 16);
    return KBLOGGERMEDIANAME + hexChecksum;
}

QString KBloggerMedia::cachedFileUrl()
{
    //mCachedFileUrl = cachePath();
    return cachePath() + cachedFilename();
}

QString KBloggerMedia::cachePath()
{
    return KStandardDirs::locateLocal("appdata", CACHE_DIR , true);
}

bool KBloggerMedia::removeCachedFile()
{
    kDebug() << "Removing:" << cachedFileUrl();
    QFile file( cachedFileUrl() );
    return file.remove();
}

void KBloggerMedia::setChecksum(quint16 checksum)
{
    kDebug();
    mChecksum = checksum;
}

quint16 KBloggerMedia::checksum()
{
    kDebug();
    return mChecksum;
}

bool KBloggerMedia::isCached()
{
    QString url = cachedFileUrl();
    if ( url.isEmpty() ) return false;
    QFile file( url );
    return file.exists();
}

void KBloggerMedia::setUrl(const KUrl& url)
{
    if ( !url.isValid() ) return;
    KBlog::BlogMedia::setStatus( KBlog::BlogMedia::Created );
    KBlog::BlogMedia::setUrl(url);
}

//Thumbail managing
void KBloggerMedia::setFullSizeImage(KBloggerMedia *media)
{
    kDebug();
    mFullSizeImage = media;
}

KBloggerMedia* KBloggerMedia::fullSizeImage()
{
    kDebug();
    return mFullSizeImage;
}

void KBloggerMedia::setThumbnail(KBloggerMedia *media)
{
    kDebug();
    mThumbnail = media;
}

KBloggerMedia* KBloggerMedia::thumbnail()
{
    kDebug();
    return mThumbnail;
}

} //namespace
