/***************************************************************************
*   Copyright (C) 2004-2007 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <qpushbutton.h>
#include <qlineedit.h>
#include <qstringlist.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qlabel.h>
#include <qwhatsthis.h>

#include <krun.h>
#include <kdialog.h>
#include <kapplication.h>
#include <klistview.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kio/netaccess.h>

#include <fieldlineedit.h>
#include <settings.h>
#include "entrywidgetuser.h"

namespace KBibTeX
{

    EntryWidgetUser::EntryWidgetUser( BibTeX::Entry *entry, BibTeX::File *bibtexfile, bool isReadOnly, QWidget *parent, const char *name )
            : EntryWidgetTab( entry, bibtexfile, isReadOnly, parent, name ), m_isModified( FALSE )
    {
        setupGUI();
    }

    EntryWidgetUser::~EntryWidgetUser()
    {
        m_listViewUserFields->clear();
    }

    bool EntryWidgetUser::isModified()
    {
        return m_isModified;
    }

    void EntryWidgetUser::updateGUI( BibTeX::Entry::EntryType /*entryType*/, bool /*enableAll*/ )
    {
        // nothing
    }

    void EntryWidgetUser::apply( )
    {
        for ( QStringList::iterator it = m_deletedUserFields.begin(); it != m_deletedUserFields.end(); it++ )
            m_entry->deleteField( *it );
        m_deletedUserFields.clear();

        for ( QListViewItemIterator it( m_listViewUserFields ); it.current(); it++ )
        {
            ValueListViewItem *vlvi = dynamic_cast<KBibTeX::ValueListViewItem*>( it.current() );
            if ( vlvi != NULL )
            {
                BibTeX::EntryField * field = m_entry->getField( vlvi->title() );
                if ( field == NULL )
                {
                    field = new BibTeX::EntryField( vlvi->title() );
                    m_entry->addField( field );
                }

                field->setValue( vlvi->value() );
            }
        }

        m_isModified = FALSE;
    }

    void EntryWidgetUser::reset( )
    {
        m_listViewUserFields->clear();
        m_deletedUserFields.clear();

        for ( QValueList<BibTeX::EntryField*>::iterator it = m_entry->begin(); it != m_entry->end(); it++ )
        {
            BibTeX::EntryField *field = *it;
            if ( field->fieldType() == BibTeX::EntryField::ftUnknown )
                new ValueListViewItem( field->fieldTypeName(), field->value(), m_listViewUserFields );
        }

        m_isModified = FALSE;
    }

    void EntryWidgetUser::updateWarnings( BibTeX::Entry::EntryType /*entryType*/, QListView * /*listViewWarnings*/ )
    {
        // no warnings neccessary for user fields
    }

    void EntryWidgetUser::setupGUI()
    {
        QGridLayout * gridLayout = new QGridLayout( this, 5, 3, KDialog::marginHint(), KDialog::spacingHint(), "gridLayout" );

        m_lineEditUserKey = new QLineEdit( this, "m_lineEditUserKey" );
        gridLayout->addWidget( m_lineEditUserKey, 0, 1 );
        QToolTip::add ( m_lineEditUserKey, i18n( "Name of the user-defined field" ) );
        QWhatsThis::add ( m_lineEditUserKey, i18n( "The name of the user-defined field. Should only contain letters and numbers." ) );
        QLabel *label = new QLabel( i18n( "&Name:" ), this );
        label->setBuddy( m_lineEditUserKey );
        gridLayout->addWidget( label, 0, 0 );

        m_pushButtonUserAdd = new QPushButton( i18n( "&Add" ), this, "m_pushButtonUserAdd" );
        gridLayout->addWidget( m_pushButtonUserAdd, 0, 2 );
        m_pushButtonUserAdd->setIconSet( QIconSet( SmallIcon( "add" ) ) );

        m_fieldLineEditUserValue = new KBibTeX::FieldLineEdit( i18n( "Value" ), KBibTeX::FieldLineEdit::itMultiLine, m_isReadOnly, this, "m_fieldLineEditUserValue" );
        gridLayout->addMultiCellWidget( m_fieldLineEditUserValue, 1, 2, 1, 2 );
        QToolTip::add ( m_fieldLineEditUserValue, i18n( "Content of the user-defined field" ) );
        QWhatsThis::add ( m_fieldLineEditUserValue, i18n( "The content of the user-defined field. May contain any text." ) );
        label = new QLabel( i18n( "&Content:" ), this );
        label->setBuddy( m_fieldLineEditUserValue );
        gridLayout->addWidget( label, 1, 0 );

        QSpacerItem* spacer = new QSpacerItem( 20, 110, QSizePolicy::Minimum, QSizePolicy::Expanding );
        gridLayout->addItem( spacer, 2, 0 );

        m_listViewUserFields = new KListView( this, "m_listViewUserFields" );
        m_listViewUserFields->addColumn( i18n( "Key" ) );
        m_listViewUserFields->addColumn( i18n( "Value" ) );
        m_listViewUserFields->setAllColumnsShowFocus( true );
        m_listViewUserFields->setFullWidth( TRUE );
        gridLayout->addMultiCellWidget( m_listViewUserFields, 3, 5, 1, 1 );

        label = new QLabel( i18n( "&List:" ), this );
        label->setBuddy( m_listViewUserFields );
        label->setAlignment( Qt::AlignTop );
        gridLayout->addWidget( label, 3, 0 );

        m_pushButtonUserDelete = new QPushButton( i18n( "&Delete" ), this, "m_pushButtonUserDelete" );
        gridLayout->addWidget( m_pushButtonUserDelete, 3, 2 );
        m_pushButtonUserDelete->setIconSet( QIconSet( SmallIcon( "delete" ) ) );

        m_pushButtonUserOpen = new QPushButton( i18n( "Op&en" ), this, "m_pushButtonUserOpen" );
        gridLayout->addWidget( m_pushButtonUserOpen, 4, 2 );
        m_pushButtonUserOpen->setIconSet( QIconSet( SmallIcon( "fileopen" ) ) );

        spacer = new QSpacerItem( 20, 110, QSizePolicy::Minimum, QSizePolicy::Expanding );
        gridLayout->addItem( spacer, 5, 2 );

        m_pushButtonUserOpen->setEnabled( FALSE );
        m_pushButtonUserAdd->setEnabled( FALSE );
        m_pushButtonUserDelete->setEnabled( FALSE );

        connect( m_listViewUserFields, SIGNAL( clicked( QListViewItem*, const QPoint&, int ) ), this, SLOT( userFieldExecute( QListViewItem* ) ) );
        connect( m_lineEditUserKey, SIGNAL( textChanged( const QString& ) ), this, SLOT( updateGUI() ) );
        connect( m_pushButtonUserAdd, SIGNAL( clicked( ) ), this, SLOT( userAddClicked( ) ) );
        connect( m_pushButtonUserDelete, SIGNAL( clicked( ) ), this, SLOT( userDeleteClicked( ) ) );
        connect( m_pushButtonUserOpen, SIGNAL( clicked() ), this, SLOT( userOpenClicked() ) );
    }

    void EntryWidgetUser::updateGUI()
    {
        QString text = m_lineEditUserKey->text();
        QListViewItem * item = m_listViewUserFields->findItem( text, 0 ) ;
        bool contains = item != NULL;
        bool isUnknown = BibTeX::EntryField::fieldTypeFromString( text ) == BibTeX::EntryField::ftUnknown;

        if ( contains )
            m_listViewUserFields->setSelected( item, TRUE );
        else
            m_listViewUserFields->clearSelection();

        m_pushButtonUserDelete->setEnabled( !m_isReadOnly && contains );
        m_pushButtonUserAdd->setEnabled( !m_isReadOnly && !text.isEmpty() && isUnknown );
        m_pushButtonUserAdd->setText( contains ? i18n( "&Apply" ) : i18n( "&Add" ) );
        m_pushButtonUserAdd->setIconSet( QIconSet( SmallIcon( contains ? "apply" : "fileopen" ) ) );

        bool validURL = FALSE;
        if ( contains )
        {
            KURL url = Settings::locateFile( item->text( 1 ), m_bibtexfile->fileName, this );
            validURL = url.isValid();
            m_internalURL = url;
        }
        m_pushButtonUserOpen->setEnabled( validURL );
    }

    void EntryWidgetUser::userFieldExecute( QListViewItem * item )
    {
        ValueListViewItem * vlvi = dynamic_cast<KBibTeX::ValueListViewItem*>( item );
        if ( vlvi != NULL )
        {
            m_lineEditUserKey->setText( vlvi->title() );
            m_fieldLineEditUserValue->setValue( vlvi->value() );
        }
    }

    void EntryWidgetUser::userAddClicked()
    {
        QString text = m_lineEditUserKey->text();
        QListViewItem * item = m_listViewUserFields->findItem( text, 0 ) ;
        ValueListViewItem * vlvi = item != NULL ? dynamic_cast<KBibTeX::ValueListViewItem*>( item ) : NULL;

        if ( vlvi != NULL )
            vlvi->setValue( m_fieldLineEditUserValue->value() );
        else
            new ValueListViewItem( text, m_fieldLineEditUserValue->value(), m_listViewUserFields );

        updateGUI();

        m_isModified = TRUE;
    }

    void EntryWidgetUser::userDeleteClicked()
    {
        QListViewItem * item = m_listViewUserFields->findItem( m_lineEditUserKey->text(), 0 );

        if ( item != NULL )
        {
            m_deletedUserFields.append( item->text( 0 ) );
            delete item;
            m_lineEditUserKey->setText( "" );
            m_fieldLineEditUserValue->setValue( new BibTeX::Value() );
            updateGUI();
        }

        m_isModified = TRUE;
    }

    void EntryWidgetUser::userOpenClicked()
    {
        if ( m_internalURL.isValid() )
        {
            BibTeX::Value * value = m_fieldLineEditUserValue->value();
            new KRun( m_internalURL, this );
            delete value;
        }
        else
            m_fieldLineEditUserValue->setEnabled( FALSE );
    }

    ValueListViewItem::ValueListViewItem( const QString& title, BibTeX::Value *value, QListView *parent )
            : QListViewItem( parent ), m_title( title ), m_value( value )
    {
        setTexts( m_title );
    }

    const BibTeX::Value *ValueListViewItem::value()
    {
        return m_value;
    }

    QString ValueListViewItem::title()
    {
        return m_title;
    }

    void ValueListViewItem::setValue( BibTeX::Value *value )
    {
        if ( value != m_value )
        {
            if ( m_value != NULL )
                delete m_value;

            if ( value != NULL )
                m_value = new BibTeX::Value( value );
            else
                m_value = new BibTeX::Value();

        }
        setTexts( m_title );
    }

    void ValueListViewItem::setTitle( const QString &title )
    {
        m_title = title;
        setTexts( title );
    }

    void ValueListViewItem::setTexts( const QString& title )
    {
        setText( 0, title );
        if ( m_value != NULL )
            setText( 1, m_value->text() );
    }

}
#include "entrywidgetuser.moc"
