/***************************************************************************
 *   Copyright (C) 2005 by Roberto Cappuccio and the Kat team              *
 *   Roberto Cappuccio : roberto.cappuccio@gmail.com                       *
 *   Praveen Kandikuppa : praveen9@gmail.com                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#ifndef _KATINDEXER_H_
#define _KATINDEXER_H_

#include <qobject.h>
#include <qthread.h>
#include <qwaitcondition.h>
#include <qstringlist.h>
#include <qdir.h>
#include <qtimer.h>
#include <qdatetime.h>

#include "inotify-qt.h"
#include "katscheduler.h"
#include "kat_export.h"

class KatEngine;
class KatCatalog;
class KatScanFolder;
class KatInfoExtractor;
class KatTempTable;

class WatcherEvent;

class KAT_EXPORT KatIndexer : public QObject, public QThread
{
Q_OBJECT

public:
    /**
        * Possible values for status of indexer
        */
    enum Status {
        /**
            * Pruning - removing old dead entries from the catalog for the files
            *           which have been deleted from the filesystem since our last run.
            */
        Prune,
        /**
            * Scanning - scanning the filesystem for files which need to be put in 
            *            the catalog, adds watch on the directories encountered. Adds
            *            any such files/dirs encountered to a temporary table which will
            *            be used in the next stage of indexing and also to report
            *            the percentage of index.
            */
        Scan,
        /**
            * Indexing - getting files to be indexed from the temporary table where they
            *            have been put in by the Scan stage and extracting info for the
            *            files and putting them in the catalog
            */
        Index,
        /**
            * Process - Processing any events generated by inotify. When the indexer is
            *           paused they are put in the table or if it is running sent to extract
            *           info.
            */
        Process,
        /**
            * Waiting - Waiting for inotify events to be triggered.
            */
        Wait,
        /**
            * Paused - Indexer is paused.
            */
        Pause,
        /**
            * Stopped - Indexer is stopped.
            */
        Stop
    };

    enum SubStatus {
        /**
            * Add - Adding files to the database
            */
        AddFiles,
        /**
            * Delete - Delete files from the database
            */
        DeleteFiles,
        /**
            * Attrib - Updating Attrib of files in the database
            */
        AttribFiles,
        /**
            * Move - Move files
            */
        MoveFiles,
        /**
            * ExtractingInfo - Extracting info for the files which have been added by the
            *                  scan stage or for which we have inotify events to indicate
            *                  they have been modified/created etc.
            */
        ExtractInfo,
        /**
            * Requested Schedule - Indexer needs to perform a database operation/needs to
            *                       extract info. Waiting for exclusive access to database/
            *                       work queue.
            */
        RequestSchedule,
        /**
            * Scheduled Timeout - Indexer has been scheduled to resume after m_waitSched msecs
            */
        ScheduleTimeout,
        /**
            * Reset
            */
        Reset
    };

    /**
        * Action to be taken on the file.
        */
    enum Action {
        /**
            * No action.
            */
        None    = 0x01,
        /**
            * Create file in the database.
            */
        Create  = 0x02,
        /**
            * Delete file from the database.
            */
        Delete  = 0x04,
        /**
            * Update file in the database by performing extracting info.
            */
        Update  = 0x08,
        /**
            * Move the file/dir to another location.
            */
        Move    = 0x10,
        /**
            * Update attributes of the file/dir.
            */
        Attrib  = 0x20      // Update attributes
    };

    /**
        * Constructor. Creates KatIndexer to work on a KatCatalog
        * Do not create KatIndexer Objects. Instead use the KatIndexerManager.
        *
        * @param obj Parent object to which status is reported.
        *
        * @param e KatEngine handle.
        *
        * @param cat KatCatalog on which to run the indexer.
        *
        * @param table KatTempTable to which all actions are logged when
        * the indexer is scanning/paused.
        *
        * @param scheduler KatScheduler, used to schedule access to database
        * and assign time to operate on extracting info.
        *
        */
    KatIndexer( QObject* obj, KatEngine* e, KatCatalog* cat, KatTempTable* table, KatScheduler* scheduler );
    ~KatIndexer();
    virtual void run();

    void setIgnore( const QStringList&, const QStringList& );

private slots:
    /* slot called when an intoify event occurs */
    void slotOnEvent( KatWatcher::EventType, QString , QString );

    /* slot called when the indexer is paused to commit any
        * outstanding inotify events to table to be processed
        * when the indexer resumes */
    void slotCommitTimeout();

    void slotInfoExtractorCompleted();

private:
    typedef QMap<QString,int> ProcessMap;
    typedef QMap<QString,QString> MoveMap;
    typedef QMap<QString,WatcherEvent> EventMap;
    typedef QMap<KatIndexer::Action,QStringList> ActionMap;

    KatTempTable *m_tempTable;
    KatEngine* m_ke;
    KatCatalog* m_catalog;
    KatScheduler* m_scheduler;

    int m_catalogId;

    KatInfoExtractor* m_kie;
    KatScanFolder* m_ksf;
    KatWatcher* m_watcher;

    bool m_running;                 // Run the indexer
    bool m_pause;                   // Pause the indexer
    bool m_committing;              // Committing events to table
    bool m_waiting;                 // Are we waiting for InfoExtractor to finish

    QWaitCondition m_pauseCond;     // Pause Condition - Waiting until resumed
    QWaitCondition m_eventCond;     // Event Condition - Waiting for events to show up
    QWaitCondition m_sleepCond;     // Sleep Condition - Waiting before dispatching events
    QWaitCondition m_infoCond;      // Info Condition - Waiting for KatInfoExtractor to finish
    QWaitCondition m_schedCond;     // Schedule Condition - Waiting for Scheduler to assign

    QMutex  m_eventMutex;           // Protect access to EventQueue
    QMutex  m_processMutex;         // Protect access to ProcessFiles list

    QStringList m_crawlQueue;       // Directories to crawl;
    QStringList m_ignoreDirs;       // Directories to ignore;
    QStringList m_ignoreFiles;

    EventMap m_waitingEvents;       // Event queue list, must only hold either Update/Attrib evets

    ProcessMap m_processFiles;      // List of files to be processed
    MoveMap m_moveFiles;            // oldpath/newpath map for the moved files

    QTimer m_commitTimer;

    int m_schedWait;                // Wait time in effect before indexer resumes

    KatIndexer::Action findAction ( const QFileInfo* );
    void indexRoutine();
    void scanRoutine();
    void processDispatch();
    int processWaiting();
    void customEvent( QCustomEvent* );
    void checkWatcherEvent( WatcherEvent );
    void checkMoveEvent( WatcherEvent, QString oldpath );
    void crawl( QString );
    int processDir( QDir );
    void crawlOnMove( QString );
    void processDirOnMove( QDir );
    bool processFiles();
    bool acquireLock( KatScheduler::Priority );
    bool interesting( QStringList&, bool );
    bool interesting( const QString&, bool );
    void cleanUp();
};

#endif // _KATINDEXER_H_
