/***************************************************************************
 *																		 *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or	 *
 *   (at your option) any later version.								   *
 *																		 *
 ***************************************************************************/

#include <qpixmap.h>
#include <qevent.h>
#include <qtimer.h>
#include <qstyle.h>
#include <qfileinfo.h>
#include <qimage.h>
#include <qtooltip.h>
#include <qdir.h>
#include <qpopupmenu.h>
#include <qbitmap.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>
#include <algorithm>

#include "action.h"
#include "chat_manager.h"
#include "chat_widget.h"
#include "config_file.h"
#include "configuration_window_widgets.h"
#include "custom_input.h"
#include "debug.h"
#include "icons_manager.h"
#include "main_configuration_window.h"
#include "message_box.h"
#include "misc.h"
#include "modules.h"
#include "userlist.h"

#include "../notify/notify.h"

#include "screenshot.h"

#define MODULE_SCREENSHOT_VERSION 0.5.0

ScreenShot* screenShot;

extern "C" int screenshot_init()
{
	kdebugf();
	screenShot = new ScreenShot();
	notification_manager->registerEvent("ssSizeLimit", "ScreenShot images size limit", CallbackNotRequired);
	MainConfigurationWindow::registerUiFile(dataPath("kadu/modules/configuration/screenshot.ui"), screenShot->configurationUiHandler());
	return 0;
}

extern "C" void screenshot_close()
{
	kdebugf();
	MainConfigurationWindow::unregisterUiFile(dataPath("kadu/modules/configuration/screenshot.ui"), screenShot->configurationUiHandler());
	notification_manager->unregisterEvent("ssSizeLimit");
	delete screenShot;
}

//-----------------------------------------------------------------------------------

ShotSizeHint::ShotSizeHint()
	: QWidget(0, 0, WStyle_Customize|Qt::WStyle_NoBorder|Qt::WStyle_StaysOnTop)
{
	QGridLayout *g = new QGridLayout(this, 1, 1);
	geom = new QLabel(this, "");
	fileSize = new QLabel(this, "0 KB");
	g->addWidget(geom, 0, 0);
	g->addWidget(fileSize, 1, 0);
}

//-----------------------------------------------------------------------------------

class ScreenShotConfigurationUiHandler : public ConfigurationUiHandler
{
public:
	virtual void mainConfigurationWindowCreated(MainConfigurationWindow *mainConfigurationWindow)
	{
		connect(mainConfigurationWindow->widgetById("screenshot/enableSizeLimit"), SIGNAL(toggled(bool)),
			mainConfigurationWindow->widgetById("screenshot/sizeLimit"), SLOT(setEnabled(bool)));

		QStringList opts = QStringList::fromStrList(QImageIO::outputFormats());
		ConfigComboBox *formats = dynamic_cast<ConfigComboBox *>(mainConfigurationWindow->widgetById("screenshot/formats"));
		formats->setItems(opts, opts);
	}
};

//-----------------------------------------------------------------------------------

ScreenShot::ScreenShot(QWidget * parent, const char * name, WFlags f)
 : QWidget(parent, name, Qt::WStyle_Customize|Qt::WStyle_NoBorder)
{
	kdebugf();
	minSize = 8;

	sizeHint = new ShotSizeHint();
	hintTimer = new QTimer();
	connect(hintTimer, SIGNAL(timeout()), this, SLOT(updateHint()));

	// Chat windows menu
	menu = new QPopupMenu();
	popups[0] = menu->insertItem(tr("Simple shot"), this, SLOT(takeShot(int)));
	popups[1] = menu->insertItem(tr("With chat window hidden"), this, SLOT(takeShot(int)));
	popups[2] = menu->insertItem(tr("Window shot"), this, SLOT(takeShot(int)));

	UiHandler = new ScreenShotConfigurationUiHandler();

	// Chat toolbar button
	screenshot_action = new Action("ScreenshotShot",
		tr("ScreenShot"), "ScreenShotAction", Action::TypeChat);
	connect(screenshot_action, SIGNAL(activated(const UserGroup*, const QWidget*, bool)),
		this, SLOT(screenshotActionActivated(const UserGroup*, const QWidget*)));

	ToolBar::addDefaultAction("Chat toolbar 1", "ScreenShotAction");

	// Rest stuff
	buttonPressed = false;
	warnedAboutSize = false;

	createDefaultConfiguration();
}


ScreenShot::~ScreenShot()
{
	kdebugf();

	delete UiHandler;
	delete screenshot_action;

	hintTimer->stop();
	delete hintTimer;
	delete sizeHint;
	delete menu;
}

void ScreenShot::screenshotActionActivated(const UserGroup *grp, const QWidget *source)
{
	kdebugf();

	tempChat = chat_manager->findChatWidget(grp);

	menu->popup(source->mapToGlobal(QPoint(0,20)));
}

void ScreenShot::mousePressEvent(QMouseEvent* e)
{
	kdebugf();
	if (e->button() == Qt::LeftButton)
	{
		if ( shotType == SINGLE_WINDOW )
		{
			releaseMouse();
			releaseKeyboard();
			hide();
			update();
			QTimer::singleShot(100, this, SLOT(takeWindowShot_Step2()));
		}
		else
		{
			region = QRect(e->pos(), e->pos());
			buttonPressed = true;

			int x = e->pos().x()+50,
				y = e->pos().y()+50;

			QRect screen = QApplication::desktop()->screenGeometry();
			if (x+150 > screen.width())
				x -= 150;

			if (y+100 > screen.height())
				y -= 100;

			sizeHint->move(x, y);

			sizeHint->geom->setText("0x0");
			sizeHint->fileSize->setText("0 KB");
			sizeHint->show();
			hintTimer->start(1000);
		}
	}
}

void ScreenShot::mouseReleaseEvent(QMouseEvent* e)
{
	kdebugf();

	if (!buttonPressed)
		return;

	hintTimer->stop();
	sizeHint->hide();

	// Uwalnianie myszki, klawiatury
	buttonPressed = false;
	releaseMouse();
	releaseKeyboard();
	drawRegionRect();

	// Normalizowanie prostokta do zrzutu
	region.setBottomRight(e->pos());
	region = region.normalize();

	// Zrzut
	QPixmap shot = QPixmap::grabWindow(winId(), region.x(), region.y(), region.width(), region.height());

	// Chowanie widgeta zrzutu i przywrcenie kursora.
	hide();
	QApplication::restoreOverrideCursor();

	handleShot(shot);
}

void ScreenShot::handleShot(QPixmap p)
{
	// Plik do zapisu:
	QDir dir(config_file.readEntry("ScreenShot", "path", ggPath("images/")));
	if (!dir.exists())
	{
		int ret = mkdir(dir.path().local8Bit().data(), 0755);
		if (ret)
		{
			printf("Error while creating directory %s:\n", dir.path().local8Bit().data());
			switch (ret)
			{
				case ENAMETOOLONG:
					printf("Pathname too long.\n");
					break;
				case EACCES:
					printf("The parent directory does not allow write permission to the process, or one of the directories in pathname did not allow search (execute) permission.\n");
					break;
				case EFAULT:
					printf("Pathname points outside your accessible address space.\n");
					break;
				case EROFS:
					printf("pathname refers to a file on a read-only filesystem.\n");
					break;
				case ENOSPC:
					printf("The new directory cannot be created because the user's disk quota is exhausted.\n");
					break;
				case EPERM:
					printf("The filesystem containing pathname does not support the creation of directories. \n");
					break;
				case EEXIST:
					printf("pathname already exists (not necessarily as a directory). This includes the case where pathname is a symbolic link, dangling or not.\n");
					break;
				default:
					printf("Unknown error.\n");
					break;
			}
			return;
		}
	}

	bool useShortJpg = config_file.readBoolEntry("ScreenShot", "use_short_jpg", false);

	QString ext = config_file.readEntry("ScreenShot", "fileFormat", "PNG").lower();
	if ( ext == "jpeg" && useShortJpg )
		ext = "jpg";

	QString path = QDir::cleanDirPath(
				dir.path() + "/" +
				config_file.readEntry("ScreenShot", "filenamePrefix", "shot") +
				QString::number(QDateTime::currentDateTime().toTime_t()) + "." + ext
		);

	bool ret = p.save(
			path, config_file.readEntry("ScreenShot", "fileFormat", "PNG").ascii(),
			config_file.readNumEntry("ScreenShot", "quality", -1)
		);

	if (!ret)
	{
		printf("Can't write file %s.\nAccess denied or other problem.", path.local8Bit().data());
		return;
	}

	if (!tempChat)
	{
		printf("ScreenShot module can't determinate Chat window. It should not happen! It's a bug!\n Please report it at Kadu Forum or to module author.\n");
		return;
	}

	QFileInfo f(path);
	int size = f.size();

	if (size == 0)
	{
		printf("File %s has 0 size!\nIt should be bigger.", path.local8Bit().data());
		return;
	}

	if (shotType == WITH_CHAT_MINIMIZED || shotType == SINGLE_WINDOW)
		restore( tempChat );

	// Wklejanie [IMAGE] do okna Chat
	if (config_file.readBoolEntry("ScreenShot", "paste_clause", true))
	{
		// Sprawdzanie rozmiaru zrzutu wobec rozmwcw
		UserListElements users = tempChat->users()->toUserListElements();
		if (users.count() > 1)
		{
			QStringList list;
			for ( uint i = 0; i < users.count(); i++ )
			{
				if ( ( users[i].protocolData("Gadu", "MaxImageSize").toInt() * 1024) < size)
					list.append(users[i].altNick()) ;
			}
			if (list.count() > 0)
			{
				if (list.count() == users.count())
				{
					MessageBox::msg(
							tr("Image size is bigger than maximal image size\nset by all of conference contacts."),
							true
						);
				}
				else
				{
					MessageBox::msg(
							tr("Image size is bigger than maximal image size\nset by some of conference contacts:\n%1.")
								.arg(list.join(", ")),
							true
						);
				}
			}
		}
		else
		{
			if ( ( users[0].protocolData("Gadu", "MaxImageSize").toInt() * 1024) < size)
			{
				bool send = MessageBox::ask(
					tr("Image size is bigger than maximal image size set by %1. Send it anyway?")
						.arg( users[0].altNick() )
					);

				if (!send)
					return;
			}
		}

		int x, y;
		tempChat->edit()->getCursorPosition(&y, &x);
		tempChat->edit()->insertAt(QString("[IMAGE ") + path + "]", y, x);
		tempChat->edit()->moveCursor(QTextEdit::MoveEnd, FALSE);
	}
	tempChat = NULL;
	checkShotsSize();
}

void ScreenShot::keyPressEvent(QKeyEvent* e)
{
	kdebugf();
	if (e->key() == Qt::Key_Escape)
	{
		releaseMouse();
		releaseKeyboard();
		hide();
	}
}

void ScreenShot::takeShot(int ident)
{
	kdebugf();

	// This code tells us which item from menu button was selected
	int id;
	for ( int i = 0; i < 3; i++ )
	{
		if (popups[i] == ident)
		{
			id = i;
			break;
		}
	}

	shotType = (ScreenShotMode)id;
	switch (shotType)
	{
		case STANDARD:
		{
			QTimer::singleShot(100, this, SLOT(takeShot_Step2()));
			tempChat->update();
			qApp->processEvents();
			break;
		}
		case WITH_CHAT_MINIMIZED:
		{
			wasMaximized = isMaximized( tempChat );
			minimize( tempChat );
			QTimer::singleShot(600, this, SLOT(takeShot_Step2()));
			break;
		}
		case SINGLE_WINDOW:
		{
			takeWindowShot();
			break;
		}
	}
}

void ScreenShot::takeShot_Step2()
{
	pixmap = QPixmap::grabWindow(QApplication::desktop()->winId());
	resize(pixmap.size());
	setPaletteBackgroundPixmap(pixmap);
	showFullScreen();
	setCursor(crossCursor);

	QTimer::singleShot(100, this, SLOT(grabMouseSlot()));
}

void ScreenShot::grabMouseSlot()
{
	kdebugf();
	grabMouse();
	grabKeyboard();
}

void ScreenShot::mouseMoveEvent(QMouseEvent* e)
{
	kdebugf();
	if (!buttonPressed)
		return;

	drawRegionRect();
	region.setBottomRight(e->pos());
	drawRegionRect();

	QRect reg = region;
	reg = reg.normalize();

	sizeHint->geom->setText(
		QString("%1x%2")
			.arg(QString::number(reg.width()))
			.arg(QString::number(reg.height()))
		);
}

void ScreenShot::updateHint()
{
	QRect reg;
	reg.setTopLeft(region.topLeft());
	reg.setBottomRight(region.bottomRight());
	reg = reg.normalize();
	QPixmap shot = QPixmap::grabWindow(winId(), reg.x(), reg.y(), reg.width(), reg.height());
	bool ret = shot.save("/tmp/kadu_screenshot_tmp.png", "PNG", -1);
	if (ret)
	{
		QFileInfo f("/tmp/kadu_screenshot_tmp.png");
		sizeHint->fileSize->setText(QString::number(f.size()/1024)+" KB");
	}
}

void ScreenShot::drawRegionRect()
{
	QPainter painter;
	painter.begin( this );
	painter.setRasterOp( NotROP );
	painter.setPen( QPen( color0, 1 ) );
	painter.setBrush( NoBrush );
	style().drawPrimitive(
			QStyle::PE_FocusRect,
			&painter,
			region,
			colorGroup(),
			QStyle::Style_Default,
			QStyleOption(colorGroup().base())
		);

	painter.end();
}

void ScreenShot::checkShotsSize()
{
	kdebugf();
	if (!config_file.readBoolEntry("ScreenShot", "dir_size_warns", true))
		return;

	int size = 0;

	int limit = config_file.readNumEntry("ScreenShot", "dir_size_limit", 10000);
	QDir dir(config_file.readEntry("ScreenShot", "path", ggPath("images")));

	QString prefix = config_file.readEntry("ScreenShot", "filenamePrefix", "shot");
	QFileInfoList* list = (QFileInfoList*)dir.entryInfoList(prefix+"*", QDir::Files);
	QFileInfo *f = list->first();
	while (f)
	{
		size += f->size();
		f = list->next();
	}

	if (size/1024 >= limit)
	{
		Notification *notification = new Notification("ssSizeLimit", "Blocking", UserListElements());
		notification->setTitle(tr("ScreenShot size limit"));
		notification->setText(tr("Images size limit exceed: %1 KB").arg(size/1024));
		notification_manager->notify(notification);
	}
}

void ScreenShot::takeWindowShot()
{
	kdebugf();
	wasMaximized = isMaximized( tempChat );
	minimize( tempChat );
	takeShot_Step2();
}

void ScreenShot::takeWindowShot_Step2()
{
	kdebugf();
	QPixmap winPixmap = grabCurrent();
	handleShot( winPixmap );
}

void ScreenShot::restore(QWidget* w)
{
	// For tabs module
	while (w->parent())
		w = static_cast<QWidget *>(w->parent());

	if (wasMaximized)
		w->showMaximized();
	else
		w->showNormal();
}

void ScreenShot::minimize(QWidget* w)
{
	// For tabs module
	while (w->parent())
		w = static_cast<QWidget *>(w->parent());

	w->showMinimized();
}

bool ScreenShot::isMaximized(QWidget* w)
{
	// For tabs module
	while (w->parent())
		w = static_cast<QWidget *>(w->parent());

	return w->isMaximized();
}

void ScreenShot::createDefaultConfiguration()
{
	config_file.addVariable("ScreenShot", "fileFormat", "PNG");
	config_file.addVariable("ScreenShot", "use_short_jpg", true);
	config_file.addVariable("ScreenShot", "quality", -1);
	config_file.addVariable("ScreenShot", "path", ggPath("images/"));
	config_file.addVariable("ScreenShot", "filenamePrefix", "shot");
	config_file.addVariable("ScreenShot", "paste_clause", true);
	config_file.addVariable("ScreenShot", "dir_size_warns", true);
	config_file.addVariable("ScreenShot", "dir_size_limit", 10000);
}

//////////////////////////////////////////////////////////////////
// Code below is copied (and changed a little) from KSnapShot,
// copyrighted by Bernd Brandstetter <bbrand@freenet.de>.
// It's licenced with GPL.

QPixmap ScreenShot::grabCurrent()
{
	Window root;
	int y, x;
	uint w, h, border, depth;
	XGrabServer( qt_xdisplay() );
	Window child = windowUnderCursor();
	XGetGeometry( qt_xdisplay(), child, &root, &x, &y, &w, &h, &border, &depth );
	Window parent;
	Window* children;
	unsigned int nchildren;
	if ( XQueryTree( qt_xdisplay(), child, &root, &parent, &children, &nchildren ) != 0 )
	{
		if( children != NULL )
			XFree( children );

		int newx, newy;
		Window dummy;
		if ( XTranslateCoordinates( qt_xdisplay(), parent, qt_xrootwin(), x, y, &newx, &newy, &dummy ))
		{
			x = newx;
			y = newy;
		}
	}
	QPixmap pm( grabWindow( child, x, y, w, h, border ) );
	XUngrabServer( qt_xdisplay() );
	return pm;
}

Window ScreenShot::windowUnderCursor()
{
	Window root;
	Window child;
	uint mask;
	int rootX, rootY, winX, winY;
	XGrabServer( qt_xdisplay() );
	XQueryPointer( qt_xdisplay(), qt_xrootwin(), &root, &child,
		&rootX, &rootY, &winX, &winY, &mask );

	if ( child == None )
		child = qt_xrootwin();

	Window real_child = findRealWindow( child );
	if ( real_child != None ) // test just in case
		child = real_child;

	return child;
}

QPixmap ScreenShot::grabWindow( Window child, int x, int y, uint w, uint h, uint border )
{
	QPixmap pm( QPixmap::grabWindow( qt_xrootwin(), x, y, w, h ) );

	int tmp1, tmp2;
	//Check whether the extension is available
	if ( XShapeQueryExtension( qt_xdisplay(), &tmp1, &tmp2 ) )
	{
		QBitmap mask( w, h );
		//As the first step, get the mask from XShape.
		int count, order;
		XRectangle* rects = XShapeGetRectangles( qt_xdisplay(), child,
							ShapeBounding, &count, &order );
		//The ShapeBounding region is the outermost shape of the window;
		//ShapeBounding - ShapeClipping is defined to be the border.
		//Since the border area is part of the window, we use bounding
		// to limit our work region
		if (rects)
		{
			//Create a QRegion from the rectangles describing the bounding mask.
			QRegion contents;
			for ( int pos = 0; pos < count; pos++ )
			contents += QRegion( rects[pos].x, rects[pos].y,
						rects[pos].width, rects[pos].height );
			XFree( rects );

			//Create the bounding box.
			QRegion bbox( 0, 0, w, h );

			if( border > 0 )
			{
				contents.translate( border, border );
				contents += QRegion( 0, 0, border, h );
				contents += QRegion( 0, 0, w, border );
				contents += QRegion( 0, h - border, w, border );
				contents += QRegion( w - border, 0, border, h );
			}

			//Get the masked away area.
			QRegion maskedAway = bbox - contents;
			QMemArray<QRect> maskedAwayRects = maskedAway.rects();

			//Construct a bitmap mask from the rectangles
			QPainter p(&mask);
			p.fillRect(0, 0, w, h, Qt::color1);
			for (uint pos = 0; pos < maskedAwayRects.count(); pos++)
				p.fillRect(maskedAwayRects[pos], Qt::color0);

			p.end();
			pm.setMask(mask);
		}
	}

	return pm;
}

// Recursively iterates over the window w and its children, thereby building
// a tree of window descriptors. Windows in non-viewable state or with height
// or width smaller than minSize will be ignored.
void ScreenShot::getWindowsRecursive( std::vector<QRect>& windows, Window w, int rx, int ry, int depth )
{
	XWindowAttributes atts;
	XGetWindowAttributes( qt_xdisplay(), w, &atts );
	if ( atts.map_state == IsViewable && atts.width >= minSize && atts.height >= minSize )
	{
		int x = 0, y = 0;
		if ( depth )
		{
			x = atts.x + rx;
			y = atts.y + ry;
		}

		QRect r( x, y, atts.width, atts.height );
		if ( std::find( windows.begin(), windows.end(), r ) == windows.end() )
		{
			windows.push_back( r );
		}

		Window root, parent;
		Window* children;
		unsigned int nchildren;

		if ( XQueryTree( qt_xdisplay(), w, &root, &parent, &children, &nchildren ) != 0 )
		{
				for ( unsigned int i = 0; i < nchildren; ++i )
				{
					getWindowsRecursive( windows, children[ i ], x, y, depth + 1 );
				}
				if ( children != NULL )
					XFree( children );
		}
	}
	//if ( depth == 0 )
	//	std::sort( windows.begin(), windows.end() );
}

Window ScreenShot::findRealWindow( Window w, int depth )
{
	if( depth > 5 )
		return None;

	static Atom wm_state = XInternAtom( qt_xdisplay(), "WM_STATE", False );
	Atom type;
	int format;
	unsigned long nitems, after;
	unsigned char* prop;
	if ( XGetWindowProperty( qt_xdisplay(), w, wm_state, 0, 0, False, AnyPropertyType, &type, &format, &nitems, &after, &prop ) == Success )
	{
		if ( prop != NULL )
			XFree( prop );

		if ( type != None )
			return w;
	}
	Window root, parent;
	Window* children;
	unsigned int nchildren;
	Window ret = None;
	if ( XQueryTree( qt_xdisplay(), w, &root, &parent, &children, &nchildren ) != 0 )
	{
		for ( unsigned int i = 0; i < nchildren && ret == None; ++i )
			ret = findRealWindow( children[ i ], depth + 1 );

		if ( children != NULL )
			XFree( children );
	}
	return ret;
}

// End of code copied from KSnapShot
//////////////////////////////////////////////////////////////////
