#ifndef KADU_MODULES_H
#define KADU_MODULES_H

#include <qhbox.h>
#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>

class QCheckBox;
class QLabel;
class QListView;
class QListViewItem;
class QTranslator;

/**
	Zastpuje klas QLibrary na specyficzne potrzeby Kadu.
	\class Library
	\brief Biblioteki dzielone.
**/
class Library : public QObject
{
	private:
		QString FileName;
		void* Handle;

	public:
		/**
			\fn Library(const QString& file_name)
			Konstruktor przydzielajcy dany plik dla tego obiektu biblioteki dzielonej.
			\param file_name nazwa pliku biblioteki dzielonej.
		**/
		Library(const QString& file_name);
		~Library();

		/**
			\fn bool load()
			aduje przydzielon bibliotek dzielon do pamici.
		**/
		bool load();

		/**
			\fn void* resolve(const QString& symbol_name)
			Tumaczy nazw symbolu na funkcj z zaadowanej biblioteki dzielonej.
			\param symbol_name nazwa symbolu do przetumaczenia.
			\return wskanik do prztumaczonego symbolu.
		**/
		void* resolve(const QString& symbol_name);

		/**
			\fn QString error()
			\return tre bdu, jaki wystpi podczas adowanie biblioteki dzielonej.
		**/
		QString error();
};

/**
	\struct ModuleInfo
	\brief Informacje o module.
**/
struct ModuleInfo
{
	QStringList depends; /*!< Jakie inne moduy s wymagane przez ten modu. */
	QStringList conflicts; /*!< Z jakimi moduami ten modu konfliktuje. */
	QStringList provides; /*!< Jak cech dostarcza ten modu. */
	QString description; /*!< Opis moduu. */
	QString author; /*!< Autor moduu. */
	QString version; /*!< Wersja moduu. */
	bool load_by_def; /*!< Czy modu jest domylnie adowany, czy nie? */
	bool base; /*!< Czy modu naley do moduw podstawowych? */
	ModuleInfo();
};

class LayoutHelper;
/**
	Klasa reprezentujca okno dialogowe, suce do adowanie i wyadowywania moduw.
	adowanie/wyadowywanie, oraz inne operacje na moduach z poziomu C++ dokonywane
	s za pomoc klasy ModulesManager. Ta klasa tworzy jedynie okno dialogowe, bdce
	interfejsem do ww. klasy dla uytkownika Kadu.
	\class ModulesDialog
	\brief "Zarzdca moduw"
**/
class ModulesDialog : public QHBox
{
	Q_OBJECT

	private:
		QListView* lv_modules;
		QLabel* l_moduleinfo;
		LayoutHelper *layoutHelper;
		QCheckBox *hideBaseModules;

	private slots:
		void loadItem();
		void unloadItem();
		void moduleAction(QListViewItem *);
		void itemsChanging();
		void getInfo();
		void refreshList();
		void keyPressEvent(QKeyEvent *);

	public:
		/**
			\fn ModulesDialog()
			Konstruktor standardowy.
		**/
		ModulesDialog();
		~ModulesDialog();
	protected:
		virtual void resizeEvent(QResizeEvent *);
};

/**
	Ta klasa odpowiada za obsug moduw Kadu.
	\class ModulesManager
	\brief Zarzdzanie moduami
**/
class ModulesManager : public QObject
{
	Q_OBJECT

	private:
		typedef int InitModuleFunc();
		typedef void CloseModuleFunc();
		/**
			\struct StaticModule
			\brief Informacje o statycznym module.
		**/
		struct StaticModule
		{
			InitModuleFunc* init; /*!< Wskanik do funkcji inicjalizujcej modu. */
			CloseModuleFunc* close; /*!< Wskanik do funkcji deinicjalizujcej modu. */
		};
		/**
			Lista statycznych moduw wypeniania
			przez kod generowany przez configure.
		**/
		QMap<QString,StaticModule> StaticModules;
		/**
			Informacje o aktywnym module
			statycznym bd zewntrznym.
			Dla moduu statycznego lib==NULL.
			\struct Module
			\brief Informacje o aktywnym module.
		**/
		struct Module
		{
			Library* lib; /*!< Wskanik do obiektu biblioteki dzielonej. */
			CloseModuleFunc* close; /*!< Wskanik do funkcji deinicjalizujcej modu. */
			QTranslator* translator; /*!< Wskanik do obiektu tumaczcego dla tego moduu. */
			ModuleInfo info; /*!< Informacje o module. */
			int usage_counter; /*!< Licznik uycia moduu. */
			Module();
		};
		/**
			Lista aktywnych moduw
			statycznych bd zewntrznych.
		**/
		QMap<QString,Module> Modules;
		/**
		**/
		ModulesDialog* Dialog;
		/**
			aduje plik z tumaczeniem. Zwraca NULL jeli wystpi
			bd.
		**/
		QTranslator* loadModuleTranslation(const QString& module_name);
		/**
			Sprawdza czy dostpne s moduy z listy
			zalenoci danego moduu. W razie czego
			stara si je zaadowa jeli s dostpne.
			@param module_info informacje o module
		**/
		bool satisfyModuleDependencies(const ModuleInfo& module_info);
		/**
			Zwiksza liczniki uycia moduw uywanych
			przez dany modu.
			@param module_info informacje o module
		**/
		void incDependenciesUsageCount(const ModuleInfo& module_info);
		/**
			Rejestruje modu statyczny. Funcja wywoywana
			dla wszystkich moduw statycznych przez kod
			wygenerowany przez configure.
		**/
		void registerStaticModule(const QString& module_name,
			InitModuleFunc* init,CloseModuleFunc* close);
		/**
			Rejestruje moduy statyczne. Kod funkcji jest
			generowany przez configure.
		**/
		void registerStaticModules();

		/**
			Skupia wszystkie tumaczenia w jednej hierarchii
		**/
		QObject *translators;

	private slots:
		void dialogDestroyed();

	public:
		/**
			\fn static void initModule()
			Inicjalizuje obsug moduw. Metoda ta wywoywana jest przy starcie Kadu, przez jego rdze.
		**/
		static void initModule();

		/**
			\fn static void closeModule()
			Deinicjalizuje obsug moduw. Metoda ta jest wywoywana przy zamykaniu Kadu, przez jego rdze.
		**/
		static void closeModule();

		/**
			\fn ModulesManager()
			Standardowy konstruktor.
		**/
		ModulesManager();
		~ModulesManager();

		/**
			\fn QStringList staticModules() const
			\return list moduw wkompilowanych
			statycznie w plik wykonywalny Kadu.
		**/
		QStringList staticModules() const;

		/**
			\fn QStringList installedModules() const
			\return list moduw zainstalowanych jako
			dzielone biblioteki (shared libraries).
		**/
		QStringList installedModules() const;

		/**
			\fn QStringList loadedModules() const
			\return list moduw zaadowanych do pamici
			jako dzielone biblioteki (shared libraries).
		**/
		QStringList loadedModules() const;

		/**
			\fn QStringList unloadedModules() const
			\return list moduw zainstalowanych jako
			dzielone biblioteki (shared libraries)
			i nie zaadowanych aktualnie do pamici.
		**/
		QStringList unloadedModules() const;

		/**
			\fn QStringList activeModules() const
			\return list aktywnych moduw.
			Uwzgldniane s zarwno aktywne moduy
			statyczne jak i zewntrzne.
		**/
		QStringList activeModules() const;

		QString moduleProvides(const QString &provides);

		/**
			\fn bool moduleInfo(const QString& module_name, ModuleInfo& info) const
			Pobiera do info informacje o danym module.
			\param[in] module_name nazwa moduu, dla ktrego naley pobra informacj.
			\param[out] info struktura, w ktrej te informacje naley umieci.
			\return true, jeli si udao, w przeciwnym wypadku false.
		**/
		bool moduleInfo(const QString& module_name, ModuleInfo& info) const;

		/**
			\fn bool moduleIsStatic(const QString& module_name) const
			Sprawdza czy podany modu jest wkompilowany statycznie.
			\param module_name nazwa moduu.
			\return true, jeli modu jest wkompilowany, false jeli nie jest.
		**/
		bool moduleIsStatic(const QString& module_name) const;

		/**
			\fn bool moduleIsInstalled(const QString& module_name) const
			Sprawdza czy podany modu jest zainstalowany
			w katalogu z moduami zewntrznymi.
			\param module_name nazwa moduu.
			\return true, jeli modu jest zainstalowany, false jeli nie jest.
		**/
		bool moduleIsInstalled(const QString& module_name) const;

		/**
			\fn bool moduleIsLoaded(const QString& module_name) const
			Sprawdza czy podany modu zewntrzny jest zaadowany.
			\param module_name nazwa moduu.
			\return true, jeli modu jest zaadowany, false jeli nie jest.
		**/
		bool moduleIsLoaded(const QString& module_name) const;

		/**
			\fn bool moduleIsActive(const QString& module_name) const
			Sprawdza czy podany modu jest aktywny.
			Dziaa dla moduw statycznych i zaadowanych
			zewntrznych.
			\param module_name nazwa moduu.
			\return true, jeli modu jest aktywny, false jeli nie jest.
		**/
		bool moduleIsActive(const QString& module_name) const;

		/**
			\fn bool conflictsWithLoaded(const QString &module_name, const ModuleInfo& module_info) const
			Sprawdza czy podany modu konfliktuje
			z jakim innym zaadowanym moduem.
			\param module_name nazwa moduu.
			\param module_info informacje o module.
			\return true, jeli modu konfliktuje, false jeli nie.
		**/
		bool conflictsWithLoaded(const QString &module_name, const ModuleInfo& module_info) const;

	public slots:
		/**
			\fn bool activateModule(const QString& module_name)
			Aktywuje modu statyczny jeli jest dostpny
			lub aduje do pamici i aktywuje modu zewntrzny.
			Przez aktywacje rozumie si wywoanie funkcji *_init z moduu.
			\param module_name nazwa moduu.
			\return true jeli aktywacja przebiega bezproblemowo, false w przeciwnym wypadku.
		**/
		bool activateModule(const QString& module_name);

		/**
			\fn bool deactivateModule(const QString& module_name, bool force=false)
			Deaktywuje modu statyczny lub deaktywuje i usuwa z pamici modu zewntrzny.
			\param module_name nazwa moduu.
			\return true jeli dezaktywacja przebiega bezproblemowo, false w przeciwnym wypadku.
		**/
		bool deactivateModule(const QString& module_name, bool force=false);

		/**
			\fn void showDialog()
			Wywietla okno dialogowe "Zarzdcy moduw", czyli tworzy i pokazuje klas ModulesDialog.
		**/
		void showDialog();

		/**
			\fn void moduleIncUsageCount(const QString& module_name)
			Zwiksza licznik uycia moduu o 1.
			\param module_name nazwa moduu.
		**/
		void moduleIncUsageCount(const QString& module_name);

		/**
			\fn void moduleDecUsageCount(const QString& module_name)
		 	Zmniejsza licznik uycia moduu o 1.
			\param module_name nazwa moduu.
		**/
		void moduleDecUsageCount(const QString& module_name);

		/**
			\fn void saveLoadedModules()
			Zapisuje do pliku konfiguracyjnego list zaadowanych
			moduw.
		**/
		void saveLoadedModules();
};

extern ModulesManager* modules_manager;

#endif
