#ifndef KADU_CONFIGURATION_WINDOW_H
#define KADU_CONFIGURATION_WINDOW_H

#include <qdom.h>
#include <qgroupbox.h>
#include <qtabwidget.h>

#include "color_button.h"
#include "hot_key.h"
#include "path_list_edit.h"
#include "select_font.h"
#include "syntax_editor.h"
#include "userbox.h"

class ConfigGroupBox;
class ConfigLineEdit;
class ConfigTab;
class ConfigurationWindow;
class ConfigWidget;

class QGridLayout;
class QGroupBox;
class QVBox;

class ConfigSection : public QObject
{
	Q_OBJECT

	QString name;
	ConfigurationWindow *configurationWindow;
	QString pixmap;

	QListBoxItem *listBoxItem;
	QMap<QString, ConfigTab *> configTabs;

	QTabWidget *mainWidget;

	ConfigTab *configTab(const QString &name, bool create = true);
	bool activated;

private slots:
	void iconThemeChanged();

public:
	ConfigSection(const QString &name, ConfigurationWindow *configurationWindow, QListBoxItem *listBoxItem, QWidget *parentConfigGroupBoxWidget,
		const QString &pixmap);
	~ConfigSection();

	void activate();

	void show() { mainWidget->show(); }
	void hide() { mainWidget->hide(); }

	ConfigGroupBox * configGroupBox(const QString &tab, const QString &groupBox, bool create = true);

	void removedConfigTab(const QString &configTabName);

};

/**
	@class ConfigGroupBox
	@author Vogel
	@short GroupBox w oknie konfiguracji

	GroupBox w oknie konfiguracji definiowany jest przez nastpujcy tag:
	<code>
		&lt;group-box caption="tytu" id="id"&gr;
			...
		&lt;/group-box&gt;
	</code>

	GroupBox moe znajdowa si tylko wewntrz tagu tab. W jego wntrzu
	mog zawiera si dowolne tagi widgetw konfigruacyjnych.

	Dodatkowo, GroupBox'a mona stworzy (lub, jeeli istnieje, uzyska)
	wywoujc funkcj configGroupBox(section, tab, groupBox) z okna konfiguracyjnego.
	Do tak uzyskanego GroupBox'a mona dodawa dowolne widgety (@see addWidget,
	@see addWidgets).
 **/
class ConfigGroupBox
{
	QString name;
	ConfigTab *configTab;

	QGroupBox *groupBox;
	QWidget *container;
	QGridLayout *gridLayout;

public:
	ConfigGroupBox(const QString &name, ConfigTab *configTab, QGroupBox *groupBox);
	~ConfigGroupBox();

	QWidget * widget() { return container; }

	/**
		Dodaje widget do GroupBoxa.
		@param widget dodawany widget (nie moe by NULL)
		@param fullSpace kiedy true, dodany widget zajmuje ca szeroko GroupBox'a,
			w przeciwnym wypadku tylko praw jego cz
	 **/
	void addWidget(QWidget *widget, bool fullSpace = false);
	/**
		Dodaje 2 widget do GroupBoxa, jeden po lewej stronie, drugi po prawej.
		@param widget1 widget dodawany z lewej strony (zazwyczaj etykieta)
		@param widget2 widget dodawany z prawej strony
	 **/
	void addWidgets(QWidget *widget1, QWidget *widget2);

	/**
		Zwraca true, gdy GroupBox nie posiada w sobie adnych widgetw.
		@return true, gdy GroupBox nie posiada w sobie adnych widgetw
	 **/
	bool empty();

};

/**
	@class ConfigurationWindow
	@author Vogel
	@short Widget okna konfigruacyjnego.

	Okno konfiguracyjne tego typu zawiera widgety konfiuracyjne podzielone wg. 3 stopnioej hierarchii.
	Pierwszym stopniem s sekcje reprezentowane przez ListBox'a z lewej strony okna (zawierajcego
	ikony i opis tekstowy). Okno zawierajce tylko jedn sekcj nie wywietla ListBox'a.
	Drugim stopniem s karty reprezentowane przez TabWidget'y, trzecim - grupy opisane przez GroupBox'y.

	Okno konfiguracyjne identyfikuje si przez jego nazw podan w konstruktorze
	(dziki tej nazwie kade okno moe osobno zapamita swoj pozycj i ostatnio
	otwartej karty).

	Okna mog teoretycznie zawiera dowolne widgety. Kady z nich, ktry dodatkowo
	dziedziczy z klasy ConfigWidget, traktowany jest w specjalny sposb. Jego
	metody loadConfiguration i saveConfiguration s wywoywane automatycznie
	przy otwieraniu okna i przy zapisywaniu konfiguracji, dziki czemu nie jest
	potrzebne adne 'rczne' podpinanie si do tych akcji.

	W momencie zapisania konfiguracji wszystkie obiekty w programie bdce instancajmi
	klasy @see ConfigurationAwareObject zostan o tym poinformowane i bd
	mogy zaktualizowa swj stan.

	Widgety w oknie mog by tworzone na 2 sposoby. Pierwszym z nich jest
	pobranie GroupBoxa za pomoc funkcji @see configGroupBox i dodawanie
	do niego widgetw za pomoc jego funkcji addWidget i addWidgets.
	Drug jest stworzenie plikw XML *.ui, ktre s wczytywane i usuwane dynamicznie
	z pomoc metod @see appendUiFile i @see removeUiFile.

	W tym drugim przypadku stosuje si pliki *.ui o nastpujcej strukturze:

	&lt;configuration-ui&gt;
		&lt;section caption="tytu" icon="nazwa_ikony"&gt;
			&lt;tab caption="tytu"&gt;
				&lt;group-box caption="tytu" id="id"&gt;
					&lt;widget ... /&gt;
				&lt;/group-box&gt;
			&lt;/tab&gt;
		&lt;/section&gt;
	&lt;/configuration-ui&gt;

	Elementy zawierajce atrybut id (nie wymagany) - czyli group-box i dowolny widget
	mog zosta pobrane przez aplikacje za pomoc metody @see widgetById.
	Widgety z moduw powinny posiada id w postaci: nazwaModuu/nazwaId.

	Atrybut catpion jest wymagany. Moliwe tagi widget s opisane w dokumentacji
	klas Config* (np.: ConfigComboBox).
 **/
class ConfigurationWindow : public QVBox
{
	Q_OBJECT

	QString Name;

	QVBox *left;
	QMap<QString, ConfigSection *> configSections;
	ConfigSection *currentSection;

	QMap<QString, QWidget *> widgets;

	QListBox *sectionsListBox;
	QWidget *container;

	void loadConfiguration(QObject *object);
	void saveConfiguration(QObject *object);

	ConfigSection *configSection(const QString &name);
	ConfigSection *configSection(const QString &icon, const QString &name, bool create = true);

	QValueList<ConfigWidget *> processUiFile(const QString &fileName, bool append = true);
	QValueList<ConfigWidget *> processUiSectionFromDom(QDomNode sectionNode, bool append = true);
	QValueList<ConfigWidget *> processUiTabFromDom(QDomNode tabNode, const QString &iconName, const QString &sectionName, bool append = true);
	QValueList<ConfigWidget *> processUiGroupBoxFromDom(QDomNode groupBoxNode, const QString &sectionName, const QString &tabName, bool append = true);
	ConfigWidget * appendUiElementFromDom(QDomNode uiElementNode, ConfigGroupBox *configGroupBox);
	void removeUiElementFromDom(QDomNode uiElementNode, ConfigGroupBox *configGroupBox);

private slots:
	void updateAndCloseConfig();
	void updateConfig();

	void changeSection(const QString &newSectionName);

protected:
	virtual void keyPressEvent(QKeyEvent *e);

public:
	/**
		Tworzy okno konfiguracyjne o danej nazwie. Nazwa wykorzystywana
		jest przy zapamitywaniu pozycji okna oraz jego ostatnio
		otwartej karty.
	 **/
	ConfigurationWindow(const QString &name, const QString &caption);
	virtual ~ConfigurationWindow();

	QString name() { return Name; }

	/**
		Pobiera GroupBox'a zawartego w nastpujcej ciece XPath:
		/section[@caption=SECTION]/tab[@caption=TAB]/group-box[@caption=GROUPBOX]

		Jeeli nie istnieje taki group-box to zostanie on stworzony, gdy create == true.
		Dla crate == false zostanie zwrcony NULL.
	 **/
	ConfigGroupBox * configGroupBox(const QString &section, const QString &tab, const QString &groupBox, bool create = true);

	/**
		Jeeli okno jest ukryte wczytuje wartoci elementw z pliku
		konfiguracyjnego i pokazuje okno.
	 **/
	virtual void show();

	/**
		Dodaje do okna konfiguracyjnego widgety wczytane z pliku fileName.
		Gdy load == true wczytuje ich wartoci z pliku konfiguracyjnego. W przeciwnym
		wypadku wczytanie wartoci nastpi po wywoaniu metody @see show.
	 **/
	QValueList<ConfigWidget *> appendUiFile(const QString &fileName, bool load = true);
	/**
		Usuwa kontrolki wczytane z danego pliku *.ui.
	 **/
	void removeUiFile(const QString &fileName);

	/**
		Zwraca element opisane id="id" we wczytanych plikach *.ui. Moliwe
		dowolne widgety konfiguracyjne i elementy group-box.
	 **/
	QWidget *widgetById(const QString &id);

	// TODO: make private or sth
	void removedConfigSection(const QString &sectionName);

signals:
	/**
		Sygna emitowany po naciniciu Ok lub Apply ale przed zapisaniem
		wartoci do pliku konfiguracyjnego. Nietypowe widgety konfiguracyjne
		powinny si podpi pod ten sygna i po jego uzyskaniu zapisa
		nowe wartoci do pliku.
	 **/
	void configurationWindowApplied();

};

#endif // KADU_CONFIGURATION_WINDOW_H
