// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the ImageMagickWriter K-3D object, which writes bitmap images using the ImageMagick Magick++ API
		\author Brett W. McCoy (bmccoy@chapelperilous.net)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/ibitmap_write_format.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/ifile_format.h>
#include <k3dsdk/module.h>
#include <k3dsdk/path.h>

#include <Magick++.h>

using namespace Magick;

namespace
{

/////////////////////////////////////////////////////////////////////////////
// imagemagick_writer

class imagemagick_writer :
	public k3d::ifile_format,
	public k3d::ibitmap_write_format,
	public k3d::ideletable
{
public:
	imagemagick_writer()
	{
	}

	unsigned long priority()
	{
		return 64;
	}

	bool query_can_handle(const k3d::filesystem::path&)
	{
		return true;
	}

        bool write_file(const k3d::filesystem::path& FilePath, const k3d::bitmap& Bitmap)
        {
		// Sanity checks ...
		return_val_if_fail(!FilePath.empty(), false);

		k3d::log() << info << "Write " << FilePath.native_console_string() << " using ImageMagickWriter" << std::endl;

		// I really hate to make a copy of the data, but we have to convert from half to float, and ImageMagick reverses the sense of the alpha channel ...
		k3d::basic_bitmap<k3d::basic_rgba<float> > buffer(Bitmap);
		for(k3d::basic_bitmap<k3d::basic_rgba<float> >::iterator p = buffer.begin(); p != buffer.end(); ++p)
			p->alpha = 1.0 - p->alpha;
		
		Image image(buffer.width(), buffer.height(), "RGBA", FloatPixel, buffer.data());
		image.depth(8);
		
		image.write(FilePath.native_filesystem_string());
		
		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::application_plugin<imagemagick_writer>,
			k3d::interface_list<k3d::ibitmap_write_format> > factory(
				k3d::uuid(0x12f48171, 0xfdff41da, 0x9901acc1, 0xabc77a1f),
				"ImageMagickWriter",
				_("ImageMagick Formats ( many )"),
				"Bitmap BitmapWriter");

		return factory;
	}

private:

};
}//namespace

/////////////////////////////////////////////////////////////////////////////
// Helper functions

namespace libk3dimagemagick
{

/////////////////////////////////////////////////////////////////////////////
// imagemagick_writer_factory

k3d::iplugin_factory& imagemagick_writer_factory()
{
	return ::imagemagick_writer::get_factory();
}

} // namespace libk3dimagemagick


