// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the K-3D main() entry point
		\author Tim Shead (tshead@k-3d.com)
*/

// Standard K-3D interface implementations for embedding
#include <k3dsdk/application_detail.h>
#include <k3dsdk/plugin_factory_collection.h>
#include <k3dsdk/render_farm.h>

// Standard K-3D interfaces and helpers
#include <k3dsdk/algebra.h>
#include <k3dsdk/auto_ptr.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/create_plugins.h>
#include <k3dsdk/create_plugins.h>
#include <k3dsdk/fstream.h>
#include <k3dsdk/gl_info.h>
#include <k3dsdk/gzstream.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/idocument.h>
#include <k3dsdk/idocument_read_format.h>
#include <k3dsdk/iuser_interface.h>
#include <k3dsdk/iuser_interface_plugin.h>
#include <k3dsdk/log.h>
#include <k3dsdk/log_control.h>
#include <k3dsdk/nodes.h>
#include <k3dsdk/options_policy.h>
#include <k3dsdk/property.h>
#include <k3dsdk/render_farm_detail.h>
#include <k3dsdk/register_application.h>
#include <k3dsdk/scripting.h>
#include <k3dsdk/shader_cache_detail.h>
#include <k3dsdk/share_detail.h>
#include <k3dsdk/system.h>
#include <k3dsdk/user_interface_detail.h>
#include <k3dsdk/utility.h>
#include <k3dsdk/version.h>
#include <k3dsdk/xml.h>

#include <iomanip>
#include <iterator>

#ifdef K3D_PLATFORM_WIN32
#include <windows.h>
#undef interface
#endif // K3D_PLATFORM_WIN32

#ifdef K3D_STATIC_BUILD
  K3D_DECLARE_EXTERNAL_MODULE(libk3dbitmap)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dblobbies)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dchannels)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dcore)
  K3D_DECLARE_EXTERNAL_MODULE(libk3ddeformation)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dfreetype2)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dgeometry)
  K3D_DECLARE_EXTERNAL_MODULE(libk3djpeg)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dlsystem)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dmesh)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dngui)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dnurbs)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dpng)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dpython)
  K3D_DECLARE_EXTERNAL_MODULE(libk3drenderman)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dscript)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dscripting)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dtiff)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dtime)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dviewport)
  K3D_DECLARE_EXTERNAL_MODULE(libk3dwindows)
#endif // K3D_STATIC_BUILD

namespace
{

typedef std::vector<std::string> string_array;

/////////////////////////////////////////////////////////////////////////////
// User options specified on the command line

bool g_show_timestamps = false;
bool g_show_process = false;
bool g_syslog = false;
bool g_color_level = false;
k3d::log_level_t g_minimum_log_level = k3d::K3D_LOG_LEVEL_WARNING;

k3d::iuser_interface_plugin* g_user_interface = 0;

k3d::filesystem::path g_options_path;
std::string g_plugin_paths;
k3d::filesystem::path g_shader_cache_path;
k3d::filesystem::path g_share_path;
k3d::filesystem::path g_user_interface_path;
k3d::filesystem::path g_null_user_interface_path;

/////////////////////////////////////////////////////////////////////////////
// handle_error

void handle_error(const std::string& Message, bool& Quit, bool& Error)
{
	Quit = true;
	Error = true;

	throw std::runtime_error(Message);
}

/////////////////////////////////////////////////////////////////////////////
// startup_message_handler

/// Called during application startup to display progress to the user
void startup_message_handler(const std::string& Message)
{
	k3d::log() << info << Message << std::endl;

	if(g_user_interface)
		g_user_interface->startup_message_handler(Message);
}

/////////////////////////////////////////////////////////////////////////////
// exit_request_handler

/// Called when the user wants to shut everything down
bool exit_request_handler()
{
	if(g_user_interface)
		g_user_interface->stop_event_loop();

	return true;
}

/////////////////////////////////////////////////////////////////////////////
// set_default_options

/// Sets-up default user options for various platforms
void set_default_options(bool& Quit, bool& Error)
{
#ifdef K3D_PLATFORM_WIN32

	// Get the path where this module is executing ...
	std::string executable(256, '\0');
	GetModuleFileName(0, const_cast<char*>(executable.data()), executable.size());
	executable.resize(strlen(executable.c_str()));
	const k3d::filesystem::path executable_path = k3d::filesystem::native_path(k3d::ustring::from_utf8(executable)).branch_path();

	const k3d::filesystem::path data_path = k3d::system::get_home_directory() / k3d::filesystem::native_path(k3d::ustring::from_utf8(".k3d"));
	g_options_path = data_path / k3d::filesystem::generic_path("options.k3d");
	g_plugin_paths = (executable_path / k3d::filesystem::generic_path("plugins")).native_utf8_string().raw();
	g_shader_cache_path = data_path / k3d::filesystem::generic_path("shadercache");
	g_share_path = executable_path / k3d::filesystem::generic_path("share");
	g_user_interface_path = executable_path / k3d::filesystem::generic_path("libk3dngui.dll");
	g_null_user_interface_path = executable_path / k3d::filesystem::generic_path("libk3dnui.dll");

	// Add the executable path to PATH
	_putenv(("PATH=" + executable_path.native_filesystem_string() + ";" + k3d::system::get_env("PATH")).c_str());

#endif // K3D_PLATFORM_WIN32
}

/////////////////////////////////////////////////////////////////////////////
// parse_logging_options

/// Looks for command-line arguments that apply to logging (so we can set them up right away)
string_array parse_command_line_logging_options(const std::string& ProcessName, const string_array Options, bool& Quit, bool& Error)
{
	// Keep track of "unused" options ...
	string_array unused;

	for(string_array::const_iterator option = Options.begin(); option != Options.end(); ++option)
	{
		if((*option) == "--show-timestamps")
		{
			g_show_timestamps = true;
		}
		else if((*option) == "--show-process")
		{
			g_show_process = true;
		}
		else if((*option) == "--syslog")
		{
			g_syslog = true;
		}
		else if((*option) == "--color")
		{
			g_color_level = true;
		}
		else if((*option) == "--log-level")
		{
			++option;
			if(option == Options.end())
			{
				handle_error("You must supply a minimum log level with the --log-level option!", Quit, Error);
				return string_array();
			}

			if((*option) == "warning")
				g_minimum_log_level = k3d::K3D_LOG_LEVEL_WARNING;
			else if((*option) == "information")
				g_minimum_log_level = k3d::K3D_LOG_LEVEL_INFO;
			else if((*option) == "debug")
				g_minimum_log_level = k3d::K3D_LOG_LEVEL_DEBUG;
			else
			{
				handle_error("Valid values for the --log-level option are \"warning\", \"information\", or \"debug\"", Quit, Error);
				return string_array();
			}
		}
		else
		{
			unused.push_back(*option);
		}
	}

	k3d::log_show_timestamps(g_show_timestamps);
	k3d::log_set_tag(g_show_process ? "[" + ProcessName + "]" : std::string());
	k3d::log_color_level(g_color_level);
	k3d::log_show_level(true);
	k3d::log_syslog(g_syslog);
	k3d::log_minimum_level(g_minimum_log_level);

	return unused;
}

/////////////////////////////////////////////////////////////////////////////
// parse_command_line_2

/// Handles "normal" command-line arguments that specify initial application state
string_array parse_command_line_2(const string_array Options, bool& Quit, bool& Error)
{
	// We return any "unused" options ...
	string_array unused;

	// For each command-line argument ...
	for(string_array::const_iterator option = Options.begin(); option != Options.end(); ++option)
	{
		if((*option) == "--ui")
		{
			++option;
			if(option == Options.end())
			{
				handle_error("You must specify the path to a user interface plugin with the --ui option!", Quit, Error);
				return string_array();
			}

			if(*option == "none")
				g_user_interface_path = g_null_user_interface_path;
			else
				g_user_interface_path = k3d::filesystem::native_path(k3d::ustring::from_utf8(*option));
		}
		else if((*option) == "--plugins")
		{
			++option;
			if(option == Options.end())
			{
				handle_error("You must supply a delimited set of directories with the --plugins option!", Quit, Error);
				return string_array();
			}

			g_plugin_paths = (*option);
		}
		else if((*option) == "--shadercache")
		{
			++option;
			if(option == Options.end())
			{
				handle_error("You must supply a directory path with the --shadercache option!", Quit, Error);
				return string_array();
			}

			g_shader_cache_path = k3d::filesystem::native_path(k3d::ustring::from_utf8(*option));
		}
		else if((*option) == "--share")
		{
			++option;
			if(option == Options.end())
			{
				handle_error("You must supply a directory path with the --share option!", Quit, Error);
				return string_array();
			}

			g_share_path = k3d::filesystem::native_path(k3d::ustring::from_utf8(*option));
		}
		else if((*option) == "--options")
		{
			++option;
			if(option == Options.end())
			{
				handle_error("You must supply a file path with the --options option!", Quit, Error);
				return string_array();
			}

			g_options_path = k3d::filesystem::native_path(k3d::ustring::from_utf8(*option));
		}
		else
		{
			unused.push_back(*option);
		}
	}

	return unused;
}

/////////////////////////////////////////////////////////////////////////////
// check_dependencies

/// Checks for any runtime resources required to continue program execution
void check_dependencies(bool& Quit, bool& Error)
{
	k3d::log() << info << "package: " << K3D_PACKAGE << std::endl;
	k3d::log() << info << "version: " << K3D_VERSION << std::endl;
	k3d::log() << info << "platform: " << K3D_HOST << std::endl;
	k3d::log() << info << "compiler: " << __VERSION__ << std::endl;
	k3d::log() << info << "build time: " << __DATE__ << " " << __TIME__ << " local" << std::endl;

#ifdef K3D_HAVE_EXPAT
	k3d::log() << info << "xml parser: expat" << std::endl;
#elif defined K3D_HAVE_LIBXML2
	k3d::log() << info << "xml parser: libxml2" << std::endl;
#else
	k3d::log() << warning "xml parser: unknown" << std::endl;
#endif

	k3d::log() << info << "options file: " << g_options_path.native_console_string() << std::endl;
#ifndef K3D_STATIC_BUILD
	k3d::log() << info << "plugin path(s): " << g_plugin_paths << std::endl;
#endif // !K3D_STATIC_BUILD
	k3d::log() << info << "shader cache path: " << g_shader_cache_path.native_console_string() << std::endl;
	k3d::log() << info << "share path: " << g_share_path.native_console_string() << std::endl;
	k3d::log() << info << "user interface: " << g_user_interface_path.native_console_string() << std::endl;
	k3d::log() << info << "home directory: " << k3d::system::get_home_directory().native_console_string() << std::endl;
	k3d::log() << info << "temp directory: " << k3d::system::get_temp_directory().native_console_string() << std::endl;

#ifdef K3D_HAVE_NLS
	k3d::log() << info << "locale directory: " << K3D_LOCALE << std::endl;
#endif // K3D_HAVE_NLS

	// The options file must be specified and must exist ...
	if(g_options_path.empty())
	{
		handle_error("Options path must be specified using --options <path>.", Quit, Error);
		return;
	}

	k3d::filesystem::create_directories(g_options_path.branch_path());
	if(!k3d::filesystem::exists(g_options_path))
	{
		k3d::filesystem::ofstream stream(g_options_path);
		stream << k3d::xml::declaration() << k3d::xml::element("k3dml") << std::endl;
	}

	if(!k3d::filesystem::exists(g_options_path))
	{
		handle_error("Options path [" + g_options_path.native_console_string() + "] does not exist and could not be created.", Quit, Error);
		return;
	}

	// Every specified plugin path must exist ...
	const k3d::system::paths_t plugin_paths = k3d::system::decompose_path_list(g_plugin_paths);
	for(k3d::system::paths_t::const_iterator plugin_path = plugin_paths.begin(); plugin_path != plugin_paths.end(); ++plugin_path)
	{
		if(!k3d::filesystem::exists(*plugin_path))
		{
			handle_error("Plugin path [" + plugin_path->native_console_string() + "] does not exist.", Quit, Error);
			return;
		}
	}

	// The shader cache path must be specified, and must already exist or be successfully created ...
	if(g_shader_cache_path.empty())
	{
		handle_error("Shader cache path must be specified using --shadercache <path>.", Quit, Error);
		return;
	}

	k3d::filesystem::create_directories(g_shader_cache_path);
	if(!k3d::filesystem::exists(g_shader_cache_path))
	{
		handle_error("Shader cache path [" + g_shader_cache_path.native_console_string() + "] does not exist and could not be created.", Quit, Error);
		return;
	}

	// The share path must exist ...
	if(!k3d::filesystem::exists(g_share_path))
	{
		handle_error("Share path [" + g_share_path.native_console_string() + "] does not exist.", Quit, Error);
		return;
	}

	// Test for an OpenGL implementation ...
	if(!k3d::gl::exists(k3d::log()))
	{
		handle_error("Could not connect to an OpenGL server.", Quit, Error);
		return;
	}
}

/////////////////////////////////////////////////////////////////////////////
// create_user_interface

/// Instantiates the (optional) user interface plugin
void create_user_interface(k3d::plugin_factory_collection& Plugins, bool& Quit, bool& Error)
{
#ifdef K3D_STATIC_BUILD

	const std::string module_name = "NGUI";
	Plugins.bind_module(module_name, libk3dngui_register_k3d_module, libk3dngui_register_k3d_plugins);

#else // K3D_STATIC_BUILD

	const std::string module_name = g_user_interface_path.native_utf8_string().raw();
	if(!g_user_interface_path.empty() && !k3d::filesystem::exists(g_user_interface_path))
	{
		handle_error("UI plugin module [" + module_name + "] does not exist", Quit, Error);
		return;
	}
	Plugins.load_module(g_user_interface_path, k3d::plugin_factory_collection::IGNORE_PROXIES);

#endif // !K3D_STATIC_BUILD

	if(Plugins.factories().empty())
	{
		handle_error("UI plugin module [" + module_name + "] does not contain any K-3D plugins", Quit, Error);
		return;
	}
	if(Plugins.factories().size() > 1)
	{
		handle_error("UI plugin module [" + module_name + "] contains more than one K-3D plugin", Quit, Error);
		return;
	}

	k3d::iunknown* const ui_plugin = k3d::create_plugin(**Plugins.factories().begin());
	if(!ui_plugin)
	{
		handle_error("UI plugin module [" + module_name + "] failed to instantiate a K-3D plugin", Quit, Error);
		return;
	}

	g_user_interface = dynamic_cast<k3d::iuser_interface_plugin*>(ui_plugin);
	if(!g_user_interface)
	{
		delete dynamic_cast<k3d::ideletable*>(ui_plugin);
		handle_error("UI plugin module [" + module_name + "] does not contain a user interface plugin", Quit, Error);
		return;
	}

	if(!dynamic_cast<k3d::iuser_interface*>(ui_plugin))
	{
		g_user_interface = 0;
		delete dynamic_cast<k3d::ideletable*>(ui_plugin);
		handle_error("UI plugin module [" + module_name + "] does not implement required interfaces", Quit, Error);
		return;
	}
}

/////////////////////////////////////////////////////////////////////////////
// load_modules

/// Loads (statically- or dynamically-linked) plugin modules
void load_modules(k3d::plugin_factory_collection& Plugins, bool& Quit, bool& Error)
{
#ifdef K3D_STATIC_BUILD

	Plugins.bind_module("Bitmap", libk3dbitmap_register_k3d_module, libk3dbitmap_register_k3d_plugins);
	Plugins.bind_module("Blobbies", libk3dblobbies_register_k3d_module, libk3dblobbies_register_k3d_plugins);
	Plugins.bind_module("Channels", libk3dchannels_register_k3d_module, libk3dchannels_register_k3d_plugins);
	Plugins.bind_module("Core", libk3dcore_register_k3d_module, libk3dcore_register_k3d_plugins);
	Plugins.bind_module("Deformation", libk3ddeformation_register_k3d_module, libk3ddeformation_register_k3d_plugins);
	Plugins.bind_module("Freetype2", libk3dfreetype2_register_k3d_module, libk3dfreetype2_register_k3d_plugins);
	Plugins.bind_module("Geometry", libk3dgeometry_register_k3d_module, libk3dgeometry_register_k3d_plugins);
	Plugins.bind_module("JPEG", libk3djpeg_register_k3d_module, libk3djpeg_register_k3d_plugins);
	Plugins.bind_module("K3DScript", libk3dscript_register_k3d_module, libk3dscript_register_k3d_plugins);
	Plugins.bind_module("LSystem", libk3dlsystem_register_k3d_module, libk3dlsystem_register_k3d_plugins);
	Plugins.bind_module("Mesh", libk3dmesh_register_k3d_module, libk3dmesh_register_k3d_plugins);
	Plugins.bind_module("NURBS", libk3dnurbs_register_k3d_module, libk3dnurbs_register_k3d_plugins);
	Plugins.bind_module("PNG", libk3dpng_register_k3d_module, libk3dpng_register_k3d_plugins);
	Plugins.bind_module("Python", libk3dpython_register_k3d_module, libk3dpython_register_k3d_plugins);
	Plugins.bind_module("RenderMan", libk3drenderman_register_k3d_module, libk3drenderman_register_k3d_plugins);
	Plugins.bind_module("Scripting", libk3dscripting_register_k3d_module, libk3dscripting_register_k3d_plugins);
	Plugins.bind_module("TIFF", libk3dtiff_register_k3d_module, libk3dtiff_register_k3d_plugins);
	Plugins.bind_module("Time", libk3dtime_register_k3d_module, libk3dtime_register_k3d_plugins);
	Plugins.bind_module("Viewport", libk3dviewport_register_k3d_module, libk3dviewport_register_k3d_plugins);
	Plugins.bind_module("Windows", libk3dwindows_register_k3d_module, libk3dwindows_register_k3d_plugins);

#else // K3D_STATIC_BUILD

	Plugins.load_modules(g_plugin_paths, true, k3d::plugin_factory_collection::LOAD_PROXIES);

#endif // !K3D_STATIC_BUILD
}

/////////////////////////////////////////////////////////////////////////////
// parse_command_line_3

/// Handles any command-line options that simulate user input (e.g. running a script / tutorial)
string_array parse_command_line_3(const string_array Options, bool& Quit, bool& Error, k3d::iapplication& Application)
{
	// Keep track of "unused" options
	string_array unused;

	// For each command-line argument ...
	for(string_array::const_iterator option = Options.begin(); option != Options.end(); ++option)
	{
		if((*option) == "--new" || (*option) == "-n")
		{
			k3d::idocument* const document = Application.create_document();
			if(!document)
			{
				handle_error("Error creating document", Quit, Error);
				return string_array();
			}
		}
		else if((*option) == "--open" || (*option) == "-o")
		{
			++option;
			if(option == Options.end())
			{
				handle_error("You must supply a K-3D document path with the --open option!", Quit, Error);
				return string_array();
			}

			k3d::auto_ptr<k3d::idocument_read_format> filter(k3d::create_plugin<k3d::idocument_read_format>(k3d::classes::DocumentReader()));
			if(!filter.get())
			{
				handle_error("Document reader plugin not installed", Quit, Error);
				return string_array();
			}

			k3d::idocument* const document = Application.create_document();
			if(!document)
			{
				handle_error("Error creating document", Quit, Error);
				return string_array();
			}

			const k3d::filesystem::path document_path = k3d::filesystem::native_path(k3d::ustring::from_utf8(*option));

			if(!filter->read_file(*document, document_path))
			{
				handle_error("Error reading document", Quit, Error);
				return string_array();
			}

			k3d::set_value(document->path(), document_path);
			k3d::set_value(document->title(), document_path.leaf());
		}
		else if((*option) == "--script")
		{
			++option;
			if(option == Options.end())
			{
				handle_error("You must supply a K-3D script path with the --script option!", Quit, Error);
				return string_array();
			}

			bool recognized = false;
			bool executed = false;
			std::string scriptname(*option);

			if(scriptname == "-")
			{
				k3d::log() << info << "Running script [STDIN]" << std::endl;
				scriptname = "STDIN";
				k3d::iscript_engine::context_t context;
				k3d::script::execute(k3d::script::code(std::cin), scriptname, context, recognized, executed);
			}
			else
			{
				k3d::log() << info << "Running script [" << scriptname << "]" << std::endl;
				k3d::filesystem::igzstream file(k3d::filesystem::native_path(k3d::ustring::from_utf8(scriptname)));
				k3d::iscript_engine::context_t context;
				k3d::script::execute(k3d::script::code(file), scriptname, context, recognized, executed);
			}

			if(!recognized)
			{
				k3d::log() << error << "Couldn't recognize scripting language for script [ " << scriptname << " ]" << std::endl;
				Quit = true;
				Error = true;
				return string_array();
			}

			if(!executed)
			{
				k3d::log() << error << "Error executing script [ " << scriptname << " ]" << std::endl;
				Quit = true;
				Error = true;
				return string_array();
			}
		}
		else if((*option) == "--exit")
		{
			Quit = true;
			return string_array();
		}
		else
		{
			unused.push_back(*option);
		}
	}

	return unused;
}

void check_unused_options(const string_array Options, bool& Quit, bool& Error)
{
	// If there aren't any leftover options, we're done ...
	if(Options.empty())
		return;

	k3d::log() << warning << "The following unknown command-line options will be ignored: ";
	std::copy(Options.begin(), Options.end(), std::ostream_iterator<std::string>(k3d::log(), " "));
	k3d::log() << std::endl;

	Quit = false;
	Error = false;
}

} // namespace

int main(int argc, char* argv[])
{
	const std::string program_name = k3d::filesystem::native_path(k3d::ustring::from_utf8(argv[0])).leaf().raw();

#ifdef K3D_HAVE_NLS

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, K3D_LOCALE);
	bind_textdomain_codeset(PACKAGE, "UTF-8");
	textdomain(PACKAGE);

#endif

	try
	{
		bool quit = false;
		bool error = false;

		// Set default values for all user-configurable options ...
		set_default_options(quit, error);
		if(quit)
			return error ? 1 : 0;

		// Put command-line options into an easier form for parsing ...
		string_array options(&argv[1], &argv[argc]);

		// Setup our logging options before going any further ...
		options = parse_command_line_logging_options(program_name, options, quit, error);
		if(quit)
			return error ? 1 : 0;

		// Look for command-line options that will configure our subsequent behavior ...
		options = parse_command_line_2(options, quit, error);
		if(quit)
			return error ? 1 : 0;

		// Make sure we have all resources required to run ...
		check_dependencies(quit, error);
		if(quit)
			return error ? 1 : 0;

		// Set the shader cache path ...
		k3d::set_shader_cache_path(g_shader_cache_path);

		// Set the share path ...
		k3d::set_share_path(g_share_path);

		// Load user options ...
		k3d::options::file_storage user_options(g_options_path);
		k3d::options::set_storage(user_options);

		// Handle creation of the required user interface plugin ...
		k3d::plugin_factory_collection ui_plugins;
		ui_plugins.connect_message_signal(sigc::ptr_fun(startup_message_handler));
		create_user_interface(ui_plugins, quit, error);
		if(quit)
			return error ? 1 : 0;
		return_val_if_fail(g_user_interface, 1);

		// Register our user interface ...
		k3d::set_user_interface(*dynamic_cast<k3d::iuser_interface*>(g_user_interface));

		// Make sure the UI gets cleaned-up properly ...
		std::auto_ptr<k3d::ideletable> user_interface_storage(dynamic_cast<k3d::ideletable*>(g_user_interface));

		// Give the UI a chance to handle command-line options ...
		options = g_user_interface->parse_command_line(options, quit, error);
		if(quit)
			return error ? 1 : 0;

		// Load plugins ...
		k3d::plugin_factory_collection plugins;
		plugins.connect_message_signal(sigc::ptr_fun(startup_message_handler));
		load_modules(plugins, quit, error);
		if(quit)
			return error ? 1 : 0;

		// Setup a render farm ...
		k3d::render_farm_implementation render_farm(g_options_path);
		k3d::set_render_farm(render_farm);

		// Create the main application object ...
		k3d::application_implementation application(plugins);

		// We want to be notified when the user requests a shutdown ...
		application.exit_signal().connect(sigc::ptr_fun(exit_request_handler));

		// Register it with the library as the global application object ...
		k3d::register_application(application.interface());

		// Switch the UI to its "normal" (post-startup) layout ...
		startup_message_handler(_("Starting user interface"));
		g_user_interface->display_user_interface();

		// Parse command-line options that control program execution ...
		options = parse_command_line_3(options, quit, error, application.interface());
		if(quit)
			return error ? 1 : 0;

		// Check for "unused" command-line options
		check_unused_options(options, quit, error);
		if(quit)
			return error ? 1 : 0;

		// Main UI event-loop ...
		g_user_interface->start_event_loop();

		return 0;
	}
	catch(std::exception& e)
	{
		std::string message = "Caught exception [";
			message += typeid(e).name();
			message += "] - ";
			message += e.what();

		k3d::log() << critical << message << std::endl;

#ifdef K3D_PLATFORM_WIN32
		MessageBox(0, message.c_str(), _("K-3D Fatal Error"), MB_OK | MB_ICONSTOP);
#endif // K3D_PLATFORM_WIN32

		return 1;
	}
	catch(...)
	{
		const std::string message = "Caught unknown exception.";

		k3d::log() << critical << message << std::endl;

#ifdef K3D_PLATFORM_WIN32
		MessageBox(0, message.c_str(), _("K-3D Fatal Error"), MB_OK | MB_ICONSTOP);
#endif // K3D_PLATFORM_WIN32

		return 1;
	}
}
