/*
 * Created on 09.07.2008
 */
package de.abelssoft.wordtools.jWordSplitter.impl;

import java.io.IOException;
import java.util.Collection;

import junit.framework.TestCase;

/**
 * Test cases for the German word splitter.
 * @author Daniel Naber
 */
public class GermanWordSplitterTest extends TestCase {
  
  private static final String TESTFILE = "test/test-de.txt";

  private GermanWordSplitter splitter;

  public void test() throws IOException {
    splitter = new GermanWordSplitter(true, TESTFILE);
    splitter.setStrictMode(true);
    expect("[xyz]", "xyz");
    expect("[Verhalten]", "Verhalten");
    expect("[Verhalten, störung]", "Verhaltenstörung");
    expect("[Verhalten, störung]", "Verhaltensstörung");
    expect("[Verhaltenxstörung]", "Verhaltenxstörung");
  }

  public void testWithHyphen() throws IOException {
    splitter = new GermanWordSplitter(true, TESTFILE);
    splitter.setStrictMode(true);
    expect("[Verhalten, Störung]", "Verhaltens-Störung");
    expect("[Sauerstoff, Flasche]", "Sauerstoff-Flasche");
    
    expect("[Sauerstoff-Foobar]", "Sauerstoff-Foobar");
    splitter.setStrictMode(false);
    expect("[Sauerstoff, Foobar]", "Sauerstoff-Foobar");
    // no term known -> not split at all despite hyphen:
    expect("[Blahbar-Foobar]", "Blahbar-Foobar");
  }

  public void testWrongCase() throws IOException {
    splitter = new GermanWordSplitter(true, TESTFILE);
    splitter.setStrictMode(true);
    // words with wrong case are also split up:
    expect("[VERHALTEN, STÖRUNG]", "VERHALTENSSTÖRUNG");
    expect("[verhalten, störung]", "verhaltensstörung");
  }
  
  public void testWithConnectionCharacter() throws IOException {
    splitter = new GermanWordSplitter(false, TESTFILE);
    splitter.setStrictMode(true);
    expect("[xyz]", "xyz");
    expect("[Verhalten]", "Verhalten");
    expect("[Verhalten, störung]", "Verhaltenstörung");
    expect("[Verhaltens, störung]", "Verhaltensstörung");   // now with "s"
    expect("[Verhaltenxstörung]", "Verhaltenxstörung");
    expect("[Verhaltenfoobar]", "Verhaltenfoobar");
  }

  public void testTooShortWords() throws IOException {
    splitter = new GermanWordSplitter(false, TESTFILE);
    splitter.setStrictMode(true);
    // too short to be split (default min word length: 4)
    expect("[Verhaltenei]", "Verhaltenei");
    expect("[Eiverhalten]", "Eiverhalten");
  }
  
  public void testNonStrictMode() throws IOException {
    splitter = new GermanWordSplitter(false, TESTFILE);
    splitter.setStrictMode(false);
    expect("[xyz]", "xyz");
    expect("[Verhalten]", "Verhalten");
    expect("[Verhalten, störung]", "Verhaltenstörung");
    expect("[Verhaltens, störung]", "Verhaltensstörung");
    // now split because of non-strict mode:
    expect("[Verhaltenx, störung]", "Verhaltenxstörung");
    expect("[xyz, störung]", "xyzstörung");
    expect("[Verhalten, xyz]", "Verhaltenxyz");
    expect("[Verhaltens, xyz]", "Verhaltensxyz");
    expect("[Verhaltens, xyz]", "Verhaltensxyz");
  }

  public void testLongerWords() throws IOException {
    splitter = new GermanWordSplitter(false, TESTFILE);
    splitter.setStrictMode(true);
    expect("[Sauerstoff, flaschen, störungs, verhalten]", "Sauerstoffflaschenstörungsverhalten");
    expect("[Sauerstoff, sauerstoff]", "Sauerstoffsauerstoff");
    expect("[Sauerstoff, sauerstoff, sauerstoff]", "Sauerstoffsauerstoffsauerstoff");
    expect("[Störungs, störung]", "Störungsstörung");
    expect("[Störungs, störungs, störung]", "Störungsstörungsstörung");
  }
  
  // TODO: Sauerstoffflasche vs Sauerstofflasche; upper vs lower case

  public void testNoCompounds() throws IOException {
    splitter = new GermanWordSplitter(false, TESTFILE);
    splitter.setStrictMode(true);
    // Kotflügel, Kot, and Flügel in the dictionary so don't split: 
    expect("[Kotflügel]", "Kotflügel");
  }
  
  public void testSpecialCases() throws IOException {
    splitter = new GermanWordSplitter(false, TESTFILE);
    expect("[]", null);
    expect("[]", "");
    expect("[]", "\t");
    expect("[]", "   ");
  }
  
  private void expect(String exptected, String input) {
    Collection<String> result = splitter.splitWord(input);
    assertEquals(exptected, result.toString());
  }

}
