"""Tests for launching a new server customised for juju using Nova"""


from twisted.internet.defer import succeed

from juju import errors
from juju.lib import serializer
from juju.lib.mocker import MATCH
from juju.lib.testing import TestCase


from juju.providers.openstack.tests import MockedProvider
from juju.providers.openstack.launch import NovaLaunchMachine


class MockedLaunchProvider(MockedProvider):

    def launch(self, machine_id, master=False, constraints=None):
        if constraints is None:
            constraints = self.constraint_set.load({'instance-type': None})
        else:
            constraints = self.constraint_set.parse(constraints)
        details = {
            'machine-id': machine_id,
            'constraints': constraints
            }
        return NovaLaunchMachine.launch(self, details, master)

    def expect_launch_setup(self, machine_id):
        self.port_manager.ensure_groups(machine_id)
        self.mocker.result(succeed(["juju-x", "juju-y"]))
        self.nova.list_flavor_details()
        self.mocker.result(succeed(self.default_flavors))

    def expect_run_server(self, machine_id, cc_match, response, flavor_id=1):
        self.nova.run_server(
            name="juju testing instance " + machine_id,
            image_id=42,
            flavor_id=flavor_id,
            security_group_names=["juju-x", "juju-y"],
            user_data=MATCH(cc_match),
            )
        self.mocker.result(succeed(response))

    def expect_available_floating_ip(self, server_id):
        self.nova.list_floating_ips()
        self.mocker.result(succeed([
            {'instance_id': None, 'ip': "198.162.1.0"},
            ]))
        self.nova.add_floating_ip(server_id, "198.162.1.0")
        self.mocker.result(succeed(None))


class _CloudConfigMatcher(object):
    """Hack around testcase and provider seperation to check cloud-config

    Really provider launch tests have no business caring about the specifics
    of the cloud-config format, but the contents varies depending on launch
    parameters.
    """

    def __init__(self, testcase, machine_id, provider, is_master):
        self._case = testcase
        self.machine_id = machine_id
        self.provider = provider
        self.is_master = is_master

    def match(self, user_data):
        """Check contents of user_data but always match if assertions pass"""
        self._case.assertEqual("#cloud-config", user_data.split("\n", 1)[0])
        cc = serializer.load(user_data)
        if self.is_master:
            zookeeper_hosts = "localhost:2181"
        else:
            zookeeper_hosts = "master.invalid:2181"
        self._case.assertEqual({
                'juju-provider-type': self.provider.provider_type,
                'juju-zookeeper-hosts': zookeeper_hosts,
                'machine-id': self.machine_id},
            cc['machine-data'])
        self._case.assertEqual([self.provider.config["authorized-keys"]],
            cc["ssh_authorized_keys"])
        if self.is_master:
            self._match_master_runcmd(cc["runcmd"])
        return True

    def _match_master_runcmd(self, runcmd):
        id_url = "".join([self.provider.api_url, "swift", "/juju_master_id"])
        if self.provider.config["ssl-hostname-verification"]:
            instance_arg = "$(curl %s)" % id_url
        else:
            instance_arg = "$(curl -k %s)" % id_url
        for cmd in runcmd:
            if cmd.startswith("juju-admin initialize "):
                self._case.assertIn(" --provider-type=openstack", cmd)
                self._case.assertIn(" --instance-id=" + instance_arg, cmd)
                break
        else:
            self._case.fail("Missing juju-admin initialize: " + repr(runcmd))


class NovaLaunchMachineTests(TestCase):

    def test_launch_requires_default_image_id(self):
        config = dict(MockedLaunchProvider.default_config)
        del config['default-image-id']
        provider = MockedLaunchProvider(self.mocker, config)
        provider.expect_zookeeper_machines(1000)
        self.mocker.replay()
        deferred = provider.launch("1")
        return self.assertFailure(deferred, errors.ProviderError)

    def get_cc_matcher(self, machine_id, provider, is_master=False):
        return _CloudConfigMatcher(self, machine_id, provider, is_master).match

    def test_start_machine_with_constraints(self):
        provider = MockedLaunchProvider(self.mocker)
        provider.expect_zookeeper_machines(1000, "master.invalid")
        provider.expect_launch_setup("1")
        provider.expect_run_server(
            "1",
            self.get_cc_matcher("1", provider),
            response={
                'id': 1001,
                'addresses': {'public': []},
                },
            flavor_id=2)
        self.mocker.replay()
        return provider.launch("1", constraints=["cpu=2", "mem=3G"])

    def test_start_machine(self):
        provider = MockedLaunchProvider(self.mocker)
        provider.expect_zookeeper_machines(1000, "master.invalid")
        provider.expect_launch_setup("1")
        provider.expect_run_server("1",
            self.get_cc_matcher("1", provider),
            response={
                'id': 1001,
                'addresses': {'public': []},
            })
        self.mocker.replay()
        return provider.launch("1")

    def test_start_machine_delay(self):
        provider = MockedLaunchProvider(self.mocker)
        provider.config["use-floating-ip"] = True
        provider.expect_zookeeper_machines(1000, "master.invalid")
        provider.expect_launch_setup("1")
        provider.expect_run_server("1",
            self.get_cc_matcher("1", provider),
            response={
                'id': 1001,
            })
        provider.nova.get_server(1001)
        self.mocker.result(succeed({
            'id': 1001,
            'addresses': {'public': []},
            }))
        provider.expect_available_floating_ip(1001)
        self.mocker.result(succeed(None))
        self.mocker.replay()
        self.patch(NovaLaunchMachine, "_DELAY_FOR_ADDRESSES", 0)
        return provider.launch("1")

    def _start_master_machine_test(self, provider):
        provider.expect_swift_public_object_url("juju_master_id")
        provider.expect_launch_setup("0")
        provider.expect_run_server("0",
            self.get_cc_matcher("0", provider, is_master=True),
            response={
                'id': 1000,
                'addresses': {'public': []},
            })
        provider.expect_swift_put("juju_master_id", "1000")
        provider.provider_actions.save_state({'zookeeper-instances': [1000]})
        self.mocker.result(succeed(None))
        self.mocker.replay()
        return provider.launch("0", master=True)

    def test_start_machine_master(self):
        provider = MockedLaunchProvider(self.mocker)
        return self._start_master_machine_test(provider)

    def test_start_machine_master_no_certs(self):
        config = dict(MockedLaunchProvider.default_config)
        config['ssl-hostname-verification'] = False
        provider = MockedLaunchProvider(self.mocker, config)
        return self._start_master_machine_test(provider)
