"""Tests for OpenStack API twisted client"""

import json

from twisted.internet import defer, reactor
from twisted.python.failure import Failure
from twisted.web import http_headers

from juju import errors
from juju.lib import mocker, testing
from juju.providers.openstack import client, credentials


class StubClient(client._OpenStackClient):

    def __init__(self):
        self.url = "http://testing.invalid"

    make_url = client._OpenStackClient._make_url


class TestMakeUrl(testing.TestCase):

    def setUp(self):
        self.client = StubClient()
        self.client.services = {
            "compute": self.client.url + "/nova",
            "object-store": self.client.url + "/swift",
            }

    def test_list_str(self):
        self.assertEqual("http://testing.invalid/nova/servers",
            self.client.make_url("compute", ["servers"]))
        self.assertEqual("http://testing.invalid/swift/container/object",
            self.client.make_url("object-store", ["container", "object"]))

    def test_list_int(self):
        self.assertEqual("http://testing.invalid/nova/servers/1000",
            self.client.make_url("compute", ["servers", 1000]))
        self.assertEqual("http://testing.invalid/nova/servers/1000/detail",
            self.client.make_url("compute", ["servers", 1000, "detail"]))

    def test_list_unicode(self):
        url = self.client.make_url("object-store", ["container", u"\xa7"])
        self.assertIsInstance(url, str)
        self.assertEqual("http://testing.invalid/swift/container/%C2%A7", url)

    def test_str(self):
        self.assertEqual("http://testing.invalid/nova/servers",
            self.client.make_url("compute", "servers"))
        self.assertEqual("http://testing.invalid/swift/container/object",
            self.client.make_url("object-store", "container/object"))

    def test_trailing_slash(self):
        self.client.services["object-store"] += "/"
        self.assertEqual("http://testing.invalid/nova/container",
            self.client.make_url("compute", "container"))
        self.assertEqual("http://testing.invalid/nova/container/object",
            self.client.make_url("compute", ["container", "object"]))


class FakeResponse(object):
    """Bare minimum needed to look like a twisted http response"""

    def __init__(self, code, headers, body=None):
        self.code = code
        self.headers = headers
        if body is None:
            self.length = 0
        else:
            self.length = len(body)
        self.body = body

    def deliverBody(self, reader):
        reader.connectionMade()
        reader.dataReceived(self.body)
        reader.connectionLost(client.ResponseDone())


class ClientTests(testing.TestCase):
    """Testing of low level client behaviour

    Rough temporary tests until client rearrangements make this easier.
    """

    class Checker(object):
        """Standin for cert checker that exists regardless of txaws"""

    class SSLError(Exception):
        """Standin for ssl exception that exists regardless of OpenSSL"""

    def setUp(self):
        super(ClientTests, self).setUp()
        self.patch(client, "SSLError", self.SSLError)
        self.patch(client, "WebVerifyingContextFactory", self.Checker)
        self.mock_agent = self.mocker.replace("twisted.web.client.Agent",
            passthrough=False)
        self.mocker.order()

    def get_credentials(self, config):
        return credentials.OpenStackCredentials(config)

    def is_checker(self, obj):
        return isinstance(obj, self.Checker)

    def make_client_legacy(self):
        config = {
            "auth-url": "https://testing.invalid",
            "auth-mode": "legacy",
            "username": "user",
            "access-key": "key",
            }
        osc = client._OpenStackClient(self.get_credentials(config), True)
        self.mock_agent(reactor, contextFactory=mocker.MATCH(self.is_checker))
        self.mocker.result(self.mock_agent)
        return config, osc

    def make_client_userpass(self):
        config = {
            "auth-url": "https://testing.invalid/v2.0/",
            "username": "user",
            "password": "pass",
            "project-name": "project",
            }
        osc = client._OpenStackClient(self.get_credentials(config), True)
        self.mock_agent(reactor, contextFactory=mocker.MATCH(self.is_checker))
        self.mocker.result(self.mock_agent)
        return config, osc

    @defer.inlineCallbacks
    def test_auth_legacy(self):
        config, osc = self.make_client_legacy()
        # TODO: check headers for correct auth values
        self.mock_agent.request("GET", config["auth-url"], mocker.ANY, None)
        response = FakeResponse(204, http_headers.Headers({
            "X-Server-Management-Url": ["http://testing.invalid/compute"],
            "X-Auth-Token": ["tok"],
            }))
        self.mocker.result(defer.succeed(response))
        self.mocker.replay()
        log = self.capture_logging()
        yield osc.authenticate()
        self.assertIsInstance(osc.token, str)
        self.assertEqual("tok", osc.token)
        self.assertEqual("http://testing.invalid/compute/path",
            osc._make_url("compute", ["path"]))
        self.assertIn("compute service not using secure", log.getvalue())

    @defer.inlineCallbacks
    def test_auth_userpass(self):
        config, osc = self.make_client_userpass()
        self.mock_agent.request("POST", "https://testing.invalid/v2.0/tokens",
            mocker.ANY, mocker.ANY)
        response = FakeResponse(200, http_headers.Headers({
            "Content-Type": ["application/json"],
            }),
            json.dumps({'access': {
                'token': {'id': "tok", 'expires': "shortly"},
                'serviceCatalog': [
                    {
                        'type': "compute",
                        'endpoints': [
                            {'publicURL': "http://testing.invalid/compute"},
                        ],
                    },
                    {
                        'type': "object-store",
                        'endpoints': [
                            {'publicURL': "http://testing.invalid/objstore"},
                        ],
                    },
                ],
            }}))
        self.mocker.result(defer.succeed(response))
        self.mocker.replay()
        log = self.capture_logging()
        yield osc.authenticate()
        self.assertIsInstance(osc.token, str)
        self.assertEqual("tok", osc.token)
        self.assertEqual("http://testing.invalid/compute/path",
            osc._make_url("compute", ["path"]))
        self.assertEqual("http://testing.invalid/objstore/path",
            osc._make_url("object-store", ["path"]))
        self.assertIn("compute service not using secure", log.getvalue())
        self.assertIn("object-store service not using secure", log.getvalue())

    def test_cert_failure(self):
        config, osc = self.make_client_legacy()
        self.mock_agent.request("GET", config["auth-url"], mocker.ANY, None)
        response = FakeResponse(204, http_headers.Headers({
            "X-Server-Management-Url": ["http://testing.invalid/compute"],
            "X-Auth-Token": ["tok"],
            }))
        self.mocker.result(defer.fail(client.ResponseFailed([
            Failure(self.SSLError()),
            ])))
        self.mocker.replay()
        deferred = osc.authenticate()
        return self.assertFailure(deferred, errors.SSLVerificationError)


class TestReauthentication(testing.TestCase):

    @defer.inlineCallbacks
    def test_detect_token_expiration(self):
        """Verify expired tokens reset is_authenticated to False on client"""

        # Note that the swift client uses the same code path as the
        # nova client in terms of token expiration management

        mock_request = self.mocker.mock()
        self.patch(client, "request", mock_request)

        # 1. First authenticate
        mock_request("GET", "https://example.com", check_certs=True,
                     extra_headers=[("X-Auth-User", "my-user"),
                                    ("X-Auth-Key", "my-key")])
        self.mocker.result(defer.succeed(
                (FakeResponse(204, http_headers.Headers({
                                "X-Server-Management-Url":
                                    ["https://example.com/compute"],
                                "X-Auth-Token": ["first-token"],
                                })),
                 None)))

        # 2. Succeed on the first pass through
        mock_request("GET", "https://example.com/compute/os-floating-ips",
                     [("X-Auth-Token", "first-token")], None, True)
        self.mocker.result(defer.succeed(
                (FakeResponse(200, http_headers.Headers({
                                "Content-Type": ["application/json"]
                                })),
                 json.dumps({"floating_ips": ["ip-1", "ip-2", "ip-3"]}))))

        # 3. Then fail upon the next request
        mock_request("GET", "https://example.com/compute/os-floating-ips",
                     [("X-Auth-Token", "first-token")], None, True)
        self.mocker.result(defer.succeed(
                (FakeResponse(401, http_headers.Headers()),
                 None)))

        # 4. This forces the client to authenticate again
        mock_request("GET", "https://example.com", check_certs=True,
                     extra_headers=[("X-Auth-User", "my-user"),
                                    ("X-Auth-Key", "my-key")])
        self.mocker.result(defer.succeed(
                    (FakeResponse(204, http_headers.Headers({
                                    "X-Server-Management-Url":
                                         ["https://example.com/compute"],
                                    "X-Auth-Token": ["second-token"],
                                    })),
                     None)))

        # 5. And finally it gets the desired result upon the last request
        mock_request("GET", "https://example.com/compute/os-floating-ips",
                     [("X-Auth-Token", "second-token")], None, True)
        body = json.dumps({"floating_ips": ["another-ip-1", "another-ip-3"]})
        response = FakeResponse(
            200,
            http_headers.Headers({
                    "Content-Type": ["application/json"]
                    }))
        self.mocker.result(defer.succeed((response, body)))

        self.mocker.replay()

        # Exercise mocks and verify assertions
        osc = client._OpenStackClient(
            credentials.OpenStackCredentials({
                    "auth-url": "https://example.com",
                    "auth-mode": "legacy",
                    "username": "my-user",
                    "access-key": "my-key",
                    }), True)
        nova_client = client._NovaClient(osc)
        ips = yield nova_client.list_floating_ips()
        self.assertTrue(osc.is_authenticated())
        self.assertEqual("first-token", osc.token)
        self.assertEqual(ips, ["ip-1", "ip-2", "ip-3"])

        e = yield self.assertFailure(nova_client.list_floating_ips(),
                                     errors.ProviderInteractionError)
        self.assertEqual(str(e), "Need to reauthenticate by retrying")
        self.assertFalse(osc.is_authenticated())

        # Now retry: the token will change and the mock is also
        # returning new data
        ips = yield nova_client.list_floating_ips()
        self.assertTrue(osc.is_authenticated())
        self.assertEqual("second-token", osc.token)
        self.assertEqual(ips, ["another-ip-1", "another-ip-3"])


class TestPlan(testing.TestCase):
    """Ideas for tests needed"""

    # auth request without auth
    # get bytes, content-length 0, return ""
    # get bytes, not ResponseDone, raise wrapped in ProviderError (with any bytes?)
    # get bytes, type json, return bytes
    # get json, content length 0, raise ProviderError
    # get json, bad header (several forms), raise ProviderError (with bytes)
    # get json, not ResponseDone, raise wrapped in ProviderError
    #    (with any bytes?)
    # get json, undecodable, raise wrapped in ProviderError with bytes
    # get json, mismatching root, raise ProviderError with bytes or json?
    # wrong code, no json header, raise ProviderError with bytes
    # wrong code, not ResponseDone, raise ProviderError from code
    #    with any bytes
    # wrong code, undecodable, raise ProviderError from code with bytes
    # wrong code, has mystery root, raise ProviderError from code with bytes or json?
    # wrong code, has good root, no message
    # wrong code, has good root, no code
    # wrong code, has good root, differing code
    # wrong code, has good root, message, and matching code
