"""Provider interface implementation for OpenStack backend

Much of the logic is implemented in sibling modules, but the overall model is
exposed here.

Still in need of work here:
* Implement constraints using the Nova flavors api. This will always mean an
  api call rather than hard coding values as is done with EC2. Things like
  memory and cpu count are broadly equivalent, but there's no guarentee what
  details are exposed and ranking by price will generally not be an option.
"""

import logging

from twisted.internet.defer import inlineCallbacks, returnValue

from juju import errors
from juju.lib.twistutils import gather_results
from juju.lib.cache import CachedValue
from juju.providers.common.base import MachineProviderBase


from .client import _OpenStackClient, _NovaClient, _SwiftClient
from . import credentials
from .files import FileStorage
from .launch import NovaLaunchMachine
from .machine import (
    NovaProviderMachine, get_server_status, machine_from_instance
    )
from .ports import NovaPortManager


log = logging.getLogger("juju.openstack")


class MachineProvider(MachineProviderBase):
    """MachineProvider for use in an OpenStack environment"""

    Credentials = credentials.OpenStackCredentials

    def __init__(self, environment_name, config):
        super(MachineProvider, self).__init__(environment_name, config)
        self.credentials = self.Credentials.from_environment(config)
        self._check_certs = self.config.get("ssl-hostname-verification", True)
        if not self._check_certs:
            log.warn("Verification of HTTPS certificates is disabled for this"
                " environment.\nSet 'ssl-hostname-verification' to ensure"
                " secure communication.")
        elif not self.credentials.url.startswith("https:"):
            log.warn("OpenStack identity service not using secure transport")
        client = _OpenStackClient(self.credentials, self._check_certs)
        self.nova = _NovaClient(client)
        self.swift = _SwiftClient(client)
        self.port_manager = NovaPortManager(self.nova, environment_name)
        # constraints are good for several hrs
        self._cached_constraint = CachedValue(3600 * 12)

    @property
    def provider_type(self):
        return "openstack"

    def get_serialization_data(self):
        """Get provider configuration suitable for serialization.

        Also fills in credential information that may have earlier been
        extracted from the environment.
        """
        data = super(MachineProvider, self).get_serialization_data()
        self.credentials.set_config_defaults(data)
        return data

    def get_file_storage(self):
        """Retrieve a Swift-backed :class:`FileStorage`."""
        return FileStorage(self.swift, self.config["control-bucket"])

    @inlineCallbacks
    def get_constraint_set(self):
        """Get the provider specific machine constraints.
        """
        # Use cached value if available.
        cs = self._cached_constraint.get()
        if cs is not None:
            returnValue(cs)
        cs = yield super(MachineProvider, self).get_constraint_set()

        # Fetch provider defined instance types (just names)
        flavors = yield self.nova.list_flavors()
        flavor_names = [f['name'] for f in flavors]
        cs.register_generics(flavor_names)
        self._cached_constraint.set(cs)
        returnValue(cs)

    def start_machine(self, machine_data, master=False):
        """Start an OpenStack machine.

        :param dict machine_data: desired characteristics of the new machine;
            it must include a "machine-id" key, and may include a "constraints"
            key to specify the underlying OS and hardware.

        :param bool master: if True, machine will initialize the juju admin
            and run a provisioning agent, in addition to running a machine
            agent.
        """
        return NovaLaunchMachine.launch(self, machine_data, master)

    @inlineCallbacks
    def get_machines(self, instance_ids=()):
        """List machines running in the provider.

        :param list instance_ids: ids of instances you want to get. Leave empty
            to list every
            :class:`juju.providers.openstack.machine.NovaProviderMachine` owned
            by this provider.

        :return: a list of
            :class:`juju.providers.openstack.machine.NovaProviderMachine`
            instances
        :rtype: :class:`twisted.internet.defer.Deferred`

        :raises: :exc:`juju.errors.MachinesNotFound`
        """
        if len(instance_ids) == 1:
            try:
                instances = [(yield self.nova.get_server(instance_ids[0]))]
            except errors.ProviderInteractionError, e:
                # XXX: Need to wire up treatment of 404s properly in client
                if True or getattr(e, "kind", None) == "itemNotFound":
                    raise errors.MachinesNotFound(set(instance_ids))
                raise
            instance_ids = frozenset(instance_ids)
        else:
            instances = yield self.nova.list_servers_detail()
            if instance_ids:
                instance_ids = frozenset(instance_ids)
                instances = [instance for instance in instances
                    if instance['id'] in instance_ids]

        # Only want to deal with servers that were created by juju, checking
        # the name begins with the prefix launch uses is good enough.
        name_prefix = "juju %s instance " % (self.environment_name,)
        machines = []
        for instance in instances:
            if (instance['name'].startswith(name_prefix) and
                    get_server_status(instance) in ("running", "pending")):
                machines.append(machine_from_instance(instance))

        if instance_ids:
            # We were asked for a specific list of machines, and if we can't
            # completely fulfil that request we should blow up.
            missing = instance_ids.difference(m.instance_id for m in machines)
            if missing:
                raise errors.MachinesNotFound(missing)

        returnValue(machines)

    @inlineCallbacks
    def _delete_machine(self, machine, full=False):
        server_id = machine.instance_id
        server = yield self.nova.get_server(server_id)
        if not server['name'].startswith(
            "juju %s instance" % self.environment_name):
            raise errors.MachinesNotFound(set([machine.instance_id]))
        yield self.nova.delete_server(server_id)
        returnValue(machine)

    def shutdown_machine(self, machine):
        if not isinstance(machine, NovaProviderMachine):
            raise errors.ProviderError(
                "Need a NovaProviderMachine to shutdown not: %r" % (machine,))
        # EC2 provider re-gets the machine to see if it's still in existance
        # and can be shutdown, instead just handle an error? 404-ish?
        return self._delete_machine(machine)

    @inlineCallbacks
    def destroy_environment(self):
        """Terminate all associated machines and security groups.

        The super defintion of this method terminates each machine in
        the environment; this needs to be augmented here by also
        removing the security group for the environment.

        :rtype: :class:`twisted.internet.defer.Deferred`
        """
        machines = yield self.get_machines()
        deleted_machines = yield gather_results(
            [self._delete_machine(m, True) for m in machines])
        yield self.save_state({})
        returnValue(deleted_machines)

    def shutdown_machines(self, machines):
        """Terminate machines associated with this provider.

        :param machines: machines to shut down
        :type machines: list of
            :class:`juju.providers.openstack.machine.NovaProviderMachine`

        :return: list of terminated
            :class:`juju.providers.openstack.machine.NovaProviderMachine`
            instances
        :rtype: :class:`twisted.internet.defer.Deferred`
        """
        # XXX: need to actually handle errors as non-terminated machines
        # and not include them in the resulting list
        return gather_results(
            [self.shutdown_machine(m) for m in machines], consume_errors=True)

    def open_port(self, machine, machine_id, port, protocol="tcp"):
        """Authorizes `port` using `protocol` on EC2 for `machine`."""
        return self.port_manager.open_port(machine, machine_id, port, protocol)

    def close_port(self, machine, machine_id, port, protocol="tcp"):
        """Revokes `port` using `protocol` on EC2 for `machine`."""
        return self.port_manager.close_port(
            machine, machine_id, port, protocol)

    def get_opened_ports(self, machine, machine_id):
        """Returns a set of open (port, proto) pairs for `machine`."""
        return self.port_manager.get_opened_ports(machine, machine_id)
