/*
JuffEd - An advanced text editor
Copyright 2007-2009 Mikhail Murzin

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License 
version 2 as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "Document.h"

#include "CommandStorage.h"
#include "JuffApp.h"
#include "Log.h"
#include "Functions.h"

#include <QtCore/QFileInfo>
#include <QtCore/QProcess>
#include <QtGui/QAbstractButton>
#include <QtGui/QAction>
#include <QtGui/QMessageBox>

namespace Juff {

Document::Document(const QString& fName) : QObject() {
	if ( fName.isEmpty() ) {
		static int i = 0;
		fileName_ = QString("Noname %1").arg(i++);
	}
	else {
		fileName_ = fName;
	}
	modCheckTimer_ = new QTimer(this);
	connect(modCheckTimer_, SIGNAL(timeout()), SLOT(checkLastModified()));
	if ( !fName.isEmpty() && !isNoname(fName) ) {
		lastModified_ = QFileInfo(fName).lastModified();
		modCheckTimer_->start(1000);
	}
}

Document::~Document() {
}

bool Document::save(const QString&, const QString&, QString&) {
	lastModified_ = QFileInfo(fileName_).lastModified();
	return true;
}

bool Document::isNull() const {
	return false;
}

QString Document::fileName() const {
	return fileName_;
}

void Document::setFileName(const QString& fileName) {
	JUFFENTRY;
	
	QString oldFileName = fileName_;
	
	if ( fileName_ != fileName ) {
		fileName_ = fileName;
		lastModified_ = QFileInfo(fileName).lastModified();
		if ( !modCheckTimer_->isActive() ) {
			modCheckTimer_->start(1000);
		}
		emit fileNameChanged(oldFileName);
	}
}

QString Document::type() const {
	return type_;
}

void Document::checkLastModified() {
	if ( isNull() || !widget() || !widget()->isVisible() )
		return;
	
	QFileInfo fi(fileName_);
	if ( fi.exists() ) {
		if ( fi.lastModified() > lastModified_ ) {
			if ( checkingMutex_.tryLock() ) {
				QString question = tr("The file '%1' was modified by external program.").arg(getDocTitle(fileName_)) + "\n";
				question += tr("What do you want to do?");
				QMessageBox msgBox(QMessageBox::Question, tr("Warning"), question, 
						QMessageBox::Open | QMessageBox::Save | QMessageBox::Cancel, widget());
				QAbstractButton* btn = msgBox.button(QMessageBox::Save);
				if ( btn ) {
					btn->setText(tr("Save current"));
					QAction* saveAct = CommandStorage::instance()->action(Juff::ID_FILE_SAVE);
					if ( saveAct )
						btn->setIcon(saveAct->icon());
				}
				btn = msgBox.button(QMessageBox::Open);
				if ( btn ) {
					btn->setText(tr("Reload from disk"));
					QAction* saveAct = CommandStorage::instance()->action(Juff::ID_FILE_RELOAD);
					if ( saveAct )
						btn->setIcon(saveAct->icon());
				}
				btn = msgBox.button(QMessageBox::Cancel);
				if ( btn ) {
					btn->setText(tr("Ignore"));
				}

				int res = msgBox.exec();
				switch (res) {
					case QMessageBox::Open:
						//	Reload
						reload();
						lastModified_ = QFileInfo(fileName_).lastModified();
						break;
						
					case QMessageBox::Save:
					{
						//	Save
						QString err;
						save(fileName_, charset(), err);
						lastModified_ = QFileInfo(fileName_).lastModified();
					}
						break;
						
					case QMessageBox::Cancel:
						//	Nothing to do. In this case we just make 
						//	local "check date" equal to file's real 
						//	"last modified date" on file system (to 
						//	prevent asking "What to do" again)
						lastModified_ = fi.lastModified();
						break;
					
					default: ;
				}
				checkingMutex_.unlock();
			}
		}
	}
}

/**
*	mapCharset()
*
*	Maps charset from enca's format to Qt's charset name.
*/
QString mapCharset(const QString& encaName) {
	if ( encaName == "windows-1251" ) {
		return "Windows-1251";
	}
	else if ( encaName == "IBM866" ) {
		return "IBM 866";
	}
	else {
		return encaName;
	}
}

QString Document::guessCharset() const {
	QStringList params;
	params << "-m" << fileName();
	if ( !JuffApp::language().isEmpty() ) {
		params << "-L" << JuffApp::language().left(2);
	}

	QProcess enca;
	enca.start("enca", params);
	enca.waitForFinished();
	QString output = QString(enca.readAllStandardOutput()).simplified();
	if ( !output.isEmpty() ) {
		QString codecName = mapCharset(output);
		return codecName;
	}
	else {
		return "";
	}
}

//}	//	namespace Data
}	//	namespace Juff
