/*
    JSPWiki - a JSP-based WikiWiki clone.

    Copyright (C) 2001-2007 Janne Jalkanen (Janne.Jalkanen@iki.fi)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2.1 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package com.ecyrd.jspwiki.auth.login;

import java.io.IOException;
import java.security.Principal;
import java.util.HashSet;
import java.util.Set;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.login.LoginException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;

import com.ecyrd.jspwiki.auth.Authorizer;
import com.ecyrd.jspwiki.auth.WikiPrincipal;
import com.ecyrd.jspwiki.auth.authorize.Role;
import com.ecyrd.jspwiki.auth.authorize.WebAuthorizer;

/**
 * <p>
 * Logs in a user by extracting authentication data from an Http servlet
 * session. First, the module tries to extract a Principal object out of the
 * request directly using the servlet requests's <code>getUserPrincipal()</code>
 * method. If one is found, authentication succeeds. If there is no
 * Principal in the request, try calling <code>getRemoteUser()</code>. If
 * the <code>remoteUser</code> exists but the UserDatabase can't find a matching
 * profile, a generic WikiPrincipal is created with this value. If neither
 * <code>userPrincipal</code> nor <code>remoteUser</code> exist in the request, the login fails.
 * </p>
 * <p>
 * This module must be used with a CallbackHandler that supports the following
 * Callback types:
 * </p>
 * <ol>
 * <li>{@link HttpRequestCallback} - supplies the Http request object, from
 * which the getRemoteUser and getUserPrincipal are extracted</li>
 * <li>{@link UserDatabaseCallback} - supplies the user database for looking up
 * the value of getRemoteUser</li>
 * </ol>
 * <p>
 * After authentication, the Subject will contain principals
 * {@link com.ecyrd.jspwiki.auth.authorize.Role#ALL}
 * and {@link com.ecyrd.jspwiki.auth.authorize.Role#AUTHENTICATED},
 * plus the Principal that represents the logged-in user.</p>
 *
 * @author Andrew Jaquith
 * @since 2.3
 */
public class WebContainerLoginModule extends AbstractLoginModule
{

    protected static final Logger log      = Logger.getLogger( WebContainerLoginModule.class );

    /**
     * Logs in the user.
     * @see javax.security.auth.spi.LoginModule#login()
     */
    public boolean login() throws LoginException
    {
        HttpRequestCallback rcb = new HttpRequestCallback();
        AuthorizerCallback acb = new AuthorizerCallback();
        Callback[] callbacks = new Callback[]
        { rcb, acb };
        String userId = null;

        try
        {
            // First, try to extract a Principal object out of the request
            // directly. If we find one, we're done.
            m_handler.handle( callbacks );
            HttpServletRequest request = rcb.getRequest();
            if ( request == null )
            {
                throw new LoginException( "No Http request supplied." );
            }
            HttpSession session = request.getSession(false);
            String sid = (session == null) ? NULL : session.getId();
            Principal principal = request.getUserPrincipal();
            if ( principal == null )
            {
                // If no Principal in request, try the remoteUser
                if ( log.isDebugEnabled() )
                {
                    log.debug( "No userPrincipal found for session ID=" + sid);
                }
                userId = request.getRemoteUser();
                if ( userId == null )
                {
                    if ( log.isDebugEnabled() )
                    {
                        log.debug( "No remoteUser found for session ID=" + sid);
                    }
                    throw new FailedLoginException( "No remote user found" );
                }
                principal = new WikiPrincipal( userId, WikiPrincipal.LOGIN_NAME );
            }
            if ( log.isDebugEnabled() )
            {
                log.debug("Added Principal " + principal.getName() + ",Role.ANONYMOUS,Role.ALL" );
            }
            m_principals.add( new PrincipalWrapper( principal ) );

            // Add any container roles
            injectWebAuthorizerRoles( acb.getAuthorizer(), request );

            // If login succeeds, commit these roles
            m_principals.add( Role.AUTHENTICATED );
            m_principals.add( Role.ALL );

            // If login succeeds, remove these principals/roles
            m_principalsToOverwrite.add( WikiPrincipal.GUEST );
            m_principalsToOverwrite.add( Role.ANONYMOUS );
            m_principalsToOverwrite.add( Role.ASSERTED );

            // If login fails, remove these roles
            m_principalsToRemove.add( Role.AUTHENTICATED );

            return true;
        }
        catch( IOException e )
        {
            log.error( "IOException: " + e.getMessage() );
            return false;
        }
        catch( UnsupportedCallbackException e )
        {
            log.error( "UnsupportedCallbackException: " + e.getMessage() );
            return false;
        }
    }

    /**
     * If the current Authorizer is a
     * {@link com.ecyrd.jwpwiki.auth.authorize.WebAuthorizer},
     * this method iterates through each role returned by the
     * authorizer (via
     * {@link com.ecyrd.jwpwiki.auth.authorize.WebAuthorizer#isUserInRole( HttpServletRequest, Role)})
     * and injects the appropriate ones into the Subject.
     * @param acb the authorizer callback
     * @param rcb the HTTP request
     */
    private final void injectWebAuthorizerRoles( Authorizer authorizer, HttpServletRequest request )
    {
        Principal[] roles = authorizer.getRoles();
        Set foundRoles = new HashSet();
        if ( authorizer instanceof WebAuthorizer )
        {
            WebAuthorizer wa = (WebAuthorizer)authorizer;
            for ( int i = 0; i < roles.length; i++ )
            {
                if ( wa.isUserInRole( request, roles[i] ) )
                {
                    foundRoles.add( roles[i] );
                    if ( log.isDebugEnabled() )
                    {
                        log.debug("Added Principal " + roles[i].getName() + "." );
                    }
                }
            }
        }

        // Add these container roles if login succeeds
        m_principals.addAll( foundRoles );

        // Make sure the same ones are removed if login fails
        m_principalsToRemove.addAll( foundRoles );
    }

}