#include <Python.h>
#include <datetime.h>
#include <errno.h>

#include <time.h>
#include "pytype_events.h"
#include "pytype_basics.h"
#include "pytype_timezone.h"
#include "structmember.h"
#include "python_compatibility.h"

/*******************************************************
 * Create and delloc methods for objects
 ******************************************************/

/**
 * Create a new jppy calendar event.
 *
 * @param type the type of object to create
 *
 * @param args the arguments to the constructor
 *
 * @param kwds the keyword arguments
 *
 * @return a new jppy calendar event object
 */
extern PyObject* PyPiEvent_New(PyTypeObject *type, PyObject *args, PyObject *kwds) {
  int i;
  PyPiEvent* self;

  /* Why do we have to do this here? The one in the swig init doesn't seem
     to work ?! */
  PyDateTime_IMPORT;

  EventType.ob_type = &PyType_Type;
  self = (PyPiEvent *)type->tp_alloc(type, 0);
  new_CalendarEvent(&(self->a));
  SetBasicRecordObjectAttributeDefaults((PyObject*) self, pack_CalendarEvent);
  self->filters = NULL;

  return (PyObject*)self;
}

/**
 * Initialize a jppy calendar event.
 *
 * @param args the arguments to the constructor
 *
 * @param kwds the keyword arguments: event the event to copy
 *
 * @return -1 on failure, 0 on success
 */
extern int PyPiEvent_Init(PyObject *self, PyObject *args, PyObject *kwds) {
  PyPiEvent* fromevent = NULL;
  PyPiEvent* event = NULL;
  PyObject* filters = NULL;

  int i;

  static char *kwlist[] = {"event","record_field_filters", NULL};
  if (!PyArg_ParseTupleAndKeywords(args, kwds, "|OO", kwlist,
				   &fromevent, &filters)) {
    return -1;
  }

  event = (PyPiEvent*)self;
  /* we have to support calling __init__ more than once */
  if (event->filters != NULL) {
    Py_DECREF(event->filters);
    event->filters = NULL;
  }
  if (filters != NULL) {
    event->filters = filters;
    Py_INCREF(filters);
  }
  free_CalendarEvent(&(event->a));
  if (event->saved_br.size > 0 && event->saved_br.buf) {
    free(event->saved_br.buf);
  }

  if ((fromevent == NULL) || ((PyObject *)fromevent == Py_None)) {
    /* Initialise attributes custom to this type of object */
    new_CalendarEvent(&(event->a));
    SetBasicRecordObjectAttributeDefaults((PyObject*) event, pack_CalendarEvent);
  } else {
    if (!PyPiEvent_Check(fromevent)) {
      PyErr_SetString(PyExc_TypeError,"Must provide a Event object to share");
      return -1;
    }

    /* copy all the database agnostic record attributes */
    event->saved_br.size = fromevent->saved_br.size;
    event->saved_br.attrib = fromevent->saved_br.attrib;
    event->saved_br.rt = fromevent->saved_br.rt;
    event->saved_br.unique_id = fromevent->saved_br.unique_id;

    event->rt = fromevent->rt;
    event->unique_id = fromevent->unique_id;

    event->saved_br.buf = malloc(fromevent->saved_br.size);
    memcpy(event->saved_br.buf,
	   fromevent->saved_br.buf,
	   fromevent->saved_br.size);

    event->category = fromevent->category;
    event->unsaved_changes = fromevent->unsaved_changes;
    event->deleted = fromevent->deleted;
    event->modified = fromevent->modified;
    event->busy = fromevent->busy;
    event->secret = fromevent->secret;

    /* Copy the record */
    if(copy_CalendarEvent(&(fromevent->a), &(event->a)) < 0) {
      PyErr_SetFromErrno(PyExc_SystemError);
      return -1;
    }
  }

  return 0;
}

static PyObject * PyPiEvent_Allocate(PyTypeObject *type, int nitems) {
  PyPiEvent *event;
  if (type == &EventType) {
    event = PyObject_New(PyPiEvent, &EventType);
    return (PyObject *) event;
  } else {
    /* Is this for subclasses ? */
    event = (PyPiEvent *)PyType_GenericAlloc(type, nitems);
    return (PyObject*)event;
  }
}

/**
 * Wrap an existing Calendar event in a jppy calendar appointment. This makes
 * a copy of a into a newly allocated struct inside the python object.
 *
 * @param a the palm event object
 * @param rt
 * @param unique_id
 * @param attrib
 * @param size
 * @param buf
 * @return a newly created python event object that represents the data in a
 */
extern PyObject* PyPiEvent_Wrap(CalendarEvent_t* a, PCRecType rt,
				 unsigned int unique_id, unsigned char attrib,
				 int size, void* buf, PyObject *filters) {
  PyPiEvent* event;
  int i;

 PyObject *python_mod, *python_mdict, *event_class, *python_args, *python_kw;
  python_mod = PyImport_Import(PyString_FromString("jppy.jpilot.modern"));
  if (python_mod == NULL) {
    PyErr_Print();
    return NULL;
  }
  python_mdict = PyModule_GetDict(python_mod);
  if (python_mdict == NULL) {
    PyErr_Print();
    Py_DECREF(python_mod);
    return NULL;
  }
  Py_INCREF(python_mdict);
  Py_DECREF(python_mod);
  event_class = PyDict_GetItemString(python_mdict, "Event"); /* borrowed reference */
  if (event_class == NULL) {
    PyErr_Print();
    Py_DECREF(python_mdict);
    return NULL;
  }
  Py_INCREF(event_class);
  python_args = Py_BuildValue("()");
  python_kw = Py_BuildValue("{s:O}","record_field_filters",filters);
  event = (PyPiEvent*) PyObject_Call(event_class, python_args, python_kw);
  Py_DECREF(event_class);
  Py_DECREF(python_args);
  Py_DECREF(python_kw);
  if (event == NULL) {
    PyErr_Print();
    return NULL;
  }
  Py_INCREF(event);

  copy_CalendarEvent(a, &(event->a));

  /* set saved_br stuff, and rt and unique_id, and attrib derived
     details for the current event */
  SetSavedBrAndRTandUniqueIDandAttribs(rt, unique_id, attrib, size, buf, (PyObject *)event);

  return (PyObject*)event;
}

static void PyPiEvent_Dealloc(PyPiEvent* self) {
  free_CalendarEvent(&(self->a));
  if (self->filters != NULL) {
    Py_DECREF(self->filters);
    self->filters = NULL;
  }
  if (self->saved_br.size > 0 && self->saved_br.buf) {
    free(self->saved_br.buf);
  }
  self->ob_type->tp_free((PyObject*)self);
}


/**
 * Compare two event objects.
 *
 * @param self the first object
 * @param other the second object
 * @return standard comparison values (-1, 0, 1)
 *
 */
static int PyPiEvent_Compare(PyPiEvent* self,PyPiEvent *other) {
  time_t s, o;

  s = mktime(&(self->a.begin));
  o = mktime(&(other->a.begin));

  if (s == o) {
    return 0;
  } else if (s > 0) {
    return -1;
  } else {
    return 1;
  }
}

static char *PyPiEvent_key_list[] = {
  "event", /* boolean */
  "begin", /* datetime.date or naive datetime.datetime */
  "end", /* datetime.date or naive datetime.datetime */
  "alarm", /* datetime.timedelta */
  /* @todo split out alarm? */
  "repeatType", /* int */
  "repeatForever", /* boolean */
  "repeatEnd", /* datetime.date representing the end of the repeat */
  "repeatFrequency", /* int */
  "repeatDay", /* int */
  "repeatDays.sunday", /* boolean */
  "repeatDays.monday", /* boolean */
  "repeatDays.tuesday", /* boolean */
  "repeatDays.wednesday", /* boolean */
  "repeatDays.thursday", /* boolean */
  "repeatDays.friday", /* boolean */
  "repeatDays.saturday", /* boolean */
  "repeatWeekstart", /* int */
  "exceptions", /* datetime.date */
  "description", /* unicode string */
  "note", /* unicode string */
  "location", /* unicode string */
  "timezone", /* jppy._jpilot.__jpilot._Timezone or None */
  NULL};

static PyObject* PyPiEvent_keys(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiEvent_key_list[n]) {
    PyObject *value;
    value = PyString_FromString(PyPiEvent_key_list[n++]);
    PyList_Append(list, value);
    Py_DECREF(value);
  }
  PyPi_extend_keys_from_filters((PyPiBase*)self, list);
  return list;
}

/* forward declaration */
PyObject *PyPiEvent_GetItem(PyPiEvent* self,  PyObject* key);

static PyObject* PyPiEvent_values(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiEvent_key_list[n]) {
    PyObject *key;
    PyObject *value;
    key   = PyString_FromString(PyPiEvent_key_list[n++]);
    value = PyPiEvent_GetItem((PyPiEvent *)self, key);
    PyList_Append(list, value);
    Py_DECREF(key);
    Py_DECREF(value);
  }
  return list;
}

static PyObject* PyPiEvent_items(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiEvent_key_list[n]) {
    PyObject *key, *value, *tuple;
    key = PyString_FromString(PyPiEvent_key_list[n++]);
    value = PyPiEvent_GetItem((PyPiEvent *)self, key);
    tuple = Py_BuildValue("(OO)", key, value);
    PyList_Append(list, tuple); /* get it's own ref */
    Py_DECREF(key);
    Py_DECREF(value);
    Py_DECREF(tuple);
  }
  return list;
}

/*******************************************************
 *
 ******************************************************/

static PyMethodDef PyPiEvent_Methods[] = {
  { "keys", (PyCFunction)PyPiEvent_keys, METH_NOARGS, "Return a list of available keys"},
  { "items",(PyCFunction)PyPiEvent_items, METH_NOARGS, "Return a list of available items"},
  { "values",(PyCFunction)PyPiEvent_values, METH_NOARGS, "Return a list of available items"},
  {NULL,NULL,0,NULL} /* Sentinel */
};

static PyMemberDef PyPiEvent_Members[] = {
  PYPI_MEMBERS_HEAD,
  {NULL}  /* Sentinel */
};

static PyGetSetDef PyPiEvent_Getseters[] = {
  PYPI_GETSETERS_HEAD,
  {NULL}  /* Sentinel */
};

/**** mapping interface ****/
int PyPiEvent_Len(PyObject *self) {
  int len=0;
  // quite expensive way to do it, but we need to get filters too
  PyObject *keys = PyPiEvent_keys(self);
  len = PySequence_Size(keys);
  Py_DECREF(keys);
  return len;
}

PyObject *PyPiEvent_GetItem(PyPiEvent* self,  PyObject* key) {
  char *keystring;
  PyObject *result;

  if (!PyString_Check(key)) {
    Py_INCREF(Py_None);
    return Py_None;
  }

  if ((result = PyPi_GetItem_from_filters((PyPiBase *)self, key)) != NULL)
    return result;
  else if (PyErr_Occurred() != NULL)
    return NULL;

  Py_INCREF(key);
  keystring = PyString_AsString(key);

  GET_BOOL_ATTR(keystring, "event", a.event);
  GET_INT_ATTR(keystring, "repeatType", a.repeatType);
  GET_BOOL_ATTR(keystring, "repeatForever", a.repeatForever);
  GET_INT_ATTR(keystring, "repeatFrequency", a.repeatFrequency);
  GET_INT_ATTR(keystring, "repeatDay", a.repeatDay);
  GET_BOOL_ATTR(keystring, "repeatDays.sunday", a.repeatDays[0]);
  GET_BOOL_ATTR(keystring, "repeatDays.monday", a.repeatDays[1]);
  GET_BOOL_ATTR(keystring, "repeatDays.tuesday", a.repeatDays[2]);
  GET_BOOL_ATTR(keystring, "repeatDays.wednesday", a.repeatDays[3]);
  GET_BOOL_ATTR(keystring, "repeatDays.thursday", a.repeatDays[4]);
  GET_BOOL_ATTR(keystring, "repeatDays.friday", a.repeatDays[5]);
  GET_BOOL_ATTR(keystring, "repeatDays.saturday", a.repeatDays[6]);
  GET_INT_ATTR(keystring, "repeatWeekstart", a.repeatWeekstart);
  GET_STRING_ATTR(keystring,"description", a.description);
  GET_STRING_ATTR(keystring,"note", a.note);
  GET_STRING_ATTR(keystring, "location", a.location);

  if (strcasecmp(keystring,"alarm") == 0) {
    PyObject *units;
    Py_DECREF(key);
    if (!(self->a.alarm)) {
      Py_INCREF(Py_None);
      return Py_None;
    }
    if (self->a.advanceUnits == calendar_advMinutes) {
      return PyDelta_FromDSU(0,self->a.advance * 60,0);
    } else if (self->a.advanceUnits == calendar_advHours) {
      return PyDelta_FromDSU(0,self->a.advance * 60 * 60,0);
    } else {
      return PyDelta_FromDSU(self->a.advance,0,0);
    }
  } else if (strcasecmp(keystring,"exceptions") == 0) {
    PyObject *list;
    int i;
    Py_DECREF(key);
    list = PyList_New(self->a.exceptions);
    for (i = 0; i < self->a.exceptions; i++) {
       PyList_SetItem(list,i,PyDate_FromDate(self->a.exception[i].tm_year + 1900,
					     self->a.exception[i].tm_mon + 1,
					     self->a.exception[i].tm_mday));
    }
    return list;
  } else if (strcasecmp(keystring,"begin") == 0) {
    Py_DECREF(key);
    if (self->a.event) {
      return PyDate_FromDate(self->a.begin.tm_year + 1900,
			     self->a.begin.tm_mon + 1,
			     self->a.begin.tm_mday);
    } else {
      PyObject *pydatetime;
      pydatetime = PyDateTime_FromDateAndTime(self->a.begin.tm_year + 1900,
					      self->a.begin.tm_mon + 1,
					      self->a.begin.tm_mday,
					      self->a.begin.tm_hour,
					      self->a.begin.tm_min,
					      self->a.begin.tm_sec,
					      0);
      return pydatetime;
    }
  } else if (strcasecmp(keystring,"end") == 0) {
    Py_DECREF(key);
    if (self->a.event) {
      return PyDate_FromDate(self->a.end.tm_year + 1900,
			     self->a.end.tm_mon + 1,
			     self->a.end.tm_mday);
    } else {
      PyObject *pydatetime;
      pydatetime = PyDateTime_FromDateAndTime(self->a.end.tm_year + 1900,
					      self->a.end.tm_mon + 1,
					      self->a.end.tm_mday,
					      self->a.end.tm_hour,
					      self->a.end.tm_min,
					      self->a.end.tm_sec,
					      0);
      return pydatetime;
    }
  } else if (strcasecmp(keystring,"repeatEnd") == 0) {
    Py_DECREF(key);
    if (self->a.repeatForever) {
      Py_INCREF(Py_None);
      return Py_None;
    } else {
      return PyDate_FromDate(self->a.repeatEnd.tm_year + 1900,
			     self->a.repeatEnd.tm_mon + 1,
			     self->a.repeatEnd.tm_mday);
    }
  } else if (strcasecmp(keystring, "timezone") == 0) {
    Py_DECREF(key);
    if (NULL == self->a.tz) {
      Py_INCREF(Py_None);
      return Py_None;
    } else {
      /* This creats an extra copy of a.tz in memory, but I don't know a
       * better way around it for now */
      return PyPiTimezone_Wrap(self->a.tz);
    }
  } else {
    PyErr_Format(PyExc_KeyError,"no such key '%s'", keystring);
    Py_DECREF(key);
    return NULL;
  }
}

int PyPiEvent_SetItem(PyPiEvent* self, PyObject* key, PyObject* value) {
  char buf[255];
  char *keystring;

  if (!PyString_Check(key)) {
    PyErr_SetString(PyExc_TypeError,"key must be a String");
    return -1;
  }

  if (PyPi_SetItem_from_filters((PyPiBase *)self, key, value) > 0)
    return 0;
  else if (PyErr_Occurred() != NULL)
    return -1;

  Py_INCREF(key);
  keystring = PyString_AsString(key);

  if (value == NULL) {
    PyErr_Format(PyExc_ValueError,"Can't delete value %s", keystring);
    return -1;
  }

  SET_BOOL_ATTR(keystring, "event", a.event, value, buf, 255);
  SET_INT_ATTR(keystring, "repeatType", a.repeatType, value, buf, 255);
  SET_BOOL_ATTR(keystring, "repeatForever", a.repeatForever, value, buf, 255);
  SET_INT_ATTR(keystring, "repeatFrequency", a.repeatFrequency, value, buf, 255);
  SET_INT_ATTR(keystring, "repeatDay", a.repeatDay, value, buf, 255);
  SET_BOOL_ATTR(keystring, "repeatDays.sunday", a.repeatDays[0], value, buf, 255);
  SET_BOOL_ATTR(keystring, "repeatDays.monday", a.repeatDays[1], value, buf, 255);
  SET_BOOL_ATTR(keystring, "repeatDays.tuesday", a.repeatDays[2], value, buf, 255);
  SET_BOOL_ATTR(keystring, "repeatDays.wednesday", a.repeatDays[3], value, buf, 255);
  SET_BOOL_ATTR(keystring, "repeatDays.thursday", a.repeatDays[4], value, buf, 255);
  SET_BOOL_ATTR(keystring, "repeatDays.friday", a.repeatDays[5], value, buf, 255);
  SET_BOOL_ATTR(keystring, "repeatDays.saturday", a.repeatDays[6], value, buf, 255);
  SET_INT_ATTR(keystring, "repeatWeekstart", a.repeatWeekstart, value, buf, 255);
  SET_STRING_ATTR(keystring,"description",a.description,value, 256);
  SET_STRING_ATTR(keystring,"note",a.note,value, 4096);
  SET_STRING_ATTR(keystring, "location", a.location, value, 21);

  /*
   * @todo support alarm, dates, begin, end, final, timezone
   */

  PyErr_SetString(PyExc_KeyError,"no such key");
  Py_DECREF(key);
  return -1;
}

static PyMappingMethods PyPiEvent_Mapping = {
  (lenfunc)PyPiEvent_Len,
  (binaryfunc)PyPiEvent_GetItem,
  (objobjargproc)PyPiEvent_SetItem,
};

/*******************************************************
 * Provide a repr method
 ******************************************************/
static PyObject *PyPiEvent_Repr(PyPiEvent* self) {
  static PyObject *format = NULL;
  PyObject *attrib, *args, *result;
  int len1;
  char time[100];

  if (format == NULL) {
    format = PyString_FromString("<%s '%s' %r %s>");
    if (format == NULL)
      return NULL;
  }

  if (self->a.description) {
    len1 = strlen(self->a.description) > 25 ? 25 : strlen(self->a.description);
  } else {
    len1 = 0;
  }

  if (self->a.event) {
    strftime(time, 99, "%x", &(self->a.begin));
  } else {
    strftime(time, 99, "%c", &(self->a.begin));
  }

  args = Py_BuildValue("ss#sO",
		       (self->ob_type)->tp_name,
		       self->a.description,
		       len1,
		       time,
		       Attribute_Repr((PyObject *)self));

  if (args == NULL)
    return NULL;

  result = PyString_Format(format, args);
  Py_DECREF(args);
  return result;
}


/*******************************************************
 * Declare the type
 ******************************************************/


PyTypeObject EventType = {
  PyObject_HEAD_INIT(NULL)
  0,
  "jppy._jpilot.__jpilot.Event",
  sizeof(PyPiEvent),
  0,
  (destructor)PyPiEvent_Dealloc,      /*tp_dealloc*/
  0,                                /*tp_print*/
  0, /*tp_getattr*/
  0, /*tp_setattr*/
  (cmpfunc)PyPiEvent_Compare,     /*tp_compare*/
  (reprfunc)PyPiEvent_Repr,       /*tp_repr*/
  0,                                /*tp_as_number*/
  0,                                /*tp_as_sequence*/
  &PyPiEvent_Mapping,                                /*tp_as_mapping*/
  0,                                /*tp_hash */
  0,                         /*tp_call*/
  0,                         /*tp_str*/
  0,                         /*tp_getattro*/
  0,                         /*tp_setattro*/
  0,                         /*tp_as_buffer*/
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
  "Event objects",           /* tp_doc */
  0,		               /* tp_traverse */
  0,		               /* tp_clear */
  0,		               /* tp_richcompare */
  0,		               /* tp_weaklistoffset */
  0,		               /* tp_iter */
  0,		               /* tp_iternext */
  PyPiEvent_Methods,             /* tp_methods */
  PyPiEvent_Members,            /* tp_members */
  PyPiEvent_Getseters,          /* tp_getset */
  0,                         /* tp_base */
  0,                         /* tp_dict */
  0,                         /* tp_descr_get */
  0,                         /* tp_descr_set */
  0,                         /* tp_dictoffset */
  (initproc)PyPiEvent_Init,      /* tp_init */
  (allocfunc)PyPiEvent_Allocate,                 /* tp_alloc */
  (newfunc)PyPiEvent_New,                 /* tp_new */
  0, /* Low-level free-memory routine */
};
