// $Id: StringUtils.java,v 1.2 2001/12/07 11:41:24 ramsdell Exp $

// String processing functions.

/*
 * Copyright 1999 by John D. Ramsdell
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package edu.neu.ccs.jmk;

import java.io.File;

/**
 * String processing functions.
 * These functions are used by the makefile loader to implement
 * its string processing functions.
 * The results computed by a string processing function is appended
 * to its list argument to produce the final result.
 * @see edu.neu.ccs.jmk.GlobalTable
 * @version May 1999
 * @author John D. Ramsdell
 */
public class StringUtils
{

  // No instances of this class should be constructed.
  private StringUtils() { }

  private final static char genericSeparatorChar = '/';
  private final static char genericPathSeparatorChar = ';';

  /**
   * Localizes file related strings by translating the
   * the generic separator character '/' to the separator
   * character used on this platform, and the generic
   * path separator ':' to the path separator character
   * used on this platform.
   */
  public static String localizePaths(String paths) {
    paths = paths.replace(genericPathSeparatorChar,
			  File.pathSeparatorChar);
    paths = paths.replace(genericSeparatorChar,
			  File.separatorChar);
    return paths;
  }

  /**
   * Generalizes file related strings by performing the inverse
   * translation given by localizePaths.
   */
  public static String generalizePaths(String paths) {
    paths = paths.replace(File.pathSeparatorChar,
			  genericPathSeparatorChar);
    paths = paths.replace(File.separatorChar,
			  genericSeparatorChar);
    return paths;
  }

  /**
   * The patsubst function matches the pattern with the strings
   * in the input.  A string that does not match is copied unchanged
   * to the output.  A string that does match is replaced by a string
   * generated by replacing wild cards in the replacement with the match.
   * When the replacement does not contain a wild card, the match is
   * concatenated with the replacement.
   * @see edu.neu.ccs.jmk.Matcher
   */
  static StringList patsubst(String pattern,
			     String replacement,
			     StringList input,
			     StringList list) {
    if (input == null)
      return list;
    else {
      StringList head = new StringList("");
      StringList last = head;
      Matcher matcher = new Matcher(pattern);
      for (; input != null; input = input.getRest()) {
	String string = input.getString();
	String match = matcher.match(string);
	if (match != null) {
	  if (Matcher.isPattern(replacement))
	    string = Matcher.subst(match, replacement);
	  else
	    string = match + replacement;
	}
	StringList temp = new StringList(string);
	last.setRest(temp);
	last = temp;
      }
      last.setRest(list);
      return head.getRest();
    }
  }

  /**
   * Applies stringSubst to each element of the input.
   */
  static StringList subst(String pattern,
			  String replacement,
			  StringList input,
			  StringList list) {
    if (input == null)
      return list;
    else {
      StringList head = new StringList("");
      StringList last = head;
      for (; input != null; input = input.getRest()) {
	String string = input.getString();
	string = stringSubst(pattern, replacement, string);
	StringList temp = new StringList(string);
	last.setRest(temp);
	last = temp;
      }
      last.setRest(list);
      return head.getRest();
    }
  }

  /**
   * Substitutes the replacement for every non-overlapping occurrence
   * of the pattern in the string.
   */
  static String stringSubst(String pattern,
			    String replacement,
			    String string) {
    int i = string.indexOf(pattern);
    if (i >= 0) {		// Pattern found.
      StringBuffer sb = new StringBuffer();
      do {
	sb.append(string.substring(0, i));
	sb.append(replacement);
	string = string.substring(i + pattern.length());
	i = string.indexOf(pattern);
      }
      while (i >= 0);
      sb.append(string);
      string = sb.toString();
    }
    return string;
  }

  /**
   * Concatenates the strings in the input into one string which
   * is added to the output.  The null string is added to the output
   * when there are no strings in the input.
   */
  static StringList cat(StringList input, StringList list) {
    StringBuffer sb = new StringBuffer();
    for (; input != null; input = input.getRest())
      sb.append(input.getString());
    return new StringList(sb.toString(), list);
  }

  /**
   * Expands file name that contain the wild card '*' to a list
   * of file names.  The inputs and outputs use generic separators.
   * @see edu.neu.ccs.jmk.FileOperator#glob(String)
   * @exception CommandFailedException if wild card is misused
   */
  static StringList glob(StringList input, StringList list)
       throws CommandFailedException
  {
    if (input == null)
      return list;
    else {
      StringList head = new StringList("");
      StringList last = head;
      for (; input != null; input = input.getRest()) {
	String string = input.getString();
	String[] globbed = FileOperator.glob(localizePaths(string));
	for (int i = 0; i < globbed.length; i++) {
	  StringList temp = new StringList(generalizePaths(globbed[i]));
	  last.setRest(temp);
	  last = temp;
	}
      }
      last.setRest(list);
      return head.getRest();
    }
  }

  /**
   * This method expands a list of file names into a list of all
   * directories in those files.
   * The inputs and outputs use generic separators.
   * @see edu.neu.ccs.jmk.FileOperator#dirs(String)
   */
  static StringList dirs(StringList input, StringList list) {
    if (input == null)
      return list;
    else {
      StringList head = new StringList("");
      StringList last = head;
      for (; input != null; input = input.getRest()) {
	String string = input.getString();
	String[] dirList = FileOperator.dirs(localizePaths(string));
	for (int i = 0; i < dirList.length; i++) {
	  StringList temp = new StringList(generalizePaths(dirList[i]));
	  last.setRest(temp);
	  last = temp;
	}
      }
      last.setRest(list);
      return head.getRest();
    }
  }

  /**
   * Each input is used as a key to retrieve data from the system's
   * properties.  The results use generic separators.
   */
  static StringList getprop(StringList input, StringList list) {
    if (input == null)
      return list;
    else {
      StringList head = new StringList("");
      StringList last = head;
      for (; input != null; input = input.getRest()) {
	String string = input.getString();
	String prop = System.getProperty(string);
	if (prop != null) {
	  StringList temp = new StringList(generalizePaths(prop));
	  last.setRest(temp);
	  last = temp;
	}
      }
      last.setRest(list);
      return head.getRest();
    }
  }

  /**
   * Concatenates each string in prefixes with every string in suffixes.
   */
  static StringList join(StringList prefixes,
			 StringList suffixes,
			 StringList list) {
    StringList head = new StringList("");
    StringList last = head;
    for (StringList pres = prefixes; pres != null; pres = pres.getRest()) {
      String prefix = pres.getString();
      for (StringList sufs = suffixes; sufs != null; sufs = sufs.getRest()) {
	String suffix = sufs.getString();
	StringList temp = new StringList(prefix + suffix);
	last.setRest(temp);
	last = temp;
      }
    }
    last.setRest(list);
    return head.getRest();
  }

  /**
   * Returns true if the two input lists are equal.
   * True is a list containing the string "true".
   * False is the empty list.
   */
  static StringList equal(StringList sl1, StringList sl2, StringList list) {
    for (;;) {
      if (sl1 == null)
	return sl2 == null ? new StringList("true", list) : list;
      else if (sl2 == null)
	return list;
      else if (!sl1.getString().equals(sl2.getString()))
	return list;
      else {
	sl1 = sl1.getRest();
	sl2 = sl2.getRest();
      }
    }
  }
}
