/*
 *   Jackbeat - JACK sequencer
 *    
 *   Copyright (c) 2004-2008 Olivier Guilyardi <olivier {at} samalyse {dot} com>
 *    
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   SVN:$Id: file.c 124 2008-01-10 19:10:45Z olivier $
 */

#include "gui/common.h"

void
gui_file_do_load_sample (gui_t *gui, int track, char *filename)
{
  /* Trying to reuse memory if this sample is already loaded*/
  sample_t *sample = song_try_reuse_sample (gui->song, filename);
  
  if (sample == NULL)
   {
    /* New or modified sample : loading ... */
    gui_show_progress (gui, "Loading sample", "Hold on...");
    if ((sample = sample_new (filename, gui_progress_callback, (void *) gui)) != NULL)
     {
      song_register_sample (gui->song, sample);
     }
    gui_hide_progress (gui);
   }

  char *track_name = NULL;
  if (sample != NULL)
   {
    /* Handling track names conflict */ 
    if (sequence_track_name_exists (gui->sequence, sample->name))
     {
      while ((track_name = gui_track_ask_name (gui, sample->name, 1, 1)) 
             && (strcmp (track_name, sample->name) == 0)) 
       {
        free (track_name);
        track_name = NULL;
       }
     } else {
      track_name = strdup (sample->name);
     }

    if (track_name) {
      /* We got a sample */
      if (sequence_set_sample (gui->sequence, track, sample)) 
       {
        rc_add_sample (gui->rc, sample->filename);

        sequence_set_track_name (gui->sequence, track, track_name);
        free (track_name);

        gui_set_modified (gui, 1);
        gui_refresh (gui);
       } 
      else 
      {
        gui_display_error(gui, error_to_string(sequence_get_error(gui->sequence)));
      }
    }


    /* Trying to free some memory */
    song_cleanup_unused_samples (gui->song);
   }
  else
   {
    gui_display_error (gui,
                       "Unable to load the requested sample file.");
   }
}

static GtkWidget *
gui_file_create_selector (gui_t *gui, char *title, char *path, char save_mode)
{
  GtkWidget * selector = gtk_file_chooser_dialog_new (
    title, 
    GTK_WINDOW (gui->window), 
    save_mode ? GTK_FILE_CHOOSER_ACTION_SAVE : GTK_FILE_CHOOSER_ACTION_OPEN,
    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	  save_mode ? GTK_STOCK_SAVE : GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, 
    NULL
  );  

  if (save_mode) 
   {
    char *tmp_path = strdup (path);
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (selector), dirname(tmp_path));
    free (tmp_path);
    gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (selector), basename(path));
   }
  else
   {
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (selector), path);
   }
  
   return selector;
}

void
gui_file_show_selector (gui_t *gui, char *title, char *path, 
                         gui_file_selected_callback_t callback,
                         void * callback_data, char save_mode)
{
  gui->file_selection = gui_file_create_selector (gui, title, path, save_mode);
  if (gtk_dialog_run (GTK_DIALOG (gui->file_selection)) == GTK_RESPONSE_ACCEPT)
  {
    char *filename;
    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (gui->file_selection));
    gtk_widget_destroy (gui->file_selection);
    callback (gui, filename, callback_data);
  } else {
    gtk_widget_destroy (gui->file_selection);
  }
}


sequence_t * 
gui_file_do_load_sequence (gui_t *gui, char *filename)
{
  jab_t *jab;
  char *y = strdup (filename);
  sprintf (gui->rc->sequence_wdir, "%s/", dirname (y));
  free (y);
  gui_show_progress (gui, "Loading JAB file", "Hold on...");
  if (!(jab = jab_open (filename, JAB_READ, gui_progress_callback, (void *) gui))) {
      gui_hide_progress (gui);
      gui_display_error (gui, "Unable to load the specified file.");
      return NULL;
  }
  sequence_t *sequence;
  char *name = gui_get_next_sequence_name (gui);
  int error;
  if (jab && (sequence = jab_retrieve_sequence (jab, name, &error)))
    {
      rc_add_sequence (gui->rc, filename);
      song_register_sequence (gui->song, sequence);
      song_register_sequence_samples (gui->song, sequence);
      sequence_set_transport (sequence, gui->rc->transport_aware, gui->rc->transport_query);
      sequence_set_resampler_type (sequence, gui->rc->default_resampler_type);
      gui_hide_progress (gui);
      jab_close (jab);
      return sequence;
    }
  else
    {
      char s[256];
      sprintf (s, "Unable to load the specified file : %s", 
               error_to_string (error));
      gui_display_error (gui, s);
      gui_hide_progress (gui);
      if (jab) jab_close (jab);
      return 0;
    }
  free (name);
}

static void
gui_file_load_sequence_selected (gui_t * gui, char *filename, void * unused)
{
  sequence_t *sequence;
  if ((sequence = gui_file_do_load_sequence (gui, filename)))
    gui_new_child (gui->rc, gui->arg, gui, gui->song, sequence, filename);
  g_free (filename);
}

void
gui_file_load_sequence (gui_t * gui, guint action, GtkWidget * w)
{
  gui_file_show_selector (gui, "Load Sequence", gui->rc->sequence_wdir, gui_file_load_sequence_selected, NULL, 0);
}

static void
gui_file_export_sequence_selected (gui_t * gui, char * filename, int framerate, int sustain_type, void * unused)
{
  int len = strlen (filename);
  if ((len < 4) || (strncasecmp(filename + len - 4, ".wav", 4) != 0)) 
   {
    char *p = filename;
    filename = g_strconcat(p, ".wav", NULL);
    g_free (p);
   }
  if ((access (filename, F_OK) == -1) || gui_ask_confirmation
      (gui, "Are you sure you want to overwrite the file ?"))
    {
      gui_show_progress (gui, "Exporting sequence", "Hold on...");
      sequence_export (gui->sequence, filename, framerate, sustain_type, gui_progress_callback, (void *) gui);
      gui_hide_progress (gui);
    }
  g_free (filename);    
}

void
gui_file_export_sequence (gui_t * gui, guint action, GtkWidget * w)
{
  char filename[512];
  char *tmp;
  
  if (strlen (gui->last_export_wdir)) 
   {
    tmp = strdup(gui->filename);
    sprintf (filename, "%s/%s", gui->last_export_wdir, basename (tmp));
    free(tmp);
   } 
  else 
   {
    if (gui->filename_is_set) 
     {
      strcpy(filename, gui->filename);
     } 
    else 
     {
      getcwd (filename, 512);
      strcat (filename, "/");
      tmp = strdup(gui->filename);
      strcat (filename, basename(tmp));
      free(tmp);
     }
  }

  int len = strlen (filename);

  if ((len >= 4) && (strncasecmp(filename + len - 4, ".jab", 4) == 0)) 
    filename[len - 4] = '\0';

  strcat(filename, ".wav");

  DEBUG ("Proposed name: %s", filename);

  // Building export parameters interface
  gui->file_selection = gui_file_create_selector (gui, "Export Sequence as", filename, 1);
  GtkWidget *frame = gtk_expander_new ("Export parameters");
  gtk_expander_set_expanded (GTK_EXPANDER (frame), TRUE);
  GtkWidget *table = gtk_table_new (2, 2, FALSE);
  gtk_container_add (GTK_CONTAINER(frame), table);

  // Sustain handling type
  GtkWidget *label = gtk_label_new ("Sustain:");
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, 0, 3, 6);

  GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 1, 2, 0, 1, 0, 0, 3, 6);

  GtkWidget *loop_button = gtk_radio_button_new_with_label(NULL, "loop");
  gtk_box_pack_start_defaults (GTK_BOX(hbox), loop_button);
  GtkWidget *truncate_button = gtk_radio_button_new_with_label_from_widget(
    GTK_RADIO_BUTTON (loop_button), "truncate");
  gtk_box_pack_start_defaults (GTK_BOX(hbox), truncate_button);
  GtkWidget *keep_button = gtk_radio_button_new_with_label_from_widget(
    GTK_RADIO_BUTTON (loop_button), "keep");
  gtk_box_pack_start_defaults (GTK_BOX(hbox), keep_button);
  switch (gui->last_export_sustain_type)
   {
    case SEQUENCE_SUSTAIN_LOOP:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (loop_button), 1);
      break;
    case SEQUENCE_SUSTAIN_TRUNCATE:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (truncate_button), 1);
      break;
    case SEQUENCE_SUSTAIN_KEEP:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (keep_button), 1);
      break;
    default:
      if (sequence_is_looping(gui->sequence))
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (loop_button), 1);
      else
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (keep_button), 1);
   }

  // Framerate
  label = gtk_label_new ("Framerate:");
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, 0, 3, 6);

  GtkWidget *combo =  gtk_combo_box_entry_new_text();
  gtk_table_attach (GTK_TABLE (table), combo, 1, 2, 1, 2, 0, 0, 3, 6);

  gtk_table_set_col_spacing (GTK_TABLE (table), 0, 10);

  char str[20];
  jack_nframes_t cur_rate = sequence_get_framerate (gui->sequence);
  sprintf(str, "Same as JACK (%d)", cur_rate); 
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), str);
  if (gui->last_export_framerate == 0)
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);

  int rates[] = {22050, 32000, 44100, 48000, 88200, 96000};
  int i;
  for (i = 0; i < 6; i++) 
   {
    sprintf(str, "%d", rates[i]); 
    gtk_combo_box_append_text (GTK_COMBO_BOX (combo), str);
    if (gui->last_export_framerate == rates[i])
      gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i + 1);
   }

  if (gtk_combo_box_get_active (GTK_COMBO_BOX (combo)) == -1) 
   {
    sprintf(str, "%d", gui->last_export_framerate); 
    gtk_combo_box_append_text (GTK_COMBO_BOX (combo), str);
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i + 1);
   }

  gtk_widget_show_all(frame);
  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (gui->file_selection), frame);

  // Running export dialog
  while (1)
   {
    if (gtk_dialog_run (GTK_DIALOG (gui->file_selection)) != GTK_RESPONSE_ACCEPT)
      break;
    else
     {
      char *chosen_name;
      chosen_name = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (gui->file_selection));

      int chosen_rate, sustain_type;
      if (gtk_combo_box_get_active (GTK_COMBO_BOX (combo)) == 0) 
       {
        chosen_rate = cur_rate;
        gui->last_export_framerate = 0;
       }
      else
       {
        char *active_text = gtk_combo_box_get_active_text (GTK_COMBO_BOX (combo));
        if (((sscanf(active_text, "%d", &chosen_rate) != 1))
            || (chosen_rate < 4000) || (chosen_rate > 192000))
         {
          gui_display_error (gui, "Invalid frame rate (must be an integer between 4000 and 192000)");
          continue;
         } 
        gui->last_export_framerate = chosen_rate;
       }

      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (loop_button)))
        sustain_type = SEQUENCE_SUSTAIN_LOOP;
      else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (truncate_button)))
        sustain_type = SEQUENCE_SUSTAIN_TRUNCATE;
      else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (keep_button)))
        sustain_type = SEQUENCE_SUSTAIN_KEEP;

      gtk_widget_destroy (gui->file_selection);
      gui->file_selection = NULL;
      DEBUG("Chosen name: %s", chosen_name);
      DEBUG("Parameters: framerate: %d, sustain: %d", 
        chosen_rate, sustain_type);
      gui->last_export_sustain_type = sustain_type;
      char *s = strdup(chosen_name);
      sprintf (gui->last_export_wdir, "%s/", dirname(s));
      free (s);
      gui_file_export_sequence_selected (gui, chosen_name, chosen_rate, sustain_type, NULL);
      break;
     }
   }
  if (gui->file_selection)
    gtk_widget_destroy (gui->file_selection);
}

static void
gui_file_save_as_sequence_selected (gui_t * gui, char * filename, void * unused)
{
  int len = strlen (filename);
  if ((len < 4) || (strncasecmp (filename + len - 4, ".jab", 4) != 0))
   {
    char *p = filename;
    filename = g_strconcat(p, ".jab", NULL);
    g_free (p);
   }
  if ((access (filename, F_OK) == -1) || gui_ask_confirmation
      (gui, "Are you sure you want to overwrite the file ?"))
    {
      int success = 0;
      jab_t *jab;
      gui_show_progress (gui, "Saving sequence", "Hold on...");
      if ((jab = jab_open (filename, JAB_WRITE, gui_progress_callback, (void *) gui)))
       {
        jab_add_sequence (jab, gui->sequence);
        if (jab_close (jab)) success = 1;
       }
      gui_hide_progress (gui);
      if (success)
        {
          strcpy (gui->filename, filename);
          rc_add_sequence (gui->rc, filename);
          gui->filename_is_set = 1;
          sprintf (gui->rc->sequence_wdir, "%s/", dirname (filename));
          gui_set_modified (gui, 0);
        }
      else
        {
          gui_display_error (gui, "Unable to save the specified file.");
        }
    }
  g_free (filename);    
}

void
gui_file_save_as_sequence (gui_t * gui, guint action, GtkWidget * w)
{
  char filename[512];
  if (gui->filename_is_set) strcpy (filename, gui->filename);
  else 
   {
    char *s = strdup(gui->filename);
    sprintf (filename, "%s/%s", gui->rc->sequence_wdir, basename (s));
    free(s);
   }

  gui_file_show_selector (gui, "Save Sequence as", filename, gui_file_save_as_sequence_selected, NULL, 1);
}

void
gui_file_save_sequence (gui_t * gui, guint action, GtkWidget * w)
{
  if (gui->filename_is_set)
    {
      DEBUG("Ok, we already have a filename. Let's try and save..."); 
      int success = 0;
      jab_t *jab;
      gui_show_progress (gui, "Saving sequence", "Hold on...");
      if ((jab = jab_open (gui->filename,JAB_WRITE, gui_progress_callback,
                           (void *) gui)))
       {
        jab_add_sequence (jab, gui->sequence);
        if (jab_close (jab)) success = 1;
       }
      gui_hide_progress (gui);
      if (success) 
        gui_set_modified (gui, 0);
      else
        gui_display_error (gui, "Unable to save the current sequence.");
    }
  else
    gui_file_save_as_sequence (gui, 0, w);
}

