
/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: daemonize.c,v 1.23 2006/02/10 01:20:51 ro_han Exp $
 *****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>

#include "common.h"
#include "daemonize.h"
#include "log.h"
#include "manager.h"

void signal_handler(int sig);

gboolean daemonize()
{
    DEBUG(_("Forking to background..."));

    int child_pid;

    if (chdir("/") < 0)
    {
        DEBUG(_("Could not set working directory to /, error: %s"),
              strerror(errno));
        return FALSE;
    }

    child_pid = fork();
    switch (child_pid)
    {
    case -1:
        DEBUG(_("Could not fork, error: %s"), strerror(errno));
        break;

    case 0:
        break;

    default:
        // Parent can exit
        exit(0);
        break;
    }

    /* Create session */
    if ( setsid() < 0) {
        DEBUG(_("setsid() failed, error: %s"),
              strerror(errno));
        return FALSE;
    }

    if ( !cfg_base->pidFile ) {
        DEBUG(_("Not using a PID file, none specified..."));
    }
    else {
        umask(0);
        int pid_fd = open(cfg_base->pidFile, O_RDWR | O_CREAT, 0640);

        if (pid_fd < 0) {
            DEBUG(_("Failed to open lockfile %s: %s!"), cfg_base->pidFile,
                  strerror(errno));
            return FALSE;
        }
        if (lockf(pid_fd, F_TLOCK, 0) < 0) {
            DEBUG(_("Failed to get a lock on lockfile %s: %s!"),
                  cfg_base->pidFile, strerror(errno));
            return FALSE;
        }
        char str[16];

        snprintf(str, sizeof(str), "%d\n", getpid());
        write(pid_fd, str, strlen(str));
    }

    freopen("/dev/null", "r", stdin);
    freopen("/dev/null", "w", stdout);
    freopen("/dev/null", "w", stderr);

    signal(SIGTSTP, SIG_IGN);
    signal(SIGHUP, SIG_IGN);
    signal(SIGTTOU, SIG_IGN);
    signal(SIGTTIN, SIG_IGN);

    return TRUE;
}

void clear_pidfile(char *file)
{
    if (file != NULL)
    {
        DEBUG(_("Deleting lockfile %s"), file);
        if (unlink(file) == -1)
        {
            DEBUG(_("Failed to delete lockfile %s!"), file);
        }
    }
}

gboolean dropPrivileges(char *user, char *group)
{
    if ( !user ) {
        DEBUG(_("Can't drop privileges, user not specified."));
        return FALSE;
    }
    if ( !group ) {
        DEBUG(_("Can't drop privileges, group not specified."));
        return FALSE;
    }
    
    if (geteuid() != 0)
    {
        DEBUG(_("Not dropping privileges; we aren't root.\n"));
        return FALSE;
    }
    
    // Drop privileges
    if (chdir("/") != 0)
    {
        DEBUG(_("Could not set working directory to /, error: %s"),
              strerror(errno));
        return FALSE;
    }

    struct group *grpentry = getgrnam(group);

    if (grpentry == NULL)
    {
        DEBUG(_("Group '%s' does not appear to exist!"), group);
        return FALSE;
    }
    unsigned int gid = grpentry->gr_gid;

    if (geteuid() == 0 && setgid(gid) != 0)
    {
        DEBUG(_("Couldn't set group ID to %s, error: %s"),
              group, strerror(errno));
        return FALSE;
    }

    struct passwd *pwdentry = getpwnam(user);

    if (pwdentry == NULL)
    {
        DEBUG(_("User '%s' does not appear to exist!"), user);
        return FALSE;
    }
    unsigned int uid = pwdentry->pw_uid;


    // Set the owner of the PID file to our new user
    if ((cfg_base->pidFile != NULL) && (0 != chown(cfg_base->pidFile, uid,
gid)))
    {
        DEBUG(_("Warning: couldn't chown pidfile %s: %s\n"),
cfg_base->pidFile,
                    strerror(errno));
    }

    if (geteuid() == 0 && setuid(uid) != 0)
    {
        DEBUG(_("Couldn't set user ID to %s, error: %s"),
              user, strerror(errno));
        return FALSE;
    }

    return TRUE;
}
