/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_playlist.c,v $ -- playlist support
 * $Id: mod_playlist.c,v 1.29 2004/08/10 21:08:31 boucman Exp $
 *
 * Copyright (C) by Andreas Neuhaus
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"
#include "irmp3config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3mod.h"
#include "mod_playlist.h"

/*************************************************************************
 * GLOBALS
 */
char mod_playlist_file[512]="";
mod_playlist_t *mod_playlist_list = NULL;
mod_playlist_t *mod_playlist_tail = NULL;
mod_playlist_t *mod_playlist_shufflelist = NULL;
mod_playlist_t *mod_playlist_current = NULL;
int mod_playlist_playing = 0;
int mod_playlist_shuffle = 0;
int mod_playlist_repeat = 2;
int mod_playlist_skip_back_timeout = 2;
int mod_playlist_sort_mode = 0;

/* this is used to speed up access to playlist entries */
mod_playlist_t **mod_playlist_array = NULL;
int mod_playlist_array_count = 0;
int mod_playlist_array_capacity = MOD_PLAYLIST_CAPACITY_INC;		/* the current capacity */


/*************************************************************************
 * MODULE INFO
 */
mod_t mod_playlist = {
	"mod_playlist",
	mod_playlist_deinit,	// deinit
	NULL,			// reload
	&blank_fd,		// watchfd
	NULL,			// poll
	NULL,			// update
	mod_playlist_message,	// message
	NULL,			// SIGCHLD handler
	mod_playlist_init,
	NULL,			// avoid warning
};


/*************************************************************************
 * SORT THE PLAYLIST
 */

static int compare(const void *a, const void *b)
{
	// note: qsort passes ptr to array element
	// which itself is a ptr to the playlist entry
	char* s1=(*(mod_playlist_t**)a)->song;
	char* s2=(*(mod_playlist_t**)b)->song;

	return strcmp(s1, s2);
}


void play_entry(mod_playlist_t *song_entry) {
	if(song_entry->type) {
		mod_sendmsgf(MSGTYPE_INPUT, "playtype %s %s",song_entry->type,song_entry->song);
	} else {
		mod_sendmsgf(MSGTYPE_INPUT, "play %s",song_entry->song);
	}
	mod_playlist_playing = 1;
}

static int mod_playlist_sort (void)
// Strategy: create an array with pointers to the
// linked list. Alphabetically sort the array
// (with quicksort) and re-create the linked list.
// This is the most efficient whay I know of to
// sort a linked list - rvpaasen.
{
	int i, n;
	mod_playlist_t *p, **array;

	log_printf(LOG_VERBOSE, "mod_playlist_sort(): Sorting playlist...\n");

	// count entries
	n = 0;
	p = mod_playlist_list;
	while (p) {
		n++;
		p = p->next;
	}
	
	if (n<2) {
		return 0;
	}

	// setup playlist array
	array = (mod_playlist_t**)calloc(n, sizeof(mod_playlist_t*));
	if (!array) {
		return -1;
	}

	p = mod_playlist_list;
	for (i=0; i<n; i++) {
		array[i] = p;
		p = p->next;
	}

	qsort(array, n, sizeof(mod_playlist_t*), compare);

	// re-create the list
	mod_playlist_list = array[0];
	array[0]->prev = NULL;
	for (i=0; i<n-1; i++) {
		log_printf(LOG_NOISYDEBUG, "mod_playlist_sort(): Entry %d is %s.\n", i, array[i]->song);
                // Update the position index so that it corresponds with the order returned by info plfiles
                array[i]->plpos = i;  
		array[i]->next = array[i+1];
		array[i+1]->prev = array[i];
		// don't touch the shuffle pointers
	}
	array[n-1]->next = NULL;
        array[n-1]->plpos = n-1;  
	mod_playlist_tail = array[n-1];

	// print the last entry
	log_printf(LOG_NOISYDEBUG, "mod_playlist_sort(): Entry %d is %s.\n", i, array[i]->song);
	
	free(array);

	// set current song to first song in list
	mod_playlist_current = mod_playlist_list;

	log_printf(LOG_VERBOSE, "mod_playlist_sort(): Done sorting playlist.\n");

	return n;
}


/*************************************************************************
 * CLEAR PLAYLIST
 */
void mod_playlist_free (void)
{
	mod_playlist_t *p,*n;

	if(mod_playlist_list) {
	  for (p=mod_playlist_list; p; p=n) {
	     n=p->next;
	     free(p->song);
	     free(p->type);
	     free(p);
	  }
    }
	mod_playlist_file[0]=0;
	mod_playlist_list = NULL;
	mod_playlist_tail = NULL;
	mod_playlist_shufflelist = NULL;
	mod_playlist_current = NULL;
	mod_playlist_array_count = 0;
	log_printf(LOG_DEBUG, "mod_playlist_free(): cleared the playlist\n");
}


/*************************************************************************
 * ADD TO PLAYLIST
 */

int mod_playlist_add_typed (char *song,char *type)
{
	mod_playlist_t *newentry;
	int len=strlen(song)+1;

	// do we need some more memory for the new entry ?
	if(mod_playlist_array_count >= mod_playlist_array_capacity) {
		mod_playlist_t **na;

		na = (mod_playlist_t **)realloc(mod_playlist_array, (mod_playlist_array_capacity + MOD_PLAYLIST_CAPACITY_INC)*sizeof(mod_playlist_t *));

		if(!na) {
			return -1;
		}
	
		mod_playlist_array = na;
		mod_playlist_array_capacity += MOD_PLAYLIST_CAPACITY_INC;
	}

	// setup new playlist entry
	newentry = malloc(sizeof(mod_playlist_t));
	if (!newentry) {
		return -1;
	}
	newentry->song = malloc(len);
	if (!newentry->song) {
		free(newentry);
		return -1;
	}
	memcpy(newentry->song, song, len);
	if(type) {
		len = strlen(type)+1;
		newentry->type = malloc(len);
		if (!newentry->type) {
			free(newentry->song);
			free(newentry);
			return -1;
		}
		memcpy(newentry->type, type, len);
	} else {
		newentry->type = NULL;
	}
	newentry->plpos = mod_playlist_array_count;
	newentry->next = NULL;
	newentry->prev = mod_playlist_tail;
	newentry->shufflenext = NULL;
	newentry->shuffleprev = NULL;

	// link new entry into playlist chain
	if(mod_playlist_tail) {
		mod_playlist_tail->next = newentry;
	}else {
		mod_playlist_list = newentry;
	}
	if (!mod_playlist_current) {
		mod_playlist_current = mod_playlist_list;
	}

	// add entry to playlist array
	mod_playlist_array[mod_playlist_array_count] = newentry;
	mod_playlist_array_count++;
        mod_playlist_tail = newentry;

	log_printf(LOG_NOISYDEBUG, "mod_playlist_add(): added '%s' with type '%s' at position %d\n", newentry->song,newentry->type, newentry->plpos);

	return 0;
}


int mod_playlist_add (char *song)
{
	int retval;
	retval = mod_playlist_add_typed(song, NULL);
	return retval;
}

/*************************************************************************
 * LOAD PLAYLIST
 */
int mod_playlist_load (char *name, int display)
{
	char *c, *basename, root[512], song[512];
	FILE *fh;
	int  count, rof;

	if (!name || !*name) {
		return -1;
	}
	
	// get root path of playlist
	strncpy(root, name, sizeof(root)-1);
	c = strrchr(root, '/');
	if (c) {
		*c = 0;
	}
	else {
		getcwd(root, sizeof(root)-1);
	}

	// read playlist file
	fh = fopen(name, "r");
	if (fh == NULL) {
		log_printf(LOG_DEBUG, "mod_playlist_load(): unable to open playlist '%s'\n", name);
		return -2;
	}

	strncpy(mod_playlist_file, name, sizeof(mod_playlist_file)-1);
	mod_playlist_file[sizeof(mod_playlist_file)-1]=0;

	count = 0;
	rof=strlen(root);
	root[rof++]='/';
	root[rof]=0;
	
	while (freadtrimedline(fh, song, sizeof(song)) >= 0) {
		if(song[0]!=0) {
			log_printf(LOG_DEBUG, "mod_playlist_load(): song is now '%s'\n", song);
			if (*song != '/' && (strncmp(song, "http://", 7) != 0)) {
				strncpy(root, song, sizeof(root)-rof-1);
				root[sizeof(root)-1]=0;
				mod_playlist_add(root);
			} else {
				mod_playlist_add(song);
			}
			count++;
		}
	}
	fclose(fh);

	log_printf(LOG_DEBUG, "mod_playlist_load(): added %d songs\n", count);

	if (display) {
		// display playlist name on lcd?
		if ((basename = rindex(name, (int) '/'))) {
			mod_sendmsgf(MSGTYPE_EVENT, "loadplaylist %s", (basename++));
		}
		else {
			mod_sendmsgf(MSGTYPE_EVENT, "loadplaylist %s", name);
		}
	}

	return count;
}


/*************************************************************************
 * LOAD PLAYLIST FROM DIRECTORY
 */
int mod_playlist_loaddir_scan (char *root, char *subdir, char *pattern)
{
	DIR *dir;
	struct dirent *dirent;
	struct stat st;
	char buf[512];
	char** subdirs;
	int isubdirs, nsubdirs;
	int count, i;

	// Scan the directory
	snprintf(buf, sizeof(buf)-1, "%s%s", root, subdir);
	log_printf(LOG_DEBUG, "mod_playlist_loaddir_scan(): scanning dir '%s'\n", buf);
	dir = opendir(buf);
	if (!dir) {
		log_printf(LOG_DEBUG, "mod_playlist_loaddir_scan(): unable to read '%s'!\n", buf);
		return 0;
	}

	strncpy(mod_playlist_file, buf, sizeof(mod_playlist_file)-1);
	mod_playlist_file[sizeof(mod_playlist_file)-1]=0;

	count = 0;
	isubdirs = 0;
	nsubdirs = 0;
	subdirs = NULL;
	
	while (1) {
		dirent = readdir(dir);
		if (!dirent) {
			break;
		}
		if (!strcmp(dirent->d_name, ".") || !strcmp(dirent->d_name, "..")) {
			continue;
		}
		snprintf(buf, sizeof(buf)-1, "%s%s%s", root, subdir, dirent->d_name);
		if (stat(buf, &st)) {
			log_printf(LOG_DEBUG, "mod_playlist_loaddir_scan(): cannot stat '%s', skipping!\n", buf);
			continue;
		}
		if (S_ISDIR(st.st_mode)) {
		
			/* add the subdirectory to the dirs-to-scan list */
			if (isubdirs == nsubdirs) {
				/* increase the subdir array size */
				log_printf(LOG_DEBUG, "mod_playlist_loaddir_scan(): subdirectory count %d reached, increasing capacity to %d\n", isubdirs, nsubdirs + MOD_PLAYLIST_SUBDIR_CAPACITY_INC);
				nsubdirs+=MOD_PLAYLIST_SUBDIR_CAPACITY_INC;
				subdirs = (char**)realloc(subdirs, nsubdirs * sizeof(char**));
				if (subdirs == NULL) {
					log_printf(LOG_ERROR, "mod_playlist_loaddir_scan(): not enough memory for %d subdirectories\n", nsubdirs);
				}
			}
			
			if (subdirs != NULL) {
				/* add the subdir */
				subdirs[isubdirs] = strdup(dirent->d_name);
				log_printf(LOG_DEBUG, "mod_playlist_loaddir_scan(): added subdirectory %d: %s\n", isubdirs, subdirs[isubdirs]);
				if (subdirs[isubdirs] != NULL) {
					isubdirs++;
				} else {
					log_printf(LOG_ERROR, "mod_playlist_loaddir_scan(): not enough memory for subdirectory #%d\n", isubdirs);
				}
			}
			
		} else {
		
			/* Add the song if it matches our search string */
			snprintf(buf, sizeof(buf)-1, "%s%s", subdir, dirent->d_name);
			if (!fnmatch(pattern, buf, FNM_NOESCAPE)) {
				snprintf(buf, sizeof(buf)-1, "%s%s%s", root, subdir, dirent->d_name);
				count++;
				mod_playlist_add(buf);
		  }
		  
		}
	}
	closedir(dir);

	/* process subdirs */
	for (i=0; i<isubdirs; i++) {
		snprintf(buf, sizeof(buf)-1, "%s%s/", subdir, subdirs[i]);
		count += mod_playlist_loaddir_scan(root, buf, pattern);
		free(subdirs[i]);
	}
	
	if (subdirs != NULL) {
		free(subdirs);
	}

	return count;
}

int mod_playlist_loaddir (char *pattern)
{
	char root[512], search[512], buf[512];
	char *c;
	int count;

	if (!pattern || !*pattern)
		return -2;

	strncpy(root, pattern, sizeof(root)-1);
	c = strchr(root, '*');
	if (!c) {
		log_printf(LOG_DEBUG, "mod_playlist_loaddir(): no wildcard specified!\n");
		return -1;
	}
	strcpy(buf, c);
	*c = 0;
	c = strrchr(root, '/');
	if (!c) {
		log_printf(LOG_DEBUG, "mod_playlist_loaddir(): no root path!\n");
		return -1;
	}
	strncpy(search, c+1, sizeof(search)-1);
	strncat(search, buf, sizeof(search)-1);
	*c = 0;
	log_printf(LOG_DEBUG, "mod_playlist_loaddir(): Searching for '%s' in '%s'\n", search, root);

	count = mod_playlist_loaddir_scan(root, "/", search);
	log_printf(LOG_DEBUG, "mod_playlist_loaddir(): Added %d songs\n", count);
	return count;
}


/*************************************************************************
 * SKIP TO SONG (relative)
 */
int mod_playlist_skip (int offset, int quiet)
{
	int i, count;

	if (!offset || !mod_playlist_current || mod_playlist_repeat==1) {
		return 0;
	}
	count = 0;

	// skip forward
	if (offset > 0) {
		for (i=0; i<offset; i++) {
			if (!(mod_playlist_shuffle ? mod_playlist_current->shufflenext : mod_playlist_current->next) && mod_playlist_repeat==2)
				mod_playlist_current = mod_playlist_shuffle ? mod_playlist_shufflelist : mod_playlist_list;
			else
				mod_playlist_current = mod_playlist_shuffle ? mod_playlist_current->shufflenext : mod_playlist_current->next;
			count++;
			if (!mod_playlist_current)
				break;
		}

	// skip backwards
	} else if (offset < 0) {
		for (i=0; i<-offset; i++) {
			if (!(mod_playlist_shuffle ? mod_playlist_current->shuffleprev : mod_playlist_current->prev) && mod_playlist_repeat==2) {
				mod_playlist_current = mod_playlist_shuffle ? mod_playlist_shufflelist : mod_playlist_list;
				while (mod_playlist_current && (mod_playlist_shuffle ? mod_playlist_current->shufflenext : mod_playlist_current->next)) {
					mod_playlist_current = (mod_playlist_shuffle ? mod_playlist_current->shufflenext : mod_playlist_current->next);
				}
			} else {
				mod_playlist_current = mod_playlist_shuffle ? mod_playlist_current->shuffleprev : mod_playlist_current->prev;
			}
			count--;
			if (!mod_playlist_current) {
				break;
			}
		}
	}

	if (!quiet) {
		log_printf(LOG_NORMAL, "Playlist skipped %d songs\n", count);
	}
	return count;
}


/*************************************************************************
 * SKIP TO DIRECTORY IN PLAYLIST
 */ 
int mod_playlist_skipdir (int offset)
{
	int i, count, dircount;
	mod_playlist_t *current;
	char dir[512], buf[512], *p;

	if (!offset || !mod_playlist_current || mod_playlist_repeat==1) {
		return 0;
	}
	count = 0;
	dircount = 0;

	// remember current song
	current = mod_playlist_current;

	// skip forward
	if (offset > 0) {
	
		for (i=0; i<offset; i++) {
			strncpy(dir, mod_playlist_current->song, sizeof(dir)-1);
			p = strrchr(dir, '/');
			if (!p) {
				p = dir;
			}
			*p = 0;
			do {
				count += mod_playlist_skip(1, 1);
				if (!mod_playlist_current || mod_playlist_current == current) {
					break;
				}
				strncpy(buf, mod_playlist_current->song, sizeof(buf)-1);
				p = strrchr(buf, '/');
				if (!p) {
					p = buf;
				}
				*p = 0;
			} while (!strcmp(buf, dir));
			dircount++;
			if (!mod_playlist_current) {
				break;
			}
		}

	// skip backwards
	} else if (offset < 0) {
	
		for (i=-1; i<-offset; i++) {
			strncpy(dir, mod_playlist_current->song, sizeof(dir)-1);
			p = strrchr(dir, '/');
			if (!p) {
				p = dir;
			}
			*p = 0;
			do {
				count += mod_playlist_skip(-1, 1);
				if (!mod_playlist_current || mod_playlist_current == current) {
					break;
				}
				strncpy(buf, mod_playlist_current->song, sizeof(buf)-1);
				p = strrchr(buf, '/');
				if (!p) {
					p = buf;
				}
				*p = 0;
			} while (!strcmp(buf, dir));
			dircount--;
			if (!mod_playlist_current) {
				break;
			}
		}
		dircount++;
		count += mod_playlist_skip(1, 1);
	}

	log_printf(LOG_NORMAL, "Playlist skipped %d songs (%d directories)\n", count, dircount);
	return count;
}


/*************************************************************************
 * LIST FILES IN PLAYLIST
 */ 
int mod_playlist_listfiles (void) {
	int i=0;
	mod_playlist_t *n;
	
	if (mod_playlist_list) {
		n=mod_playlist_list;
		while(n) {
			mod_sendmsgf(MSGTYPE_INFO,"plfiles: %s", n->song);
			i++;
			n=n->next;
		}
	}
	return i;
}

/*************************************************************************
 * LIST DIRECTORIES IN PLAYLIST
 */ 
int mod_playlist_listdirs (void) {
	int dircount;
	mod_playlist_t *current;
	char dir1[512], dir2[512], *p, *q;

	if (!mod_playlist_list) {
		return 0;
	}
	
	dircount = 0;

	// get first directory
	current=mod_playlist_list;
	strncpy(dir1, current->song, sizeof(dir1)-1);
	p = strrchr(dir1, '/');
	if (!p) {
		p = dir1;
	}
	*p = 0;	p = dir1;
	
	do {
		// walk through the list
		current=current->next;

		if (current) {
			// get next directory
			strncpy(dir2, current->song, sizeof(dir2)-1);
			q = strrchr(dir2, '/');
			if (!q) {
				q = dir2;
			}
			*q = 0;	q = dir2;

		} else {
			q = dir2;
			*q = 0;
		}
		
		// if this differs from the previous then output it
		if (strcmp(p, q)) {
			mod_sendmsgf(MSGTYPE_INFO,"pldirs: %s", p);
			strcpy(p,q);
			dircount++;
		}
		
	} while (current);
	
	return dircount;
}


/*************************************************************************
 * SETUP SHUFFLED PLAYLIST
 */
void mod_playlist_shufflecalc (int modifycurrent)
{
	mod_playlist_t	**entries;
	mod_playlist_t	*shuffle_prev=NULL, *p;
	int		num_entries, count, pos;

	if (!mod_playlist_shuffle) {
		return;
	}
	
	// first make a copy of the orignal playlist_array
	num_entries=mod_playlist_array_count;
	if(!num_entries) return;
	entries=(mod_playlist_t **)malloc(sizeof(mod_playlist_t *)*num_entries);
	if(entries==NULL) {
		return;
	}

	memcpy(entries, mod_playlist_array, sizeof(mod_playlist_t *)*num_entries);

	// now let's shuffle
	srand(time(NULL));

	count=0;
	while(count < num_entries) {
	
		// get next random entrie from playlist
		pos=rand() % num_entries;
		while((p=entries[pos])==NULL) {
			// this entry is already used, so find the nearest unused one
			pos++;

			if(pos>=num_entries) {
				pos=0;
			}
		}

		entries[pos]=NULL;	// we are done with this entry
		count++;

		if(shuffle_prev==NULL) {
			shuffle_prev=p;
			p->shuffleprev=NULL;

			mod_playlist_shufflelist=p;
		} else {
			p->shuffleprev=shuffle_prev;
			shuffle_prev->shufflenext=p;
		}

		p->shufflenext=NULL;
		shuffle_prev=p;

		if(num_entries > 20 && (count*4 > num_entries)) {
			// now is a good time to compact the entries array
			int	i;
			
			pos=0;
			for(i=0; i<num_entries; i++) {
				if(entries[i]!=NULL) {
					entries[pos++]=entries[i];
				}
			}

			if(pos != (num_entries - count)) {
				log_printf(LOG_ERROR, "mod_playlist_shuffle(): something went wrong compacting the shuffle entries array\n");
			}

			num_entries-=count;
			count=0;
		}
	}

	// set current song if desired
	if (modifycurrent) {
		mod_playlist_current = mod_playlist_shufflelist;
	}

	free(entries);

	log_printf(LOG_DEBUG, "mod_playlist_shuffle(): set up shuffled playlist\n");
}


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_playlist_message (int msgtype, char *rawmsg,const char __attribute__((unused))*sender)
{
	static int secpos, secremain;
	char *c1, *c2, *c3,*c4, msg[512];
	int i;

	strncpy(msg,rawmsg,sizeof(msg));	// pad msg with nulls

	// handle input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = strtok(msg, " \t");
		if(!c1) return;

		if ( !strcasecmp(c1, "query")) {
			mod_sendmsg(MSGTYPE_QUERY,rawmsg+6);
			// process playlist commands
		} else if (!strcasecmp(c1, "playlist")) {
			c2 =  strtok(NULL, " \t") ;
			if(!c2) return;
			// load playlist from file
			if ( !strcasecmp(c2, "load") ) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				mod_playlist_free();
				log_printf(LOG_NORMAL, "Loading playlist '%s'\n", c3);
				if (mod_playlist_load(c3, 1) <= 0) {
					log_printf(LOG_ERROR, "Unable to load playlist!\n");
				}
				else {
					if (mod_playlist_sort_mode) {
						// sort the playlist
						mod_playlist_sort();
					}
					mod_playlist_shufflecalc(1);
					if(mod_playlist_current) {
						play_entry(mod_playlist_current);
					} else	{
						mod_sendmsg(MSGTYPE_PLAYER, "stop");
						log_printf(LOG_NORMAL, "Empty playlist - playing stopped!\n");
						mod_playlist_playing = 0;
					}
				}

				// load playlist from file without removing old playlist
			} else if ( !strcasecmp(c2, "loadplus") ) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				log_printf(LOG_NORMAL, "Loading additional playlist '%s'\n", c3);
				if ((i=mod_playlist_load(c3, 0)) <= 0) {
					log_printf(LOG_ERROR, "Unable to load additional playlist!\n");
				} else {
					if (mod_playlist_sort_mode) {
						// sort the playlist
						mod_playlist_sort();
					}
					mod_playlist_shufflecalc(1);
					if (!mod_playlist_playing) {
						// automatically start playing
						play_entry(mod_playlist_current);
					}
				}
				mod_sendmsgf(MSGTYPE_INFO, "browser info %d added (%d)", i, mod_playlist_array_count);

				// load playlist by scanning dirs
			} else if ( !strcasecmp(c2, "loaddir") ) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				mod_playlist_free();
				log_printf(LOG_NORMAL, "Loading playlist from pattern '%s'\n", c3);
				if (mod_playlist_loaddir(c3) <= 0) {
					log_printf(LOG_ERROR, "Unable to load playlist!\n");
				} else {
					if (mod_playlist_sort_mode) {
						// sort the playlist
						mod_playlist_sort();
					}
					mod_playlist_shufflecalc(1);
					if(mod_playlist_current) {
						play_entry(mod_playlist_current);
					} else {
						mod_sendmsg(MSGTYPE_PLAYER, "stop");
						log_printf(LOG_NORMAL, "Empty playlist - playing stopped!\n");
						mod_playlist_playing = 0;
					}
				}

				// load playlist by scanning dirs without removing old playlist
			} else if ( !strcasecmp(c2, "loaddirplus") ) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				log_printf(LOG_NORMAL, "Loading additional files to playlist from pattern '%s'\n", c3);
				if ((i=mod_playlist_loaddir(c3)) <= 0) {
					log_printf(LOG_ERROR, "Unable to load playlist!\n");
				} else {
					if (mod_playlist_sort_mode) {
						// sort the playlist
						mod_playlist_sort();
					}
					mod_playlist_shufflecalc(1);
					if (!mod_playlist_playing) {
						// automatically start playing
						play_entry(mod_playlist_current);
					}
				}
				mod_sendmsgf(MSGTYPE_INFO, "browser info %d added (%d)", i, mod_playlist_array_count);

				// add song to playlist
			} else if ( !strcasecmp(c2, "add") ) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				log_printf(LOG_NORMAL, "add song '%s'\n", c3);
				if (mod_playlist_add(c3) < 0) {
					log_printf(LOG_ERROR, "Unable to add!\n");
				} else {
					if (mod_playlist_sort_mode) {
						// sort the playlist
						mod_playlist_sort();
					}
				}

				// add song to playlist with additional lcd output
			} else if (!strcasecmp(c2, "addplus") ) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				log_printf(LOG_NORMAL, "add additional song '%s'\n", c3);
				if (mod_playlist_add(c3) < 0) {
					log_printf(LOG_ERROR, "Unable to add!\n");
					mod_sendmsgf(MSGTYPE_INFO, "browser info 0 added (%d)", mod_playlist_array_count);
				} else {
					if (mod_playlist_sort_mode) {
						// sort the playlist
						mod_playlist_sort();
					}
					mod_playlist_shufflecalc(1);
					if (!mod_playlist_playing) {
						//automatically start playing
						play_entry(mod_playlist_current);
					}
					mod_sendmsgf(MSGTYPE_INFO, "browser info 1 added (%d)", mod_playlist_array_count);
				}
			} else if (!strcasecmp(c2, "addtype") ) {
				c3 = strtok(NULL, " \t");
				if(!c3) return ;
				c4 = strtok(NULL, "");
				if(!c4) return ;
				log_printf(LOG_NORMAL, "add song '%s', type '%s'\n", c3,c4);
				if (mod_playlist_add_typed(c4,c3) < 0) {
					log_printf(LOG_ERROR, "Unable to add!\n");
				} else {
					if (mod_playlist_sort_mode) {
						// sort the playlist
						mod_playlist_sort();
					}
				}

				// add song to playlist with additional lcd output
			} else if (!strcasecmp(c2, "addtypeplus")) {
				c3 = strtok(NULL, " \t");
				if(!c3) return ;
				c4 = strtok(NULL, "");
				log_printf(LOG_NORMAL, "add additional song '%s'with type '%s'\n", c3,c4);
				if (mod_playlist_add_typed(c4,c3) < 0) {
					log_printf(LOG_ERROR, "Unable to add!\n");
					mod_sendmsgf(MSGTYPE_INFO, "browser info 0 added (%d)", mod_playlist_array_count);
				} else {
					if (mod_playlist_sort_mode) {
						// sort the playlist
						mod_playlist_sort();
					}
					mod_playlist_shufflecalc(1);
					if (!mod_playlist_playing) {
						//automatically start playing
						play_entry(mod_playlist_current);
					}
					mod_sendmsgf(MSGTYPE_INFO, "browser info 1 added (%d)", mod_playlist_array_count);
				}


			} else if ( !strcasecmp(c2, "done")) {
				log_printf(LOG_NORMAL, "Playlist done\n");
				if (mod_playlist_sort_mode) {
					mod_playlist_sort();
				}
				mod_playlist_shufflecalc(1);
				if(mod_playlist_current) {
					play_entry(mod_playlist_current);
				} else {
					mod_sendmsg(MSGTYPE_PLAYER, "stop");
					log_printf(LOG_NORMAL, "Empty playlist - playing stopped!\n");
					mod_playlist_playing = 0;
				}

				// sort playlist
			} else if (!strcasecmp(c2, "sort")) {
				i = mod_playlist_sort();
				mod_sendmsgf(MSGTYPE_INFO,"sorted: %d", i);

				// clear playlist
			} else if (!strcasecmp(c2, "clear")) {
				mod_sendmsg(MSGTYPE_PLAYER, "stopall");
				mod_playlist_free();
				mod_playlist_playing = 0;

				// clear playlist w/o stop playing
			} else if (!strcasecmp(c2, "clearnostop")) {
				mod_playlist_free();

				// jump/skip within the playlist
			} else if (!strcasecmp(c2, "jump") ) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				if (*c3!='+' && *c3!='-') {
					mod_playlist_current = mod_playlist_list;
				}

				if 		(*c3 == '-') 	i=atoi(c3);
				else if (*c3 == '+') 	i=atoi(c3+1);
				else 					i=atoi(c3);

				if (i != -1 || secpos <= mod_playlist_skip_back_timeout || mod_playlist_skip_back_timeout == -1) {
					mod_playlist_skip(i, 0);
				}
				if (mod_playlist_current) {
					play_entry(mod_playlist_current);
				} else {
					mod_sendmsg(MSGTYPE_PLAYER, "stop");
					mod_playlist_playing = 0;
				}

				// jump/skip within playlist directories
			} else if (!strcasecmp(c2, "jumpdir") ) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				if (*c3!='+' && *c3!='-')
					mod_playlist_current = mod_playlist_list;
				if 		(*c3 == '-') 	i=atoi(c3);
				else if (*c3 == '+') 	i=atoi(c3+1);
				else 					i=atoi(c3);
				mod_playlist_skipdir(i);
				if (mod_playlist_current)
					play_entry(mod_playlist_current);
				else {
					mod_sendmsg(MSGTYPE_PLAYER, "stop");
					mod_playlist_playing = 0;
				}

				// toggle shuffle mode
			} else if ( !strcasecmp(c2, "shuffle")) {
				c3 = strtok(NULL, "");
				if (!c3) {
					mod_playlist_shuffle = mod_playlist_shuffle ? 0 : 1;
				} else {
					mod_playlist_shuffle = atoi(c3);
				}
				mod_playlist_shuffle %= 2;
				if (mod_playlist_shuffle) {
					mod_playlist_shufflecalc(0);
				}
				mod_sendmsgf(MSGTYPE_EVENT,"shuffle %d", mod_playlist_shuffle);
				log_printf(LOG_NORMAL, "Shuffle %s\n", mod_playlist_shuffle ? "on" : "off");

				// toggle repeat mode
			} else if (!strcasecmp(c2, "repeat")) {
				c3 = strtok(NULL, "");
				if (!c3) {
					mod_playlist_repeat++;
				} else {
					mod_playlist_repeat = atoi(c3);
				}
				mod_playlist_repeat %= 3;
				mod_sendmsgf(MSGTYPE_EVENT,"repeat %d", mod_playlist_repeat);
				log_printf(LOG_NORMAL, "Repeat %s\n", mod_playlist_repeat==0 ? "off" : (mod_playlist_repeat==1 ? "current" : "all"));
			}
	
			// receive stop
		} else if (!strcasecmp(c1, "stop")) {
		    mod_playlist_playing = 0;
		}

		// handle player messages
	} else if (msgtype == MSGTYPE_PLAYER) {
		c1 =  strtok(msg, " \t") ;
		if(!c1) return;

		// reset times
		if ( !strcasecmp(c1, "time")) {
			c2 = strtok(NULL, " \t");
			c3 = strtok(NULL, "");
			secpos = c2 ? atoi(c2) : 0;
			secremain = c3 ? atoi(c3) : 0;
		}
	} else if (msgtype == MSGTYPE_QUERY) {
		c1 =  strtok(msg, " \t");
		if(!c1) return;
		if ( !strcasecmp(c1,"plpos")) {
			if (!mod_playlist_list) {
				mod_sendmsg(MSGTYPE_INFO,"plpos: 0 0");
			} else {
				mod_sendmsgf(MSGTYPE_INFO,"plpos: %d %d", mod_playlist_current->plpos+1, mod_playlist_array_count);
			}
		} else if ( !strcasecmp(c1, "playlist")) {
			mod_sendmsgf(MSGTYPE_INFO,"playlist %s", mod_playlist_file);
		} else if ( !strcasecmp(c1, "shuffle")) {
			mod_sendmsgf(MSGTYPE_INFO,"shuffle %d", mod_playlist_shuffle);
		} else if ( !strcasecmp(c1, "repeat")) {
			mod_sendmsgf(MSGTYPE_INFO,"repeat %d", mod_playlist_repeat);
		} else if ( !strcasecmp(c1, "plfiles")) {
			i = mod_playlist_listfiles();
			mod_sendmsgf(MSGTYPE_INFO,"plfiles: total %d",i);
		} else if ( !strcasecmp(c1,"pldirs")) {
			i = mod_playlist_listdirs();
			mod_sendmsgf(MSGTYPE_INFO,"pldirs: total %d",i);
		}
	} else if (msgtype == MSGTYPE_EVENT) {
		c1 =  strtok(msg, " \t") ;
		if(!c1) return;
		if(!strcasecmp(c1,"endofsong")) {
			mod_playlist_skip(1, 0);
			if (mod_playlist_current) {
				play_entry(mod_playlist_current);
			} else {
				mod_sendmsg(MSGTYPE_PLAYER, "stop");
				mod_playlist_playing = 0;
			}
		} else if(!strcasecmp(c1,"player")) {
			c2 =  strtok(NULL, " \t") ;
			if(!c2) return;
			if(!strcasecmp(c2,"error")) {
				mod_playlist_skip(1, 0);
				if (mod_playlist_current) {
					play_entry(mod_playlist_current);
				} else {
					mod_sendmsg(MSGTYPE_PLAYER, "stop");
					mod_playlist_playing = 0;
				}
			}
		}
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_playlist_init (void)
{
	log_printf(LOG_DEBUG, "mod_playlist_init(): initializing\n");

	// allocate memory for playlist array
	mod_playlist_array = (mod_playlist_t **)malloc(mod_playlist_array_capacity*sizeof(mod_playlist_t *));
	if(!mod_playlist_array) {
		return "error allocating memory for mod_playlist_array";
	}


	// get repeat-mode from config file. defaulting to 2 = all
	mod_playlist_repeat = 
	  strcasecmp("off",config_getstr("playlist_repeat", "all"))==0?0:
	  strcasecmp("current",config_getstr("playlist_repeat", "all"))==0?1:2;

	// after playlist_skip_back_timeout seconds "jump -1" will 
	// jump to the start of the current song instead of the previous song.
	// set the variable to -1 to disable this behaviour.
	mod_playlist_skip_back_timeout = atoi(config_getstr("playlist_skip_back_timeout", "2"));

	// set automatic sort mode
	mod_playlist_sort_mode = !strcasecmp("auto", config_getstr("playlist_sort_mode", "manual"));
	if (mod_playlist_sort_mode) {
	    log_printf(LOG_DEBUG, "mod_playlist_init(): automatic playlist sorting is enabled.\n");
	} else {
	    log_printf(LOG_DEBUG, "mod_playlist_init(): automatic playlist sorting is disabled.\n");
	}

	// tell other modules what repeat-mode we are in.
	mod_sendmsgf(MSGTYPE_EVENT, "repeat %d", mod_playlist_repeat);

	// shuffle mode
	if (!strcasecmp(config_getstr("playlist_shuffle", "no"), "yes")) {
	    mod_sendmsg(MSGTYPE_INPUT, "playlist shuffle 1");
	}

	// initialize playlist
	mod_playlist_free();

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_playlist_deinit (void)
{
	// clear up playlist
	mod_playlist_free();

	// free playlist array
	free(mod_playlist_array);
	mod_playlist_array=NULL;

	log_printf(LOG_DEBUG, "mod_playlist_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
