// --------------------------------------------------------------------
// Small widgets
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipebase.h"
#include "widgets.h"

// --------------------------------------------------------------------

IpeAction::IpeAction(int cmd, const QIcon &icon,
		     const QString &text, QObject *parent)
  : QAction(icon, text, parent)
{
  iCommand = cmd;
  connect(this, SIGNAL(triggered()), SLOT(forwardTrigger()));
}

IpeAction::IpeAction(int cmd, const QString &text, QObject *parent)
  : QAction(text, parent)
{
  iCommand = cmd;
  connect(this, SIGNAL(triggered()), SLOT(forwardTrigger()));
}

void IpeAction::forwardTrigger()
{
  emit triggered(iCommand);
}

// --------------------------------------------------------------------


IpeSpinBox::IpeSpinBox(QWidget *parent, int minValue, int maxValue, int step)
  : QSpinBox(parent)
{
  setFocusPolicy(Qt::NoFocus);
  lineEdit()->setFocusPolicy(Qt::NoFocus);
  setMinimum(minValue);
  setMaximum(maxValue);
  setSingleStep(step);
}

void IpeSpinBox::set(int val, int maxValue)
{
  blockSignals(true);
  // make sure current value doesn't violate new maximum
  setValue(minimum());
  setMaximum(maxValue);
  setValue(val);
  blockSignals(false);
}

void IpeSpinBox::set(int val)
{
  blockSignals(true);
  setValue(val);
  blockSignals(false);
}

void IpeSpinBox::keyPressEvent(QKeyEvent *event)
{
  int key = event->key();

  // only a few keys are permitted
  if (('0' <= key && key <= '9') ||
      key == Qt::Key_Escape || key == Qt::Key_Enter ||
      key == Qt::Key_Backspace ||
      key == Qt::Key_Period || key == Qt::Key_Return ||
      key == Qt::Key_Tab || key == Qt::Key_Backtab ||
      key == Qt::Key_Left || key == Qt::Key_Right ||
      key == Qt::Key_Up || key == Qt::Key_Down)
    // dispatch normally
    QSpinBox::keyPressEvent(event);
  else
    event->ignore();
}

// --------------------------------------------------------------------

ArrowButton::ArrowButton() : QToolButton()
{
  setCheckable(true);
  connect(this, SIGNAL(toggled(bool)), this, SLOT(updateArrow(bool)));
}

void ArrowButton::updateArrow(bool on)
{
  if (on)
    setIcon(iArrow);
  else
    setIcon(QPixmap());
}

void ArrowButton::setArrow(QPixmap &arr)
{
  iArrow = arr;
}

// --------------------------------------------------------------------

DecimalSpinBox::DecimalSpinBox(int minValue, int maxValue, int step,
			       int decimals, QWidget *parent)
  : IpeSpinBox(parent, minValue, maxValue, step)
{
  iDecimals = decimals;
}

QString DecimalSpinBox::textFromValue(int value) const
{
  QString s1, s2;
  s1.sprintf("%d.", value / 1000);
  s2.sprintf("%03d", value % 1000);
  return s1 + s2.left(iDecimals);
}

QValidator::State DecimalSpinBox::validate(QString &text, int &pos) const
{
  IpeLex lex(IpeString(text.mid(pos).toUtf8()));
  (void) lex.GetFixed();
  lex.SkipWhitespace();
  return lex.Eos() ? QValidator::Acceptable : QValidator::Invalid;
}

int DecimalSpinBox::valueFromText(const QString &text) const
{
  IpeLex lex(IpeString(text.toUtf8()));
  IpeFixed d = lex.GetFixed();
  lex.SkipWhitespace();
  if (lex.Eos()) {
    ipeDebug("Converted %s -> %d", text.toAscii().data(), d.Internal());
    return d.Internal();
  }
    ipeDebug("Conversion error %s", text.toAscii().data());
  return 0;
}

// --------------------------------------------------------------------

ZoomSpinBox::ZoomSpinBox(QWidget *parent)
  : IpeSpinBox(parent, 10000, 1000000)
{
  // lineEdit()->setReadOnly(true);
  // setFocusPolicy(Qt::NoFocus);
  setSuffix(tr(" dpi"));
}

void ZoomSpinBox::stepBy(int steps)
{
  int res = value();
  while (steps > 0) {
    res = int(res * 1.3 + 0.5);
    if (res > maximum())
      res = maximum();
    steps--;
  }
  while (steps < 0) {
    res = int(res / 1.3 + 0.5);
    if (res < minimum())
      res = minimum();
    steps++;
  }
  setValue(res);
}

QString ZoomSpinBox::textFromValue(int value) const
{
  QString s;
  s.sprintf("%d", int(value * 0.001 + 0.5));
  return s;
}

int ZoomSpinBox::valueFromText(const QString &text) const
{
  return 1000 * QSpinBox::valueFromText(text);
}

// --------------------------------------------------------------------

void LayerBox::mouseReleaseEvent(QMouseEvent *e)
{
  QListWidgetItem *i = itemAt(e->pos());
  ipeDebug("MousePress % x", ulong(i));
  // ignore mouse outside items
  if (!i)
    return;
  if (e->button() == Qt::LeftButton) {
    QListWidget::mouseReleaseEvent(e);
  } else
    emit rightButtonPressed(i, e->globalPos());
}

// --------------------------------------------------------------------

IpeComboBox::IpeComboBox()
  : QComboBox()
{
  setFocusPolicy(Qt::NoFocus);
}

void IpeComboBox::keyPressEvent(QKeyEvent *event)
{
  event->ignore();
}

// --------------------------------------------------------------------

