/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "connman-manager.h"

#include <string.h>

#include "connman-manager-xml.h"
#include "connman-technology-xml.h"
#include "connman.h"
#include "marshal.h"
#include "connman-service.h"

G_DEFINE_TYPE(ConnmanManager, connman_manager, G_TYPE_OBJECT)

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE((o), CONNMAN_TYPE_MANAGER, \
			       ConnmanManagerPrivate))

typedef struct _ConnmanManagerPrivate ConnmanManagerPrivate;

struct _ConnmanManagerPrivate {
  /*
   * I'm not sure if a hashtable is the best choise, especially because we
   * need traverse it quite often. But currently the number of services is
   * so low that it doesn't really matter.
   */
  GHashTable *services;
  GHashTable *technologies;

  GDBusProxy *proxy;
  gboolean connected;
  gboolean initialised;

  guint watch_id;

  /* used only by get_services() */
  ConnmanService **array;

  /* current default service, visible to clients */
  ConnmanService *default_service;

  /*
   * the new default service but waiting for it to be ready, not visible to
   * clients
   */
  ConnmanService *pending_default_service;

  ConnmanTechnologyState wifi_state;
  ConnmanTechnologyState ethernet_state;
  ConnmanTechnologyState cellular_state;
  ConnmanTechnologyState bluetooth_state;

  gboolean offline_mode;
};

struct technology_type_string
{
  const gchar *str;
  ConnmanTechnologyType type;
};

static const struct technology_type_string technology_type_map[] = {
  { "ethernet",		CONNMAN_TECHNOLOGY_TYPE_ETHERNET },
  { "wifi",		CONNMAN_TECHNOLOGY_TYPE_WIFI },
  { "bluetooth",	CONNMAN_TECHNOLOGY_TYPE_BLUETOOTH },
  { "cellular",		CONNMAN_TECHNOLOGY_TYPE_CELLULAR },
};

struct technology_state_string
{
  const gchar *str;
  ConnmanTechnologyState state;
};

static const struct technology_state_string technology_state_map[] = {
  { "offline",		CONNMAN_TECHNOLOGY_STATE_OFFLINE },
  { "available",	CONNMAN_TECHNOLOGY_STATE_AVAILABLE },
  { "enabled",		CONNMAN_TECHNOLOGY_STATE_ENABLED },
  { "connected",	CONNMAN_TECHNOLOGY_STATE_CONNECTED },
};

struct technology  {
  GDBusProxy *proxy;
  ConnmanTechnologyType type;
  ConnmanTechnologyState state;
};

enum {
  SERVICE_ADDED_SIGNAL,
  SERVICE_REMOVED_SIGNAL,
  LAST_SIGNAL,
};

static guint signals[LAST_SIGNAL] = { 0 };

enum
{
  /* reserved */
  PROP_0,

  PROP_CONNECTED,
  PROP_DEFAULT_SERVICE,
  PROP_WIFI_STATE,
  PROP_ETHERNET_STATE,
  PROP_CELLULAR_STATE,
  PROP_BLUETOOTH_STATE,
  PROP_OFFLINE_MODE,
};

static const gchar *technology_type2str(ConnmanTechnologyType type)
{
  const struct technology_type_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(technology_type_map); i++) {
    s = &technology_type_map[i];
    if (s->type == type)
      return s->str;
  }

  g_warning("%s(): unknown technology type %d", __func__, type);

  return "unknown";
}

static void enable_technology_cb(GObject *object, GAsyncResult *res,
				 gpointer user_data)
{
  GSimpleAsyncResult *simple = G_SIMPLE_ASYNC_RESULT(user_data);
  GDBusProxy *proxy = G_DBUS_PROXY(object);
  GError *error = NULL;

  g_dbus_proxy_call_finish(proxy, res, &error);

  if (error != NULL) {
    g_simple_async_result_set_from_error(simple, error);
    g_error_free(error);
    goto out;
  }

 out:
  g_simple_async_result_complete(simple);
  g_object_unref(simple);
}

void connman_manager_enable_technology(ConnmanManager *self,
				       ConnmanTechnologyType type,
				       GCancellable *cancellable,
				       GAsyncReadyCallback callback,
				       gpointer user_data)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GSimpleAsyncResult *simple;
  GVariant *parameters;
  const gchar *s;

  g_return_if_fail(CONNMAN_IS_MANAGER(self));
  g_return_if_fail(priv != NULL);

  simple = g_simple_async_result_new(G_OBJECT(self), callback,
				     user_data,
				     connman_manager_enable_technology);

  s = technology_type2str(type);
  parameters = g_variant_new("(s)", s);

  /* FIXME: cancel the call on dispose */
  g_dbus_proxy_call(priv->proxy, "EnableTechnology", parameters,
  		    G_DBUS_CALL_FLAGS_NONE, CONNECT_TIMEOUT, cancellable,
  		    enable_technology_cb, simple);
}

void connman_manager_enable_technology_finish(ConnmanManager *self,
					      GAsyncResult *res,
					      GError **error)
{
  GSimpleAsyncResult *simple = G_SIMPLE_ASYNC_RESULT(res);

  g_return_if_fail(CONNMAN_IS_MANAGER(self));
  g_return_if_fail(g_simple_async_result_get_source_tag(simple) ==
		   connman_manager_enable_technology);

  g_simple_async_result_propagate_error(simple, error);
}

static void disable_technology_cb(GObject *object, GAsyncResult *res,
				 gpointer user_data)
{
  GSimpleAsyncResult *simple = G_SIMPLE_ASYNC_RESULT(user_data);
  GDBusProxy *proxy = G_DBUS_PROXY(object);
  GError *error = NULL;

  g_dbus_proxy_call_finish(proxy, res, &error);

  if (error != NULL) {
    g_simple_async_result_set_from_error(simple, error);
    g_error_free(error);
    goto out;
  }

 out:
  g_simple_async_result_complete(simple);
  g_object_unref(simple);
}

void connman_manager_disable_technology(ConnmanManager *self,
				       ConnmanTechnologyType type,
				       GCancellable *cancellable,
				       GAsyncReadyCallback callback,
				       gpointer user_data)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GSimpleAsyncResult *simple;
  GVariant *parameters;
  const gchar *s;

  g_return_if_fail(CONNMAN_IS_MANAGER(self));
  g_return_if_fail(priv != NULL);

  simple = g_simple_async_result_new(G_OBJECT(self), callback,
				     user_data,
				     connman_manager_disable_technology);

  s = technology_type2str(type);
  parameters = g_variant_new("(s)", s);

  /* FIXME: cancel the call on dispose */
  g_dbus_proxy_call(priv->proxy, "DisableTechnology", parameters,
  		    G_DBUS_CALL_FLAGS_NONE, CONNECT_TIMEOUT, cancellable,
  		    disable_technology_cb, simple);
}

void connman_manager_disable_technology_finish(ConnmanManager *self,
					       GAsyncResult *res,
					       GError **error)
{
  GSimpleAsyncResult *simple = G_SIMPLE_ASYNC_RESULT(res);

  g_return_if_fail(CONNMAN_IS_MANAGER(self));
  g_return_if_fail(g_simple_async_result_get_source_tag(simple) ==
		   connman_manager_disable_technology);

  g_simple_async_result_propagate_error(simple, error);
}

static void connect_service_cb(GObject *object, GAsyncResult *res,
			       gpointer user_data)
{
  GSimpleAsyncResult *simple = G_SIMPLE_ASYNC_RESULT(user_data);
  GDBusProxy *proxy = G_DBUS_PROXY(object);
  GError *error = NULL;

  g_dbus_proxy_call_finish(proxy, res, &error);

  if (error != NULL) {
    g_simple_async_result_set_from_error(simple, error);
    g_error_free(error);
    goto out;
  }

 out:
  g_simple_async_result_complete(simple);
  g_object_unref(simple);
}

void connman_manager_connect_service(ConnmanManager *self,
				     ConnmanServiceType type,
				     ConnmanServiceMode mode,
				     ConnmanServiceSecurity security,
				     guint8 *ssid,
				     guint ssid_len,
				     GCancellable *cancellable,
				     GAsyncReadyCallback callback,
				     gpointer user_data)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  gchar ssid_buf[ssid_len + 1];
  GSimpleAsyncResult *simple;
  GVariantBuilder *builder;
  GVariant *parameters;

  g_return_if_fail(CONNMAN_IS_MANAGER(self));
  g_return_if_fail(priv != NULL);

  simple = g_simple_async_result_new(G_OBJECT(self), callback,
                                      user_data,
				     connman_manager_connect_service);

  /*
   * FIXME: ssid can contain any character possible, including null, but
   * for now null terminate the buffer with null and send it as a string to
   * connman. But this needs to be properly fixed at some point.
   */
  memcpy(ssid_buf, ssid, ssid_len);
  ssid_buf[ssid_len] = '\0';

  builder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));
  g_variant_builder_add(builder, "{sv}", "Type",
			g_variant_new_string(connman_service_type2str(type)));
  g_variant_builder_add(builder, "{sv}", "Mode",
			g_variant_new_string(connman_service_mode2str(mode)));
  g_variant_builder_add(builder, "{sv}", "Security",
			g_variant_new_string(connman_service_security2str(security)));
  g_variant_builder_add(builder, "{sv}", "SSID",
			g_variant_new_string(ssid_buf));

  parameters = g_variant_new("(a{sv})", builder);

  /* FIXME: cancel the call on dispose */
  g_dbus_proxy_call(priv->proxy, "ConnectService", parameters,
  		    G_DBUS_CALL_FLAGS_NONE, CONNECT_TIMEOUT, cancellable,
  		    connect_service_cb, simple);

  /* FIXME: doesn't builder leak here? */
}

void connman_manager_connect_service_finish(ConnmanManager *self,
					    GAsyncResult *res,
					    GError **error)
{
  GSimpleAsyncResult *simple = G_SIMPLE_ASYNC_RESULT(res);

  g_return_if_fail(CONNMAN_IS_MANAGER(self));
  g_return_if_fail(g_simple_async_result_get_source_tag(simple) ==
		   connman_manager_connect_service);

  g_simple_async_result_propagate_error(simple, error);
}

gboolean connman_manager_get_connected(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_MANAGER(self), FALSE);
  g_return_val_if_fail(priv != NULL, FALSE);

  return priv->connected;
}

static void iterate_create_list(gpointer key, gpointer value,
				gpointer user_data)
{
  GList **l = user_data;
  ConnmanService *service = value;

  *l = g_list_append(*l, service);
}

/**
 * connman_manager_get_services:
 *
 * @self: self
 *
 * Returns: (array zero-terminated=1) (array length=false) (element-type Connman.Service) (transfer none): Null terminated list of services,
 * owned by ConnmanManager. Caller must create a reference if it uses them.
 */
ConnmanService **connman_manager_get_services(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  ConnmanService *service;
  GList *l;
  gint i;

  g_return_val_if_fail(CONNMAN_IS_MANAGER(self), NULL);
  g_return_val_if_fail(priv != NULL, NULL);

  if (!priv->connected)
    return NULL;

  if (priv->services == NULL)
    return NULL;

  l = NULL;

  g_hash_table_foreach(priv->services, iterate_create_list, &l);

  g_free(priv->array);

  /* allocate space for the pointers and the null termination */
  priv->array = g_malloc((sizeof(ConnmanService *) * g_list_length(l)) + 1);

  if (priv->array == NULL)
    return NULL;

  for (i = 0; l != NULL; l = l->next) {
    service = l->data;

    /*
     * return only services which are ready yet, there will be
     * a separate service added signal for the ones which are not ready yet
     */
    if (!connman_service_is_ready(service))
      continue;

    priv->array[i++] = service;
  }

  priv->array[i] = NULL;

  g_list_free(l);
  l = NULL;

  return priv->array;
}

/**
 * connman_manager_get_service:
 *
 * @self: self
 * @path: path
 *
 * Returns: (transfer none): The service or null if not found.
 */
ConnmanService *connman_manager_get_service(ConnmanManager *self,
					    const gchar *path)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_MANAGER(self), NULL);
  g_return_val_if_fail(priv != NULL, NULL);

  return g_hash_table_lookup(priv->services, path);
}

/**
 * connman_manager_get_default_service:
 *
 * @self: self
 *
 * Returns: (transfer none): The service or null if not found.
 */
ConnmanService *connman_manager_get_default_service(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_MANAGER(self), NULL);
  g_return_val_if_fail(priv != NULL, NULL);

  return priv->default_service;
}

ConnmanTechnologyState connman_manager_get_wifi_state(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_MANAGER(self),
		       CONNMAN_TECHNOLOGY_STATE_UNKNOWN);
  g_return_val_if_fail(priv != NULL, CONNMAN_TECHNOLOGY_STATE_UNKNOWN);

  return priv->wifi_state;
}

ConnmanTechnologyState connman_manager_get_ethernet_state(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_MANAGER(self),
		       CONNMAN_TECHNOLOGY_STATE_UNKNOWN);
  g_return_val_if_fail(priv != NULL, CONNMAN_TECHNOLOGY_STATE_UNKNOWN);

  return priv->ethernet_state;
}

ConnmanTechnologyState connman_manager_get_cellular_state(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_MANAGER(self),
		       CONNMAN_TECHNOLOGY_STATE_UNKNOWN);
  g_return_val_if_fail(priv != NULL, CONNMAN_TECHNOLOGY_STATE_UNKNOWN);

  return priv->cellular_state;
}

ConnmanTechnologyState connman_manager_get_bluetooth_state(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_MANAGER(self),
		       CONNMAN_TECHNOLOGY_STATE_UNKNOWN);
  g_return_val_if_fail(priv != NULL, CONNMAN_TECHNOLOGY_STATE_UNKNOWN);

  return priv->bluetooth_state;
}

gboolean connman_manager_get_offline_mode(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_val_if_fail(CONNMAN_IS_MANAGER(self), FALSE);
  g_return_val_if_fail(priv != NULL, FALSE);

  return priv->offline_mode;
}

void connman_manager_set_offline_mode(ConnmanManager *self, gboolean mode)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(CONNMAN_IS_MANAGER(self));
  g_return_if_fail(priv != NULL);

  g_object_set(self, "offline-mode", mode, NULL);
}

static void update_default_service(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  gboolean changed = FALSE;
  ConnmanService *pending;

  g_return_if_fail(priv != NULL);

  pending = priv->pending_default_service;

  if (pending == NULL) {
    if (priv->default_service == NULL)
      goto out;

    /* remove current default service because there aren't any services */
    g_object_unref(priv->default_service);
    priv->default_service = NULL;
    changed = TRUE;
    goto check;
  }

  /* no need to do anything if the service is already the default */
  if (priv->default_service == pending)
    goto check;

  if (!connman_service_is_ready(pending))
    /* wait for the ready signal from the service */
    goto check;

  switch(connman_service_get_state(pending)) {
  case CONNMAN_SERVICE_STATE_IDLE:
  case CONNMAN_SERVICE_STATE_FAILURE:
  case CONNMAN_SERVICE_STATE_ASSOCIATION:
  case CONNMAN_SERVICE_STATE_CONFIGURATION:
  case CONNMAN_SERVICE_STATE_DISCONNECT:
    /* pending isn't connected and we can't use it as default, yet  */
    goto check;
  case CONNMAN_SERVICE_STATE_READY:
  case CONNMAN_SERVICE_STATE_LOGIN:
  case CONNMAN_SERVICE_STATE_ONLINE:
    break;
  }

  /* pending service is now ready, let's switch to it */
  if (priv->default_service != NULL)
    g_object_unref(priv->default_service);

  /* the ownership moves from pending_default_service to default_service */
  priv->default_service = pending;
  priv->pending_default_service = NULL;
  changed = TRUE;

 check:
  if (priv->default_service == NULL)
    goto out;

  /*
   * Let's check the state of the default service and whether it needs to
   * be removed.
   */
  switch(connman_service_get_state(priv->default_service)) {
  case CONNMAN_SERVICE_STATE_IDLE:
  case CONNMAN_SERVICE_STATE_FAILURE:
  case CONNMAN_SERVICE_STATE_ASSOCIATION:
  case CONNMAN_SERVICE_STATE_CONFIGURATION:
  case CONNMAN_SERVICE_STATE_DISCONNECT:
    /* remove the default service because it's not valid anymore */
    g_object_unref(priv->default_service);
    priv->default_service = NULL;
    changed = TRUE;
    break;
  case CONNMAN_SERVICE_STATE_READY:
  case CONNMAN_SERVICE_STATE_LOGIN:
  case CONNMAN_SERVICE_STATE_ONLINE:
    break;
  }

 out:
  /* don't send any signals until connected to connmand */
  if (priv->connected && changed)
    g_object_notify(G_OBJECT(self), "default-service");
}

static void iterate_service_ready(gpointer key, gpointer value,
				  gpointer user_data)
{
  gboolean *not_ready = user_data;
  ConnmanService *service = value;

  if (!connman_service_is_ready(service))
    *not_ready = TRUE;
}

static gboolean all_services_ready(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  gboolean not_ready = FALSE;

  g_hash_table_foreach(priv->services, iterate_service_ready, &not_ready);

  return !not_ready;
}

static void service_ready(ConnmanService *service, GParamSpec *pspec,
			  gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  if (!connman_service_is_ready(service))
    return;

  if (priv->pending_default_service != NULL &&
      priv->pending_default_service == service)
    update_default_service(self);

  /*
   * Send service added signals only after we are connected. The client
   * needs to call get_services() when it has received the connected
   * signal.
   */
  if (!priv->connected) {
    if (all_services_ready(self)) {
      priv->connected = TRUE;
      g_object_notify(G_OBJECT(self), "connected");
    }

    return;
  }

  g_signal_emit(self, signals[SERVICE_ADDED_SIGNAL], 0, service);
}

static void service_state_notify(ConnmanService *service, GParamSpec *pspec,
				 gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  if (priv->pending_default_service == service ||
      priv->default_service == service)
    update_default_service(self);
}

static ConnmanService *create_service(ConnmanManager *self,
				      const gchar *path)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  ConnmanService *service;

  service = connman_service_new(path);

  g_hash_table_insert(priv->services, g_strdup(path), service);

  /* FIXME: disconnect ready handler */
  g_signal_connect(service, "notify::ready", G_CALLBACK(service_ready),
		   self);
  g_signal_connect(service, "notify::state",
		   G_CALLBACK(service_state_notify), self);

  return service;
}

static void remove_service(ConnmanManager *self, ConnmanService *service)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  const gchar *path = connman_service_get_path(service);

  /*
   * Send service removed signals only when connected, connected
   * property with false value implies that services are lost.
   */
  if (priv->connected)
    g_signal_emit(self, signals[SERVICE_REMOVED_SIGNAL], 0, path);

  g_signal_handlers_disconnect_by_func(service,
				       G_CALLBACK(service_state_notify),
				       self);

  g_object_unref(service);
}

static GHashTable *create_services_list(ConnmanManager *self)
{
  return g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
}

static void notify_service_removal(gpointer key, gpointer value,
				   gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  ConnmanService *service = CONNMAN_SERVICE(value);

  remove_service(self, service);
}

static void remove_services_list(ConnmanManager *self, GHashTable *list)
{
  g_hash_table_foreach(list, notify_service_removal, self);
  g_hash_table_destroy(list);
}

static void update_services(ConnmanManager *self, GVariant *variant)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GHashTable *old_services;
  ConnmanService *service;
  gchar *path;
  GVariantIter iter;
  gint count;

  old_services = priv->services;
  priv->services = create_services_list(self);

  g_variant_iter_init(&iter, variant);
  count = 0;

  while (g_variant_iter_next(&iter, "o", &path)) {
    /* try find an existing service */
    if (old_services != NULL) {
      service = g_hash_table_lookup(old_services, path);

      if (service != NULL) {
	/* there's already an object for this service */
	g_hash_table_remove(old_services, path);
	g_hash_table_insert(priv->services, path, service);
	goto next;
      }
    }

    service = create_service(self, path);
    g_free(path);

  next:
    if (count++ == 0) {
      if (priv->pending_default_service != NULL)
	g_object_unref(priv->pending_default_service);

      priv->pending_default_service = g_object_ref(service);
      update_default_service(self);
    }
  }

  /* remove services which are not visible anymore */
  if (old_services != NULL) {
    remove_services_list(self, old_services);
    old_services = NULL;
  }
}

static ConnmanTechnologyType technology_str2type(const gchar *type)
{
  const struct technology_type_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(technology_type_map); i++) {
    s = &technology_type_map[i];
    if (g_strcmp0(s->str, type) == 0)
      return s->type;
  }

  g_warning("unknown technology type %s", type);

  return CONNMAN_TECHNOLOGY_TYPE_UNKNOWN;
}

static ConnmanTechnologyState technology_str2state(const gchar *state)
{
  const struct technology_state_string *s;
  guint i;

  for (i = 0; i < G_N_ELEMENTS(technology_state_map); i++) {
    s = &technology_state_map[i];
    if (g_strcmp0(s->str, state) == 0)
      return s->state;
  }

  g_warning("unknown technology state %s", state);

  return CONNMAN_TECHNOLOGY_STATE_UNKNOWN;
}

static void technology_notify_state(ConnmanManager *self,
				    struct technology *tech)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  ConnmanTechnologyState *state;
  const gchar *property = NULL;

  g_return_if_fail(tech != NULL);
  g_return_if_fail(tech->state != CONNMAN_TECHNOLOGY_STATE_UNKNOWN);

  switch (tech->type) {
  case CONNMAN_TECHNOLOGY_TYPE_UNKNOWN:
    /* type is not yet retrieved, can happen with get_properties() */
    return;
  case CONNMAN_TECHNOLOGY_TYPE_ETHERNET:
    state = &priv->ethernet_state;
    property = "ethernet-state";
    break;
  case CONNMAN_TECHNOLOGY_TYPE_WIFI:
    state = &priv->wifi_state;
    property = "wifi-state";
    break;
  case CONNMAN_TECHNOLOGY_TYPE_BLUETOOTH:
    state = &priv->bluetooth_state;
    property = "bluetooth-state";
    break;
  case CONNMAN_TECHNOLOGY_TYPE_CELLULAR:
    state = &priv->cellular_state;
    property = "cellular-state";
    break;
  default:
    g_warning("%s(): unknown type %d", __func__, tech->type);
    return;
  }

  if (*state == tech->state)
    return;

  g_return_if_fail(property != NULL);

  *state = tech->state;
  g_object_notify(G_OBJECT(self), property);
}

static void technology_state_updated(ConnmanManager *self,
				     struct technology *tech,
				     GVariant *value)
{
  const gchar *s;

  s = g_variant_get_string(value, NULL);
  tech->state = technology_str2state(s);

  technology_notify_state(self, tech);

  g_variant_unref(value);
}

static void technology_type_updated(ConnmanManager *self,
				    struct technology *tech,
				    GVariant *value)
{
  const gchar *s;

  s = g_variant_get_string(value, NULL);
  tech->type = technology_str2type(s);

  g_variant_unref(value);
}

static void technology_property_updated(ConnmanManager *self,
					GDBusProxy *proxy,
					const gchar *property,
					GVariant *value)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  struct technology *tech;

  tech = g_hash_table_lookup(priv->technologies,
			     g_dbus_proxy_get_object_path(proxy));

  if (tech == NULL) {
    g_warning("Did not find tech for %s %s update",
	      g_dbus_proxy_get_object_path(proxy), property);
    g_variant_unref(value);
    return;
  }

  if (g_strcmp0(property, CONNMAN_TECHNOLOGY_PROPERTY_STATE) == 0)
    technology_state_updated(self, tech, value);
  else if (g_strcmp0(property, CONNMAN_TECHNOLOGY_PROPERTY_TYPE) == 0)
    technology_type_updated(self, tech, value);
  else
    /* unknown property */
    g_variant_unref(value);
}

static void technology_property_changed(ConnmanManager *self,
				  GDBusProxy *proxy,
				  GVariant *parameters)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GVariant *value;
  gchar *name;

  g_return_if_fail(priv != NULL);

  g_variant_get(parameters, "(sv)", &name, &value);

  /* callee takes the ownership of the variant */
  technology_property_updated(self, proxy, name, value);

  g_free(name);
}

static void technology_g_signal(GDBusProxy *proxy,
			  const gchar *sender_name,
			  const gchar *signal_name,
			  GVariant *parameters,
			  gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);

  /*
   * gdbus documentation is not clear who owns the variant so take it, just
   * in case
   */
  g_variant_ref(parameters);

  if (g_strcmp0(signal_name, "PropertyChanged") == 0)
    technology_property_changed(self, proxy, parameters);

  g_variant_unref(parameters);
}


static void technology_get_properties_cb(GObject *source_object,
					 GAsyncResult *res,
					 gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GDBusProxy *proxy = G_DBUS_PROXY(source_object);
  GVariant *result, *value, *dict;
  struct technology *tech;
  GError *error = NULL;
  GVariantIter iter;
  gchar *name;

  g_return_if_fail(self != NULL);
  g_return_if_fail(priv != NULL);
  g_return_if_fail(proxy != NULL);

  result = g_dbus_proxy_call_finish(proxy, res, &error);

  if (error != NULL) {
    g_warning("Failed to get connman properties: %s", error->message);
    g_error_free(error);
    return;
  }

  if (result == NULL)
    return;

  /* get the dict */
  dict = g_variant_get_child_value(result, 0);

  /* go through the dict */
  g_variant_iter_init(&iter, dict);

  while (g_variant_iter_next(&iter, "{sv}", &name, &value)) {
    /* callee owns the variant now */
    technology_property_updated(self, proxy, name, value);

    g_free(name);
  }

  tech = g_hash_table_lookup(priv->technologies,
			     g_dbus_proxy_get_object_path(proxy));

  if (tech == NULL) {
    g_warning("Did not find tech for %s during the first state notify",
	      g_dbus_proxy_get_object_path(proxy));
    goto out;
  }

  /*
   * this needs to come after all properties are updated so that the type
   * is known
   */
  technology_notify_state(self, tech);

 out:
  g_variant_unref(dict);
  g_variant_unref(result);
}


static void technology_get_properties(ConnmanManager *self, GDBusProxy *proxy)
{
  g_return_if_fail(self != NULL);
  g_return_if_fail(proxy != NULL);

  g_dbus_proxy_call(proxy, "GetProperties", NULL,
  		    G_DBUS_CALL_FLAGS_NONE, -1, NULL,
  		    technology_get_properties_cb, self);

  /* FIXME: howto cancel the call during dispose? */
}

static void technology_create_proxy_cb(GObject *source_object,
				       GAsyncResult *res,
				       gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GDBusNodeInfo *introspection_data;
  GDBusInterfaceInfo *info;
  GError *error = NULL;
  GDBusProxy *proxy;

  g_return_if_fail(priv != NULL);

  proxy = g_dbus_proxy_new_finish(res, &error);

  if (error != NULL) {
    g_warning("Failed to get technology proxy: %s", error->message);
    g_error_free(error);
    return;
  }

  if (proxy == NULL) {
    g_warning("Failed to get technology proxy, but no errors");
    return;
  }

  g_signal_connect(proxy, "g-signal", G_CALLBACK(technology_g_signal), self);

  introspection_data = g_dbus_node_info_new_for_xml(connman_technology_xml,
						    NULL);
  g_return_if_fail(introspection_data != NULL);

  info = introspection_data->interfaces[0];
  g_dbus_proxy_set_interface_info(proxy, info);

  technology_get_properties(self, proxy);
}

static struct technology *technology_added(ConnmanManager *self,
					   const gchar *path)
{
  struct technology *tech;

  tech = g_malloc0(sizeof(*tech));

  g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM, G_DBUS_PROXY_FLAGS_NONE,
			   NULL, CONNMAN_SERVICE_NAME, path,
			   CONNMAN_TECHNOLOGY_INTERFACE, NULL,
			   technology_create_proxy_cb, self);

  return tech;
}

static void technology_removed(ConnmanManager *self, struct technology *tech)
{
  tech->state = CONNMAN_TECHNOLOGY_STATE_UNAVAILABLE;
  technology_notify_state(self, tech);

  if (tech->proxy != NULL)
    g_object_unref(tech->proxy);

  /* a precaution */
  memset(tech, 0, sizeof(*tech));

  g_free(tech);
}

static void technology_notify_removal(gpointer key, gpointer value,
				      gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  struct technology *tech = value;

  g_return_if_fail(self != NULL);
  g_return_if_fail(tech != NULL);

  technology_removed(self, tech);
}

static GHashTable *create_technology_table(ConnmanManager *self)
{
  return g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
}

static void destroy_technology_table(ConnmanManager *self, GHashTable *list)
{
  g_hash_table_foreach(list, technology_notify_removal, self);
  g_hash_table_destroy(list);
}

static void technologies_updated(ConnmanManager *self, GVariant *variant)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  struct technology *tech;
  GHashTable *old_techs;
  gchar *path;
  GVariantIter iter;

  g_return_if_fail(priv->technologies != NULL);

  old_techs = priv->technologies;
  priv->technologies = create_technology_table(self);

  g_variant_iter_init(&iter, variant);

  while (g_variant_iter_next(&iter, "o", &path)) {
    /* try find an existing proxy */
    if ((tech = g_hash_table_lookup(old_techs, path)) != NULL) {
	/* there's already an object for this service */
	g_hash_table_remove(old_techs, path);
    } else {
      tech = technology_added(self, path);
    }

    g_hash_table_insert(priv->technologies, path, tech);

    /* hashtable now owns path */
  }

  /* remove services which are not visible anymore */
  destroy_technology_table(self, old_techs);

  g_variant_unref(variant);
}

static void offline_mode_updated(ConnmanManager *self, GVariant *value)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  gboolean mode;

  mode = g_variant_get_boolean(value);

  if (priv->offline_mode == mode)
    return;

  priv->offline_mode = mode;
  g_object_notify(G_OBJECT(self), "offline-mode");

  g_variant_unref(value);
}

static void update_property(ConnmanManager *self, const gchar *property,
			    GVariant *variant)
{
  if (g_strcmp0(property, CONNMAN_PROPERTY_SERVICES) == 0) {
    update_services(self, variant);
  } else if (g_strcmp0(property, CONNMAN_PROPERTY_TECHNOLOGIES) == 0) {
    technologies_updated(self, variant);
  } else if (g_strcmp0(property, CONNMAN_PROPERTY_OFFLINE_MODE) == 0) {
    offline_mode_updated(self, variant);
  } else {
    /* unknown property */
    g_variant_unref(variant);
    variant = NULL;
  }
}

static void property_changed(ConnmanManager *self, GVariant *parameters)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GVariant *variant;
  gchar *name;

  g_return_if_fail(priv != NULL);

  g_variant_get(parameters, "(sv)", &name, &variant);

  /* callee takes the ownership of the variant */
  update_property(self, name, variant);

  g_free(name);
}

static void connman_manager_g_signal(GDBusProxy *proxy,
				   const gchar *sender_name,
				   const gchar *signal_name,
				   GVariant *parameters,
				   gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);

  /*
   * gdbus documentation is not clear who owns the variant so take it, just
   * in case
   */
  g_variant_ref(parameters);

  if (g_strcmp0(signal_name, "PropertyChanged") == 0)
    property_changed(self, parameters);

  g_variant_unref(parameters);
}

static void get_properties_cb(GObject *source_object, GAsyncResult *res,
			      gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GVariant *result, *variant, *dict;
  GError *error = NULL;
  GVariantIter iter, dict_iter;
  gchar *name;

  g_return_if_fail(self != NULL);
  g_return_if_fail(priv != NULL);

  result = g_dbus_proxy_call_finish(priv->proxy, res, &error);

  if (error != NULL) {
    g_warning("Failed to get connman properties: %s", error->message);
    g_error_free(error);
    return;
  }

  if (result == NULL)
    return;

  g_variant_iter_init(&iter, result);

  /* get the dict */
  dict = g_variant_iter_next_value(&iter);

  /* go through the dict */
  g_variant_iter_init(&dict_iter, dict);

  while (g_variant_iter_next(&dict_iter, "{sv}", &name, &variant)) {
    /* callee owns the variant now */
    update_property(self, name, variant);

    g_free(name);
  }

  if (!priv->initialised)
    priv->initialised = TRUE;

  if (!priv->connected && g_hash_table_size(priv->services) == 0) {
    /* no services so we can change to connected immediately */
    priv->connected = TRUE;
    g_object_notify(G_OBJECT(self), "connected");
  }

  g_variant_unref(dict);
  g_variant_unref(result);
}


static void get_properties(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(self != NULL);

  g_dbus_proxy_call(priv->proxy, "GetProperties", NULL,
  		    G_DBUS_CALL_FLAGS_NONE, -1, NULL,
  		    get_properties_cb, self);
}

static void connman_appeared(GDBusConnection *connection, const gchar *name,
			     const gchar *name_owner, gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);

  get_properties(self);
}


static void connman_vanished(GDBusConnection *connection, const gchar *name,
			     gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  /* if connman is not available, send a disconnected signal during init */

  if (priv->initialised && !priv->connected)
    return;

  priv->connected = FALSE;

  if (priv->services != NULL) {
    remove_services_list(self, priv->services);
    priv->services = NULL;
  }

  if (!priv->initialised)
    priv->initialised = TRUE;

  g_object_notify(G_OBJECT(self), "connected");
}


static void create_proxy_cb(GObject *source_object, GAsyncResult *res,
			    gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GDBusNodeInfo *introspection_data;
  GDBusInterfaceInfo *info;
  GError *error = NULL;

  g_return_if_fail(priv != NULL);

  priv->proxy = g_dbus_proxy_new_finish(res, &error);

  if (error != NULL) {
    g_warning("Failed to get connman proxy: %s", error->message);
    g_error_free(error);
    return;
  }

  if (priv->proxy == NULL) {
    g_warning("Failed to get connman proxy, but no errors");
    return;
  }

  g_signal_connect(priv->proxy, "g-signal", G_CALLBACK(connman_manager_g_signal),
		   self);

  introspection_data = g_dbus_node_info_new_for_xml(connman_manager_xml,
						    NULL);
  g_return_if_fail(introspection_data != NULL);

  info = introspection_data->interfaces[0];
  g_dbus_proxy_set_interface_info(priv->proxy, info);

  priv->watch_id = g_bus_watch_name(G_BUS_TYPE_SYSTEM, CONNMAN_SERVICE_NAME,
				    G_BUS_NAME_WATCHER_FLAGS_NONE,
				    connman_appeared,
				    connman_vanished,
				    self,
				    NULL);
}

static void set_dbus_property_cb(GObject *object,
				 GAsyncResult *res,
				 gpointer user_data)
{
  ConnmanManager *self = CONNMAN_MANAGER(user_data);
  GDBusProxy *proxy = G_DBUS_PROXY(object);
  GError *error = NULL;

  g_dbus_proxy_call_finish(proxy, res, &error);

  if (error != NULL) {
    g_warning("ConnmanManager SetProperty() failed: %s", error->message);
    g_error_free(error);
  }

  /* trick to avoid destroying self during async call */
  g_object_unref(self);
}

static void set_dbus_property(ConnmanManager *self,
			      const gchar *property,
			      GVariant *value)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GVariant *parameters;

  g_return_if_fail(CONNMAN_IS_MANAGER(self));
  g_return_if_fail(priv != NULL);

  parameters = g_variant_new("(sv)", property, value);

  g_dbus_proxy_call(priv->proxy, "SetProperty", parameters,
  		    G_DBUS_CALL_FLAGS_NONE, -1, NULL,
  		    set_dbus_property_cb, g_object_ref(self));
}

static void update_offline_mode(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);
  GVariant *value;

  value = g_variant_new("b", priv->offline_mode);

  set_dbus_property(self, "OfflineMode", value);
}

static void set_property(GObject *object, guint property_id,
			 const GValue *value, GParamSpec *pspec)
{
  ConnmanManager *self = CONNMAN_MANAGER(object);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_return_if_fail(priv != NULL);

  switch(property_id) {
  case PROP_CONNECTED:
      priv->connected = g_value_get_boolean(value);
      break;
  case PROP_DEFAULT_SERVICE:
      priv->default_service = g_value_get_object(value);
      break;
  case PROP_WIFI_STATE:
    priv->wifi_state = g_value_get_uint(value);
    break;
  case PROP_ETHERNET_STATE:
    priv->ethernet_state = g_value_get_uint(value);
    break;
  case PROP_CELLULAR_STATE:
    priv->cellular_state = g_value_get_uint(value);
    break;
  case PROP_BLUETOOTH_STATE:
    priv->bluetooth_state = g_value_get_uint(value);
    break;
  case PROP_OFFLINE_MODE:
    priv->offline_mode = g_value_get_boolean(value);
    update_offline_mode(self);
    break;
  default:
    /* We don't have any other property... */
    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
    break;
  }
}

static void get_property(GObject *object, guint property_id,
			 GValue *value, GParamSpec *pspec)
{
  ConnmanManager *self = CONNMAN_MANAGER(object);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  switch(property_id) {
  case PROP_CONNECTED:
    g_value_set_boolean(value, priv->connected);
    break;
  case PROP_DEFAULT_SERVICE:
    g_value_set_object(value, priv->default_service);
    break;
  case PROP_WIFI_STATE:
    g_value_set_uint(value, priv->wifi_state);
    break;
  case PROP_ETHERNET_STATE:
    g_value_set_uint(value, priv->ethernet_state);
    break;
  case PROP_CELLULAR_STATE:
    g_value_set_uint(value, priv->cellular_state);
    break;
  case PROP_BLUETOOTH_STATE:
    g_value_set_uint(value, priv->bluetooth_state);
    break;
  case PROP_OFFLINE_MODE:
    g_value_set_boolean(value, priv->offline_mode);
    break;
  default:
    /* We don't have any other property... */
    G_OBJECT_WARN_INVALID_PROPERTY_ID(object, property_id, pspec);
    break;
  }
}

static void connman_manager_dispose(GObject *object)
{
  ConnmanManager *self = CONNMAN_MANAGER(object);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  if (priv->services != NULL) {
    remove_services_list(self, priv->services);
    priv->services = NULL;
  }

  if (priv->technologies != NULL) {
    destroy_technology_table(self, priv->technologies);
    priv->technologies = NULL;
  }

  if (priv->proxy != NULL) {
    g_object_unref(priv->proxy);
    priv->proxy = NULL;
  }

  if (priv->watch_id != 0) {
    g_bus_unwatch_name(priv->watch_id);
    priv->watch_id = 0;
  }

  if (priv->default_service != NULL) {
    g_object_unref(priv->default_service);
    priv->default_service = NULL;
  }

  if (priv->pending_default_service != NULL) {
    g_object_unref(priv->pending_default_service);
    priv->pending_default_service = NULL;
  }

  G_OBJECT_CLASS(connman_manager_parent_class)->dispose(object);
}

static void connman_manager_finalize(GObject *object)
{
  ConnmanManager *self = CONNMAN_MANAGER(object);
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  g_free(priv->array);
  priv->array = NULL;

  G_OBJECT_CLASS(connman_manager_parent_class)->finalize(object);
}

static void connman_manager_class_init(ConnmanManagerClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
  GParamSpec *pspec;
  guint id;

  g_type_class_add_private(klass, sizeof(ConnmanManagerPrivate));

  gobject_class->dispose = connman_manager_dispose;
  gobject_class->finalize = connman_manager_finalize;
  gobject_class->set_property = set_property;
  gobject_class->get_property = get_property;

  id = g_signal_new("service-added",
		    G_TYPE_FROM_CLASS(klass),
		    G_SIGNAL_RUN_LAST,
		    0, NULL, NULL,
		    _marshal_VOID__BOXED,
		    G_TYPE_NONE, 1, CONNMAN_TYPE_SERVICE);
  signals[SERVICE_ADDED_SIGNAL] = id;

  id = g_signal_new("service-removed",
		    G_TYPE_FROM_CLASS(klass),
		    G_SIGNAL_RUN_LAST,
		    0, NULL, NULL,
		    _marshal_VOID__STRING,
		    G_TYPE_NONE, 1, G_TYPE_STRING);
  signals[SERVICE_REMOVED_SIGNAL] = id;

  pspec = g_param_spec_boolean("connected",
			       "ConnmanManager's connected property",
			       "Set connected state", FALSE,
			       G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_CONNECTED, pspec);

  pspec = g_param_spec_object("default-service",
			     "ConnmanManager's default service",
			     "Set default service",
			     CONNMAN_TYPE_SERVICE,
			     G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_DEFAULT_SERVICE, pspec);


  pspec = g_param_spec_uint("wifi-state",
			    "ConnmanManager's wifi state property",
			    "The state of the technology",
			    0, G_MAXUINT,
			    CONNMAN_TECHNOLOGY_STATE_UNKNOWN,
			    G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_WIFI_STATE, pspec);

  pspec = g_param_spec_uint("ethernet-state",
			    "ConnmanManager's ethernet state property",
			    "The state of the technology",
			    0, G_MAXUINT,
			    CONNMAN_TECHNOLOGY_STATE_UNKNOWN,
			    G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_ETHERNET_STATE, pspec);

  pspec = g_param_spec_uint("cellular-state",
			    "ConnmanManager's cellular state property",
			    "The state of the technology",
			    0, G_MAXUINT,
			    CONNMAN_TECHNOLOGY_STATE_UNKNOWN,
			    G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_CELLULAR_STATE, pspec);

  pspec = g_param_spec_uint("bluetooth-state",
			    "ConnmanManager's bluetooth state property",
			    "The state of the technology",
			    0, G_MAXUINT,
			    CONNMAN_TECHNOLOGY_STATE_UNKNOWN,
			    G_PARAM_READABLE);
  g_object_class_install_property(gobject_class, PROP_BLUETOOTH_STATE, pspec);

  pspec = g_param_spec_boolean("offline-mode",
			       "ConnmanManager's offline mode property",
			       "Set offline mode", FALSE,
			       G_PARAM_READWRITE);
  g_object_class_install_property(gobject_class, PROP_OFFLINE_MODE, pspec);
}

static void connman_manager_init(ConnmanManager *self)
{
  ConnmanManagerPrivate *priv = GET_PRIVATE(self);

  priv->services = NULL;
  priv->technologies = create_technology_table(self);

  priv->proxy = NULL;
  priv->connected = FALSE;
  priv->initialised = FALSE;

  priv->default_service = NULL;
  priv->pending_default_service = NULL;

  g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM, G_DBUS_PROXY_FLAGS_NONE,
			   NULL, CONNMAN_SERVICE_NAME, CONNMAN_MANAGER_PATH,
			   CONNMAN_MANAGER_INTERFACE, NULL,
			   create_proxy_cb, self);
}

ConnmanManager *connman_manager_new(void)
{
  return g_object_new(CONNMAN_TYPE_MANAGER, NULL);
}
