# -*- coding: iso-8859-1 -*-
###############################################################################
# begin                : Sat Dec 14 2002
# copyright            : (C) 2003 by Ricardo Niederberger Cabral
# email                : nieder at mail dot ru
#
###############################################################################
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###############################################################################

"""Database module

    Main database module. Responsible for image metadata, grouping, similarity grouping.
    Also responsible for providing GUI with transparent ways for generating thumbnails, loading image files and embedded metadata.
"""

try:
    import sys
    import os
    import traceback
    import md5                          # hashing filenames to generate thumbnails
    import random                       # for generating unique id's
    import time                         # to test performance increase during development, and generating batch names
    import marshal                      # for loading and saving img databases
    from string import *                # yeah, this is really crappy, i know
    import Error
except:
    Error.PrintTB("Error importing the necessary python modules. Unable to continue.")
    sys.exit()
try:
    import EXIF
except:
    Error.PrintTB("Error importing EXIF module.")
try:
    from qt import *
except:
    Error.PrintTB("You system doesn't seem to have PyQT installed. Please install it before running this application. See http://www.riverbankcomputing.co.uk/pyqt/download.php and http://imgseek.sourceforge.net/requirements.html")

try:
    import imgdb
except:
    Error.PrintTB("Warning: Unable to load the C++ extension \"imgdb.so\" module. Make sure you installed imgSeek correctly, and email any install bug to \"imgseek-devel@lists.sourceforge.net\".")
    sys.exit()

####### Optional modules
## IPTC from PIL
try:
    import IptcExtract
except:
    pass
try:
    import Image
except:
    pass

class AddFilter:
    """This class should hold all restrictions that a user would to apply on a database Add """

    def __init__(self,env):
        # igntext and minsize and mindim are file criteria to ignore
        self.igntext = ""
        self.minsize = None
        self.mindim = 0
        self.ignext = env.doc_ext
        self.exexif = 0
        self.exiptc = 0
        self.followsym = 1
        self.mounted = 0
        self.probeext = 0
        self.addfname = ""
        self.volid = 1
        self.groupid = 1
        self.removeEmptyGroup = 0       # if restr.removeEmptyGroup is set and this dir has no images, then remove the group this dir should belong to

    def getFromDialog(self,dlg):
        """get restrictions from the add tab"""
        ## not sure if this code should be on imgSeek::addImageBtn_clicked or here.
        pass

class ImgDB:
    """ this is the class holding all logical db data

    A better description of the most important members are on reset()
    """
    def tr(self,msg):
        try:
            if hasattr(self.env, 'qapp'):
                return self.env.qapp.translate('main',msg)
            else:
                return msg
        except:
            Error.PrintTB("translating at imgdb")
            return msg
    
    def __init__(self,env):
        self.dbversion = 10             # saved on the beginning of marshalled db files. For version db detection and handling
        self.fname = ""                 # the filename of the currently opened database
        self.env = env
        self.imgdb = imgdb
        #### image metadata:
        self.imageParms = ["Description",
                           "Dimensions",
                           "Filesize",
                           "Filename",
                           "Format",
                           "Modify date",
                           "Database date",
                           "Modify_date_epoch",
                           "Mounted"] # default meta keys all image should have

        self.blankMetaDict = {}           # blank dictionary with available keys. To be copied later to every new db file
        for par in self.imageParms:
            self.blankMetaDict[par] = None
        self.dirty = 0
        self.imgexts = [ 'jpeg', 'jpg', 'gif', 'png', 'rgb', 'pbm', 'pgm', 'ppm', 'tiff', 'tif', 'rast', 'xbm', 'bmp' ] # to help determining img format from extension
        #### Thumbnail stuff
        import platform
        # as explained on http://www.brunningonline.net/simon/blog/archives/001168.html
        if platform.release().find('98') != -1: # Windows 98 doesn't understand ~home dirs ...
            self.thdir = os.path.join(os.path.split(sys.argv[0])[0],".thumbnails","")
            self.thdir2 = os.path.join(os.path.split(sys.argv[0])[0],".thumbnails","normal","")
        else:
            self.thdir = os.path.expanduser(os.path.join("~",".thumbnails","")) # path for storing thumbnails. Calculate it only once here.
            self.thdir2 = os.path.expanduser(os.path.join("~",".thumbnails","normal","")) # path for storing thumbnails. Calculate it only once here.
        # just to make sure the thumbnail dir exists, so I dont need to be checking it over and over on self.getThumb()
        if not os.path.exists(self.thdir):
            os.mkdir(self.thdir)
        if not os.path.exists(self.thdir2):
            os.mkdir(self.thdir2)

        #### Callbacks (lists of functions that should be called on changes
        self.wnd = None
        self.app = None
        # cbs maps what could change to a list of functions that will get callled when that thing changes
        # these functions should accept a map describing what changed on db
        self.cbs = {}
        self.cbs["Batch"] = []
        self.cbs["Dir"] = []
        self.cbs["Group"] = []
        self.cbs["SimGroup"] = []
        self.cbs["Img"] = []
        self.cbs["Volume"] = []
        self.cbs["Meta"] = []
        self.cbs["Database"] = []         # generic changes like, dbname, location. etc
        self.sysdirbmCb = []              # list of functions that should get called when a bookmark is changed. #TODO9: this should go to the new callback system (dbCallbacks.py)

        ### Init C++ db engine
        self.doNotSave = 0                # flag to abort automatic db saving
        self.reset()
        imgdb.initDbase()
        ### Misc inits
        self.readonlyfields = ["Filesize","Filename","Format","Dimensions","Modify date","Database date","Volume","Mounted"] # default readonly metadata
        self.invisiblefields = ["ViewRotate","Modify_date_epoch"] # fields which shouldn't even appear on the metadata dialog

    def changedDB(self,what = None):
        """helper function, call it whenever something on db is changed and the GUI should know of

        @param what: describe the change
        @type what: dict
        """
        self.dirty = 1
        if not what:
            print "Changes should have a reason."
            return
        try:
            for cb in self.cbs[what["scope"]]:
                cb(what)
        except:
            Error.PrintTB("db callback system error - no scope")

    def remove_dead(self):
        """removes any non existant file on database """
        remd = []
        self.doNotSave = 1
        for fidx in self.img.keys():
            if (self.meta[fidx].get("Mounted","no") == "no") and (not os.path.exists(self.img[fidx][0])): # only remove unmounted imgs
                self.removeFile(fidx)
                remd.append(fidx)
        self.doNotSave = 0
        self.dirty = 1
        self.changedDB({"scope":"Img","reason":"removed","subjects":remd})
        return len(remd)

    def rescan(self):
        """recalculate coefficients for all images on the database and regenerate the db buckets  """
        print "This hasn't been implemented yet !!!"

    def reset(self):
        """ reset current db, should be called on File-> New """
        ## reseting all data structures
        self.img = {}                   # key is id, value is [0-fullpath, 1-parent dir id, 2-[group ids]]
        self.meta = {}                  # key is id, value is dictionary of meta values
        self.dirs = {}                  # key id, value is [0-full path (/ at end),1-parent dir id,2-child dir ids,3-child files ids,4-child imgs filenames,5-description,6-volumeid]
        self.volumes = {}               # key id, value is [0-list of dir ids on this volume,1-base path,2-description,3-name]
        self.batches = {}               # key id, value is [name,contents]
                                        # Batch contents types:
                                        # "SysDir", "Dir", "Group",  "SimGroup",  "Img", "Volume", "File" (id is the path)
        self.groups = {}                # key is group id, value is [0-name,1-description,2-parent id, 3-child imgs id, 4-child groups]
        self.opened = 0                 # db has been succesfully opened
        self.dirty = 0                  # has db been changed since last change ?
        self.openGroups = []            # group ids with listvierw item expanded (so ui will set the listview state when it changed)
        self.openDirs = []              # dir ids with listvierw item expanded (so ui will set the listview state when it changed)
        self.simgroups = {}             # key is id value is [0-group name,1-main img id,2-list of similar images ids].  This list points to the group map of the currently selected similarity grouping mode
        self.contSimGroups = {}         # content similarity groups
        self.colorSimGroups = {}        # color similarity groups
        self.dateSimGroups = {}         # date similarity groups
        self.fileSimGroups = {}         # filename similarity groups
        # UI related structures:
        self.metafields = self.imageParms[:] # history of entered metafields
        self.textqueryhistory = []      # key is text str, value is list of [field,value] pairs
        self.sysdirbm = []              # browse by sysdir bookmark list. List of paths
        self.filenamedict = {}          # key is filename value is id
        self.fullfilenamedict = {}      # key is filename value is id

        ###### Standard items on datastructures:
        # standard group where all images are added to if nothing else is specified
        self.groups[1] = ["Orphan","Images which haven't been assigned to any other group.",-1,[],[]]
        self.volumes[1] = [[],os.sep,str(self.tr("Default volume")),str(self.tr("Local filesystem"))]
        self.batches[1] = [str(self.tr("Temporary work batch")),[]]
        self.kbsize = 0                 # current database file size
        self.curBatch = 1               # current batch being added.
        self.dirty = 1                  # if db has changed and should be saved later
        self.agressive = 0              # on agressive mode, all viewed images are automatically added to db
        imgdb.resetdb()
        self.changedDB({"scope":"Database","reason":"reset"})

    def opendb(self,fname):
        """open metadata db, losing current db. Will also tell imgdb C++ module to load corresponding image data

        @param fname: full path filename
        @type fname: string

        #TODO2: run more sanity checks on database
        """
        self.fname = os.path.expanduser(fname)
        if not os.path.exists(self.fname):
            print self.tr("Starting with a new database.")
            self.reset()
            return 0
        print "Opening database file "+self.fname+" ..."
        if not imgdb.loaddb(self.fname+".img"):
            print self.tr("Error loading image database")
        rdbversion = 1                  # database version read from db file
        try:                            # get db file size for statistical purposes
            self.kbsize = os.stat(self.fname).st_size
        except:
            self.kbsize = 0
        ##### Load db contents one by one, changing the order and loading whatever necessary (for each db version)
        try:
            f = open(self.fname,"rb")
            rdbversion = marshal.load(f)
            if rdbversion < 3:
                self.v1files = marshal.load(f)
                self.v1buckets = marshal.load(f)
                self.v1dirs = marshal.load(f)
                self.v1dirtree = marshal.load(f)
                self.v1groups = marshal.load(f)
                if rdbversion>1:
                    self.v1meta = marshal.load(f)
            else:                       # load db > v3 < v5
                self.img = marshal.load(f)
                self.meta = marshal.load(f)
                if rdbversion < 5:
                    self.buckets = marshal.load(f)
                self.dirs = marshal.load(f)
                self.volumes = marshal.load(f)
                if rdbversion==3: marshal.load(f) # deprecated. Was removed on db version 4
                self.groups = marshal.load(f)
                self.contSimGroups = marshal.load(f)
                self.metafields = marshal.load(f)
                self.textqueryhistory = marshal.load(f)
                self.openGroups = marshal.load(f)
            if rdbversion==4:
                self.sig = marshal.load(f)
            if rdbversion>=4:
                self.batches = marshal.load(f)
                self.sysdirbm = marshal.load(f)
            if rdbversion==5:           # fix list of groups each image belongs to
                print "Importing version 5 database"
            if rdbversion>=7:
                self.filenamedict = marshal.load(f)
                self.fullfilenamedict = marshal.load(f)
                self.openDirs = marshal.load(f)
            self.syncFilenames()
            self.syncFullFilenames()
            if rdbversion>8:
                self.colorSimGroups = marshal.load(f)
                self.dateSimGroups = marshal.load(f)
                self.fileSimGroups = marshal.load(f)
            f.close()
            self.batches[1] = ["Temporary work batch",[]]
            self.opened = 1
        except:
            Error.PrintTB("Error opening database, starting with an empty one.")
            self.reset()
            return 0

        ##### Incrementally treat each db version, making sure we end up with a db compatible with the latest
        if rdbversion==1 or rdbversion==2:
            # import a dbversion 1 db and increment it to the next version so the next for catches it and do the proper imports for the next version
            print "Importing version 2 image database to version 3. This will require imgSeek to rescan all images you had on your database."
            print "Please wait..."
            for dir in self.v1dirtree.keys():
                if not self.v1dirtree[dir]: # is root dir
                    self.addDir(1,dir,1,[],1)
            # import descriptions
            if rdbversion == 2:
                for k in self.v1meta.keys():
                    try:
                        if self.v1meta[k]["Description"]:
                            for fid in self.meta.keys():
                                if find(self.v1meta[k]["Filename"],self.meta[fid]["Filename"]) != -1:
                                    self.meta[fid]["Description"] = self.v1meta[k]["Description"]
                    except:
                        pass
            self.dirty = 1
        if rdbversion == 3 or rdbversion == 4:
            print "Importing database from previous version. This will require imgSeek to rescan all images you had on your database."
            print "Adding %d images. Please wait..." % len(self.img.keys())
            for id in self.img.keys():
                if not imgdb.addImage(id,self.img[id][0],self.getThumbName(self.img[id][0]),not os.path.exists(self.getThumbName(self.img[id][0]))):
                    print "Error adding image:",self.img[id][0]
            self.dirty = 1
        if rdbversion < 7:               # make sure fullpathdict and other caches are syncd and existant
            print "Database import: gathering full path for every image..."
            self.syncFilenames()
        if rdbversion < 9:               # make sure all imgs have the "Modify_date_epoch" metadata
            print "Database import: gathering modify date for every image..."
            self.refreshModifyDate()
        if rdbversion < 10:               # make sure all imgs have the "Mounted" and "Volume" metadata
            print "Database import: setting Volume metadata for all images..."
            for vid in self.volumes.keys():
                lst = []
                self.crawlVolumeForImg(vid,lst)
                for id in lst:
                    self.meta[id]["Volume"] = self.volumes[vid][3]

            print "Database import: setting all images without 'Mounted' metadata as Not mounted..."
            for id in self.img.keys():
                if not self.meta[id].has_key("Mounted"):
                    self.meta[id]["Mounted"] = "no"

        self.simgroups = self.colorSimGroups
        print self.tr("Done.")
        self.changedDB({"scope":"Database","reason":"opened","subject":fname})
        #TODO7: add dirbm to cb system
        for cb in self.sysdirbmCb: cb()
        return 1

    def refreshModifyDate(self):
        """ call to refresh the Modify_date_epoch and Modify date metafields"""
        for id in self.img.keys():
            try:
                fname = self.img[id][0]
                self.meta[id]["Modify date"] = time.strftime("%Y/%m/%d %H:%M:%S", time.gmtime(os.stat(fname).st_mtime))
                self.meta[id]["Modify_date_epoch"] = os.stat(fname).st_mtime # for time clustering
            except:
                Error.PrintTB("Error setting Modify Date... Ignore this error if file is non-existant, and run \"Remove dead files\" later.")

    def fixImgGroups(self,id):
        """ call to make sure the list of groups this img id belongs to is right

        @param id: image id to be fixed
        """
        #TODO5: improve it. check if parents know all children and viceversa
        self.img[id][2] = []
        for gid in self.groups.keys():
            if id in self.groups[gid][3]:
                self.img[id][2].append(gid)

    def changefname(self,nname):
        """ call to change db filename name"""
        self.fname = nname
        self.dirty = 1
        self.changedDB({"scope":"Database","reason":"changedname","subject":nname})

    def generateImageID(self):
        """ need to return something unique"""
        #TODO3: is this random hack safe ? should I hash a resource somehow ? should I check if this ID already exists somewhere ? How much will extra checks slow things down ?
        try:
            a = random.randrange(999999999)
        except:
            try:
                a = random.randrange(99999999)
            except:
                try:
                    a = random.randrange(9999999)
                except:
                    a = random.randrange(999999)
        # 1 is a reserved id for default items
        while a==1: a = self.generateImageID()
        return a

    def savedb(self):
        """save all metadata/grouping/etc. Will also tell imgdb C++ module to save img coeffs.
        """
        if not imgdb.savedb(self.fname+".img"):
            print "Error saving image database"
        try:
            f = open(self.fname,"wb")
            marshal.dump(self.dbversion,f)
            marshal.dump(self.img,f)
            marshal.dump(self.meta,f)
            marshal.dump(self.dirs,f)
            marshal.dump(self.volumes,f)
            marshal.dump(self.groups,f)
            marshal.dump(self.contSimGroups,f)
            marshal.dump(self.metafields,f)
            marshal.dump(self.textqueryhistory,f)
            marshal.dump(self.openGroups,f)
            marshal.dump(self.batches,f)
            marshal.dump(self.sysdirbm,f)
            marshal.dump(self.filenamedict,f)
            marshal.dump(self.fullfilenamedict,f)
            marshal.dump(self.openDirs,f)
            marshal.dump(self.colorSimGroups,f)
            marshal.dump(self.dateSimGroups,f)
            marshal.dump(self.fileSimGroups,f)
            f.close()
        except:
            Error.PrintTB("Error saving metadata.")
            return 0
        self.dirty = 0
        self.changedDB({"scope":"Database","reason":"saved","subject":self.fname})
        print self.tr("Database saved.")
        return 1

    def extIsImg(self,file,probe = 0):
        """tests if this file has a supported image format extension or is an image.

        @param file: full filename
        @type file: string
        @param probe: if true, makes it check even if file has no extension
        @type probe: int
        """
        return self.env.extIsImg(file,probe)

    def updateGroups(self,app,simthresd = 50,by = "Color (Fast)"):
        """groups by similarity

        @param simthresd:  is the similarity threshold for choosing groups
        @type simthresd: integer
        """
        if by in [str(self.tr("Content (Slow/Accurate)")),str(self.tr("Color (Fast)"))]:
            if by == str(self.tr("Content (Slow/Accurate)")):
                if self.env.wnd and len(self.img.keys())>1000: # only nag user if he has a big db
                    res = QMessageBox.information( self.env.wnd, "imgSeek",str(self.tr("Grouping may take a while on large collections, and once it's started, can't be stopped, do you want to continue ?")), QString(self.tr("&Yes")), QString(self.tr("No")))
                    if res: return
                self.contSimGroups = {}
                self.simgroups = self.contSimGroups
                simthresd = -(38.70/100.0)*simthresd
                clusters = imgdb.clusterSim(simthresd,0)
            else:                       # Color (Fast)
                self.colorSimGroups = {}
                self.simgroups = self.colorSimGroups
                simthresd = 100-simthresd
                simthresd = (38.70/100.0)*simthresd/10
                clusters = imgdb.clusterSim(simthresd,1)
            for i in range(imgdb.getLongList2Size(clusters)):
                conlist = imgdb.popLong2List(clusters) # cluster content
                ngid = self.generateImageID()
                file = imgdb.popLongList(conlist)
                try:
                    self.simgroups[ngid] = [self.meta[file]["Filename"],file,[]]
                except:
                    print "Error setting new simgroup for file", file
                    continue
                for j in range(imgdb.getLongListSize(conlist)):
                    self.simgroups[ngid][2].append(imgdb.popLongList(conlist))
        ########################################
        if by == str(self.tr("Date")):
            #sort by age
            ids = []
            for id in self.meta.keys():
                try:
                    ids.append([id,self.meta[id]["Modify_date_epoch"]])
                except:
                    # error fetching Modify date for this file, determine it now:
                    fname = self.img[id][0]
                    self.meta[id]["Modify date"] = time.strftime("%Y/%m/%d %H:%M:%S", time.gmtime(os.stat(fname).st_mtime))
                    self.meta[id]["Modify_date_epoch"] = os.stat(fname).st_mtime # for time clustering
                    ids.append([id,self.meta[id]["Modify_date_epoch"]])
            try:                        # PYTHON-FOO !!! - sort images by date
                ids.sort(lambda a,b:(a[1]>b[1])*2-1) # *2 -1 so it returns -1 and 1
            except:
                Error.PrintTB("Error sorting, perhaps a Modify date entry is missing. Retrying...")
                self.refreshModifyDate()
                try:
                    ids.sort(lambda a,b:(a[1]>b[1])*2-1)
                except:
                    Error.PrintTB("Unable to sort after 2nd attempt")
                    return
            #calculate local gap averages and compare
            from math import log
            d = 10
            if (17+simthresd-50<0): simthresd = -(17-50)+1
            K = log(17+simthresd-50)
            gaps = []
            for i in range(len(ids)-1):
                try:
                    loggn = log(ids[i+1][1]-ids[i][1])
                except:
                    continue
                sum = 0
                dec = 0
                for j in range(-d,d+1):
                    if i+j+1 < len(ids):
                        if ids[i+j+1][1] != ids[i+j][1]:
                            sum = sum+log(abs(ids[i+j+1][1]-ids[i+j][1]))
                    else:
                        dec = dec+1
                fact = 1.0/(2*d+1-dec)
                sum = sum*fact+K
                if loggn>sum:           # found cluster gap
                    gaps.append(ids[i+1][0])
            self.dateSimGroups = {}
            self.simgroups = self.dateSimGroups
            ngid = self.generateImageID()
            file = ids[0][0]
            self.simgroups[ngid] = [self.meta[file]["Filename"],file,[]]
            for idp in ids:
                if idp[0] in gaps:
                    ngid = self.generateImageID()
                    file = idp[0]
                    self.simgroups[ngid] = [self.meta[file]["Filename"],file,[]]
                self.simgroups[ngid][2].append(idp[0])
        ########################################
        if by == str(self.tr("Filename")):
            # simply sort them alphabetically then split it evenly in groups
            try:
                self.syncFilenames()
                targetids = []
                for it in self.filenamedict.keys():
                    targetids.append((lower(it),self.filenamedict[it]))
                targetids.sort(lambda n1,n2: (n1[0]>n2[0])-1 )
                self.fileSimGroups = {}
                self.simgroups = self.fileSimGroups
                from math import sqrt
                numres = int(sqrt(len(targetids)))+20
                if numres>len(targetids) or int(len(targetids)/numres) <2:
                    numres = int(len(targetids)/3)
                targetids = map(lambda f:f[1],targetids)
                for idx in range(int(len(targetids)/numres)):
                    ngid = self.generateImageID()
                    file = targetids[idx*numres]
                    self.simgroups[ngid] = [self.meta[file]["Filename"],file,targetids[idx*numres:(idx+1)*numres]]
                ngid = self.generateImageID()
                file = targetids[int(len(targetids)/numres)*numres]
                self.simgroups[ngid] = [self.meta[file]["Filename"],file,targetids[int(len(targetids)/numres)*numres:]]
            except:
                Error.PrintTB("Maybe your collection is too small ( less than 30 files ). Try adding more files")
        self.changedDB({"scope":"SimGroup","reason":"updated"})

    def hashStr(self,str):
        """returns MD5 hex representation """
        return md5.new(str).hexdigest()[:8]

    def getThumbName(self,fname):
        """return the thumbnail full path for the image
        given by fname (fname is full path) """
        return self.thdir2+ md5.new("file://"+fname).hexdigest()+".png"

    def extension(self, fname):
        """ returns the .3 extension of a given filename, including the dot"""
        return fname[-(len(fname)-rfind(fname,'.')):]

    def uniquename(self, fname,ext,path=None,attempt=0):
        """ returns unique 8.3 filename for a file with filename fname
        placed on specified path.

        @param fname: source filename
        @param path: if non-Null, this function will make sure that
        there is no file on path with the returned unique filename
        @param ext: extension for the resulting filename (".png" for example)
        @rtype: string
        """
        if not path:   # return unique name without checking
                return md5.new(fname).hexdigest()[-8:]+ext
        else:
                if attempt > 64: raise "Possible error generating unique file name."
                uname = md5.new(fname + str(attempt)).hexdigest()[-8:] + ext
                if os.path.exists(path+uname) or os.path.exists(path+os.sep+uname):
                        return self.uniquename(fname,ext,path, attempt+1)
                return uname

    def getThumbDB(self,id,force = 0, onlyNewer = 1):
        """returns a QPixmap with the thumbnail for an img in db

        @param id: db image id.
        @param force: if true, thumbnail will be recreated even if it already exists on ~/.thumbails
        @param onlyNewer: if true and force==1, thumbnail will be recreated only if thumbnail file date doesn't match img file date
        """
        if self.meta[id].has_key("Mounted") and self.meta[id]["Mounted"] == "yes":
            tfname = self.img[id][0]
            return self.getThumb(self.img[id][0],force=force, mountDiff = self.meta[id]["Database date"])
        else:
            return self.getThumb(self.img[id][0],force=force)

    def getThumb(self,fname,imgDims = None,force = 0, mountDiff="", onlyNewer = 1): #TODO7: imgDims is deprecated
        """use to get a QPixmap with the thumbnail for a file

        Using the standard from http://triq.net/~pearl/thumbnail-spec/

        @param fname: full path for image
        @param force: if true, thumbnail will be regenerated (ie, cached copy will be ignored and rewritten)
        @param mountDiff: extra salt to be used when generating thumbnail fname
        @param onlyNewer: if true and force==1, thumbnail will be recreated only if thumbnail file date doesn't match img file date
        @return: a QPixmap() instance with the proportional thumbnail.If thummbn found on cache, open it, otherwise create one.
        """
        aImage = None
        thname = self.thdir2+ md5.new("file://"+fname+mountDiff).hexdigest()+".png"
        (base,ext) = os.path.splitext(fname)
        fext = ext[1:].lower()
        try:
            if force and onlyNewer and (os.stat(thname).st_mtime > os.stat(fname).st_mtime ):
                force = 0
        except:
            Error.PrintTB()
            force = 0

        if not os.path.exists(thname) or force:
            if fext in self.env.doc_ext:
                print "Not generating thumbnail for document file. Email \"imgseek-devel@lists.sourceforge.net\" if you think imgSeek should thumbnail documents."
                return None
            if fext in self.env.qt_ext:
                if not imgdb.magickThumb(fname,thname):
                    # fast thumbnail failed, trying slow one
                    aImage = QImage()
                    if not aImage.load(fname):
                        print "Error loading image file while creating a thumbnail for it:" + fname
                        return None
                    aImage = aImage.smoothScale(128,128,QImage.ScaleMin)
                    aImage.save(thname,"PNG")
            elif self.env.hasImMagick and (fext in self.env.magick_ext):
                imgdb.magickThumb(fname,thname)
            elif self.env.hasPIL and (fext in self.env.pil_ext):
                im = Image.open(fname)
                nw = 128
                nh = 128
                sz = im.size
                if sz[0] > sz[1]:
                    nw = 128
                    nh = int(128*sz[1]/sz[0])
                else:
                    nh = 128
                    nw = int(128*sz[0]/sz[1])
                im = im.resize((nw,nh))
                im.save(thname)
                del im
        if not aImage: # load it from thumbnail file
            aImage = QPixmap()
            if not aImage.load(thname):
                print "Error loading thumbnail file " + thname
        else:
            aPix = QPixmap()
            aPix.convertFromImage(aImage)
            return aPix
        return aImage

    def addFile(self,fname,newid,dirid,ext, restr = None):
        """adds this filename to database.

        First, loading it, then calculating its haar transform
        and then adding the image index to all respective buckets.
        Each self.files list element is [filename,avg lum]

        @param fname: full filename
        @type fname: string
        @param newid: id this file should be assigned to
        @param dirid: id of the dir this file belongs to
        @param mindim: if this file has dimensions smaller than mindim, it's not added. (width and height will be checked separately)
        @type mindim: int
        @param ext: file extension or image type, should be the result of a call to extIsImg()
        @return True if image succesfully added
        """
        if self.dirs.has_key(dirid):
            if fname in self.dirs[dirid][4]:
                return 2
        if restr:
            mindim = restr.mindim
        else:
            mindim = 0
        if self.env.verbose: print "Adding file: " + fname
        ret = -1
        isMounted = 0
        if (restr and not restr.mounted) or (not restr): # not mounted. Generate thumbnail
            isMounted = 0
            thname = self.thdir2 + md5.new("file://"+fname).hexdigest()+".png"
            ret = imgdb.addImage(newid,fname,thname,not os.path.exists(thname),mindim)
        else:                           # mounted. Do not cache thumbnail
            isMounted = 1
            ret = imgdb.addImage(newid,fname,"null",0,mindim)
        if not ret:
            print "Error adding image:",fname
            return 0
        if ret == 2:                      # dimension too small or too white
            print "Ignored (small dimensions): "+fname
            return 0
        imgDims = [imgdb.getImageWidth(newid), imgdb.getImageHeight(newid)]
        ## init metadata
        try:                            # *** NOTE *** when changing this code, also change ImgDB.py:initImgMetadata **********************
            self.meta[newid] = self.blankMetaDict.copy()
            if isMounted:
                self.meta[newid]["Mounted"] = "yes"
            else:
                self.meta[newid]["Mounted"] = "no"
            self.meta[newid]["Filename"] = os.path.split(fname)[-1]
            self.meta[newid]["Filesize"] = self.prettysize(os.stat(fname).st_size)
            self.meta[newid]["Format"] = ext
            if restr:
                self.meta[newid]["Volume"] = self.volumes[restr.volid][3]
            self.meta[newid]["Database date"] = time.asctime(time.localtime())
            self.meta[newid]["Modify date"] = time.strftime("%Y/%m/%d %H:%M:%S", time.gmtime(os.stat(fname).st_mtime))
            self.meta[newid]["Modify_date_epoch"] = os.stat(fname).st_mtime # for time clustering
            if imgDims:
                self.meta[newid]["Dimensions"] = str(imgDims[0])+" x "+str(imgDims[1])
        except:
            traceback.print_exc()
            print "Error setting basic metadata for "+fname
            self.meta[newid]["Filename"] = fname
        return 1

    def visited(self, fname):
        """called when an image is visited on disk. This will add it automatically to db.
        Will also check if user is on agressive mode.
        """
        if not self.agressive:          # not on agressive mode
            return
        if fname in self.fullfilenamedict:
            return                      # file already seen / on db
        path = os.path.abspath(os.path.expanduser(os.path.split(fname)[0]))+os.sep      # make sure we have absolute path, so the stubdir code below won't choke on "wallpaper/"-like paths
        file = os.path.split(fname)[-1]
        groupid = 1                     # will belong to orphan. TODO3: make them belong to a "group of the day"
        ndid = -1
        restr = None
        hasImage = 0
        childfiles = []
        childfilenames = []
        parts = split(path,os.sep)[1:-1]
        volid = 1                       # default volume
        cparts = []
        acc = ""
        for part in parts:              # build dir list incrementally
            acc = acc+part+os.sep
            cparts.append(os.sep+acc[:])
        for part in cparts[:-1]:        # for every dir before the last one
            self.addStubDir(part,volid)
        if not cparts:                  # user tried to add '/'
            cparts=[os.sep]
        ndid = self.addStubDir(cparts[-1],volid)
        ext = self.extIsImg(path+file,probe = 0)
        nfid = self.generateImageID()
        while self.img.has_key(nfid):
            nfid = self.generateImageID()
        self.img[nfid] = [path+file,ndid,[groupid]] # add img to dbase
        ret = 0
        try:
            ret = self.addFile(path+file,nfid,ndid,ext, restr)
        except:
            traceback.print_exc()
            print "Unhandled error while adding file. Please report this bug to \"imgseek-devel@lists.sourceforge.net\""
        if ret == 1:
            hasImage = 1
            childfiles.append(nfid)
            childfilenames.append(path+file)
            self.groups[groupid][3].append(nfid)
        elif ret in [0,2]: # error adding image
            del self.img[nfid]
            print 'Unable to automatically add %s to database' % fname

        if hasImage: # only add to databse if cp module could add image
            if ndid not in self.volumes[volid][0]:self.volumes[volid][0].append(ndid)
            if ndid in self.dirs.keys():
                self.dirs[ndid][3] = self.dirs[ndid][3]+childfiles
                self.dirs[ndid][4] = self.dirs[ndid][4]+childfilenames
                self.dirs[ndid][6] = volid
            else:
                print "Dir should already be in dbase"
                self.dirs[ndid] = [path, 0, [], [], childfilenames, "", volid]
            self.fullfilenamedict[fname] = nfid
            #self.curdb.changedDB({"scope":"Group","reason":"addedimage","save":0})
            #self.curdb.changedDB({"scope":"Dir","reason":"added"})

    def initImgMetadata(self,fname,newid,imgDims = None):
        """add basic metadata to img

        @param newid: image id
        @param fname: image fullpath fname, used to extract short filename
        @param imgDims: image dimension
        @type imgDims: tuple (width,height)

        """
        self.meta[newid] = self.blankMetaDict.copy()
        try:
            self.meta[newid]["Filename"] = os.path.split(fname)[-1]
            self.meta[newid]["Filesize"] = self.prettysize(os.stat(fname).st_size)
            self.meta[newid]["Format"] = self.extIsImg(fname)
            self.meta[newid]["Database date"] = time.asctime(time.localtime())
            self.meta[newid]["Modify date"] = time.strftime("%Y/%m/%d %H:%M:%S", time.gmtime(os.stat(fname).st_mtime))
            self.meta[newid]["Modify_date_epoch"] = os.stat(fname).st_mtime # for time clustering
            if imgDims:
                self.meta[newid]["Dimensions"] = str(imgDims[0])+" x "+str(imgDims[1])
        except:
            traceback.print_exc()
            print "Error setting basic metadata for "+fname
            self.meta[newid]["Filename"] = fname

    def prettysize(self,uint):
        """returns a string after converting the integer uint and adding periods to enhance readability

        @param uint: int to be transformed
        @type uint: int
        @return: pretty string (human readable)
        @rtype: string
        """
        a = str(uint)
        b = ""
        for i in range(len(a)):
            if not i%3:
                b = "."+b
            b = a[len(a)-1-i]+b
        return b[:-1] + " bytes"

    def addStubDir(self,path,volid):
        """ make sure this path exists on db

        if it already exists, return its id, otherwise, create new dir and return new id

        @param path: full dir path
        @param volid: volume id this dir belongs to
        """
        ndid = -1
        for dirid in self.volumes[volid][0]:
            if self.dirs[dirid][0]==path:
                ndid = dirid                 # this dir is already on dbase
                break
        if ndid == -1:                  # its a new dir
            ndid = self.generateImageID()
            # guess parent
            parent = -1
            for dirid2 in self.volumes[volid][0]:
                if path[:rfind(path[:-1],os.sep)+1]==self.dirs[dirid2][0]:
                    parent = dirid2
                    break
            self.dirs[ndid] = [path,parent,[],[],[],"",volid]
            if parent != -1:
                self.dirs[parent][2].append(ndid)
            self.volumes[volid][0].append(ndid)
        return ndid

    def addDir(self,volid,path,groupid,aborted = [],recursive = 0,calledRecurs=-1,pb=None,restr=None):
        """Add to db all the files on this dir.

        aborted is an empty list that should be nonempty when you want this adddir call to be aborted
        calledRecurs is the id of the dir that should be the parent to the one at path (ie, the one being processed)
        pb is a QProgressDialog
        #TODO5: all addDir options should be a dictionary and a dict instance should be passed as the only parm
        #TODO4: This function & friends are MESSY, rewrite. Also: most parameters are deprecated, as they are now passed through restr
        """
        if aborted: return -1
        #startt = time.time()
        path = os.path.abspath(path)      # make sure we have absolute path, so the stubdir code below won't choke on "wallpaper/"-like paths
        if self.env.verbose: print "Scanning dir: " + path
        try:
            dfiles = os.listdir(path)
        except OSError, (errno, strerror):
            print "OS error(%s): %s" % (errno, strerror)
            dfiles = []
        except:
            dfiles = []
            print "Unhandled exception while listing path. Continuing anyway..."
            Error.PrintTB()
        hasImage = 0
        if not restr:
            restr = AddFilter(self.env)
        groupid = restr.groupid
        if not len(dfiles):
            if calledRecurs==-1 and restr.removeEmptyGroup and not hasImage and self.groups.has_key(groupid):
                self.doNotSave = 1
                self.removeGroup(groupid)
                self.doNotSave = 0
            return -1
        if path[-1] != os.sep[0]: path = path+os.sep
        myLabel = path
        mySteps = len(dfiles)
        hasImage = 0
        if self.wnd:                    # init progress bars
            if calledRecurs==-1:
                addDirPB = self.wnd.progressBar
                self.wnd.progressList.clear()
                self.wnd.progressList.insertItem("Started adding, please wait...")
            else:
                self.wnd.progressLabel.setText(myLabel)
                addDirPB = self.wnd.subprogressBar
            addDirPB.setTotalSteps(mySteps)
            addDirPB.setProgress(0)
        ndid = -1
        childdirs = []
        parts = split(path,os.sep)[1:-1]
        cparts = []
        acc = ""
        for part in parts:              # build dir list incrementally
            acc = acc+part+os.sep
            cparts.append(os.sep+acc[:])
        for part in cparts[:-1]:        # for every dir before the last one
            self.addStubDir(part,volid)
        if not cparts:                  # user tried to add '/'
            cparts=[os.sep]
        ndid = self.addStubDir(cparts[-1],volid)
        if ndid == -1:                  # its a new dir
            print "[ERROR] New dir should have an id"
            ndid = self.generateImageID()
        childfiles = []
        childfilenames = []
        addcnt = 0
        for file in dfiles:             # note: file is not full path. Use path+file when needed
            addcnt = addcnt+1
            if not (addcnt % 20) and self.wnd: # update progress on GUI
                addDirPB.setProgress(addcnt)
                if self.app: self.app.processEvents()
                if self.wnd.abortedAdd:
                    aborted.append(1)
                    self.wnd.abortedAdd = 0
                    break
            if os.path.isdir(path+file) and recursive:
                if file[0]=='.': continue # do not add dirs starting with . (also excludes .thumbnail dirs)
                if os.path.islink(path+file) and not restr.followsym: continue # don't follow symbolic links to directories                
                # update paths tree
                thasimg = self.addDir(volid,path+file,groupid,aborted,recursive,ndid,restr = restr)
                if self.wnd:            # update progress on GUI
                    self.wnd.progressLabel.setText(myLabel)
                    addDirPB.setTotalSteps(mySteps)
                    addDirPB.setProgress(addcnt)
                if thasimg == -1: continue
                childdirs.append(thasimg)
                hasImage = 1
                continue
            # is file
            ext = self.extIsImg(path+file,probe = restr.probeext)
            if ext: #its a file, now just check if its an image
                if file[0]=='.': continue # do not add imgs starting with . (also excludes .thumbnail dirs)
                if restr.igntext and find(path+file,restr.igntext) != -1: continue
                if restr.ignext and ext in restr.ignext: continue
                try:
                    if restr.minsize and os.stat(path+file).st_size < restr.minsize*1024: continue
                except:
                    # what the hell am I supposed to do with a file that cant even be stat'd ?
                    continue            # yeah, you guessed: ignore it
                nfid = self.generateImageID()
                while self.img.has_key(nfid):
                    nfid = self.generateImageID()
                self.img[nfid] = [path+file,ndid,[groupid]] # add img to dbase
                ret = 0
                try:
                    ret = self.addFile(path+file,nfid,ndid,ext, restr)
                except:
                    traceback.print_exc()
                    print "Unhandled error while adding file. Please report this bug to \"imgseek-devel@lists.sourceforge.net\""
                if ret==1:
                    hasImage = 1
                    childfiles.append(nfid)
                    childfilenames.append(path+file)
                    self.groups[groupid][3].append(nfid)
                    ### extract optional metadata
                    if (ext in self.env.exif_ext) and restr.exexif:
                        try:
                            self.extractEXIF(nfid)
                        except:
                            traceback.print_exc()
                            print "Error extracting EXIF metadata for:",path+file
                    if (ext in self.env.iptc_ext) and self.env.hasIPTC and restr.exiptc:
                        try:
                            self.extractIPTC(nfid)
                        except:
                            traceback.print_exc()
                            print "Error extracting IPTC metadata for:",path+file
                elif not ret: #error adding image or image already on dbase
                    if self.wnd: self.wnd.progressList.insertItem("Unable to add: "+path+file)
                    del self.img[nfid]
                elif ret==2:
                    del self.img[nfid]

        if hasImage: # only add to databse if has image isinde
            if ndid not in self.volumes[volid][0]:self.volumes[volid][0].append(ndid)
            if ndid in self.dirs.keys():
                for ddd in childdirs:
                    while ddd in self.dirs[ndid][2]:
                        self.dirs[ndid][2].remove(ddd)
                self.dirs[ndid][2] = self.dirs[ndid][2]+childdirs
                self.dirs[ndid][3] = self.dirs[ndid][3]+childfiles
                self.dirs[ndid][4] = self.dirs[ndid][4]+childfilenames
                self.dirs[ndid][6] = volid
            else:
                print "Dir should already be in dbase"
                self.dirs[ndid] = [path,calledRecurs,childdirs,childfiles,childfilenames,"",volid]
            for dirid in childdirs:     # make sure all child dirs has the new dir as their parent. This is needed on the correction made
                                        # on the beginning, where a parent is added to dbase after the child. If this isn't done, the child
                                        # that was already on dbase, will still has another dir (almos always -1 (root on volume) as a parent
                try:
                    self.dirs[dirid][1] = ndid
                except:
                    pass
        self.dirty = 1
        if self.env.verbose: print "Added dir: " + path
        if self.wnd:
            if calledRecurs==-1:
                self.wnd.progressList.insertItem("Finished successfully.")
        # if restr.removeEmptyGroup is set and this dir has no images, then remove the group this dir should belong to
        if calledRecurs==-1 and restr.removeEmptyGroup and not hasImage and self.groups.has_key(groupid):
            self.doNotSave = 1
            self.removeGroup(groupid)
            self.doNotSave = 0
        if hasImage: return ndid
        else: return -1

    def extractEXIF(self,fid):
        """hook for exif library.

        @param fid: set metadata for the image with this id.
        """
        try:
            file = open(self.img[fid][0], 'rb')
        except:
            print 'Unable to open file in order to extract EXIF data.'
            return
        data = EXIF.process_file(file)
        if data:
            x = data.keys()
            for i in x:
                if i in ('JPEGThumbnail', 'TIFFThumbnail'):
                    continue
                try:
                    self.meta[fid][i] = data[i].printable
                    if i not in self.metafields:                        
                        self.metafields.append(i)
                except:
                    try:
                        self.meta[fid][i] = data[i]
                    except:
                        print "Error extracting EXIF field:",i
            if data.has_key('JPEGThumbnail'):
                self.meta[fid]['JPEGThumbnail'] = "True"
            else:
                self.meta[fid]['JPEGThumbnail'] = "False"

    def extractIPTC(self,fid):
        """hook for iptc PIL library.

        @param fid: set metadata for the image with this id.
        """
        try:
            data = IptcExtract.getiptcinfo(self.img[fid][0])
        except:
            traceback.print_exc()
            print 'Error extracting IPTC data.'
            return
        if not data:
            return
        else:
            metacache={}
            for k, v in data.items():
                try:
                    if not k[1]: continue # skip key 0, which is \x00\x02 for example. I don't know what it means
                    i = IptcExtract.infomap[k[1]]
                    if metacache.has_key(i):
                        metacache[i] = metacache[i] + "," + str(v)
                    else:
                        metacache[i] = str(v)
                except:
                    print "----------\nError extracting IPTC field: %s %s\nIf you know what this field means according to IPTC standards, please tell us at \"imgseek-devel@lists.sourceforge.net\"\n----------"% (k, repr(v))
            for k in metacache.keys():
                self.meta[fid][k] = metacache[k]

    def getTotalFiles(self):
        """ number of images in db

        @return: number of images in db
        @rtype: int
        """
        return len(self.img.keys())

    def TextReport(self):
        """get database statistics

        @return: human readable statistics about db
        @rtype: string
        """
        ntext = "Database status:\n"
        ntext = ntext+str(self.tr("Images: %d\n"))%len(self.img.keys())
        ntext = ntext+str(self.tr("Volumes: %d\n"))%len(self.volumes.keys())
        ntext = ntext+str(self.tr("Directories: %d\n"))%len(self.dirs.keys())
        ntext = ntext+str(self.tr("Groups: %d\n"))%len(self.groups.keys())
        ntext = ntext+str(self.tr("Work Batches: %d\n"))%len(self.batches.keys())
        ntext = ntext+str(self.tr("Similarity groups: %d\n"))%len(self.simgroups.keys())
        ntext = ntext+str(self.tr("Metadata fields cached: %d\n"))%len(self.metafields)
        ntext = ntext+str(self.tr("Text query history entries: %d\n"))%len(self.textqueryhistory)
        ntext = ntext+str(self.tr("Database size: %d bytes\n"))%self.kbsize
        return ntext

    def newGroup(self,text,pargid = -1):
        """create new group

        @param text: new name
        @param pargid: parent group id
        @return: new group id
        """
        nid = self.generateImageID()
        self.groups[nid] = [text,"",pargid,[],[],nid]
        if pargid != -1:
            # tell parent he has a new child
            self.groups[pargid][4].append(nid)
        self.dirty = 1
        self.changedDB({"scope":"Group","reason":"new","subject":nid})
        return nid

    def removeGroup(self,gid):
        """remove group from database

        child imgs will be moved to Orphan group, so they don't disappear completely from the grouping system

        @param gid: group id to be removed
        """
        if gid==1:
            print "Attempt to remove default Orphan group, aborting remove process."
            return 0
        self.doNotSave = 1
        # move child imgs to default orphan group (id = 1)
        try:
            # check each children to see if it will become orphan
            for id in self.groups[gid][3]:
                if gid not in self.img[id][2]:
                    print "DB Inconsistency ! Image doesn't know about a parent group."
                else:
                    if len(self.img[id][2]) == 0:
                        print "DB Inconsistency ! Image doesn't know any parent group."
                    if len(self.img[id][2]) == 1: # only move me to orphan if this (the group being removed) was my last group
                        self.moveFileGroup(id,1) # move it to orphan group
        except:
            print "Strange error removing group and adding children to Orphan group"
            traceback.print_exc()
        # tell parent i died
        if self.groups[gid][2] != -1:
            try:
                self.groups[self.groups[gid][2]][4].remove(gid)
            except:
                traceback.print_exc()
                print "Error removing group from it's parent."
        # remove children groups
        for sid in self.groups[gid][4]:
            self.removeGroup(sid)
        del self.groups[gid]
        self.doNotSave = 0
        self.changedDB({"scope":"Group","reason":"removed","subject":gid})
        self.dirty = 1
        return 1

    def removeVolume(self,gid):
        """remove volume and all it's subdirs

        @param gid: volume id
        """
        self.doNotSave = 1
        for sid in self.volumes[gid][0]:
            self.removeDir(sid)
        del self.volumes[gid]
        self.dirty = 1
        self.doNotSave = 0
        self.changedDB({"scope":"Volume","reason":"removed","subject":gid})

    def removeDir(self,gid):
        """remove dir and all it's images

        @param gid: dir id
        """
        #remove child dirs
        self.doNotSave = 1
        for sid in self.dirs[gid][2]:
            self.removeDir(sid)
        #remove child imgs
        try:
            for sid in self.dirs[gid][3]:
                self.removeFile(sid)
        except:
            traceback.print_exc()
        #tell parent dir im no longer a child
        try:
            if self.dirs[gid][1] != -1:
                self.dirs[self.dirs[gid][1]][2].remove(gid)
        except:
            traceback.print_exc()
        #remove from volume
        try:
            self.volumes[self.dirs[gid][6]][0].remove(gid)
        except:
            traceback.print_exc()
        #remove the dir itself
        del self.dirs[gid]
        self.doNotSave = 0
        self.changedDB({"scope":"Dir","reason":"removed","subject":gid})
        self.dirty = 1

    def newVolume(self,txt):
        """ create a new volume with the provided name and return its id. -1 is returned on a failure

        @param txt: new volume name
        """
        for grp in self.volumes.keys():
            if self.volumes[grp][3]==txt:
                return -1
        nid = self.generateImageID()
        while nid in self.volumes.keys():
            nid = self.generateImageID()
        self.volumes[nid] = [[],"",str(self.tr("No description given.")),txt]
        self.changedDB({"scope":"Volume","reason":"new","subject":nid})
        return nid

    def renameVolume(self,vid,text):
        """ rename volume

        @param vid: volume id to be changed
        @param text: new volume name
        """
        self.volumes[vid][3] = text
        self.dirty = 1
        self.changedDB({"scope":"Volume","reason":"renamed","subject":vid})

    def renameGroup(self,vid,text):
        self.groups[vid][0] = text
        self.dirty = 1
        self.changedDB({"scope":"Group","reason":"renamed","subject":vid})

    def describeVolume(self,vid,text):
        self.volumes[vid][2] = text
        self.dirty = 1
        self.changedDB({"scope":"Volume","reason":"described","subject":vid})

    def describeGroup(self,vid,text):
        self.groups[vid][1] = text
        self.dirty = 1
        self.changedDB({"scope":"Group","reason":"described","subject":vid})

    def removeFile(self,fid):
        """ call to remove a given image from db.
        This function won't fire a chagedDb event, because the function calling this
        one is responsible for that (e.g. removeDir)

        @param fid: id of the file that will be removed
        """
        imgdb.removeID(fid)
        for did in self.dirs.keys():
            try:
                self.dirs[did][3].remove(fid)
                self.dirs[did][4].remove(self.img[fid][0])
            except:
                pass
        for did in self.groups.keys():
            try:
                self.groups[did][3].remove(fid)
            except:
                pass
        for did in self.simgroups.keys():
            try:
                self.simgroups[did][2].remove(fid)
                if self.simgroups[did][1]==fid:
                    del self.simgroups[did]
            except:
                pass
        try:
            del self.meta[fid]
        except:
            pass
        try:
            del self.img[fid]
        except:
            pass
        #self.changedDB({"scope":"Img","reason":"removed","subject":fid})
        self.dirty = 1

    def moveGroup(self,fromid,toid):
        """ call to move groups

        @param fromid: id of the group that will be moved
        @param toid: id of the group that will become the parent of fromid
        """
        assert fromid in self.groups
        if toid != -1:
            assert toid in self.groups
        if fromid == 1:
            print "Attempt to move Orphan group. Ignored request"
            return
        if fromid == toid:
            return 0
        if toid in self.groups[fromid][4]:
            print "Invalid move. Parent -> Child"
            return 0
        if self.isGroupAncestor(toid,fromid):
            print "Attempt to move/copy parent to a child, ignoring request."
            return 0
        if self.groups[fromid][2] != -1: # tell my parent that i've moved
            try:
                self.groups[self.groups[fromid][2]][4].remove(fromid)
            except:
                Error.PrintTB("Error telling my parent that i've moved")
                return 0
        self.groups[fromid][2] = toid   # now I have a new parent
        if toid != -1:                  # tell my new parent about me
            self.groups[toid][4].append(fromid)
        self.dirty = 1
        return 1

    def addImageToGroup(self,id,gid):
        """append image to a group

        @param id: image id
        @param gid: group id. This image will now also belong to this group
        """
        if id not in self.groups[gid][3]: # update img's parent group
            self.groups[gid][3].append(id)
        if gid not in self.img[id][2]:    # update img's list of parent groups
            self.img[id][2].append(gid)
        self.changedDB({"scope":"Group","reason":"addedimage","subject":gid,"target":id})

    def moveFileGroup(self,fromid,toid,fromgroup = -1):
        """ call to make file fromid to be a child of group toid
        if fromgroup is defined, erase this image from the group fromgroup (this would be a MOVE operation)
        """
        if fromid == toid:              # source=dest, that makes no sense
            return 0
        if toid == -1:
            print "Invalid move: Only groups may be copied/moved to root. Ignoring request."
            return 0
        if fromgroup != -1:             # fromgroup was defined, so we should remove image from src group
            try:
                self.groups[fromgroup][3].remove(fromid)
            except:
                pass
            try:
                self.img[fromid][2].remove(fromgroup)
            except:
                pass
        self.addImageToGroup(fromid, toid)
        self.dirty = 1
        return 1

    def crawlVolumeForDir(self,vid,lst):
        """crawl this volume adding to lst all dir id's found"""
        for dirid in self.volumes[vid][0]:
            self.crawlDirForDir(dirid,lst)

    def crawlDirForDir(self,dirid,lst):
        """crawl this volume adding to lst all dir id's found"""
        for chid in self.dirs[dirid][2]:
            self.crawlDirForDir(chid,lst)
        lst.append(dirid)

    def crawlVolumeForImg(self,vid,lst):
        """crawl this volume adding to lst all img id's found"""
        for dirid in self.volumes[vid][0]:
            self.crawlDirForImg(dirid,lst,0)

    def crawlDirForImg(self,dirid,lst,recurse = 1):
        """crawl this volume adding to lst all img id's found"""
        if recurse:
            for chid in self.dirs[dirid][2]:
                self.crawlDirForImg(chid,lst)
        for chid in self.dirs[dirid][3]:
            lst.append(chid)

    def crawlGroupForImg(self,dirid,lst):
        """crawl this volume adding to lst all img id's found"""
        for chid in self.groups[dirid][4]:
            self.crawlGroupForImg(chid,lst)
        for chid in self.groups[dirid][3]:
            lst.append(chid)

    def crawlVolumeForFile(self,vid,lst):
        """crawl this volume adding to lst all img id's found"""
        for dirid in self.volumes[vid][0]:
            self.crawlDirForFile(dirid,lst,0)

    def crawlDirForFile(self,dirid,lst,recurse = 1):
        """crawl this volume adding to lst all img id's found"""
        if recurse:
            for chid in self.dirs[dirid][2]:
                self.crawlDirForFile(chid,lst)
        for chid in self.dirs[dirid][3]:
            lst.append(self.img[chid][0])

    def crawlSysDirForFile(self,dirid,lst,recurse = 1):
        """crawl this volume adding to lst all img id's found"""
        if dirid[-1]!='/':dirid = dirid+'/'
        dfiles = os.listdir(dirid)
        for file in dfiles:
            if os.path.isdir(dirid+file) and recurse:
                self.crawlSysDirForFile(dirid+file,lst,recurse)
                continue
            ext = self.extIsImg(file)
            if ext: #its a file, now just check if its an image
                if file[0]=='.': continue # do not add imgs starting with . (also excludes .thumbnail dirs)
                lst.append(dirid+file)

    def crawlGroupForFile(self,dirid,lst):
        """crawl this volume adding to lst all img id's found"""
        for chid in self.groups[dirid][4]:
            self.crawlGroupForFile(chid,lst)
        for chid in self.groups[dirid][3]:
            lst.append(self.img[chid][0])

    def crawlBatchForImg(self,bid,idlist,dumpids = 1):
        """ here bid is the a batch id of a batch in db
        if dumpids==1 it will dump ids, otherwise it will dump filenames (fullpaths)
        """
        for it in self.batches[bid][1]:
            if it[0]=="Volume":
                self.crawlVolumeForImg(it[1],idlist)
            if it[0]=="Dir":
                self.crawlDirForImg(it[1],idlist)
            if it[0]=="Group":
                self.crawlGroupForImg(it[1],idlist)
            if it[0]=="Img":
                idlist.append(it[1])

        if not dumpids:                 # caller wants fullpaths instead of ids
            for i in range(len(idlist)):
                idlist[i] = self.img[idlist[i]][0]

    def crawlBatchForFile(self,bid,idlist):
        """ here bid is the a batch id of a batch in db
        will dump on idlist all image files
        """
        for it in self.batches[bid][1]:
            if it[0]=="Volume":
                self.crawlVolumeForFile(it[1],idlist)
            if it[0]=="Dir":
                self.crawlDirForFile(it[1],idlist)
            if it[0]=="SysDir":
                self.crawlSysDirForFile(it[1],idlist)
            if it[0]=="Group":
                self.crawlGroupForFile(it[1],idlist)
            if it[0]=="Img":
                idlist.append(self.img[it[1]][0])
            if it[0]=="File":
                idlist.append(it[1])


    def crawlBatchContentsForImg(self,bid,idlist):
        """here bid is a list of items, with the same format a batch content would have """
        for it in bid:
            if it[0]=="Volume":
                self.crawlVolumeForImg(it[1],idlist)
            if it[0]=="Dir":
                self.crawlDirForImg(it[1],idlist)
            if it[0]=="Group":
                self.crawlGroupForImg(it[1],idlist)
            if it[0]=="Img":
                idlist.append(it[1])

    def saveBatchMeta(self,dct,bid):
        """apply data on dict dct to the batch """
        idlist = []
        self.crawlBatchForImg(bid,idlist)
        for id in idlist:
            for fi in dct.keys():
                self.meta[id][fi] = dct[fi]

    def BatchToText(self,bid):
        """ returns a pair of strings, which represents this batch in text form"""
        try:
            cnt = self.batches[bid][1]
        except:
            traceback.print_exc()
            print "Attempt to show invalid batch"
            return
        ret = []
        for it in cnt:
            tp = "Unknown"
            tx = ""
            if it[0] == "SysDir":
                tp = str(self.tr("System directory"))
                tx = it[1]
            if it[0] == "Dir":
                tp = str(self.tr("Database directory"))
                tx= "%s - %d image(s)" % (self.dirs[it[1]][0],len(self.dirs[it[1]][3]))
            if it[0] == "Group":
                tp = str(self.tr("Group"))
                tx= "%s - %d image(s)" % (self.groups[it[1]][0],len(self.groups[it[1]][3]))
            if it[0] == "SimGroup":
                tp = str(self.tr("Similarity Group"))
                tx= "%s - %d image(s)" % (self.img[self.simgroups[it[1]][1]][0],len(self.simgroups[it[1]][2]))
            if it[0] == "Img":
                tp = str(self.tr("Database image"))
                tx= "%s - %s" % (self.img[it[1]][0],self.meta[it[1]]["Filesize"])
            if it[0] == "Volume":
                tp = str(self.tr("Volume"))
                tx= "%s - %d dirs(s)" % (self.volumes[it[1]][3],len(self.volumes[it[1]][0]))
            if it[0] == "File":
                tp = str(self.tr("System file"))
                tx = it[1]
            ret.append([tp,tx])
        return ret

    def removeItemBatch(self,idx,bid):
        """call it to remove item idx from batch bid """
        try:
            del self.batches[bid][1][idx]
        except:
            print "Error removing item from batch"
            return
        self.changedDB({"scope":"Batch","reason":"removeditem","subject":bid,"target":idx})
        self.dirty = 1

    def addItemBatch(self,it,bid = -1):
        """call to populade a batch with an item

        @param it: map of type {'type','id'}
        @param bid: destination batch id
        @return: 0 if item already in this batch, 1 is successful and -1 on unkown error.
        """
        it = [it["type"],it["id"]]
        if bid==-1: bid = self.curBatch
        try:
            if self.idInBatch(it,bid):
                return 0
            self.batches[bid][1].append(it)
        except:
            traceback.print_exc()
            print "Error adding  item to batch"
            return -1
        self.changedDB({"scope":"Batch","reason":"addeditem","subject":bid,"target":it})
        return 1

    def renameFiles(self,renpairs):
        """call when files (fullpath) were physically renamed
        renpairs is dict. key is old fname value is [new fname,QListView item] -- QListView item can be None
        """
        listItemsToRemove = []          # will collect the list o QListView items to be removed from the widget when the rename process is finished
        for old in renpairs:
            if not self.renameFile(old, renpairs[old][0], warn = 0):
                print "Error renaming:%s" % old
            else:
                listItemsToRemove.append(renpairs[old][1])
        self.changedDB({"scope":"Batch","reason":"changeditem"})
        self.changedDB({"scope":"Img","reason":"multiplerenamed"})

        return filter(None,listItemsToRemove) # filter here is to remove all None objects from list

    def renameFile(self,old,new,warn = 1):
        """call when file (fullpath) old is renamed to new
        if warn = 1,issue the proper changedDB calls
        """
        if self.fullfilenamedict.has_key(old): # if renamed file is on db
            oldbasepath = os.path.split(old)[0]+os.sep
            newbasepath = os.path.split(new)[0]+os.sep
            fid = self.fullfilenamedict[old] # holds file id being moved

            # check all db dirs to see if the new file now belongs to any of them
            for did in self.dirs.keys():
                if newbasepath == self.dirs[did][0]:
                    self.dirs[did][4].append(new)
                    self.dirs[did][3].append(fid)
                try:
                    # the following line will raise an exception if old is not
                    # found, but that's ok. I guess it's cheaper to just let it
                    # happen, than to keep doing "if old in bla_list" checks and
                    # then removing the item in question.
                    self.dirs[did][4].remove(old) # remove from nominal (full path) list
                    self.dirs[did][3].remove(fid) # remove from id list
                except:                 # not on this dir.... move along
                    pass
            try:
                self.fullfilenamedict[new] = fid
                del self.fullfilenamedict[old]
                id = self.fullfilenamedict[new]
                self.meta[id]["Filename"] = os.path.split(new)[-1]
                self.filenamedict[self.meta[id]["Filename"]] = id
                del self.filenamedict[os.path.split(old)[-1]]
                self.img[id][0] = new
            except:
                Error.PrintTB("Renaming files")
                return 0
            if warn:
                self.changedDB({"scope":"Batch","reason":"changeditem"})
                self.changedDB({"scope":"Img","reason":"renamed","subject":id})
        return 1

    def removeBatch(self,bid):
        """call it to remove batch bid """
        try:
            del self.batches[bid]
        except:
            print "Error removing batch"
            return
        self.changedDB({"scope":"Batch","reason":"removed","subject":bid})
        self.dirty = 1

    def resetBatch(self,bid):
        """call it to remove batch bid """
        try:
            self.batches[bid][1] = []
        except:
            print "Error reseting batch"
            return
        self.changedDB({"scope":"Batch","reason":"reseted","subject":bid})

    def resetBatchHistory(self):
        """ remove all batches from db """
        self.batches = {1:[time.asctime(time.localtime()),[]]}
        self.changedDB({"scope":"Batch","reason":"resetedbatchhistory"})

    def addBatch(self,name = ""):
        """call create new batch in db

        @param name: new batch name. If undefined, the current time will be used
        @type name: string
        @return: new batch id
        """
        bid = 0
        try:
            if not name: name = time.asctime(time.localtime())
            bid = self.generateImageID()
            self.batches[bid] = [name,[]]
        except:
            traceback.print_exc()
            print "Error adding  batch"
            return 0
        self.changedDB({"scope":"Batch","reason":"new","subject":bid})
        self.dirty = 1
        return bid

    def idInBatch(self,it,bid = -1):
        """checks if the given id is in one of the groups/dirs already included on a batch """
        if bid==-1: bid = self.curBatch
        for cid in self.batches[bid][1]:
            try:
                if cid[1]==it[1]:
                    if cid[0]==it[0]:
                        return 1
            except:
                traceback.print_exc()
        return 0

    def refreshDB(self,cb = None,restr = None):
        """call to scan db dirs for new files

        @param cb: Defaults do None. If present, should be a QProgressBar or QProgressDialog with a QApp parent().
        @return: count of new files found and sucessfully added to db
        """
        if not restr:
            restr = AddFilter(self.env)
        restr.removeEmptyGroup = 1
        restr.groupid = 1
        restr.removeEmptyGroup = 0

        for vid in self.volumes.keys():
            lst = []
            self.crawlVolumeForDir(vid,lst)
            for did in lst:
                path = self.dirs[did][0]
                if not os.path.exists(path): continue
                newid = self.addDir(vid,path,1,[],0,-1,restr = restr)
        self.remove_dead()
        if cb:
            cb.cancel()
            cb.hide()
            cb.close()
            del cb
        return count

    def addSysDirBookmark(self,txt):
        self.sysdirbm.append(txt)
        self.dirty = 1
        for cb in self.sysdirbmCb: cb()

    def isGroupAncestor(self,chid,pid):
        """ check if a group pid is the ancestor (directly or inderectly, no matter how many generations apart) of another group chid.

        To avoid infinite loops when moving/copying groups

        @param pid: id of the parent group to check.
        @param chid: id of the child group to check
        @return: result
        @rtype: boolean
        """
        for son in self.groups[pid][4]:
            if son==chid:
                return 1
            else:
                return self.isGroupAncestor(chid,son)

    def delSysDirBookmark(self,id):
        """remove a bookmark

        @param id: 0-based index of the bmark to be removed
        """
        try:
            del self.sysdirbm[id]
            self.dirty = 1
            for cb in self.sysdirbmCb: cb()
        except:
            traceback.print_exc()
            print "Error removing the supplied bookmark id"

    def removeAllMeta(self,id):
        for fd in self.meta[id].keys():
            if fd not in self.readonlyfields:
                del self.meta[id][fd]
        self.changedDB({"scope":"Meta","reason":"removedall","subject":id})

    def syncFilenames(self,force = 0):
        """call it to update dictionary of filenames:id """
        if not force and (len(self.filenamedict.keys())==len(self.img.keys())):return # do not sync if almost nothing changed
        self.filenamedict = {}
        for id in self.img.keys():
            try:
                self.filenamedict[self.meta[id]["Filename"]] = id
            except:
                print "Image without metadata found. Trying to rebuild it."
                self.initImgMetadata(self.img[id][0],id)

    def syncFullFilenames(self,force = 0):
        """call it to update dictionary of filenames:id """
        if not force and (len(self.fullfilenamedict.keys())==len(self.img.keys())):return # do not sync if almost nothing changed
        self.fullfilenamedict = {}
        for id in self.img.keys():
            try:
                self.fullfilenamedict[self.img[id][0]] = id
            except:
                print "Image without metadata found. Trying to rebuild it."
                self.initImgMetadata(self.img[id][0],id)

    def queryFilename(self,fname,numres = 10,target = None,thresd=0.5):
        """query for similar filenames

        @param fname: full
        @param numres: maximum number of results
        @param target: list of file names to search on
        @return: list of similar filenames
        @rtype: list
        """
        if not target:
            self.syncFilenames()
            target = self.filenamedict
        import difflib                      # to match similar filenames
        try:
            fname = os.path.split(fname)[-1]
        except:
            pass

        mats = difflib.get_close_matches(fname, target.keys(), numres,thresd)
        mats = mats[1:]
        fres = []
        for mat in mats:
            fres.append([target[mat],0])
        return fres

    def queryRandomImage(self,count):
        """query for random images """
        ret = []
        if not self.img.keys():
            return ret
        for i in range(count):
            rid = self.img.keys()[random.randrange(len(self.img.keys()))]
            if rid not in ret:
                ret.append(rid)

        return [(rid,0) for rid in ret]

    def queryImage(self,filen,numres,scanned,removeFirst = 1):
        """query for similar images

        @param filen: full filename
        @param numres: maximum number of results
        @param scanned: true if image is a photo. False if it's a drawing
        @return: list of similar images. List of pairs [result id,result score]  Score is 0-100
        @rtype: list
        """
        imgdb.queryImgFile(filen,numres,not scanned)
        nres = imgdb.getNumResults()
        res = []
        for i in range(nres):
            rid = imgdb.getResultID()
            rsc = imgdb.getResultScore()
            # Where does the magic number 38.70 come from?
            rsc = -100.0*rsc/38.70
            #sanity checks
            if rsc<0:rsc = 0
            if rsc>100:rsc = 100
            res.append([rid,rsc])
        res.reverse()
        if not res: return res
        if removeFirst:
            return res[1:]
        else:
            return res

    def queryData(self,id,numres):
        """query for similar images. Source is a db image

        @param id: img id
        @param numres: maximum number of results
        @return: list of similar images. List of pairs [result id,result score]  Score is 0-100
        @rtype: list
        """
        ### NOTE: some code here should be on a separate function (so it can also get called by queryImage(). It's not in order
        # to avoid an extra function call when querying

        imgdb.queryImgID(id,numres)
        nres = imgdb.getNumResults()
        res = []
        for i in range(nres):
            rid = imgdb.getResultID()
            rsc = imgdb.getResultScore()
            rsc = -100.0*rsc/38.70
            #sanity checks
            if rsc<0:rsc = 0
            if rsc>100:rsc = 100
            res.append([rid,rsc])
        res.reverse()
        if not res: return res
        return res[1:]

    def moveObjectsToGroup(self, ids,gid):
        """ call to move a group of objects to gid

        @param ids: a list of maps.
        @param gid: destination group
        """
        exitc = 0
        self.doNotSave = 1
        try:
            for id in ids:
                if id["type"]=="Group":
                    self.moveGroup(id["id"],gid)
                if id["type"]=="Img":
                    if id.has_key("FromId"):
                        self.moveFileGroup(id["id"],gid,id["FromId"])
                    else:
                        self.moveFileGroup(id["id"],gid)
            exitc = 1
        except:
            traceback.print_exc()
            exitc = 1
        self.doNotSave = 0
        self.changedDB({"scope":"Group","reason":"movedimages"})
        return exitc

    def copyGroup(self,sid,gid):
        """call to copy a group into another

        @param sid: source group id
        @param gid: destination group id. New group will be a child of group gid
        @return: new group id (ie, the id of the new group which is a copy of sid)
        """
        if self.isGroupAncestor(gid,sid):
            print "Attempt to move/copy parent to a child, ignoring request."
            return -1
        nid = self.newGroup(self.groups[sid][0][:],pargid = gid)
        self.groups[nid][1] = self.groups[sid][1][:]
        self.groups[nid][3] = self.groups[sid][3][:]
        for sg in self.groups[sid][4]:
            self.copyGroup(sg,nid)
        return nid

    def copyObjectsToGroup(self, ids,gid):
        """ call to copy a group of objects to a destination group

        @param gid: destination group
        @param ids:  is a list of maps. {'type':{Group,Img},'id':obj_id}
        @return: True if something got changed
        """

        exitc = 0
        self.doNotSave = 1
        try:
            for id in ids:
                if id["type"]=="Group":
                    self.copyGroup(id["id"],gid)
                if id["type"]=="Img":
                    self.moveFileGroup(id["id"],gid)
            exitc = 1
        except:
            traceback.print_exc()
            exitc = 1
        self.doNotSave = 0
        self.changedDB({"scope":"Group","reason":"copiedimages"})
        return exitc

    def textquery(self,txt,numres = 10):
        """start text query with logical operators.

        @param txt: txt[1] = {'AND','OR'} is the logic to be used. txt[0] is a list of pairs [field_name,query_keyword]
        @param numres: maximum number of results
        @type numres: int
        @return: list of pairs [img_id,0]. (0 is fixed and should be ignored.) What matters is the img_id integer, which means this image is a query result
        """
        #TODO2: document the logic for this function. I can't even understand it now...
        res = []
        metaq = txt[0]
        logic = txt[1]
        pres = []

        for fi in range(len(metaq)):
            pres.append([])
        if not pres:                    # there is no sense in a query with no keywords
            return

        for fil in self.meta.keys():
            for fi in range(len(pres)):
                try:
                    field = metaq[fi][0]
                    if not self.meta[fil].has_key(field) or not self.meta[fil][field]:
                        continue
                    else:
                        if find(lower(self.meta[fil][field]),lower(metaq[fi][1])) != -1:
                            pres[fi].append(fil)

                except:
                    traceback.print_exc()
                    pass

        if logic == "AND":
            min = 0
            for fi in range(len(pres)):
                if len(pres[fi]) < len(pres[min]):
                    min = fi
            for id in pres[min]:
                failed = 0
                for fi in range(len(pres)):
                    if id not in pres[fi]:
                        failed = 1
                        break
                if not failed and id not in res: res.append(id)

        elif logic == "OR":
            for fi in range(len(pres)):
                for id in pres[fi]:
                    if id not in res: res.append(id)

        fres = []
        for id in res:
            fres.append([id,0])
            if len(fres)>=numres:
                return fres
        return fres

    def closedb(self):
        """close database files safely. Make sure this is called when the program exists. If any database is dirty, it will be saved.
        """
        self.savedb()
        self.dirty = 0
        imgdb.closeDbase()

    def regenerateThumbs(self):
        """call it to iterate all db images and force de re-generation of their cached thumbnail.
        """
        for id in self.img.keys():
            self.getThumbDB(id,force = 1, onlyNewer = 1)

    def importMeta(self,header,data,key,fullpath = 1):
        """import data from an array of arrays into the internal metadata dict.

        This function is called directly by the Import CSV wizard.
        Will iterate through every element in data, trying to find the corresponding image according to the key field indicated by the key
        parameter, and then import the data for the file.

        @param header: list of meta fields (strings), to which data will be associated to on every imported file
        @param data: list of lists of strings.
        @param fullpath: if True, match key field to full path instead of filename
        @param key: index of column to match when looking which file in db to set this data
        @type key: int
        @return: Human readable text informing how many items were imported.
        @rtype: string
        """
        resp = ""
        count = 0
        self.syncFilenames()
        for it in data:
            try:
                id = -1
                if fullpath:
                    for itd in self.img.keys():
                        if self.img[itd][0]==it[key]:
                            id = itd
                            break
                else:
                    if self.filenamedict.has_key(it[key]):
                        id = self.filenamedict[it[key]]
                if id==-1:
                    continue
                for hid in range(len(header)):
                    if header[hid] in self.readonlyfields: continue
                    if hid==key: continue
                    self.meta[id][header[hid]] = it[hid]
                count = count+1
                self.changedDB({"scope":"Meta","reason":"changed","subject":id})
            except:
                traceback.print_exc()
                continue
        resp = resp+str(self.tr("%d item(s) imported."))%count
        return resp

    def openImage(self,fname):
        """abstract method for opening an image. This function should call whatever is available to open the given image

        @param fname: full path for an image
        @return QImage instance
        """
        (base,ext) = os.path.splitext(fname)
        fext = ext[1:].lower()
        aImage = QImage()
        if fext in self.env.qt_ext:
            if not aImage.load(fname):
                print "Error loading image file " + fname
        elif self.env.hasImMagick and (fext in self.env.magick_ext):
            imgdb.convert(fname,self.thdir2+".cachev.bmp")
            if not aImage.load(self.thdir2+".cachev.bmp"):
                print "Error loading cached image file " + fname
        elif self.env.hasPIL and (fext in self.env.pil_ext):
            im = Image.open(fname)
            im.save(self.thdir2+".cachev.bmp")
            del im
            if not aImage.load(self.thdir2+".cachev.bmp"):
                print "Error loading cached image file " + fname
        return aImage

    def openPixmap(self,fname):
        """abstract method for opening an image. This function should call whatever is available to open the given image

        @param fname: full path for an image
        @return QPixmap instance
        """
        (base,ext) = os.path.splitext(fname)
        fext = ext[1:].lower()
        aImage = QPixmap()
        if fext in self.env.qt_ext:
            if not aImage.load(fname):
                print "Error loading image file " + fname
        elif self.env.hasImMagick and (fext in self.env.magick_ext):
            imgdb.convert(fname,self.thdir2+".cachev.bmp")
            if not aImage.load(self.thdir2+".cachev.bmp"):
                print "Error loading cached image file " + fname
        elif self.env.hasPIL and (fext in self.env.pil_ext):
            im = Image.open(fname)
            im.save(self.thdir2+".cachev.bmp")
            del im
            if not aImage.load(self.thdir2+".cachev.bmp"):
                print "Error loading cached image file " + fname
        return aImage

    def scanForMetaTags(self,bid):
        """rescan all images in this batch for metadata on the files

        @param bid: batch id
        """
        ids = []
        self.crawlBatchForImg(bid,ids)
        for nfid in ids:
            fname = self.img[nfid][0]
            try:
                if not self.meta.has_key(nfid):
                    self.meta[nfid] = self.blankMetaDict.copy()
                try:
                    self.initImgMetadata(fname,nfid)
                except:
                    traceback.print_exc()
                    print "Error setting basic metadata for "+fname
                    self.meta[nfid]["Filename"] = fname
            except:
                print "Error gathering metadata for ",fname
                traceback.print_exc()
            ### extract optional metadata
            ext = self.extIsImg(fname,probe = 1)
            if (ext in self.env.exif_ext):
                try:
                    self.extractEXIF(nfid)
                except:
                    traceback.print_exc()
                    print "Error extracting EXIF metadata for:",fname
            if (ext in self.env.iptc_ext) and self.env.hasIPTC:
                try:
                    self.extractIPTC(nfid)
                except:
                    traceback.print_exc()
                    print "Error extracting IPTC metadata for:",fname

    def createGroupsFromSymGroups(self):
        """create a new logical group for every similarity group
        """
        self.doNotSave = 1
        for sid in self.simgroups.keys():
            ngid = self.newGroup("["+self.meta[self.simgroups[sid][1]]["Filename"]+"]")
            self.groups[ngid][1] = str(self.tr("Images similar to "))+self.img[self.simgroups[sid][1]][0]
            for imid in self.simgroups[sid][2]:
                self.groups[ngid][3].append(imid)
        self.doNotSave = 0
        self.changedDB({"scope":"Group","reason":"added"})
