#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import zipfile
import hashlib
import base64
import string
import stat
import sys
import os

from twisted.internet.defer import Deferred
from twisted.internet import reactor

from imagestore.lib.twistedutil import mergeDeferreds

from imagestore.eucaservice import (
    EucaService, EucaServiceError, EucaInfo, GetEucaInfo,
    BundleAndUploadImageTask, EucaTools, EucaToolsError, writeCredentialsZip)
from imagestore import eucaservice

from imagestore.model import ImageRegistration
from imagestore.lib.tests import TestCase, mocker
from imagestore.tests.helpers import ServiceTestCase


class EucaServiceTest(ServiceTestCase):


    def createTime(self):
        self.currentTime += 1
        return self.currentTime

    def setUp(self):
        self.basePath = self.makeDir()
        self.currentTime = 41
        self.service = EucaService(reactor, self.basePath,
                                   createTime=self.createTime)

        eucaservice.writeCredentialsZip = writeFakeCredentialsZip

    def tearDown(self):
        eucaservice.writeCredentialsZip = writeCredentialsZip

    def assertMD5(self, path, expectedMD5):
        file = open(path)
        data = file.read()
        file.close()
        obtainedMD5 = hashlib.md5(data).hexdigest()
        self.assertEquals(obtainedMD5, expectedMD5)

    def assertMode(self, path, expectedMode):
        obtainedMode = stat.S_IMODE(os.stat(path).st_mode)
        self.assertEquals(obtainedMode, expectedMode)

    def testGetEucaInfo(self):
        deferred1 = self.service.addTask(GetEucaInfo("admin"))
        def callback(eucaInfo):
            self.assertEquals(eucaInfo.userId, "000100729354")
            self.assertEquals(eucaInfo.accessKey,
                              "WKy3rMzOWPouVOxK1p3Ar1C2uRBwa2FBXnCw")
            self.assertEquals(eucaInfo.secretKey,
                              "DR8tdQVYOeObuDLsKwYkkzmjRcz9cfflbcVtmQ")

            join = os.path.join

            self.assertEquals(eucaInfo.privateKeyPath,
                              join(self.basePath, "admin-user-private-key.pem"))
            self.assertMD5(eucaInfo.privateKeyPath,
                           "0a173e5381df3aee296c4cd308114591")
            self.assertMode(eucaInfo.privateKeyPath, 0600)

            self.assertEquals(eucaInfo.certificatePath,
                              join(self.basePath, "admin-user-certificate.pem"))
            self.assertMD5(eucaInfo.certificatePath,
                           "e6ed34899c11bd032566ddfa4298c0a1")
            self.assertMode(eucaInfo.certificatePath, 0600)

            self.assertEquals(eucaInfo.cloudCertificatePath,
                              join(self.basePath, "cloud-certificate.pem"))
            self.assertMD5(eucaInfo.cloudCertificatePath,
                           "cc9c834817a747f3f1f65db8b4708bd1")
            self.assertMode(eucaInfo.cloudCertificatePath, 0600)

            self.assertEquals(eucaInfo.urlForS3,
                              "http://10.0.1.50:8773/services/Walrus")
            self.assertEquals(eucaInfo.urlForEC2,
                              "http://10.0.1.50:8773/services/Eucalyptus")

        deferred1.addCallback(callback)

        self.service.start()
        deferred2 = self.service.stop()

        return mergeDeferreds([deferred1, deferred2])

    def testGetEucaInfoTwice(self):
        testDoneDeferred = Deferred()

        deferred1 = self.service.addTask(GetEucaInfo("admin"))
        def callback1(eucaInfo):
            # On the first call, we write some bad data which is twice as
            # large as the original file, to ensure it'll get overwritten
            # in the next call.
            path = eucaInfo.privateKeyPath
            file = open(path, "w")
            file.write("x" * (os.path.getsize(path) * 2))
            file.close()

        deferred1.addCallback(callback1)

        def doSecondCall(result):
            # We want it to actually retrieve and write it again.
            self.service.clearEucaInfoCache()

            deferred2 = self.service.addTask(GetEucaInfo("admin"))
            def callback2(eucaInfo):
                path = eucaInfo.privateKeyPath
                self.assertMD5(path, "0a173e5381df3aee296c4cd308114591")
            deferred2.addCallback(callback2)

            # Now that both tasks have been added to the service, we can
            # ask the service to stop.
            deferred3 = self.service.stop()

            # We then wait on both the asserting callback and the service
            # stop, before considering the test concluded.
            deferred4 = mergeDeferreds([deferred2, deferred3])
            deferred4.chainDeferred(testDoneDeferred)

        # Only perform the second call once the first callback has finished,
        # otherwise we'd overwrite the same file twice in quick succession,
        # and callback1() might be run after both tasks were already finished.
        deferred1.addCallback(doSecondCall)

        self.service.start()

        return testDoneDeferred

    def testGetCachedEucaInfo(self):
        testDoneDeferred = Deferred()

        deferred1 = self.service.addTask(GetEucaInfo("admin"))
        def callback1(eucaInfo):
            def raiseError(*args, **kwargs):
                raise RuntimeError("Shouldn't call this!")
            eucaservice.writeCredentialsZip = raiseError
        deferred1.addCallback(callback1)

        def doSecondCall(result):
            deferred2 = self.service.addTask(GetEucaInfo("admin"))
            def callback2(eucaInfo):
                self.assertEquals(eucaInfo.userId, "000100729354")
            deferred2.addCallback(callback2)

            # Now that both tasks have been added to the service, we can
            # ask the service to stop.
            deferred3 = self.service.stop()

            # We then wait on both the asserting callback and the service
            # stop, before considering the test concluded.
            deferred4 = mergeDeferreds([deferred2, deferred3])
            deferred4.chainDeferred(testDoneDeferred)

        # Only perform the second call once the first callback has finished.
        deferred1.addCallback(doSecondCall)

        def showError(failure):
            failure.printTraceback(sys.stderr)
        deferred1.addErrback(showError)

        self.service.start()

        return testDoneDeferred

    def testBadZipFile(self):
        path = self.makeFile("not-a-zip")

        def writeMyCredentialsZip(username, outputFilename,
                                  eucalyptusPrefix):
            os.rename(path, outputFilename)
        eucaservice.writeCredentialsZip = writeMyCredentialsZip

        deferred1 = self.service.addTask(GetEucaInfo("admin"))
        def errback(failure):
            failure.trap(EucaServiceError)
            self.assertEquals(failure.value.message,
                              "Credentials zip file is corrupted")
            return failure
        deferred1.addErrback(errback)
        self.assertFailure(deferred1, EucaServiceError)

        return self.runServicesAndWaitForDeferred([self.service], deferred1)

    def testEmptyZipFile(self):
        path = self.makeFile()
        zip = zipfile.ZipFile(path, "w")
        zip.writestr("somefile", "Ho ho")
        zip.close()

        def writeMyCredentialsZip(username, outputFilename,
                                  eucalyptusPrefix):
            os.rename(path, outputFilename)
        eucaservice.writeCredentialsZip = writeMyCredentialsZip

        deferred1 = self.service.addTask(GetEucaInfo("admin"))
        def errback(failure):
            failure.trap(EucaServiceError)
            self.assertEquals(failure.value.message,
                              "'eucarc' not found in credentials zip file")
            return failure
        deferred1.addErrback(errback)
        self.assertFailure(deferred1, EucaServiceError)

        return self.runServicesAndWaitForDeferred([self.service], deferred1)

    def testCorruptedZipFile(self):
        path = self.makeFile()
        zip = zipfile.ZipFile(path, "w")
        zip.writestr("eucarc", "some content")
        zip.close()

        file = open(path, "w+")
        file.seek(20)
        file.write("hi!")
        file.close()

        def writeMyCredentialsZip(username, outputFilename,
                                  eucalyptusPrefix):
            os.rename(path, outputFilename)
        eucaservice.writeCredentialsZip = writeMyCredentialsZip

        deferred1 = self.service.addTask(GetEucaInfo("admin"))
        def errback(failure):
            failure.trap(EucaServiceError)
            self.assertEquals(failure.value.message,
                              "Credentials zip file is corrupted")
            return failure
        deferred1.addErrback(errback)
        self.assertFailure(deferred1, EucaServiceError)

        return self.runServicesAndWaitForDeferred([self.service], deferred1)

    def testWriteCredentialsZip(self):
        path = self.makeFile("")
        getStatusOutputMock = self.mocker.replace("commands.getstatusoutput")
        getStatusOutputMock("/usr/sbin/euca_conf --get-credentials %s" % path)
        self.mocker.result((0, "!OUTPUT!"))
        getStatusOutputMock(mocker.ARGS, mocker.KWARGS)
        self.mocker.count(0)
        self.mocker.replay()
        writeCredentialsZip("admin", path)

        # It must erase the old file because euca_conf fails to replace it.
        self.assertFalse(os.path.isfile(path))

    def testWriteCredentialsZipError(self):
        getStatusOutputMock = self.mocker.replace("commands.getstatusoutput")
        getStatusOutputMock("/usr/sbin/euca_conf --get-credentials !FILE!")
        self.mocker.result((1, "!OUTPUT!"))
        getStatusOutputMock(mocker.ARGS, mocker.KWARGS)
        self.mocker.count(0)
        self.mocker.replay()
        try:
            writeCredentialsZip("admin", "!FILE!")
        except EucaToolsError, e:
            self.assertEquals(str(e),
                              "Error from euca_conf --get-credentials:\n"
                              "!OUTPUT!")
        else:
            self.fail("EucaToolsError not raised")

    def testWriteCredentialsZipWithPrefix(self):
        getStatusOutputMock = self.mocker.replace("commands.getstatusoutput")
        getStatusOutputMock("/prefix/usr/sbin/euca_conf "
                            "--get-credentials !FILE!")
        self.mocker.result((0, "!OUTPUT!"))
        getStatusOutputMock(mocker.ARGS, mocker.KWARGS)
        self.mocker.count(0)
        self.mocker.replay()
        writeCredentialsZip("admin", "!FILE!", "/prefix")

    def testGetEucaInfoWithPrefix(self):
        class Stop(Exception): pass

        writeMock = self.mocker.replace(eucaservice.writeCredentialsZip)
        writeMock("admin", mocker.ANY, "/prefix")
        self.mocker.throw(Stop)

        # No other calls to writeCredentialsZip() are expected.
        writeMock(mocker.ARGS, mocker.KWARGS)
        self.mocker.count(0)

        self.mocker.replay()

        service = EucaService(reactor, self.basePath, "/prefix")
        deferred = service.addTask(GetEucaInfo("admin"))

        def testDone(failure):
            failure.trap(Stop)
            deferred = service.stop()
            deferred.addCallback(lambda result: failure)
            return deferred
        deferred.addErrback(testDone)

        service.start()

        return self.assertFailure(deferred, Stop)

    def testBundleAndUpload(self):
        imageRegistration = self.createImageRegistration(1)

        imageRegistration.eki.path = "/kernel/path"
        imageRegistration.eri.path = "/ramdisk/path"
        imageRegistration.emi.path = "/image/path"

        eucaTools = self.mocker.patch(EucaTools)

        temporaryDirs = []
        def match(argument):
            temporaryDirs.append(argument)
            return os.path.isdir(argument)

        ANYDIR = mocker.MATCH(match)

        eucaTools.bundleKernel("/kernel/path", ANYDIR, "kernel")
        self.mocker.result("/kernel/manifest/path")
        eucaTools.uploadBundle("/kernel/manifest/path", "image-store-42")
        eucaTools.registerBundle("/kernel/manifest/path", "image-store-42")
        self.mocker.result("the-eki")

        eucaTools.bundleRamdisk("/ramdisk/path", ANYDIR, "ramdisk")
        self.mocker.result("/ramdisk/manifest/path")
        eucaTools.uploadBundle("/ramdisk/manifest/path", "image-store-42")
        eucaTools.registerBundle("/ramdisk/manifest/path", "image-store-42")
        self.mocker.result("the-eri")

        eucaTools.bundleImage("/image/path", ANYDIR,
                              "the-eki", "the-eri", "image")
        self.mocker.result("/image/manifest/path")
        eucaTools.uploadBundle("/image/manifest/path", "image-store-42")
        eucaTools.registerBundle("/image/manifest/path", "image-store-42")
        self.mocker.result("the-emi")

        self.mocker.replay()

        task = BundleAndUploadImageTask(imageRegistration)
        deferred1 = self.service.addTask(task)
        def callback(reg):
            self.assertEquals(reg.eki.eid, "the-eki")
            self.assertEquals(reg.eri.eid, "the-eri")
            self.assertEquals(reg.emi.eid, "the-emi")
            
            for path in temporaryDirs:
                self.assertFalse(os.path.exists(path),
                                 "Temporary directory %s still exists" % path)

            self.assertEquals(len(set(temporaryDirs)), 3)

        deferred1.addCallback(callback)
        self.service.start()
        deferred2 = self.service.stop()
        return mergeDeferreds([deferred1, deferred2])


class EucaServiceFakeModeTest(ServiceTestCase):

    def setUp(self):
        self.basePath = self.makeDir()
        self.service = EucaService(reactor, self.basePath,
                                   fakeMode=True, fakeSecretKey="!FAKEKEY!")

    def testGetEucaInfo(self):
        deferred1 = self.service.addTask(GetEucaInfo("admin"))
        def callback(eucaInfo):
            self.assertEquals(eucaInfo.userId, "01234567890")
            self.assertEquals(eucaInfo.accessKey, "AcCeSsKeY123")
            self.assertEquals(eucaInfo.secretKey, "!FAKEKEY!")
            self.assertEquals(eucaInfo.privateKeyPath, "/fake/path")
            self.assertEquals(eucaInfo.certificatePath, "/fake/path")
            self.assertEquals(eucaInfo.cloudCertificatePath, "/fake/path")
            self.assertEquals(eucaInfo.urlForS3, "http://fake/url")
            self.assertEquals(eucaInfo.urlForEC2, "http://fake/url")

        deferred1.addCallback(callback)
        self.service.start()
        deferred2 = self.service.stop()
        return mergeDeferreds([deferred1, deferred2])

    def testBundleAndUploadImageTest(self):
        randintMock = self.mocker.replace("random.randint")
        randintMock(100000, 999999)
        self.mocker.result(123456)
        self.mocker.replay()

        imageRegistration = self.createImageRegistration(1)
        task = BundleAndUploadImageTask(imageRegistration)
        deferred1 = self.service.addTask(task)
        def callback(reg):
            self.assertEquals(reg.eki.eid, "eki-123456")
            self.assertEquals(reg.eri.eid, "eri-123456")
            self.assertEquals(reg.emi.eid, "emi-123456")

        deferred1.addCallback(callback)
        self.service.start()
        deferred2 = self.service.stop()
        return mergeDeferreds([deferred1, deferred2])


FAKE_TOOL = string.Template("""\
#!/usr/bin/env python
import sys, os

DEBUG_FILE = ${DEBUG_FILE}
CREATE_FILE = ${CREATE_FILE}

debugFile = open(DEBUG_FILE, "w")
debugFile.write(repr(sys.argv[1:]) + "\\n")
for name in ("PATH EC2_USER_ID EC2_PRIVATE_KEY EC2_CERT EUCALYPTUS_CERT "
             "EC2_ACCESS_KEY EC2_SECRET_KEY EC2_URL S3_URL").split():
    debugFile.write("%s=%s\\n" % (name, os.environ.get(name)))
debugFile.close()

if CREATE_FILE:
    os.makedirs(os.path.split(CREATE_FILE)[0])
    open(CREATE_FILE, "w").close()

${EXTRA_CODE}

sys.stdout.write(${STDOUT_DATA})
sys.stderr.write(${STDERR_DATA})
sys.exit(${STATUS_CODE})
""")

EXPECTED_ENVIRON = """
PATH=/bin:/usr/bin/:/sbin:/usr/sbin
EC2_USER_ID=!USERID!
EC2_PRIVATE_KEY=!PKPATH!
EC2_CERT=!CERTPATH!
EUCALYPTUS_CERT=!CLOUDCERTPATH!
EC2_ACCESS_KEY=!ACCESSKEY!
EC2_SECRET_KEY=!SECRETKEY!
EC2_URL=!EC2URL!
S3_URL=!S3URL!
"""

class FakeEucaInfo(EucaInfo):
    userId = "!USERID!"
    accessKey = "!ACCESSKEY!"
    secretKey = "!SECRETKEY!"
    privateKeyPath = "!PKPATH!"
    certificatePath = "!CERTPATH!"
    cloudCertificatePath = "!CLOUDCERTPATH!"
    urlForS3 = "!S3URL!"
    urlForEC2 = "!EC2URL!"


class EucaToolsTest(TestCase):

    def setUp(self):
        self.basePath = self.makeDir()
        self.debugDataPath = {}
        self.eucaTools = EucaTools(FakeEucaInfo())

    def useDebugTool(self, toolName, stdoutData="", stderrData="",
                     statusCode=0, extraCode="", createFile=""):
        self.debugDataPath[toolName] = debugDataPath = self.makeFile()
        toolContent = FAKE_TOOL.substitute(DEBUG_FILE=repr(debugDataPath),
                                           STDOUT_DATA=repr(stdoutData),
                                           STDERR_DATA=repr(stderrData),
                                           STATUS_CODE=statusCode,
                                           CREATE_FILE=repr(createFile),
                                           EXTRA_CODE=extraCode)
        path = self.makeFile(toolContent)
        os.chmod(path, 0700)
        self.eucaTools.setToolPath(toolName, path)

    def assertDebugData(self, toolName, expectedData):
        debugDataPath = self.debugDataPath[toolName]
        if not os.path.isfile(debugDataPath):
            self.fail("Tool %s not executed!" % (toolName,))
        else:
            file = open(debugDataPath)
            obtainedData = file.read()
            file.close()

            def strip(data):
                return "\n".join(line.strip() for line in data.splitlines()
                                 if line)

            def merge(data):
                return "|".join(data.split())

            obtainedData = strip(obtainedData)
            expectedData = strip(expectedData)

            if merge(obtainedData) != merge(expectedData):
                self.fail("\n\n"
                          "-- Expected -------------------------------------\n"
                          "%s\n\n"
                          "-- Obtained -------------------------------------\n"
                          "%s\n" % (expectedData, obtainedData))

    def testBundleKernel(self):
        bundlePath = self.makeFile()
        bundleManifest = bundlePath + "/test.manifest.xml"
        self.useDebugTool("euca-bundle-image",
                          createFile=bundleManifest)
        result = self.eucaTools.bundleKernel("!FILE!", bundlePath)
        self.assertDebugData("euca-bundle-image",
            """
            ['--user', '!USERID!', '--image', '!FILE!',
             '--destination', '%s', '--kernel', 'true']%s
            """ % (bundlePath, EXPECTED_ENVIRON))
        self.assertEquals(result, bundleManifest)

    def testBundleKernelWithPrefix(self):
        bundlePath = self.makeFile()
        bundleManifest = bundlePath + "/test.manifest.xml"
        self.useDebugTool("euca-bundle-image",
                          createFile=bundleManifest)
        result = self.eucaTools.bundleKernel("!FILE!", bundlePath, "!PREFIX!")
        self.assertDebugData("euca-bundle-image",
            """
            ['--user', '!USERID!', '--image', '!FILE!',
             '--destination', '%s', '--kernel', 'true',
             '--prefix', '!PREFIX!']%s
            """ % (bundlePath, EXPECTED_ENVIRON))
        self.assertEquals(result, bundleManifest)

    def testBundleKernelError(self):
        # If the manifest is not created, it should error out.
        self.useDebugTool("euca-bundle-image",
                          stdoutData="!STDOUT!\n",
                          stderrData="!STDERR!\n")
        try:
            self.eucaTools.bundleKernel(file="!FILE!",
                                        bundlePath="!BUNDLEPATH!")
        except EucaToolsError, e:
            self.assertDebugData("euca-bundle-image",
                """
                ['--user', '!USERID!', '--image', '!FILE!',
                 '--destination', '!BUNDLEPATH!', '--kernel', 'true']%s
                """ % EXPECTED_ENVIRON)
            self.assertIn("Manifest wasn't generated", str(e))
            self.assertIn("!STDOUT!", str(e))
            self.assertIn("!STDERR!", str(e))
        else:
            self.fail("Did not raise EucaToolsError")

    def testBundleRamdisk(self):
        bundlePath = self.makeFile()
        bundleManifest = bundlePath + "/test.manifest.xml"
        self.useDebugTool("euca-bundle-image",
                          createFile=bundleManifest)
        result = self.eucaTools.bundleRamdisk("!FILE!", bundlePath)
        self.assertDebugData("euca-bundle-image",
            """
            ['--user', '!USERID!', '--image', '!FILE!',
             '--destination', '%s', '--ramdisk', 'true']%s
            """ % (bundlePath, EXPECTED_ENVIRON))
        self.assertEquals(result, bundleManifest)

    def testBundleRamdiskWithPrefix(self):
        bundlePath = self.makeFile()
        bundleManifest = bundlePath + "/test.manifest.xml"
        self.useDebugTool("euca-bundle-image",
                          createFile=bundleManifest)
        result = self.eucaTools.bundleRamdisk("!FILE!", bundlePath, "!PREFIX!")
        self.assertDebugData("euca-bundle-image",
            """
            ['--user', '!USERID!', '--image', '!FILE!',
             '--destination', '%s', '--ramdisk', 'true',
             '--prefix', '!PREFIX!']%s
            """ % (bundlePath, EXPECTED_ENVIRON))
        self.assertEquals(result, bundleManifest)

    def testBundleRamdiskError(self):
        # If the manifest is not created, it should error out.
        self.useDebugTool("euca-bundle-image",
                          stdoutData="!STDOUT!\n",
                          stderrData="!STDERR!\n")
        try:
            self.eucaTools.bundleRamdisk(file="!FILE!",
                                         bundlePath="!BUNDLEPATH!")
        except EucaToolsError, e:
            self.assertDebugData("euca-bundle-image",
                """
                ['--user', '!USERID!', '--image', '!FILE!',
                 '--destination', '!BUNDLEPATH!', '--ramdisk', 'true']%s
                """ % EXPECTED_ENVIRON)
            self.assertIn("Manifest wasn't generated", str(e))
            self.assertIn("!STDOUT!", str(e))
            self.assertIn("!STDERR!", str(e))
        else:
            self.fail("Did not raise EucaToolsError")

    def testBundleImage(self):
        bundlePath = self.makeFile()
        bundleManifest = bundlePath + "/test.manifest.xml"
        self.useDebugTool("euca-bundle-image",
                          createFile=bundleManifest)
        result = self.eucaTools.bundleImage("!FILE!", bundlePath,
                                            "!EKI!", "!ERI!")
        self.assertDebugData("euca-bundle-image",
            """
            ['--user', '!USERID!', '--image', '!FILE!',
             '--destination', '%s', '--kernel', '!EKI!', '--ramdisk', '!ERI!']%s
            """ % (bundlePath, EXPECTED_ENVIRON))
        self.assertEquals(result, bundleManifest)

    def testBundleImageWithPrefix(self):
        bundlePath = self.makeFile()
        bundleManifest = bundlePath + "/test.manifest.xml"
        self.useDebugTool("euca-bundle-image",
                          createFile=bundleManifest)
        result = self.eucaTools.bundleImage("!FILE!", bundlePath,
                                            "!EKI!", "!ERI!", "!PREFIX!")
        self.assertDebugData("euca-bundle-image",
            """
            ['--user', '!USERID!', '--image', '!FILE!',
             '--destination', '%s', '--kernel', '!EKI!', '--ramdisk', '!ERI!',
             '--prefix', '!PREFIX!']%s
            """ % (bundlePath, EXPECTED_ENVIRON))
        self.assertEquals(result, bundleManifest)

    def testBundleImageError(self):
        # If the manifest is not created, it should error out.
        self.useDebugTool("euca-bundle-image",
                          stdoutData="!STDOUT!\n",
                          stderrData="!STDERR!\n")
        try:
             self.eucaTools.bundleImage("!FILE!", "!BUNDLEPATH!",
                                        "!EKI!", "!ERI!")
        except EucaToolsError, e:
            self.assertDebugData("euca-bundle-image",
                """
                ['--user', '!USERID!', '--image', '!FILE!',
                 '--destination', '!BUNDLEPATH!',
                 '--kernel', '!EKI!', '--ramdisk', '!ERI!']%s
                """ % EXPECTED_ENVIRON)
            self.assertIn("Manifest wasn't generated", str(e))
            self.assertIn("!STDOUT!", str(e))
            self.assertIn("!STDERR!", str(e))
        else:
            self.fail("Did not raise EucaToolsError")

    def testUploadBundle(self):
        bundleManifest = self.makeFile("")
        self.useDebugTool("euca-upload-bundle")
        self.eucaTools.uploadBundle(bundleManifest, "!BUCKET!")
        self.assertDebugData("euca-upload-bundle",
            "['--manifest', '%s', '--bucket', '!BUCKET!'] %s"
            % (bundleManifest, EXPECTED_ENVIRON))

    def testUploadBundleError(self):
        bundleManifest = self.makeFile("")
        self.useDebugTool("euca-upload-bundle",
                          stdoutData="!STDOUT!\n",
                          stderrData="!STDERR!\n",
                          statusCode=12)
        try:
             self.eucaTools.uploadBundle(bundleManifest, "!BUCKET!")
        except EucaToolsError, e:
            self.assertDebugData("euca-upload-bundle",
                "['--manifest', '%s', '--bucket', '!BUCKET!'] %s"
                % (bundleManifest, EXPECTED_ENVIRON))
            self.assertEquals("Command 'euca-upload-bundle' returned "
                              "status code 12:\n!STDERR!\n!STDOUT!\n", str(e))
        else:
            self.fail("Did not raise EucaToolsError")

    def testUploadBundleMissingManifestError(self):
        try:
             self.eucaTools.uploadBundle("!MANIFEST!", "!BUCKET!")
        except EucaToolsError, e:
            self.assertEquals(str(e), "Bundle manifest '!MANIFEST!' "
                                      "not found for upload.")
        else:
            self.fail("Did not raise EucaToolsError")

    def testRegisterBundle(self):
        bundleManifest = self.makeFile("", basename="test.manifest.xml")
        self.useDebugTool("euca-register", stdoutData=" IMAGE\t !EMI!\n \n")
        result = self.eucaTools.registerBundle(bundleManifest, "!BUCKET!")
        self.assertDebugData("euca-register",
            "['!BUCKET!/test.manifest.xml'] %s"
            % (EXPECTED_ENVIRON,))
        self.assertEquals(result, "!EMI!")

    def testRegisterBundleOutputError1(self):
        bundleManifest = self.makeFile("", basename="test.manifest.xml")
        self.useDebugTool("euca-register", stdoutData="WHAT !EMI!")
        try:
            self.eucaTools.registerBundle(bundleManifest, "!BUCKET!")
        except EucaToolsError, e:
            self.assertDebugData("euca-register",
                "['!BUCKET!/test.manifest.xml'] %s"
                % (EXPECTED_ENVIRON,))
            self.assertEquals(str(e),
                              "Unexpected output registering image:\n"
                              "WHAT !EMI!")
        else:
            self.fail("Did not raise EucaToolsError")

    def testRegisterBundleOutputError2(self):
        bundleManifest = self.makeFile("", basename="test.manifest.xml")
        self.useDebugTool("euca-register", stdoutData="IMAGE !EMI! WHO")
        try:
            self.eucaTools.registerBundle(bundleManifest, "!BUCKET!")
        except EucaToolsError, e:
            self.assertDebugData("euca-register",
                "['!BUCKET!/test.manifest.xml'] %s"
                % (EXPECTED_ENVIRON,))
            self.assertEquals(str(e),
                              "Unexpected output registering image:\n"
                              "IMAGE !EMI! WHO")
        else:
            self.fail("Did not raise EucaToolsError")


def writeFakeCredentialsZip(username, outputFilename, eucalyptusPrefix=""):
    assert username == "admin", "Only admin supported for now."
    assert eucalyptusPrefix == "", "Received an unexpected eucalyptus prefix"
    file = open(outputFilename, "w")
    file.write(base64.decodestring(CREDENTIALS_ZIP))
    file.close()


CREDENTIALS_ZIP = \
"""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==
"""
