#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import os
import hashlib

from urlparse import urlunsplit

from twisted.internet import reactor
from twisted.internet.defer import maybeDeferred

from imagestore.lib.twistedutil import mergeDeferreds
from imagestore.lib.tests import TestCase
from imagestore.lib.service import ServiceHub

from imagestore.model import Image, ImageFile, ImagePart, ImageRegistration


def path2url(path):
    return urlunsplit(("file", "", path, "", ""))


def sha256(path):
    from hashlib import sha256
    return sha256(open(path).read()).hexdigest()


def createImage(id):
    return Image(uri="http://example/%d" % id,
                 title="Test Image %d" % id,
                 version="1.%d" % id,
                 summary="Test image number %d." % id)


def createImagePart(id, kind="kernel"):
    prefix = "eki"
    if kind == "ramdisk":
        prefix = "eri"
    elif kind == "image":
        prefix = "emi"

    content = "X" * id
    part = ImagePart()
    part.eid = "%s-%08x" % (prefix, id)
    part.kind = kind
    part.sha256 = hashlib.sha256(content).hexdigest()
    part.size = len(content)
    return part


def createRegistration(id, image=None):
    if image is None:
        image = createImage(id)
    registration = ImageRegistration(image)
    registration.eki = createImagePart(id, kind="kernel")
    registration.eri = createImagePart(id, kind="ramdisk")
    registration.emi = createImagePart(id, kind="image")
    return registration


class ImageTestCase(TestCase):

    def createImage(self, id, withFiles=False):
        image = createImage(id)
        if withFiles:
            kernel = self.makeFile("X" * 100, basename="kernel")
            ramdisk = self.makeFile("Y" * 100, basename="ramdisk")
            machineImage = self.makeFile("Z" * 10000, basename="image")

            image["files"] = [
                ImageFile({
                    "url": path2url(kernel),
                    "size-in-bytes": os.path.getsize(kernel),
                    "kind": "kernel",
                    "sha256": sha256(kernel)}),
                ImageFile({
                    "url": path2url(ramdisk),
                    "size-in-bytes": os.path.getsize(ramdisk),
                    "kind": "ramdisk",
                    "sha256": sha256(ramdisk)}),
                ImageFile({
                    "url": path2url(machineImage),
                    "size-in-bytes": os.path.getsize(machineImage),
                    "kind": "image",
                    "sha256": sha256(machineImage)})
                ]
        return image

    def createImageRegistration(self, id):
        image = self.createImage(id, withFiles=True)

        byKind = {}
        for imageFile in image["files"]:
            kind = imageFile["kind"]
            byKind[kind] = imageFile

        imageRegistration = ImageRegistration(image)
        imageRegistration.eki.sha256 = byKind["kernel"]["sha256"]
        imageRegistration.eki.size = byKind["kernel"]["size-in-bytes"]
        imageRegistration.eki.kind = "kernel"
        imageRegistration.eki.eid = "eki-%08x" % id

        imageRegistration.eri.sha256 = byKind["ramdisk"]["sha256"]
        imageRegistration.eri.size = byKind["ramdisk"]["size-in-bytes"]
        imageRegistration.eri.kind = "ramdisk"
        imageRegistration.eri.eid = "eri-%08x" % id

        imageRegistration.emi.sha256 = byKind["image"]["sha256"]
        imageRegistration.emi.size = byKind["image"]["size-in-bytes"]
        imageRegistration.emi.kind = "image"
        imageRegistration.emi.eid = "emi-%08x" % id

        return imageRegistration


class ServiceTestCase(ImageTestCase):

    def startStopService(self, service):
        service.start()
        deferred = service.stop()
        while not deferred.called:
            reactor.runUntilCurrent()

    def runServicesAndWaitForDeferred(self, services, deferred):
        for service in services:
            service.start()
        def callback(result):
            return mergeDeferreds(
                [maybeDeferred(service.stop) for service in services])
        def errback(failure):
            deferred = mergeDeferreds(
                [maybeDeferred(service.stop) for service in services])
            deferred.addCallback(lambda result: failure)
            return deferred
        deferred.addCallbacks(callback, errback)
        return deferred
