#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import logging
import os
import base64

from twisted.web.resource import Resource
from twisted.web import server
from twisted.internet import reactor
from twisted.internet.defer import succeed

from imagestore.model import ImageStoreResponse
from imagestore.lib.service import ServiceError
from imagestore.lib.signer import Signer


class Root(Resource):

    def __init__(self, presenter):
        Resource.__init__(self)
        self.putChild("api", APIResource(presenter))


class APIResource(Resource):

    def __init__(self, presenter):
        Resource.__init__(self)
        self.putChild("dashboard", DashboardResource(presenter))
        self.putChild("search", SearchResource(presenter))
        self.putChild("states", StatesResource(presenter))
        self.putChild("images", ImagesResource(presenter))


class DashboardResource(Resource):

    def __init__(self, presenter):
        Resource.__init__(self)
        self._presenter = presenter

    def render_GET(self, request):
        return renderDeferred(request, self._presenter.getDashboard())


class SearchResource(Resource):

    def __init__(self, presenter):
        Resource.__init__(self)
        self._presenter = presenter

    def render_GET(self, request):
        query = request.args.get("q")
        if query:
            query = query[0]
        return renderDeferred(request, self._presenter.search(query))


class StatesResource(Resource):

    def __init__(self, presenter):
        Resource.__init__(self)
        self._presenter = presenter

    def render_POST(self, request):
        imageUris = request.args.get("image-uri", [])
        deferred = self._presenter.getStates(imageUris)
        def buildResponse(states):
            return ImageStoreResponse({"states": states})
        deferred.addCallback(buildResponse)
        return renderDeferred(request, deferred)


class ImagesResource(Resource):

    isLeaf = False

    def __init__(self, presenter):
        Resource.__init__(self)
        self._presenter = presenter

    def getChild(self, name, request):
        imageUri = base64.urlsafe_b64decode(name)
        return ImageResource(self._presenter, imageUri)


class ImageResource(Resource):

    def __init__(self, presenter, imageUri):
        Resource.__init__(self)
        self._presenter = presenter
        self._imageUri = imageUri

    def getChild(self, name, request):
        if name == "install":
            return ImageInstallResource(self._presenter, self._imageUri)
        if name == "cancel":
            return ImageCancelResource(self._presenter, self._imageUri)
        if name == "clear-error":
            return ImageClearErrorResource(self._presenter, self._imageUri)

    def render_GET(self, request):
        return createErrorResponse("Individual image data not yet "
                                   "provided by the proxy API")


class SignatureProtectedResource(Resource):

    def __init__(self, presenter):
        Resource.__init__(self)
        self._presenter = presenter
        if hasattr(self, "renderSigned_POST"):
            self.render_POST = self.renderSigned
        #if hasattr(self, "renderSigned_GET"):
        #    self.render_GET = self.renderSigned

    def _sendError(self, request, hint=None):
        if not hint:
            message = "Bad request signature"
        else:
            message = "Bad request signature (%s)" % hint
        response = ImageStoreResponse({"error-message": message})
        request.write(str(response))
        request.finish()

    def _check(self, request, secretKey):
        method = request.method
        host = request.getHeader("Host")
        path = request.path
        params = dict(request.args)
        #print "sign(%s,%s,%s): %s" % (
        #    method, host, path,
        #    Signer(secretKey).sign(method, host, path, params,
        #                           expires=False, nonce=False).get("Signature"))
        return Signer(secretKey).check(method, host, path, params)

    def renderSigned(self, request):
        clientId = request.args.get("ClientId")
        if not clientId:
            logging.info("ClientId not provided in request for %s" %
                         (request.uri,))
            self._sendError(request, "ClientId not provided")
        else:
            def gotSecretKey(secretKey):
                logging.debug("Got secret key %r from presenter." % secretKey)
                if not secretKey:
                    self._sendError(request, "unknown ClientId")
                elif not self._check(request, secretKey):
                    logging.info("Refused bad signature on request for %s" %
                                 (request.uri,))
                    self._sendError(request)
                else:
                    logging.debug("Got valid signature on request for %s" %
                                  (request.uri,))
                    handler = getattr(self, "renderSigned_%s" % request.method)
                    handler(request)
            def gotFailure(failure):
                logging.error("Error getting credentials on request for %s: %s"
                              % (request.uri, failure.getBriefTraceback(),))
                result = createErrorResponse("Proxy failed to retrieve "
                                             "Eucalyptus credentials")
                request.write(result)
                request.finish()
            logging.debug("Asking presenter for secret key.")
            deferred = self._presenter.getSecretKey(clientId[0])
            deferred.addCallbacks(gotSecretKey, gotFailure)
        return server.NOT_DONE_YET


class ImageActionResource(SignatureProtectedResource):

    presenterMethod = None

    def __init__(self, presenter, imageUri):
        SignatureProtectedResource.__init__(self, presenter)
        self._imageUri = imageUri

    def renderSigned_POST(self, request):
        presenterMethod = getattr(self._presenter, self.presenterMethod)
        deferred = presenterMethod(self._imageUri)

        # Eat any errors.  Since we won't chain this deferred anywhere
        # else because we want it to run in background, we just forget
        # about errors here.  The presenter and its services should
        # have saved any important errors in the image state to let the
        # user know about them in a future moment.
        deferred.addErrback(lambda failure: None)

        request.args["image-uri"] = [self._imageUri]
        return StatesResource(self._presenter).render_POST(request)


class ImageInstallResource(ImageActionResource):
    presenterMethod = "startInstall"

class ImageCancelResource(ImageActionResource):
    presenterMethod = "cancelChanges"

class ImageClearErrorResource(ImageActionResource):
    presenterMethod = "clearError"


def renderDeferred(request, deferred):
    def callback(result):
        request.write(str(result))
        request.finish()
    def errback(failure):
        logging.error("Error handling request: %s" %
                      failure.getBriefTraceback())
        if failure.check(ServiceError):
            message = failure.value.message
        else:
            message = "An error occurred in the image store proxy. " \
                      "Please check the logs and report."
        request.write(createErrorResponse(message))
        request.finish()
    deferred.addCallbacks(callback, errback)
    return server.NOT_DONE_YET


def createErrorResponse(message):
    return str(ImageStoreResponse({"error-message": message}))
