#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#if defined(lint) && defined(HAVE_NOTE_H)
#include <note.h>
#endif /* lint && HAVE_NOTE_H */

#include <iiimp-data.h>
#include <iiimp-opcode.h>

#include "iiimp-dataP.h"


static int
iiimp_file_operation_open_flags_i2l(
    int	open_flags)
{
    int	f;

    f = 0;

    if (IIIMP_FILE_OPERATION_OPEN_O_RDONLY & open_flags) f |= O_RDONLY;
    if (IIIMP_FILE_OPERATION_OPEN_O_WRONLY & open_flags) f |= O_WRONLY;
    if (IIIMP_FILE_OPERATION_OPEN_O_RDWR & open_flags) f |= O_RDWR;
    if (IIIMP_FILE_OPERATION_OPEN_O_APPEND & open_flags) f |= O_APPEND;
    if (IIIMP_FILE_OPERATION_OPEN_O_CREAT & open_flags) f |= O_CREAT;
    if (IIIMP_FILE_OPERATION_OPEN_O_TRUNC & open_flags) f |= O_TRUNC;
    if (IIIMP_FILE_OPERATION_OPEN_O_EXCL & open_flags) f |= O_EXCL;

    return f;
}


static int
iiimp_file_operation_open_flags_l2i(
    int	open_flags)
{
    int	f;

    f = 0;

    if (O_RDONLY & open_flags) f |= IIIMP_FILE_OPERATION_OPEN_O_RDONLY;
    if (O_WRONLY & open_flags) f |= IIIMP_FILE_OPERATION_OPEN_O_WRONLY;
    if (O_RDWR & open_flags) f |= IIIMP_FILE_OPERATION_OPEN_O_RDWR;
    if (O_APPEND & open_flags) f |= IIIMP_FILE_OPERATION_OPEN_O_APPEND;
    if (O_CREAT & open_flags) f |= IIIMP_FILE_OPERATION_OPEN_O_CREAT;
    if (O_TRUNC & open_flags) f |= IIIMP_FILE_OPERATION_OPEN_O_TRUNC;
    if (O_EXCL & open_flags) f |= IIIMP_FILE_OPERATION_OPEN_O_EXCL;

    return f;
}


static int
iiimp_file_operation_fcntl_cmd_i2l(
    int	cmd)
{
    switch (cmd) {
    case IIIMP_FILE_OPERATION_FCNTL_F_SETFD:
	return F_SETFD;
    case IIIMP_FILE_OPERATION_FCNTL_F_SETFL:
	return F_SETFL;
    case IIIMP_FILE_OPERATION_FCNTL_F_GETLK:
	return F_GETLK;
    case IIIMP_FILE_OPERATION_FCNTL_F_SETLK:
	return F_SETLK;
    case IIIMP_FILE_OPERATION_FCNTL_F_SETLKW:
	return F_SETLKW;
    default:
	return -1;
    }
}


static int
iiimp_file_operation_fcntl_cmd_l2i(
    int	cmd)
{
    switch (cmd) {
    case F_SETFD:
	return IIIMP_FILE_OPERATION_FCNTL_F_SETFD;
    case F_SETFL:
	return IIIMP_FILE_OPERATION_FCNTL_F_SETFL;
    case F_GETLK:
	return IIIMP_FILE_OPERATION_FCNTL_F_GETLK;
    case F_SETLK:
	return IIIMP_FILE_OPERATION_FCNTL_F_SETLK;
    case F_SETLKW:
	return IIIMP_FILE_OPERATION_FCNTL_F_SETLKW;
    default:
	return -1;
    }
}


static int
iiimp_file_operation_fcntl_setfd_flag_i2l(
    int	flag)
{
    switch (flag) {
    case IIIMP_FILE_OPERATION_FCNTL_SETFD_FLAG_FD_CLOEXEC:
	return FD_CLOEXEC;
    default:
	return 0;
    }
}


static int
iiimp_file_operation_fcntl_setfd_flag_l2i(
    int	flag)
{
    switch (flag) {
    case FD_CLOEXEC:
	return IIIMP_FILE_OPERATION_FCNTL_SETFD_FLAG_FD_CLOEXEC;
    default:
	return 0;
    }
}


static int
iiimp_file_operation_fcntl_flock_type_i2l(
    int	type)
{
    switch (type) {
    case IIIMP_FILE_OPERATION_FCNTL_FLOCK_TYPE_F_RDLCK:
	return F_RDLCK;
    case IIIMP_FILE_OPERATION_FCNTL_FLOCK_TYPE_F_WRLCK:
	return F_WRLCK;
    case IIIMP_FILE_OPERATION_FCNTL_FLOCK_TYPE_F_UNLCK:
	return F_UNLCK;
    case IIIMP_FILE_OPERATION_FCNTL_FLOCK_TYPE_F_UNLKSYS:
	return F_UNLKSYS;
    default:
	return -1;
    }
}


static int
iiimp_file_operation_fcntl_flock_type_l2i(
    int	type)
{
    switch (type) {
    case F_RDLCK:
	return IIIMP_FILE_OPERATION_FCNTL_FLOCK_TYPE_F_RDLCK;
    case F_WRLCK:
	return IIIMP_FILE_OPERATION_FCNTL_FLOCK_TYPE_F_WRLCK;
    case F_UNLCK:
	return IIIMP_FILE_OPERATION_FCNTL_FLOCK_TYPE_F_UNLCK;
    case F_UNLKSYS:
	return IIIMP_FILE_OPERATION_FCNTL_FLOCK_TYPE_F_UNLKSYS;
    default:
	return -1;
    }
}


static int
iiimp_file_operation_whence_i2l(
    int	whence)
{
    switch (whence) {
    case IIIMP_FILE_OPERATION_WHENCE_SEEK_SET:
	return SEEK_SET;
    case IIIMP_FILE_OPERATION_WHENCE_SEEK_CUR:
	return SEEK_CUR;
    case IIIMP_FILE_OPERATION_WHENCE_SEEK_END:
	return SEEK_END;
    default:
	return -1;
    }
}


static int
iiimp_file_operation_whence_l2i(
    int	whence)
{
    switch (whence) {
    case SEEK_SET:
	return IIIMP_FILE_OPERATION_WHENCE_SEEK_SET;
    case SEEK_CUR:
	return IIIMP_FILE_OPERATION_WHENCE_SEEK_CUR;
    case SEEK_END:
	return IIIMP_FILE_OPERATION_WHENCE_SEEK_END;
    default:
	return -1;
    }
}


static int
iiimp_file_operation_access_amode_i2l(
    int	amode)
{
    switch (amode) {
    case IIIMP_FILE_OPERATION_ACCESS_F_OK:
	return F_OK;
    case IIIMP_FILE_OPERATION_ACCESS_X_OK:
	return X_OK;
    case IIIMP_FILE_OPERATION_ACCESS_W_OK:
	return W_OK;
    case IIIMP_FILE_OPERATION_ACCESS_R_OK:
	return R_OK;
    default:
	return -1;
    }
}


static int
iiimp_file_operation_access_amode_l2i(
    int	amode)
{
    switch (amode) {
    case F_OK:
	return IIIMP_FILE_OPERATION_ACCESS_F_OK;
    case X_OK:
	return IIIMP_FILE_OPERATION_ACCESS_X_OK;
    case W_OK:
	return IIIMP_FILE_OPERATION_ACCESS_W_OK;
    case R_OK:
	return IIIMP_FILE_OPERATION_ACCESS_R_OK;
    default:
	return -1;
    }
}


static int
iiimp_file_operation_error_number_i2l(
    int	error_number)
{
    switch (error_number) {
    case IIIMP_FILE_OPERATION_ERRNO_NONE:
	return 0;
#if defined(EPERM)
    case IIIMP_FILE_OPERATION_ERRNO_EPERM:
	return EPERM;
#endif /* EPERM */
#if defined(ENOENT)
    case IIIMP_FILE_OPERATION_ERRNO_ENOENT:
	return ENOENT;
#endif /* ENOENT */
#if defined(EINTR)
    case IIIMP_FILE_OPERATION_ERRNO_EINTR:
	return EINTR;
#endif /* EINTR */
#if defined(EIO)
    case IIIMP_FILE_OPERATION_ERRNO_EIO:
	return EIO;
#endif /* EIO */
#if defined(ENXIO)
    case IIIMP_FILE_OPERATION_ERRNO_ENXIO:
	return ENXIO;
#endif /* ENXIO */
#if defined(EBADF)
    case IIIMP_FILE_OPERATION_ERRNO_EBADF:
	return EBADF;
#endif /* EBADF */
#if defined(EAGAIN)
    case IIIMP_FILE_OPERATION_ERRNO_EAGAIN:
	return EAGAIN;
#endif /* EAGAIN */
#if defined(ENOMEM)
    case IIIMP_FILE_OPERATION_ERRNO_ENOMEM:
	return ENOMEM;
#endif /* ENOMEM */
#if defined(EACCES)
    case IIIMP_FILE_OPERATION_ERRNO_EACCES:
	return EACCES;
#endif /* EACCES */
#if defined(EFAULT)
    case IIIMP_FILE_OPERATION_ERRNO_EFAULT:
	return EFAULT;
#endif /* EFAULT */
#if defined(EBUSY)
    case IIIMP_FILE_OPERATION_ERRNO_EBUSY:
	return EBUSY;
#endif /* EBUSY */
#if defined(EEXIST)
    case IIIMP_FILE_OPERATION_ERRNO_EEXIST:
	return EEXIST;
#endif /* EEXIST */
#if defined(EXDEV)
    case IIIMP_FILE_OPERATION_ERRNO_EXDEV:
	return EXDEV;
#endif /* EXDEV */
#if defined(ENOTDIR)
    case IIIMP_FILE_OPERATION_ERRNO_ENOTDIR:
	return ENOTDIR;
#endif /* ENOTDIR */
#if defined(EISDIR)
    case IIIMP_FILE_OPERATION_ERRNO_EISDIR:
	return EISDIR;
#endif /* EISDIR */
    case IIIMP_FILE_OPERATION_ERRNO_EINVAL:
	return EINVAL;
#if defined(ENFILE)
    case IIIMP_FILE_OPERATION_ERRNO_ENFILE:
	return ENFILE;
#endif /* ENFILE */
#if defined(EMFILE)
    case IIIMP_FILE_OPERATION_ERRNO_EMFILE:
	return EMFILE;
#endif /* EMFILE */
#if defined(ETXTBSY)
    case IIIMP_FILE_OPERATION_ERRNO_ETXTBSY:
	return ETXTBSY;
#endif /* ETXTBSY */
#if defined(EFBIG)
    case IIIMP_FILE_OPERATION_ERRNO_EFBIG:
	return EFBIG;
#endif /* EFBIG */
#if defined(ENOSPC)
    case IIIMP_FILE_OPERATION_ERRNO_ENOSPC:
	return ENOSPC;
#endif /* ENOSPC */
#if defined(EROFS)
    case IIIMP_FILE_OPERATION_ERRNO_EROFS:
	return EROFS;
#endif /* EROFS */
#if defined(EMLINK)
    case IIIMP_FILE_OPERATION_ERRNO_EMLINK:
	return EMLINK;
#endif /* EMLINK */
#if defined(EDEADLK)
    case IIIMP_FILE_OPERATION_ERRNO_EDEADLK:
	return EDEADLK;
#endif /* EDEADLK */
#if defined(ENOLCK)
    case IIIMP_FILE_OPERATION_ERRNO_ENOLCK:
	return ENOLCK;
#endif /* ENOLCK */
#if defined(EDQUOT)
    case IIIMP_FILE_OPERATION_ERRNO_EDQUOT:
	return EDQUOT;
#endif /* EDQUOT */
#if defined(ENAMETOOLONG)
    case IIIMP_FILE_OPERATION_ERRNO_ENAMETOOLONG:
	return ENAMETOOLONG;
#endif /* ENAMETOOLONG */
#if defined(EOVERFLOW)
    case IIIMP_FILE_OPERATION_ERRNO_EOVERFLOW:
	return EOVERFLOW;
#endif /* EOVERFLOW */
#if defined(ENOSYS)
    case IIIMP_FILE_OPERATION_ERRNO_ENOSYS:
	return ENOSYS;
#endif /* ENOSYS */
#if defined(ELOOP)
    case IIIMP_FILE_OPERATION_ERRNO_ELOOP:
	return ELOOP;
#endif /* ELOOP */
    default:
	return EINVAL;
    }
}


static int
iiimp_file_operation_error_number_l2i(
    int	error_number)
{
    switch (error_number) {
    case 0:
	return IIIMP_FILE_OPERATION_ERRNO_NONE;
#if defined(EPERM)
    case EPERM:
	return IIIMP_FILE_OPERATION_ERRNO_EPERM;
#endif /* EPERM */
#if defined(ENOENT)
    case ENOENT:
	return IIIMP_FILE_OPERATION_ERRNO_ENOENT;
#endif /* ENOENT */
#if defined(EINTR)
    case EINTR:
	return IIIMP_FILE_OPERATION_ERRNO_EINTR;
#endif /* EINTR */
#if defined(EIO)
    case EIO:
	return IIIMP_FILE_OPERATION_ERRNO_EIO;
#endif /* EIO */
#if defined(ENXIO)
    case ENXIO:
	return IIIMP_FILE_OPERATION_ERRNO_ENXIO;
#endif /* ENXIO */
#if defined(EBADF)
    case EBADF:
	return IIIMP_FILE_OPERATION_ERRNO_EBADF;
#endif /* EBADF */
#if defined(EAGAIN)
    case EAGAIN:
	return IIIMP_FILE_OPERATION_ERRNO_EAGAIN;
#endif /* EAGAIN */
#if defined(ENOMEM)
    case ENOMEM:
	return IIIMP_FILE_OPERATION_ERRNO_ENOMEM;
#endif /* ENOMEM */
#if defined(EACCES)
    case EACCES:
	return IIIMP_FILE_OPERATION_ERRNO_EACCES;
#endif /* EACCES */
#if defined(EFAULT)
    case EFAULT:
	return IIIMP_FILE_OPERATION_ERRNO_EFAULT;
#endif /* EFAULT */
#if defined(EBUSY)
    case EBUSY:
	return IIIMP_FILE_OPERATION_ERRNO_EBUSY;
#endif /* EBUSY */
#if defined(EEXIST)
    case EEXIST:
	return IIIMP_FILE_OPERATION_ERRNO_EEXIST;
#endif /* EEXIST */
#if defined(EXDEV)
    case EXDEV:
	return IIIMP_FILE_OPERATION_ERRNO_EXDEV;
#endif /* EXDEV */
#if defined(ENOTDIR)
    case ENOTDIR:
	return IIIMP_FILE_OPERATION_ERRNO_ENOTDIR;
#endif /* ENOTDIR */
#if defined(EISDIR)
    case EISDIR:
	return IIIMP_FILE_OPERATION_ERRNO_EISDIR;
#endif /* EISDIR */
    case EINVAL:
	return IIIMP_FILE_OPERATION_ERRNO_EINVAL;
#if defined(ENFILE)
    case ENFILE:
	return IIIMP_FILE_OPERATION_ERRNO_ENFILE;
#endif /* ENFILE */
#if defined(EMFILE)
    case EMFILE:
	return IIIMP_FILE_OPERATION_ERRNO_EMFILE;
#endif /* EMFILE */
#if defined(ETXTBSY)
    case ETXTBSY:
	return IIIMP_FILE_OPERATION_ERRNO_ETXTBSY;
#endif /* ETXTBSY */
#if defined(EFBIG)
    case EFBIG:
	return IIIMP_FILE_OPERATION_ERRNO_EFBIG;
#endif /* EFBIG */
#if defined(ENOSPC)
    case ENOSPC:
	return IIIMP_FILE_OPERATION_ERRNO_ENOSPC;
#endif /* ENOSPC */
#if defined(EROFS)
    case EROFS:
	return IIIMP_FILE_OPERATION_ERRNO_EROFS;
#endif /* EROFS */
#if defined(EMLINK)
    case EMLINK:
	return IIIMP_FILE_OPERATION_ERRNO_EMLINK;
#endif /* EMLINK */
#if defined(EDEADLK)
    case EDEADLK:
	return IIIMP_FILE_OPERATION_ERRNO_EDEADLK;
#endif /* EDEADLK */
#if defined(ENOLCK)
    case ENOLCK:
	return IIIMP_FILE_OPERATION_ERRNO_ENOLCK;
#endif /* ENOLCK */
#if defined(EDQUOT)
    case EDQUOT:
	return IIIMP_FILE_OPERATION_ERRNO_EDQUOT;
#endif /* EDQUOT */
#if defined(ENAMETOOLONG)
    case ENAMETOOLONG:
	return IIIMP_FILE_OPERATION_ERRNO_ENAMETOOLONG;
#endif /* ENAMETOOLONG */
#if defined(EOVERFLOW)
    case EOVERFLOW:
	return IIIMP_FILE_OPERATION_ERRNO_EOVERFLOW;
#endif /* EOVERFLOW */
#if defined(ENOSYS)
    case ENOSYS:
	return IIIMP_FILE_OPERATION_ERRNO_ENOSYS;
#endif /* ENOSYS */
#if defined(ELOOP)
    case ELOOP:
	return IIIMP_FILE_OPERATION_ERRNO_ELOOP;
#endif /* ELOOP */
    default:
	return IIIMP_FILE_OPERATION_ERRNO_EINVAL;
    }
}


IIIMP_file_operation_data_fcntl *
iiimp_file_operation_data_fcntl_setfd_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	file_descriptor_flags)
{
    IIIMP_file_operation_data_fcntl *	data;
    size_t				nbyte;

    data = (IIIMP_file_operation_data_fcntl *)
	malloc(sizeof(IIIMP_file_operation_data_fcntl));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    nbyte = 4;

    data->nbyte = nbyte;
    data->value.file_descriptor_flags = file_descriptor_flags;

    return data;
}


IIIMP_file_operation_data_fcntl *
iiimp_file_operation_data_fcntl_setfl_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	file_status_flags)
{
    IIIMP_file_operation_data_fcntl *	data;
    size_t				nbyte;

    data = (IIIMP_file_operation_data_fcntl *)
	malloc(sizeof(IIIMP_file_operation_data_fcntl));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    nbyte = 4;

    data->nbyte = nbyte;
    data->value.file_status_flags = file_status_flags;

    return data;
}


IIIMP_file_operation_data_fcntl *
iiimp_file_operation_data_fcntl_flock_new(
    IIIMP_data_s *	data_s,
    IIIMP_card16	type,
    IIIMP_card16	whence,
    IIIMP_card32	start,
    IIIMP_card32	len)
{
    IIIMP_file_operation_data_fcntl *	data;
    size_t				nbyte;

    data = (IIIMP_file_operation_data_fcntl *)
	malloc(sizeof(IIIMP_file_operation_data_fcntl));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    nbyte = (2 + 2 + 4 + 4);

    data->nbyte = nbyte;
    data->value.flock.type = type;
    data->value.flock.whence = whence;
    data->value.flock.start = start;
    data->value.flock.len = len;

    return data;
}


IIIMP_file_operation_data_stat *
iiimp_file_operation_data_stat_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	mode,
    IIIMP_card32	ino,
    IIIMP_card32	dev,
    IIIMP_card32	rdev,
    IIIMP_card32	nlink,
    IIIMP_string *	user_name,
    IIIMP_card32	uid,
    IIIMP_string *	group_name,
    IIIMP_card32	gid,
    IIIMP_card32	size,
    IIIMP_card32	atime_tv_sec,
    IIIMP_card32	atime_tv_nsec,
    IIIMP_card32	mtime_tv_sec,
    IIIMP_card32	mtime_tv_nsec,
    IIIMP_card32	ctime_tv_sec,
    IIIMP_card32	ctime_tv_nsec,
    IIIMP_card32	blksize,
    IIIMP_card32	blocks)
{
    IIIMP_file_operation_data_stat *	data;
    size_t				nbyte;

    data = (IIIMP_file_operation_data_stat *)
	malloc(sizeof(IIIMP_file_operation_data_stat));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    nbyte = (4 + 4 + 4 + 4 + 4);	/* mode + ino + dev + rdev + nlink */
    if (NULL == user_name) {
	nbyte += 4;
    } else {
	nbyte += user_name->nbyte;
    }
    nbyte += 4;				/* uid */
    if (NULL == group_name) {
	nbyte += 4;
    } else {
	nbyte += group_name->nbyte;
    }
    nbyte += 4;				/* gid */
    nbyte += 4;				/* size */
    nbyte += (4 + 4);			/* atime */
    nbyte += (4 + 4);			/* mtime */
    nbyte += (4 + 4);			/* ctime */
    nbyte += (4 + 4);			/* blksize + blocks */

    data->nbyte = nbyte;
    data->mode = mode;
    data->ino = ino;
    data->dev = dev;
    data->rdev = rdev;
    data->nlink = nlink;
    data->user_name = user_name;
    data->uid = uid;
    data->group_name = group_name;
    data->gid = gid;
    data->size = size;
    data->atime.tv_sec = atime_tv_sec;
    data->atime.tv_nsec = atime_tv_nsec;
    data->mtime.tv_sec = mtime_tv_sec;
    data->mtime.tv_nsec = mtime_tv_nsec;
    data->ctime.tv_sec = ctime_tv_sec;
    data->ctime.tv_nsec = ctime_tv_nsec;
    data->blksize = blksize;
    data->blocks = blocks;

    return data;
}


IIIMP_file_operation_data_dir *
iiimp_file_operation_data_dir_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	ino,
    IIIMP_card32	off,
    IIIMP_string *	name)
{
    IIIMP_file_operation_data_dir *	data;
    size_t				nbyte;

    data = (IIIMP_file_operation_data_dir *)
	malloc(sizeof(IIIMP_file_operation_data_dir));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    nbyte = (4 + 4);
    if (NULL == name) {
	nbyte += 4;
    } else {
	nbyte += name->nbyte;
    }

    data->nbyte = nbyte;
    data->ino = ino;
    data->off = off;
    data->name = name;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_close_new(
    IIIMP_data_s *		data_s,
    IIIMP_card32		file_descriptor)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = 4;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_CLOSE;
    data->value_nbyte = value_nbyte;
    data->value.close.file_descriptor = file_descriptor;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_creat_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path,
    IIIMP_card32	mode)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (path->nbyte + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_CREAT;
    data->value_nbyte = value_nbyte;
    data->value.creat.path = path;
    data->value.creat.mode = mode;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_ftruncate_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	file_descriptor,
    IIIMP_card32	length)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_FTRUNCATE;
    data->value_nbyte = value_nbyte;
    data->value.ftruncate.file_descriptor = file_descriptor;
    data->value.ftruncate.length = length;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_lseek_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	file_descriptor,
    IIIMP_card32	offset,
    IIIMP_card32	whence)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4 + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_LSEEK;
    data->value_nbyte = value_nbyte;
    data->value.lseek.file_descriptor = file_descriptor;
    data->value.lseek.offset = offset;
    data->value.lseek.whence = whence;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_open_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path,
    IIIMP_card32	oflag,
    IIIMP_card32	mode)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (path->nbyte + 4 + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_OPEN;
    data->value_nbyte = value_nbyte;
    data->value.open.path = path;
    data->value.open.oflag = oflag;
    data->value.open.mode = mode;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_read_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	file_descriptor,
    IIIMP_card32	length)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_READ;
    data->value_nbyte = value_nbyte;
    data->value.read.file_descriptor = file_descriptor;
    data->value.read.nbyte = length;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_truncate_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path,
    IIIMP_card32	length)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (path->nbyte + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_TRUNCATE;
    data->value_nbyte = value_nbyte;
    data->value.truncate.path = path;
    data->value.truncate.length = length;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_write_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	file_descriptor,
    size_t		buf_nbyte,
    const uchar_t *	buf)
{
    IIIMP_file_operation_request *	data;
    uchar_t *				value;
    size_t				nbyte;
    size_t				value_nbyte;

    if ((0 < buf_nbyte) && (NULL == buf)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    if (0 < buf_nbyte) {
	value = (uchar_t *)malloc(buf_nbyte);
	if (NULL == value) {
	    free(data);
	    data_s->status = IIIMP_DATA_MALLOC_ERROR;
	    return NULL;
	}
	(void)memcpy(value, buf, buf_nbyte);
    }

    value_nbyte = 4;
    if (0 < buf_nbyte) {
	value_nbyte += (4 + buf_nbyte + PAD(buf_nbyte));
    } else {
	value_nbyte += 4;
    }
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_WRITE;
    data->value_nbyte = value_nbyte;
    data->value.write.nbyte = value_nbyte;
    data->value.write.file_descriptor = file_descriptor;
    data->value.write.value.nbyte = buf_nbyte;
    data->value.write.value.ptr = value;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_link_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	existing_path,
    IIIMP_string *	new_path)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if ((NULL == existing_path) || (NULL == new_path)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (existing_path->nbyte + new_path->nbyte);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_LINK;
    data->value_nbyte = value_nbyte;
    data->value.link.existing_path = existing_path;
    data->value.link.new_path = new_path;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_readlink_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = path->nbyte;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_READLINK;
    data->value_nbyte = value_nbyte;
    data->value.readlink.path = path;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_rename_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	old_path,
    IIIMP_string *	new_path)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if ((NULL == old_path) || (NULL == new_path)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (old_path->nbyte + new_path->nbyte);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_RENAME;
    data->value_nbyte = value_nbyte;
    data->value.rename.old_path = old_path;
    data->value.rename.new_path = new_path;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_symlink_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	existing_path,
    IIIMP_string *	new_path)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if ((NULL == existing_path) || (NULL == new_path)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (existing_path->nbyte + new_path->nbyte);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_SYMLINK;
    data->value_nbyte = value_nbyte;
    data->value.symlink.existing_path = existing_path;
    data->value.symlink.new_path = new_path;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_unlink_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = path->nbyte;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_UNLINK;
    data->value_nbyte = value_nbyte;
    data->value.unlink.path = path;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_chmod_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path,
    IIIMP_card32	mode)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (path->nbyte + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_CHMOD;
    data->value_nbyte = value_nbyte;
    data->value.chmod.path = path;
    data->value.chmod.mode = mode;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_chown_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path,
    IIIMP_string *	user_name,
    IIIMP_card32	user_id,
    IIIMP_string *	group_name,
    IIIMP_card32	group_id)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = path->nbyte;
    if (NULL == user_name) {
	value_nbyte += 4;
    } else {
	value_nbyte += user_name->nbyte;
    }
    value_nbyte += 4;
    if (NULL == group_name) {
	value_nbyte += 4;
    } else {
	value_nbyte += group_name->nbyte;
    }
    value_nbyte += 4;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_CHOWN;
    data->value_nbyte = value_nbyte;
    data->value.chown.path = path;
    data->value.chown.user_name = user_name;
    data->value.chown.user_id = user_id;
    data->value.chown.group_name = group_name;
    data->value.chown.group_id = group_id;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_fchmod_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	file_descriptor,
    IIIMP_card32	mode)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_FCHMOD;
    data->value_nbyte = value_nbyte;
    data->value.fchmod.file_descriptor = file_descriptor;
    data->value.fchmod.mode = mode;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_fchown_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	file_descriptor,
    IIIMP_string *	user_name,
    IIIMP_card32	user_id,
    IIIMP_string *	group_name,
    IIIMP_card32	group_id)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = 4;
    if (NULL == user_name) {
	value_nbyte += 4;
    } else {
	value_nbyte += user_name->nbyte;
    }
    value_nbyte += 4;
    if (NULL == group_name) {
	value_nbyte += 4;
    } else {
	value_nbyte += group_name->nbyte;
    }
    value_nbyte += 4;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_FCHOWN;
    data->value_nbyte = value_nbyte;
    data->value.fchown.file_descriptor = file_descriptor;
    data->value.fchown.user_name = user_name;
    data->value.fchown.user_id = user_id;
    data->value.fchown.group_name = group_name;
    data->value.fchown.group_id = group_id;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_lchown_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path,
    IIIMP_string *	user_name,
    IIIMP_card32	user_id,
    IIIMP_string *	group_name,
    IIIMP_card32	group_id)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = path->nbyte;
    if (NULL == user_name) {
	value_nbyte += 4;
    } else {
	value_nbyte += user_name->nbyte;
    }
    value_nbyte += 4;
    if (NULL == group_name) {
	value_nbyte += 4;
    } else {
	value_nbyte += group_name->nbyte;
    }
    value_nbyte += 4;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_LCHOWN;
    data->value_nbyte = value_nbyte;
    data->value.lchown.path = path;
    data->value.lchown.user_name = user_name;
    data->value.lchown.user_id = user_id;
    data->value.lchown.group_name = group_name;
    data->value.lchown.group_id = group_id;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_utime_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path,
    IIIMP_card32	actime,
    IIIMP_card32	modtime)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (path->nbyte + 4 + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_UTIME;
    data->value_nbyte = value_nbyte;
    data->value.utime.path = path;
    data->value.utime.utimbuf_data.actime = actime;
    data->value.utime.utimbuf_data.modtime = modtime;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_utimes_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path,
    IIIMP_card32	tv_sec0,
    IIIMP_card32	tv_usec0,
    IIIMP_card32	tv_sec1,
    IIIMP_card32	tv_usec1)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (path->nbyte + 4 + 4 + 4 + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_UTIMES;
    data->value_nbyte = value_nbyte;
    data->value.utimes.path = path;
    data->value.utimes.timeval_data.tv_sec0 = tv_sec0;
    data->value.utimes.timeval_data.tv_usec0 = tv_usec0;
    data->value.utimes.timeval_data.tv_sec1 = tv_sec1;
    data->value.utimes.timeval_data.tv_usec1 = tv_usec1;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_fstat_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	file_descriptor)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = 4;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_FSTAT;
    data->value_nbyte = value_nbyte;
    data->value.fstat.file_descriptor = file_descriptor;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_lstat_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = path->nbyte;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_LSTAT;
    data->value_nbyte = value_nbyte;
    data->value.lstat.path = path;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_stat_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = path->nbyte;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_STAT;
    data->value_nbyte = value_nbyte;
    data->value.stat.path = path;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_access_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	path,
    IIIMP_card32	amode)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == path) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (path->nbyte + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_ACCESS;
    data->value_nbyte = value_nbyte;
    data->value.access.path = path;
    data->value.access.amode = amode;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_closedir_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	directory_handle)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = 4;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_CLOSEDIR;
    data->value_nbyte = value_nbyte;
    data->value.closedir.directory_handle = directory_handle;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_opendir_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	dirname)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == dirname) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = dirname->nbyte;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_OPENDIR;
    data->value_nbyte = value_nbyte;
    data->value.opendir.dirname = dirname;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_readdir_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	directory_handle)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = 4;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_READDIR;
    data->value_nbyte = value_nbyte;
    data->value.readdir.directory_handle = directory_handle;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_rewinddir_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	directory_handle)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = 4;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_REWINDDIR;
    data->value_nbyte = value_nbyte;
    data->value.rewinddir.directory_handle = directory_handle;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_seekdir_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	directory_handle,
    IIIMP_card32	loc)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_SEEKDIR;
    data->value_nbyte = value_nbyte;
    data->value.seekdir.directory_handle = directory_handle;
    data->value.seekdir.loc = loc;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_telldir_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	directory_handle)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = 4;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_TELLDIR;
    data->value_nbyte = value_nbyte;
    data->value.telldir.directory_handle = directory_handle;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_mkdir_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	dirname,
    IIIMP_card32	mode)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == dirname) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (dirname->nbyte + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_MKDIR;
    data->value_nbyte = value_nbyte;
    data->value.mkdir.dirname = dirname;
    data->value.mkdir.mode = mode;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_rmdir_new(
    IIIMP_data_s *	data_s,
    IIIMP_string *	dirname)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == dirname) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = dirname->nbyte;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_RMDIR;
    data->value_nbyte = value_nbyte;
    data->value.rmdir.dirname = dirname;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_fcntl_new(
    IIIMP_data_s *			data_s,
    IIIMP_card32			file_descriptor,
    IIIMP_card32			cmd,
    IIIMP_file_operation_data_fcntl *	arg)
{
    IIIMP_file_operation_request *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    if (NULL == arg) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_request *)
	malloc(sizeof(IIIMP_file_operation_request));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4 + arg->nbyte);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_FCNTL;
    data->value_nbyte = value_nbyte;
    data->value.fcntl.file_descriptor = file_descriptor;
    data->value.fcntl.cmd = cmd;
    data->value.fcntl.arg = arg;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_result *
iiimp_file_operation_result_simple_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	type,
    IIIMP_card32	return_value,
    IIIMP_card32	error_number)
{
    IIIMP_file_operation_result *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_result *)
	malloc(sizeof(IIIMP_file_operation_result));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4);
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = type;
    data->value_nbyte = value_nbyte;
    data->value.simple.return_value = return_value;
    data->value.simple.error_number = error_number;
    data->next = NULL;

    return data;
}


IIIMP_file_operation_result *
iiimp_file_operation_result_void_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	type
)
{
    IIIMP_file_operation_result *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_result *)
	malloc(sizeof(IIIMP_file_operation_result));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = 0;
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = type;
    data->value_nbyte = value_nbyte;
    data->next = NULL;

    return data;
}


/*
 * iiimp_file_operation_result_close_new()
 */

/*
 * iiimp_file_operation_result_creat_new()
 */

/*
 * iiimp_file_operation_result_ftruncate_new()
 */

/*
 * iiimp_file_operation_result_lseek_new()
 */

/*
 * iiimp_file_operation_result_open_new()
 */


IIIMP_file_operation_result *
iiimp_file_operation_result_read_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	error_number,
    size_t		buf_nbyte,
    const uchar_t *	buf)
{
    IIIMP_file_operation_result *	data;
    uchar_t *				value;
    size_t				nbyte;
    size_t				value_nbyte;

    if ((0 < buf_nbyte) && (NULL == buf)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    data = (IIIMP_file_operation_result *)
	malloc(sizeof(IIIMP_file_operation_result));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    if (0 < buf_nbyte) {
	value = (uchar_t *)malloc(buf_nbyte);
	if (NULL == value) {
	    free(data);
	    data_s->status = IIIMP_DATA_MALLOC_ERROR;
	    return NULL;
	}
	(void)memcpy(value, buf, buf_nbyte);
    }

    value_nbyte = 4;
    if (0 < buf_nbyte) {
	value_nbyte += (4 + buf_nbyte + PAD(buf_nbyte));
    } else {
	value_nbyte += 4;
    }
    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_READ;
    data->value_nbyte = value_nbyte;
    data->value.read.error_number = error_number;
    data->value.read.value.nbyte = buf_nbyte;
    data->value.read.value.ptr = value;
    data->next = NULL;

    return data;
}


/*
 * iiimp_file_operation_result_truncate_new()
 */

/*
 * iiimp_file_operation_result_write_new()
 */

/*
 * iiimp_file_operation_result_link_new()
 */


IIIMP_file_operation_result *
iiimp_file_operation_result_readlink_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32	return_value,
    IIIMP_card32	error_number,
    IIIMP_string *	path)
{
    IIIMP_file_operation_result *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_result *)
	malloc(sizeof(IIIMP_file_operation_result));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4);
    if (NULL == path) {
	value_nbyte += 4;
    } else {
	value_nbyte += path->nbyte;
    }

    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_READLINK;
    data->value_nbyte = value_nbyte;
    data->value.readlink.return_value = return_value;
    data->value.readlink.error_number = error_number;
    data->value.readlink.path = path;
    data->next = NULL;

    return data;
}


/*
 * iiimp_file_operation_result_rename_new()
 */

/*
 * iiimp_file_operation_result_symlink_new()
 */

/*
 * iiimp_file_operation_result_unlink_new()
 */

/*
 * iiimp_file_operation_result_chmod_new()
 */

/*
 * iiimp_file_operation_result_chown_new()
 */

/*
 * iiimp_file_operation_result_fchmod_new()
 */

/*
 * iiimp_file_operation_result_fchown_new()
 */

/*
 * iiimp_file_operation_result_lchown_new()
 */

/*
 * iiimp_file_operation_result_utime_new()
 */

/*
 * iiimp_file_operation_result_utimes_new()
 */


IIIMP_file_operation_result *
iiimp_file_operation_result_stat_common_new(
    IIIMP_data_s *			data_s,
    IIIMP_card32			type,
    IIIMP_card32			return_value,
    IIIMP_card32			error_number,
    IIIMP_file_operation_data_stat *	stat)
{
    IIIMP_file_operation_result *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_result *)
	malloc(sizeof(IIIMP_file_operation_result));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4);
    if (NULL == stat) {
	value_nbyte += 4;
    } else {
	value_nbyte += stat->nbyte;
    }

    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = type;
    data->value_nbyte = value_nbyte;
    data->value.stat.return_value = return_value;
    data->value.stat.error_number = error_number;
    data->value.stat.stat = stat;
    data->next = NULL;

    return data;
}

/*
 * iiimp_file_operation_result_fstat_new()
 */

/*
 * iiimp_file_operation_result_lstat_new()
 */

/*
 * iiimp_file_operation_result_stat_new()
 */

/*
 * iiimp_file_operation_result_access_new()
 */

/*
 * iiimp_file_operation_result_closedir_new()
 */

/*
 * iiimp_file_operation_result_opendir_new()
 */


IIIMP_file_operation_result *
iiimp_file_operation_result_readdir_new(
    IIIMP_data_s *			data_s,
    IIIMP_card32			error_number,
    IIIMP_file_operation_data_dir *	dir)
{
    IIIMP_file_operation_result *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_result *)
	malloc(sizeof(IIIMP_file_operation_result));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = 4;
    if (NULL == dir) {
	value_nbyte += 4;
    } else {
	value_nbyte += dir->nbyte;
    }

    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_READDIR;
    data->value_nbyte = value_nbyte;
    data->value.readdir.error_number = error_number;
    data->value.readdir.dir = dir;
    data->next = NULL;

    return data;
}


/*
 * iiimp_file_operation_result_rewinddir_new()
 */

/*
 * iiimp_file_operation_result_seekdir_new()
 */

/*
 * iiimp_file_operation_result_telldir_new()
 */

/*
 * iiimp_file_operation_result_mkdir_new()
 */

/*
 * iiimp_file_operation_result_rmdir_new()
 */


IIIMP_file_operation_result *
iiimp_file_operation_result_fcntl_new(
    IIIMP_data_s *			data_s,
    IIIMP_card32			return_value,
    IIIMP_card32			error_number,
    IIIMP_card32			cmd,
    IIIMP_file_operation_data_fcntl *	arg)
{
    IIIMP_file_operation_result *	data;
    size_t				nbyte;
    size_t				value_nbyte;

    data = (IIIMP_file_operation_result *)
	malloc(sizeof(IIIMP_file_operation_result));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    value_nbyte = (4 + 4 + 4);
    if (NULL == arg) {
	value_nbyte += 4;
    } else {
	value_nbyte += arg->nbyte;
    }

    nbyte = (4 + 4 + value_nbyte);

    data->nbyte = nbyte;
    data->type = IIIMP_FILE_OPERATION_FCNTL;
    data->value_nbyte = value_nbyte;
    data->value.fcntl.return_value = return_value;
    data->value.fcntl.error_number = error_number;
    data->value.fcntl.cmd = cmd;
    data->value.fcntl.arg = arg;
    data->next = NULL;

    return data;
}


static void
iiimp_file_operation_data_fcntl_setfd_delete(
    IIIMP_data_s *		data_s,
    IIIMP_file_operation_data_fcntl *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_data_fcntl_setfl_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_fcntl *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_data_fcntl_flock_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_fcntl *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_data_fcntl_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_fcntl *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_data_stat_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_stat *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    iiimp_string_delete(data_s, data->user_name);
    iiimp_string_delete(data_s, data->group_name);
    free(data);
    return;
}


static void
iiimp_file_operation_data_dir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_dir *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    iiimp_string_delete(data_s, data->name);
    free(data);
    return;
}


static void
iiimp_file_operation_request_close_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_creat_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.creat.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_ftruncate_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_lseek_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_open_delete(
    IIIMP_data_s *	data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.open.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_read_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_truncate_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.truncate.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_write_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data->value.write.value.ptr);
    free(data);
    return;
}


static void
iiimp_file_operation_request_link_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    iiimp_string_delete(data_s, data->value.link.existing_path);
    iiimp_string_delete(data_s, data->value.link.new_path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_readlink_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.readlink.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_rename_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.rename.old_path);
    iiimp_string_delete(data_s, data->value.rename.new_path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_symlink_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.symlink.existing_path);
    iiimp_string_delete(data_s, data->value.symlink.new_path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_unlink_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.unlink.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_chmod_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.chmod.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_chown_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.chown.path);
    iiimp_string_delete(data_s, data->value.chown.user_name);
    iiimp_string_delete(data_s, data->value.chown.group_name);
    free(data);
    return;
}


static void
iiimp_file_operation_request_fchmod_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_fchown_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.fchown.user_name);
    iiimp_string_delete(data_s, data->value.fchown.group_name);
    free(data);
    return;
}


static void
iiimp_file_operation_request_lchown_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.lchown.path);
    iiimp_string_delete(data_s, data->value.lchown.user_name);
    iiimp_string_delete(data_s, data->value.lchown.group_name);
    free(data);
    return;
}


static void
iiimp_file_operation_request_utime_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    iiimp_string_delete(data_s, data->value.utime.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_utimes_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    iiimp_string_delete(data_s, data->value.utimes.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_fstat_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_lstat_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.lstat.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_stat_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.stat.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_access_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.access.path);
    free(data);
    return;
}


static void
iiimp_file_operation_request_closedir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_opendir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.opendir.dirname);
    free(data);
    return;
}


static void
iiimp_file_operation_request_readdir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_rewinddir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_seekdir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_telldir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_request_mkdir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.mkdir.dirname);
    free(data);
    return;
}


static void
iiimp_file_operation_request_rmdir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_string_delete(data_s, data->value.rmdir.dirname);
    free(data);
    return;
}


static void
iiimp_file_operation_request_fcntl_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    iiimp_file_operation_data_fcntl_delete(data_s, data->value.fcntl.arg);
    free(data);
    return;
}


void
iiimp_file_operation_request_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    if (NULL == data) return;

    switch(data->type) {
    case IIIMP_FILE_OPERATION_CLOSE:
	iiimp_file_operation_request_close_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_CREAT:
	iiimp_file_operation_request_creat_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_FTRUNCATE:
	iiimp_file_operation_request_ftruncate_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_LSEEK:
	iiimp_file_operation_request_lseek_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_OPEN:
	iiimp_file_operation_request_open_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_READ:
	iiimp_file_operation_request_read_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_TRUNCATE:
	iiimp_file_operation_request_truncate_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_WRITE:
	iiimp_file_operation_request_write_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_LINK:
	iiimp_file_operation_request_link_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_READLINK:
	iiimp_file_operation_request_readlink_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_RENAME:
	iiimp_file_operation_request_rename_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_SYMLINK:
	iiimp_file_operation_request_symlink_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_UNLINK:
	iiimp_file_operation_request_unlink_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_CHMOD:
	iiimp_file_operation_request_chmod_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_CHOWN:
	iiimp_file_operation_request_chown_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_FCHMOD:
	iiimp_file_operation_request_fchmod_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_FCHOWN:
	iiimp_file_operation_request_fchown_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_LCHOWN:
	iiimp_file_operation_request_lchown_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_UTIME:
	iiimp_file_operation_request_utime_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_UTIMES:
	iiimp_file_operation_request_utimes_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_FSTAT:
	iiimp_file_operation_request_fstat_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_LSTAT:
	iiimp_file_operation_request_lstat_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_STAT:
	iiimp_file_operation_request_stat_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_ACCESS:
	iiimp_file_operation_request_access_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_CLOSEDIR:
	iiimp_file_operation_request_closedir_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_OPENDIR:
	iiimp_file_operation_request_opendir_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_READDIR:
	iiimp_file_operation_request_readdir_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_REWINDDIR:
	iiimp_file_operation_request_rewinddir_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_SEEKDIR:
	iiimp_file_operation_request_seekdir_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_TELLDIR:
	iiimp_file_operation_request_telldir_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_MKDIR:
	iiimp_file_operation_request_mkdir_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_RMDIR:
	iiimp_file_operation_request_rmdir_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_FCNTL:
	iiimp_file_operation_request_fcntl_delete(data_s, data);
	break;

    default:
	break;
    }
    return;
}


static void
iiimp_file_operation_result_simple_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


static void
iiimp_file_operation_result_void_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data);
    return;
}


/*
 * iiimp_file_operation_result_close_delete()
 */

/*
 * iiimp_file_operation_result_creat_delete()
 */

/*
 * iiimp_file_operation_result_ftruncate_delete()
 */

/*
 * iiimp_file_operation_result_lseek_delete()
 */

/*
 * iiimp_file_operation_result_open_delete()
 */


static void
iiimp_file_operation_result_read_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	data)
{
#if defined(lint) && defined(HAVE_NOTE_H)
    NOTE(ARGUNUSED(data_s))
#endif /* lint && HAVE_NOTE_H */

    free(data->value.read.value.ptr);
    free(data);
    return;
}


/*
 * iiimp_file_operation_result_truncate_delete()
 */

/*
 * iiimp_file_operation_result_write_delete()
 */

/*
 * iiimp_file_operation_result_link_delete()
 */


static void
iiimp_file_operation_result_readlink_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	data)
{
    iiimp_string_delete(data_s, data->value.readlink.path);
    free(data);
    return;
}


/*
 * iiimp_file_operation_result_rename_delete()
 */

/*
 * iiimp_file_operation_result_symlink_delete()
 */

/*
 * iiimp_file_operation_result_unlink_delete()
 */

/*
 * iiimp_file_operation_result_chmod_delete()
 */

/*
 * iiimp_file_operation_result_chown_delete()
 */

/*
 * iiimp_file_operation_result_fchmod_delete()
 */

/*
 * iiimp_file_operation_result_fchown_delete()
 */

/*
 * iiimp_file_operation_result_lchown_delete()
 */

/*
 * iiimp_file_operation_result_utime_delete()
 */

/*
 * iiimp_file_operation_result_utimes_delete()
 */


static void
iiimp_file_operation_result_stat_common_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	data)
{
    iiimp_file_operation_data_stat_delete(data_s, data->value.stat.stat);
    free(data);
    return;
}

/*
 * iiimp_file_operation_result_fstat_delete()
 */

/*
 * iiimp_file_operation_result_lstat_delete()
 */

/*
 * iiimp_file_operation_result_stat_delete()
 */

/*
 * iiimp_file_operation_result_access_delete()
 */

/*
 * iiimp_file_operation_result_closedir_delete()
 */

/*
 * iiimp_file_operation_result_opendir_delete()
 */


static void
iiimp_file_operation_result_readdir_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	data)
{
    iiimp_file_operation_data_dir_delete(data_s, data->value.readdir.dir);
    free(data);
    return;
}


/*
 * iiimp_file_operation_result_rewinddir_delete()
 */

/*
 * iiimp_file_operation_result_seekdir_delete()
 */

/*
 * iiimp_file_operation_result_telldir_delete()
 */

/*
 * iiimp_file_operation_result_mkdir_delete()
 */

/*
 * iiimp_file_operation_result_rmdir_delete()
 */


static void
iiimp_file_operation_result_fcntl_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	data)
{
    iiimp_file_operation_data_fcntl_delete(data_s, data->value.fcntl.arg);
    free(data);
    return;
}


void
iiimp_file_operation_result_delete(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	data)
{
    if (NULL == data) return;

    switch(data->type) {
    case IIIMP_FILE_OPERATION_CLOSE:
    case IIIMP_FILE_OPERATION_CREAT:
    case IIIMP_FILE_OPERATION_FTRUNCATE:
    case IIIMP_FILE_OPERATION_LSEEK:
    case IIIMP_FILE_OPERATION_OPEN:
	iiimp_file_operation_result_simple_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_READ:
	iiimp_file_operation_result_read_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_TRUNCATE:
    case IIIMP_FILE_OPERATION_WRITE:
    case IIIMP_FILE_OPERATION_LINK:
	iiimp_file_operation_result_simple_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_READLINK:
	iiimp_file_operation_result_readlink_delete(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_RENAME:
    case IIIMP_FILE_OPERATION_SYMLINK:
    case IIIMP_FILE_OPERATION_UNLINK:
    case IIIMP_FILE_OPERATION_CHMOD:
    case IIIMP_FILE_OPERATION_CHOWN:
    case IIIMP_FILE_OPERATION_FCHMOD:
    case IIIMP_FILE_OPERATION_FCHOWN:
    case IIIMP_FILE_OPERATION_LCHOWN:
    case IIIMP_FILE_OPERATION_UTIME:
    case IIIMP_FILE_OPERATION_UTIMES:
	iiimp_file_operation_result_simple_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_FSTAT:
    case IIIMP_FILE_OPERATION_LSTAT:
    case IIIMP_FILE_OPERATION_STAT:
	iiimp_file_operation_result_stat_common_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_ACCESS:
	iiimp_file_operation_result_simple_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_CLOSEDIR:
    case IIIMP_FILE_OPERATION_OPENDIR:
	iiimp_file_operation_result_simple_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_READDIR:
	iiimp_file_operation_result_readdir_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_REWINDDIR:
    case IIIMP_FILE_OPERATION_SEEKDIR:
	iiimp_file_operation_result_void_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_TELLDIR:
	iiimp_file_operation_result_simple_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_MKDIR:
    case IIIMP_FILE_OPERATION_RMDIR:
	iiimp_file_operation_result_simple_delete(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_FCNTL:
	iiimp_file_operation_result_fcntl_delete(data_s, data);
	break;

    default:
	break;
    }
    return;
}


static void
iiimp_file_operation_data_fcntl_setfd_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_fcntl *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		file_descriptor_flags;

    rest = *nbyte;
    p = *ptr;

    file_descriptor_flags =
	iiimp_file_operation_fcntl_setfd_flag_l2i(
	    m->value.file_descriptor_flags);

    PUT32(file_descriptor_flags, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_data_fcntl_setfl_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_fcntl *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->value.file_status_flags, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_data_fcntl_flock_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_fcntl *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		type;
    int		whence;

    rest = *nbyte;
    p = *ptr;

    type = iiimp_file_operation_fcntl_flock_type_l2i(m->value.flock.type);
    whence = iiimp_file_operation_whence_l2i(m->value.flock.whence);

    PUT16(type, rest, p, data_s->byte_swap);
    PUT16(whence, rest, p, data_s->byte_swap);
    PUT32(m->value.flock.start, rest, p, data_s->byte_swap);
    PUT32(m->value.flock.len, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_data_stat_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_stat *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->mode, rest, p, data_s->byte_swap);
    PUTU32(m->ino, rest, p, data_s->byte_swap);
    PUTU32(m->dev, rest, p, data_s->byte_swap);
    PUTU32(m->rdev, rest, p, data_s->byte_swap);
    PUTU32(m->nlink, rest, p, data_s->byte_swap);
    if (NULL == m->user_name) {
	PUTU32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_string_pack(data_s, m->user_name, &rest, &p);
    }
    PUT32(m->uid, rest, p, data_s->byte_swap);
    if (NULL == m->group_name) {
	PUTU32(0, rest, p, data_s->byte_swap);
	iiimp_string_pack(data_s, m->group_name, &rest, &p);
    }
    PUT32(m->gid, rest, p, data_s->byte_swap);
    PUT32(m->size, rest, p, data_s->byte_swap);
    PUT32(m->atime.tv_sec, rest, p, data_s->byte_swap);
    PUT32(m->atime.tv_nsec, rest, p, data_s->byte_swap);
    PUT32(m->mtime.tv_sec, rest, p, data_s->byte_swap);
    PUT32(m->mtime.tv_nsec, rest, p, data_s->byte_swap);
    PUT32(m->ctime.tv_sec, rest, p, data_s->byte_swap);
    PUT32(m->ctime.tv_nsec, rest, p, data_s->byte_swap);
    PUT32(m->blksize, rest, p, data_s->byte_swap);
    PUT32(m->blocks, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_data_dir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_dir *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->ino, rest, p, data_s->byte_swap);
    PUT32(m->off, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->name, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_close_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.close.file_descriptor, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_creat_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.creat.path, &rest, &p);
    PUT32(m->value.creat.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_ftruncate_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.ftruncate.file_descriptor, rest, p, data_s->byte_swap);
    PUT32(m->value.ftruncate.length, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_lseek_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		whence;

    rest = *nbyte;
    p = *ptr;

    whence = iiimp_file_operation_whence_l2i(m->value.lseek.whence);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.lseek.file_descriptor, rest, p, data_s->byte_swap);
    PUT32(m->value.lseek.offset, rest, p, data_s->byte_swap);
    PUT32(whence, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_open_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		oflag;

    rest = *nbyte;
    p = *ptr;

    oflag = iiimp_file_operation_open_flags_l2i(m->value.open.oflag);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.open.path, &rest, &p);
    PUT32(oflag, rest, p, data_s->byte_swap);
    PUT32(m->value.open.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_read_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.read.file_descriptor, rest, p, data_s->byte_swap);
    PUT32(m->value.read.nbyte, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_truncate_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.truncate.path, &rest, &p);
    PUT32(m->value.truncate.length, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_write_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		pad;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.write.file_descriptor, rest, p, data_s->byte_swap);
    PUT32(m->value.write.value.nbyte, rest, p, data_s->byte_swap);
    (void)memcpy(p, m->value.write.value.ptr, m->value.write.value.nbyte);
    p += m->value.write.value.nbyte;
    rest -= m->value.write.value.nbyte;
    for (pad = PAD(m->value.write.value.nbyte); 0 < pad; --pad) {
	PUTU8(0, rest, p, data_s->byte_swap);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_link_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.link.existing_path, &rest, &p);
    iiimp_string_pack(data_s, m->value.link.new_path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_readlink_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.readlink.path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_rename_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.rename.old_path, &rest, &p);
    iiimp_string_pack(data_s, m->value.rename.new_path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_symlink_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.symlink.existing_path, &rest, &p);
    iiimp_string_pack(data_s, m->value.symlink.new_path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_unlink_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.unlink.path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_chmod_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.chmod.path, &rest, &p);
    PUTU32(m->value.chmod.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_chown_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.chown.path, &rest, &p);
    if (NULL == m->value.chown.user_name) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_string_pack(data_s, m->value.chown.user_name, &rest, &p);
    }
    PUT32(m->value.chown.user_id, rest, p, data_s->byte_swap);
    if (NULL == m->value.chown.group_name) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_string_pack(data_s, m->value.chown.group_name, &rest, &p);
    }
    PUT32(m->value.chown.group_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_fchmod_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.fchmod.file_descriptor, rest, p, data_s->byte_swap);
    PUTU32(m->value.fchmod.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_fchown_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.fchown.file_descriptor, rest, p, data_s->byte_swap);
    if (NULL == m->value.fchown.user_name) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_string_pack(data_s, m->value.fchown.user_name, &rest, &p);
    }
    PUT32(m->value.fchown.user_id, rest, p, data_s->byte_swap);
    if (NULL == m->value.fchown.group_name) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_string_pack(data_s, m->value.fchown.group_name, &rest, &p);
    }
    PUT32(m->value.fchown.group_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_lchown_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.lchown.path, &rest, &p);
    if (NULL == m->value.lchown.user_name) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_string_pack(data_s, m->value.lchown.user_name, &rest, &p);
    }
    PUT32(m->value.lchown.user_id, rest, p, data_s->byte_swap);
    if (NULL == m->value.lchown.group_name) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_string_pack(data_s, m->value.lchown.group_name, &rest, &p);
    }
    PUT32(m->value.lchown.group_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_utime_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.utime.path, &rest, &p);
    PUT32(m->value.utime.utimbuf_data.actime, rest, p, data_s->byte_swap);
    PUT32(m->value.utime.utimbuf_data.modtime, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_utimes_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.utime.path, &rest, &p);
    PUT32(m->value.utimes.timeval_data.tv_sec0, rest, p, data_s->byte_swap);
    PUT32(m->value.utimes.timeval_data.tv_usec0, rest, p, data_s->byte_swap);
    PUT32(m->value.utimes.timeval_data.tv_sec1, rest, p, data_s->byte_swap);
    PUT32(m->value.utimes.timeval_data.tv_usec1, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_fstat_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.fstat.file_descriptor, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_lstat_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.lstat.path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_stat_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.stat.path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_access_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		amode;

    rest = *nbyte;
    p = *ptr;

    amode = iiimp_file_operation_access_amode_l2i(m->value.access.amode);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.access.path, &rest, &p);
    PUT32(amode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_closedir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.closedir.directory_handle, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_opendir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.opendir.dirname, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_readdir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.readdir.directory_handle, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_rewinddir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.rewinddir.directory_handle, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_seekdir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.seekdir.directory_handle, rest, p, data_s->byte_swap);
    PUT32(m->value.seekdir.loc, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_telldir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.telldir.directory_handle, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_mkdir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.mkdir.dirname, &rest, &p);
    PUT32(m->value.mkdir.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_rmdir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    iiimp_string_pack(data_s, m->value.rmdir.dirname, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_request_fcntl_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		cmd;

    rest = *nbyte;
    p = *ptr;

    cmd = iiimp_file_operation_fcntl_cmd_l2i(m->value.fcntl.cmd);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.fcntl.file_descriptor, rest, p, data_s->byte_swap);
    PUT32(cmd, rest, p, data_s->byte_swap);

    switch (m->value.fcntl.cmd) {
    case F_SETFD:
	iiimp_file_operation_data_fcntl_setfd_pack(data_s, m->value.fcntl.arg,
						   &rest, &p);
	break;
    case F_SETFL:
	iiimp_file_operation_data_fcntl_setfl_pack(data_s, m->value.fcntl.arg,
						   &rest, &p);
	break;
    case F_GETLK:
    case F_SETLK:
    case F_SETLKW:
	iiimp_file_operation_data_fcntl_flock_pack(data_s, m->value.fcntl.arg,
						   &rest, &p);
	break;
    default:
	break;
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


void
iiimp_file_operation_request_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    if (NULL == m) return;

    switch(m->type) {
    case IIIMP_FILE_OPERATION_CLOSE:
	iiimp_file_operation_request_close_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_CREAT:
	iiimp_file_operation_request_creat_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_FTRUNCATE:
	iiimp_file_operation_request_ftruncate_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_LSEEK:
	iiimp_file_operation_request_lseek_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_OPEN:
	iiimp_file_operation_request_open_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_READ:
	iiimp_file_operation_request_read_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_TRUNCATE:
	iiimp_file_operation_request_truncate_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_WRITE:
	iiimp_file_operation_request_write_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_LINK:
	iiimp_file_operation_request_link_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_READLINK:
	iiimp_file_operation_request_readlink_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_RENAME:
	iiimp_file_operation_request_rename_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_SYMLINK:
	iiimp_file_operation_request_symlink_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_UNLINK:
	iiimp_file_operation_request_unlink_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_CHMOD:
	iiimp_file_operation_request_chmod_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_CHOWN:
	iiimp_file_operation_request_chown_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_FCHMOD:
	iiimp_file_operation_request_fchmod_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_FCHOWN:
	iiimp_file_operation_request_fchown_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_LCHOWN:
	iiimp_file_operation_request_lchown_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_UTIME:
	iiimp_file_operation_request_utime_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_UTIMES:
	iiimp_file_operation_request_utimes_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_FSTAT:
	iiimp_file_operation_request_fstat_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_LSTAT:
	iiimp_file_operation_request_lstat_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_STAT:
	iiimp_file_operation_request_stat_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_ACCESS:
	iiimp_file_operation_request_access_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_CLOSEDIR:
	iiimp_file_operation_request_closedir_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_OPENDIR:
	iiimp_file_operation_request_opendir_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_READDIR:
	iiimp_file_operation_request_readdir_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_REWINDDIR:
	iiimp_file_operation_request_rewinddir_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_SEEKDIR:
	iiimp_file_operation_request_seekdir_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_TELLDIR:
	iiimp_file_operation_request_telldir_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_MKDIR:
	iiimp_file_operation_request_mkdir_pack(data_s, m, nbyte, ptr);
	break;
    case IIIMP_FILE_OPERATION_RMDIR:
	iiimp_file_operation_request_rmdir_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_FCNTL:
	iiimp_file_operation_request_fcntl_pack(data_s, m, nbyte, ptr);
	break;

    default:
	break;
    }
    return;
}


static void
iiimp_file_operation_result_simple_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		error_number;

    rest = *nbyte;
    p = *ptr;

    error_number =
	iiimp_file_operation_error_number_l2i(m->value.simple.error_number);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.simple.return_value, rest, p, data_s->byte_swap);
    PUT32(error_number, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


static void
iiimp_file_operation_result_void_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;

    rest = *nbyte;
    p = *ptr;

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}


/*
 * iiimp_file_operation_result_close_pack()
 */

/*
 * iiimp_file_operation_result_creat_pack()
 */

/*
 * iiimp_file_operation_result_ftruncate_pack()
 */

/*
 * iiimp_file_operation_result_lseek_pack()
 */

/*
 * iiimp_file_operation_result_open_pack()
 */


static void
iiimp_file_operation_result_read_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		pad;
    int		error_number;

    rest = *nbyte;
    p = *ptr;

    error_number =
	iiimp_file_operation_error_number_l2i(m->value.read.error_number);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(error_number, rest, p, data_s->byte_swap);
    PUT32(m->value.read.value.nbyte, rest, p, data_s->byte_swap);
    (void)memcpy(p, m->value.read.value.ptr, m->value.read.value.nbyte);
    p += m->value.read.value.nbyte;
    rest -= m->value.read.value.nbyte;
    for (pad = PAD(m->value.read.value.nbyte); 0 < pad; --pad) {
	PUTU8(0, rest, p, data_s->byte_swap);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


/*
 * iiimp_file_operation_result_truncate_pack()
 */

/*
 * iiimp_file_operation_result_write_pack()
 */

/*
 * iiimp_file_operation_result_link_pack()
 */


static void
iiimp_file_operation_result_readlink_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		error_number;

    rest = *nbyte;
    p = *ptr;

    error_number =
	iiimp_file_operation_error_number_l2i(m->value.readlink.error_number);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.readlink.return_value, rest, p, data_s->byte_swap);
    PUT32(error_number, rest, p, data_s->byte_swap);
    if (NULL == m->value.readlink.path) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_string_pack(data_s, m->value.readlink.path, &rest, &p);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


/*
 * iiimp_file_operation_result_rename_pack()
 */

/*
 * iiimp_file_operation_result_symlink_pack()
 */

/*
 * iiimp_file_operation_result_unlink_pack()
 */

/*
 * iiimp_file_operation_result_chmod_pack()
 */

/*
 * iiimp_file_operation_result_chown_pack()
 */

/*
 * iiimp_file_operation_result_fchmod_pack()
 */

/*
 * iiimp_file_operation_result_fchown_pack()
 */

/*
 * iiimp_file_operation_result_lchown_pack()
 */

/*
 * iiimp_file_operation_result_utime_pack()
 */

/*
 * iiimp_file_operation_result_utimes_pack()
 */


static void
iiimp_file_operation_result_stat_common_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		error_number;

    rest = *nbyte;
    p = *ptr;

    error_number =
	iiimp_file_operation_error_number_l2i(m->value.stat.error_number);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.stat.return_value, rest, p, data_s->byte_swap);
    PUT32(error_number, rest, p, data_s->byte_swap);
    if (NULL == m->value.stat.stat) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_file_operation_data_stat_pack(data_s, m->value.stat.stat, &rest, &p);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}

/*
 * iiimp_file_operation_result_fstat_pack()
 */

/*
 * iiimp_file_operation_result_lstat_pack()
 */

/*
 * iiimp_file_operation_result_stat_pack()
 */

/*
 * iiimp_file_operation_result_access_pack()
 */

/*
 * iiimp_file_operation_result_closedir_pack()
 */

/*
 * iiimp_file_operation_result_opendir_pack()
 */


static void
iiimp_file_operation_result_readdir_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		error_number;

    rest = *nbyte;
    p = *ptr;

    error_number =
	iiimp_file_operation_error_number_l2i(m->value.readdir.error_number);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(error_number, rest, p, data_s->byte_swap);
    if (NULL == m->value.readdir.dir) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	iiimp_file_operation_data_dir_pack(data_s, m->value.readdir.dir, &rest, &p);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


/*
 * iiimp_file_operation_result_rewinddir_pack()
 */

/*
 * iiimp_file_operation_result_seekdir_pack()
 */

/*
 * iiimp_file_operation_result_telldir_pack()
 */

/*
 * iiimp_file_operation_result_mkdir_pack()
 */

/*
 * iiimp_file_operation_result_rmdir_pack()
 */


static void
iiimp_file_operation_result_fcntl_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    uchar_t *	p;
    size_t	rest;
    int		i;
    int		error_number;
    int		cmd;

    rest = *nbyte;
    p = *ptr;

    error_number =
	iiimp_file_operation_error_number_l2i(m->value.fcntl.error_number);
    cmd = iiimp_file_operation_fcntl_cmd_l2i(m->value.fcntl.cmd);

    PUT32(m->type, rest, p, data_s->byte_swap);
    PUT32(m->value_nbyte, rest, p, data_s->byte_swap);
    PUT32(m->value.fcntl.return_value, rest, p, data_s->byte_swap);
    PUT32(error_number, rest, p, data_s->byte_swap);
    PUT32(cmd, rest, p, data_s->byte_swap);

    if (NULL == m->value.fcntl.arg) {
	PUT32(0, rest, p, data_s->byte_swap);
    } else {
	switch (m->value.fcntl.cmd) {
	case F_SETFD:
	    iiimp_file_operation_data_fcntl_setfd_pack(data_s,
						       m->value.fcntl.arg,
						       &rest, &p);
	    break;
	case F_SETFL:
	    iiimp_file_operation_data_fcntl_setfl_pack(data_s,
						       m->value.fcntl.arg,
						       &rest, &p);
	    break;
	case F_GETLK:
	case F_SETLK:
	case F_SETLKW:
	    iiimp_file_operation_data_fcntl_flock_pack(data_s,
						       m->value.fcntl.arg,
						       &rest, &p);
	    break;
	default:
	    break;
	}
    }

    *nbyte = rest;
    *ptr = p;

    return;
}


void
iiimp_file_operation_result_pack(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    size_t *				nbyte,
    uchar_t **				ptr)
{
    if (NULL == m) return;

    switch(m->type) {
    case IIIMP_FILE_OPERATION_CLOSE:
    case IIIMP_FILE_OPERATION_CREAT:
    case IIIMP_FILE_OPERATION_FTRUNCATE:
    case IIIMP_FILE_OPERATION_LSEEK:
    case IIIMP_FILE_OPERATION_OPEN:
	iiimp_file_operation_result_simple_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_READ:
	iiimp_file_operation_result_read_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_TRUNCATE:
    case IIIMP_FILE_OPERATION_WRITE:
    case IIIMP_FILE_OPERATION_LINK:
	iiimp_file_operation_result_simple_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_READLINK:
	iiimp_file_operation_result_readlink_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_RENAME:
    case IIIMP_FILE_OPERATION_SYMLINK:
    case IIIMP_FILE_OPERATION_UNLINK:
    case IIIMP_FILE_OPERATION_CHMOD:
    case IIIMP_FILE_OPERATION_CHOWN:
    case IIIMP_FILE_OPERATION_FCHMOD:
    case IIIMP_FILE_OPERATION_FCHOWN:
    case IIIMP_FILE_OPERATION_LCHOWN:
    case IIIMP_FILE_OPERATION_UTIME:
    case IIIMP_FILE_OPERATION_UTIMES:
	iiimp_file_operation_result_simple_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_FSTAT:
    case IIIMP_FILE_OPERATION_LSTAT:
    case IIIMP_FILE_OPERATION_STAT:
	iiimp_file_operation_result_stat_common_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_ACCESS:
    case IIIMP_FILE_OPERATION_CLOSEDIR:
    case IIIMP_FILE_OPERATION_OPENDIR:
	iiimp_file_operation_result_simple_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_READDIR:
	iiimp_file_operation_result_readdir_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_REWINDDIR:
    case IIIMP_FILE_OPERATION_SEEKDIR:
	iiimp_file_operation_result_void_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_TELLDIR:
    case IIIMP_FILE_OPERATION_MKDIR:
    case IIIMP_FILE_OPERATION_RMDIR:
	iiimp_file_operation_result_simple_pack(data_s, m, nbyte, ptr);
	break;

    case IIIMP_FILE_OPERATION_FCNTL:
	iiimp_file_operation_result_fcntl_pack(data_s, m, nbyte, ptr);
	break;

    default:
	break;
    }
    return;
}


IIIMP_file_operation_data_fcntl *
iiimp_file_operation_data_fcntl_setfd_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_data_fcntl *	d;
    size_t				rest;
    const uchar_t *			p;
    int					file_descriptor_flags;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < 4)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_data_fcntl *)
	malloc(sizeof (IIIMP_file_operation_data_fcntl));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(file_descriptor_flags, rest, p, data_s->byte_swap);
    d->value.file_descriptor_flags =
	iiimp_file_operation_fcntl_setfd_flag_l2i(file_descriptor_flags);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_data_fcntl *
iiimp_file_operation_data_fcntl_setfl_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_data_fcntl *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_data_fcntl *)
	malloc(sizeof (IIIMP_file_operation_data_fcntl));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->value.file_status_flags, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_data_fcntl *
iiimp_file_operation_data_fcntl_flock_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_data_fcntl *	d;
    size_t				rest;
    const uchar_t *			p;
    int					type;
    int					whence;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (2 + 2 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_data_fcntl *)
	malloc(sizeof (IIIMP_file_operation_data_fcntl));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET16(type, rest, p, data_s->byte_swap);
    d->value.flock.type = iiimp_file_operation_fcntl_flock_type_i2l(type);
    GET16(whence, rest, p, data_s->byte_swap);
    d->value.flock.whence = iiimp_file_operation_whence_i2l(whence);
    GET32(d->value.flock.start, rest, p, data_s->byte_swap);
    GET32(d->value.flock.len, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_data_stat *
iiimp_file_operation_data_stat_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_data_stat *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_data_stat *)
	malloc(sizeof (IIIMP_file_operation_data_stat));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GETU32(d->mode, rest, p, data_s->byte_swap);
    GETU32(d->ino, rest, p, data_s->byte_swap);
    GETU32(d->dev, rest, p, data_s->byte_swap);
    GETU32(d->rdev, rest, p, data_s->byte_swap);
    GETU32(d->nlink, rest, p, data_s->byte_swap);
    d->user_name = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->user_name) || (rest < (4 + 4))) {
	iiimp_string_delete(data_s, d->user_name);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->uid, rest, p, data_s->byte_swap);
    d->group_name = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->group_name) ||
	(rest < (4 + 4 + 4 + 4 + 4 + 4 + 4 + 4 + 4 + 4))) {
	iiimp_string_delete(data_s, d->user_name);
	iiimp_string_delete(data_s, d->group_name);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->gid, rest, p, data_s->byte_swap);
    GET32(d->size, rest, p, data_s->byte_swap);
    GET32(d->atime.tv_sec, rest, p, data_s->byte_swap);
    GET32(d->atime.tv_nsec, rest, p, data_s->byte_swap);
    GET32(d->mtime.tv_sec, rest, p, data_s->byte_swap);
    GET32(d->mtime.tv_nsec, rest, p, data_s->byte_swap);
    GET32(d->ctime.tv_sec, rest, p, data_s->byte_swap);
    GET32(d->ctime.tv_nsec, rest, p, data_s->byte_swap);
    GET32(d->blksize, rest, p, data_s->byte_swap);
    GET32(d->blocks, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_data_dir *
iiimp_file_operation_data_dir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_data_dir *	d;
    size_t				rest;
    const uchar_t *			p;
    int					count;
    int					i;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_data_dir *)
	malloc(sizeof (IIIMP_file_operation_data_dir));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GETU32(d->ino, rest, p, data_s->byte_swap);
    GET32(d->off, rest, p, data_s->byte_swap);
    d->name = iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->name) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_close_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.close.file_descriptor, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_creat_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.creat.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.creat.path || (rest < 4))) {
	iiimp_string_delete(data_s, d->value.creat.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GETU32(d->value.creat.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_ftruncate_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.ftruncate.file_descriptor, rest, p, data_s->byte_swap);
    GET32(d->value.ftruncate.length, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_lseek_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;
    int					whence;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.lseek.file_descriptor, rest, p, data_s->byte_swap);
    GET32(d->value.lseek.offset, rest, p, data_s->byte_swap);
    GET32(whence, rest, p, data_s->byte_swap);
    d->value.lseek.whence = iiimp_file_operation_whence_l2i(whence);


    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_open_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;
    int					oflag;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < 4)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.open.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.open.path) || (rest < (4 + 4))) {
	iiimp_string_delete(data_s, d->value.creat.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(oflag, rest, p, data_s->byte_swap);
    d->value.open.oflag = iiimp_file_operation_open_flags_i2l(oflag);
    GET32(d->value.open.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_read_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.read.file_descriptor, rest, p, data_s->byte_swap);
    GET32(d->value.read.nbyte, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_truncate_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.truncate.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.truncate.path) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.truncate.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.truncate.length, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_write_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.write.file_descriptor, rest, p, data_s->byte_swap);
    GET32(d->value.write.value.nbyte, rest, p, data_s->byte_swap);
    if (rest < d->value.write.value.nbyte) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    if (0 < d->value.write.value.nbyte) {
	d->value.write.value.ptr =
	    (uchar_t *)malloc(d->value.write.value.nbyte);
	(void)memcpy(d->value.write.value.ptr, p, d->value.write.value.nbyte);
	p += (d->value.write.value.nbyte + PAD(d->value.write.value.nbyte));
	rest -= (d->value.write.value.nbyte + PAD(d->value.write.value.nbyte));

    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_link_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.link.existing_path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.link.existing_path) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.link.existing_path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    d->value.link.new_path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.link.new_path) {
	iiimp_string_delete(data_s, d->value.link.existing_path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_readlink_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.readlink.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.readlink.path) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_rename_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.rename.old_path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.rename.old_path) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.rename.old_path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    d->value.rename.new_path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.rename.new_path) {
	iiimp_string_delete(data_s, d->value.rename.old_path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_symlink_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.symlink.existing_path =
	iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.symlink.existing_path) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.symlink.existing_path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    d->value.symlink.new_path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.symlink.new_path) {
	iiimp_string_delete(data_s, d->value.symlink.existing_path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_unlink_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.unlink.path =iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.unlink.path) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_chmod_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.chmod.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.chmod.path) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.chmod.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GETU32(d->value.chmod.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_chown_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.chown.path =iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.chown.path) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.chown.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    d->value.chown.user_name = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.chown.user_name) || (rest < (4 + 4))) {
	iiimp_string_delete(data_s, d->value.chown.path);
	iiimp_string_delete(data_s, d->value.chown.user_name);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.chown.user_id, rest, p, data_s->byte_swap);
    d->value.chown.group_name = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.chown.group_name) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.chown.path);
	iiimp_string_delete(data_s, d->value.chown.user_name);
	iiimp_string_delete(data_s, d->value.chown.group_name);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.chown.group_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_fchmod_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.fchmod.file_descriptor, rest, p, data_s->byte_swap);
    GETU32(d->value.fchmod.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_fchown_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.fchown.file_descriptor, rest, p, data_s->byte_swap);
    d->value.fchown.user_name = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.fchown.user_name) || (rest < (4 + 4))) {
	iiimp_string_delete(data_s, d->value.fchown.user_name);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.fchown.user_id, rest, p, data_s->byte_swap);
    d->value.fchown.group_name = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.fchown.group_name) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.fchown.user_name);
	iiimp_string_delete(data_s, d->value.fchown.group_name);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.fchown.group_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_lchown_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.lchown.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.lchown.path) || (rest < (4 + 4))) {
	iiimp_string_delete(data_s, d->value.lchown.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    d->value.lchown.user_name =iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.lchown.user_name) || (rest < (4 + 4))) {
	iiimp_string_delete(data_s, d->value.lchown.path);
	iiimp_string_delete(data_s, d->value.lchown.user_name);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.lchown.user_id, rest, p, data_s->byte_swap);
    d->value.lchown.group_name = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.lchown.group_name) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.lchown.path);
	iiimp_string_delete(data_s, d->value.lchown.user_name);
	iiimp_string_delete(data_s, d->value.lchown.group_name);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.lchown.group_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_utime_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.utime.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.utime.path) || (rest < (4 + 4))) {
	iiimp_string_delete(data_s, d->value.utime.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.utime.utimbuf_data.actime, rest, p, data_s->byte_swap);
    GET32(d->value.utime.utimbuf_data.modtime, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_utimes_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.utime.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.utime.path) || (rest < (4 + 4 + 4 + 4))) {
	iiimp_string_delete(data_s, d->value.utime.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.utimes.timeval_data.tv_sec0, rest, p, data_s->byte_swap);
    GET32(d->value.utimes.timeval_data.tv_usec0, rest, p, data_s->byte_swap);
    GET32(d->value.utimes.timeval_data.tv_sec1, rest, p, data_s->byte_swap);
    GET32(d->value.utimes.timeval_data.tv_usec1, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_fstat_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.fstat.file_descriptor, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_lstat_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.lstat.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.lstat.path) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_stat_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.stat.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.stat.path) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_access_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;
    int					amode;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.access.path = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.access.path) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.access.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(amode, rest, p, data_s->byte_swap);
    d->value.access.amode = iiimp_file_operation_access_amode_l2i(amode);


    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_closedir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.closedir.directory_handle, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_opendir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.opendir.dirname = iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.opendir.dirname) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_readdir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.readdir.directory_handle, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_rewinddir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.rewinddir.directory_handle, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_seekdir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.seekdir.directory_handle, rest, p, data_s->byte_swap);
    GET32(d->value.seekdir.loc, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_telldir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.telldir.directory_handle, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_mkdir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.mkdir.dirname = iiimp_string_unpack(data_s, &rest, &p, rest);
    if ((NULL == d->value.mkdir.dirname) || (rest < 4)) {
	iiimp_string_delete(data_s, d->value.mkdir.dirname);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    GET32(d->value.mkdir.mode, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_rmdir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    d->value.rmdir.dirname = iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.rmdir.dirname) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_fcntl_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;
    int					cmd;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < 4)) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_request *)
	malloc(sizeof (IIIMP_file_operation_request));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.fcntl.file_descriptor, rest, p, data_s->byte_swap);
    GET32(cmd, rest, p, data_s->byte_swap);
    d->value.fcntl.cmd = iiimp_file_operation_fcntl_cmd_i2l(cmd);

    switch (d->value.fcntl.cmd) {
    case F_SETFD:
	d->value.fcntl.arg =
	    iiimp_file_operation_data_fcntl_setfd_unpack(data_s, &rest,
							 &p, rest);
	break;
    case F_SETFL:
	d->value.fcntl.arg =
	    iiimp_file_operation_data_fcntl_setfl_unpack(data_s, &rest,
							 &p, rest);
	break;
    case F_GETLK:
    case F_SETLK:
    case F_SETLKW:
	d->value.fcntl.arg =
	    iiimp_file_operation_data_fcntl_flock_unpack(data_s, &rest,
							 &p, rest);
	break;
    default:
	d->value.fcntl.arg = NULL;
	break;
    }
    if (NULL == d->value.fcntl.arg) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_request *
iiimp_file_operation_request_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_request *	d;
    size_t				rest;
    const uchar_t *			p;
    int					type;

    rest = nbyte_max;
    p = *ptr;

    GET32(type, rest, p, data_s->byte_swap);

    switch(type) {
    case IIIMP_FILE_OPERATION_CLOSE:
	d = iiimp_file_operation_request_close_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_CREAT:
	d = iiimp_file_operation_request_creat_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_FTRUNCATE:
	d = iiimp_file_operation_request_ftruncate_unpack(data_s, nbyte,
							  ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_LSEEK:
	d = iiimp_file_operation_request_lseek_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_OPEN:
	d = iiimp_file_operation_request_open_unpack(data_s, nbyte,
						     ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_READ:
	d = iiimp_file_operation_request_read_unpack(data_s, nbyte,
						     ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_TRUNCATE:
	d = iiimp_file_operation_request_truncate_unpack(data_s, nbyte,
							 ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_WRITE:
	d = iiimp_file_operation_request_write_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_LINK:
	d = iiimp_file_operation_request_link_unpack(data_s, nbyte,
						     ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_READLINK:
	d = iiimp_file_operation_request_readlink_unpack(data_s, nbyte,
							 ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_RENAME:
	d = iiimp_file_operation_request_rename_unpack(data_s, nbyte,
						       ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_SYMLINK:
	d = iiimp_file_operation_request_symlink_unpack(data_s, nbyte,
							ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_UNLINK:
	d = iiimp_file_operation_request_unlink_unpack(data_s, nbyte,
						       ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_CHMOD:
	d = iiimp_file_operation_request_chmod_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_CHOWN:
	d = iiimp_file_operation_request_chown_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_FCHMOD:
	d = iiimp_file_operation_request_fchmod_unpack(data_s, nbyte,
						       ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_FCHOWN:
	d = iiimp_file_operation_request_fchown_unpack(data_s, nbyte,
						       ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_LCHOWN:
	d = iiimp_file_operation_request_lchown_unpack(data_s, nbyte,
						       ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_UTIME:
	d = iiimp_file_operation_request_utime_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_UTIMES:
	d = iiimp_file_operation_request_utimes_unpack(data_s, nbyte,
						       ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_FSTAT:
	d = iiimp_file_operation_request_fstat_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_LSTAT:
	d = iiimp_file_operation_request_lstat_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_STAT:
	d = iiimp_file_operation_request_stat_unpack(data_s, nbyte,
						     ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_ACCESS:
	d = iiimp_file_operation_request_access_unpack(data_s, nbyte,
						       ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_CLOSEDIR:
	d = iiimp_file_operation_request_closedir_unpack(data_s, nbyte,
							 ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_OPENDIR:
	d = iiimp_file_operation_request_opendir_unpack(data_s, nbyte,
							ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_READDIR:
	d = iiimp_file_operation_request_readdir_unpack(data_s, nbyte,
							ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_REWINDDIR:
	d = iiimp_file_operation_request_rewinddir_unpack(data_s, nbyte,
							  ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_SEEKDIR:
	d = iiimp_file_operation_request_seekdir_unpack(data_s, nbyte,
							ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_TELLDIR:
	d = iiimp_file_operation_request_telldir_unpack(data_s, nbyte,
							ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_MKDIR:
	d = iiimp_file_operation_request_mkdir_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;
    case IIIMP_FILE_OPERATION_RMDIR:
	d = iiimp_file_operation_request_rmdir_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_FCNTL:
	d = iiimp_file_operation_request_fcntl_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;

    default:
	d = NULL;
	break;
    }
    return d;
}


IIIMP_file_operation_result *
iiimp_file_operation_result_simple_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_result *	d;
    size_t				rest;
    const uchar_t *			p;
    int					error_number;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_result *)
	malloc(sizeof (IIIMP_file_operation_result));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.simple.return_value, rest, p, data_s->byte_swap);
    GET32(error_number, rest, p, data_s->byte_swap);
    d->value.simple.error_number =
	iiimp_file_operation_error_number_i2l(error_number);

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_result *
iiimp_file_operation_result_void_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_result *	d;
    size_t				rest;
    const uchar_t *			p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_result *)
	malloc(sizeof (IIIMP_file_operation_result));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return d;
}


/*
 * iiimp_file_operation_result_close_unpack()
 */

/*
 * iiimp_file_operation_result_creat_unpack()
 */

/*
 * iiimp_file_operation_result_ftruncate_unpack()
 */

/*
 * iiimp_file_operation_result_lseek_unpack()
 */

/*
 * iiimp_file_operation_result_open_unpack()
 */



IIIMP_file_operation_result *
iiimp_file_operation_result_read_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_result *	d;
    size_t				rest;
    const uchar_t *			p;
    int					error_number;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_result *)
	malloc(sizeof (IIIMP_file_operation_result));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(error_number, rest, p, data_s->byte_swap);
    d->value.read.error_number =
	iiimp_file_operation_error_number_i2l(error_number);
    GET32(d->value.read.value.nbyte, rest, p, data_s->byte_swap);
    if (rest < d->value.read.value.nbyte) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }
    if (0 < d->value.read.value.nbyte) {
	d->value.read.value.ptr =
	    (uchar_t *)malloc(d->value.read.value.nbyte);
	(void)memcpy(d->value.read.value.ptr, p, d->value.read.value.nbyte);
	p += (d->value.read.value.nbyte + PAD(d->value.read.value.nbyte));
	rest -= (d->value.read.value.nbyte + PAD(d->value.read.value.nbyte));

    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


/*
 * iiimp_file_operation_result_truncate_unpack()
 */

/*
 * iiimp_file_operation_result_write_unpack()
 */

/*
 * iiimp_file_operation_result_link_unpack()
 */


IIIMP_file_operation_result *
iiimp_file_operation_result_readlink_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_result *	d;
    size_t				rest;
    const uchar_t *			p;
    int					error_number;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_result *)
	malloc(sizeof (IIIMP_file_operation_result));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.readlink.return_value, rest, p, data_s->byte_swap);
    GET32(error_number, rest, p, data_s->byte_swap);
    d->value.readlink.error_number =
	iiimp_file_operation_error_number_i2l(error_number);
    d->value.readlink.path =iiimp_string_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.readlink.path) {
	iiimp_string_delete(data_s, d->value.readlink.path);
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


/*
 * iiimp_file_operation_result_rename_unpack()
 */

/*
 * iiimp_file_operation_result_symlink_unpack()
 */

/*
 * iiimp_file_operation_result_unlink_unpack()
 */

/*
 * iiimp_file_operation_result_chmod_unpack()
 */

/*
 * iiimp_file_operation_result_chown_unpack()
 */

/*
 * iiimp_file_operation_result_fchmod_unpack()
 */

/*
 * iiimp_file_operation_result_fchown_unpack()
 */

/*
 * iiimp_file_operation_result_lchown_unpack()
 */

/*
 * iiimp_file_operation_result_utime_unpack()
 */

/*
 * iiimp_file_operation_result_utimes_unpack()
 */


IIIMP_file_operation_result *
iiimp_file_operation_result_stat_common_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_result *	d;
    size_t				rest;
    const uchar_t *			p;
    int					error_number;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_result *)
	malloc(sizeof (IIIMP_file_operation_result));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.stat.return_value, rest, p, data_s->byte_swap);
    GET32(error_number, rest, p, data_s->byte_swap);
    d->value.stat.error_number =
	iiimp_file_operation_error_number_i2l(error_number);
    d->value.stat.stat =
	iiimp_file_operation_data_stat_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.stat.stat) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}

/*
 * iiimp_file_operation_result_fstat_unpack()
 */

/*
 * iiimp_file_operation_result_lstat_unpack()
 */

/*
 * iiimp_file_operation_result_stat_unpack()
 */

/*
 * iiimp_file_operation_result_access_unpack()
 */

/*
 * iiimp_file_operation_result_closedir_unpack()
 */

/*
 * iiimp_file_operation_result_opendir_unpack()
 */


IIIMP_file_operation_result *
iiimp_file_operation_result_readdir_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_result *	d;
    size_t				rest;
    const uchar_t *			p;
    int					error_number;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_result *)
	malloc(sizeof (IIIMP_file_operation_result));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(error_number, rest, p, data_s->byte_swap);
    d->value.readdir.error_number =
	iiimp_file_operation_error_number_i2l(error_number);
    d->value.readdir.dir =
	iiimp_file_operation_data_dir_unpack(data_s, &rest, &p, rest);
    if (NULL == d->value.readdir.dir) {
	free(d);
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


/*
 * iiimp_file_operation_result_rewinddir_unpack()
 */

/*
 * iiimp_file_operation_result_seekdir_unpack()
 */

/*
 * iiimp_file_operation_result_telldir_unpack()
 */

/*
 * iiimp_file_operation_result_mkdir_unpack()
 */

/*
 * iiimp_file_operation_result_rmdir_unpack()
 */

IIIMP_file_operation_result *
iiimp_file_operation_result_fcntl_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_result *	d;
    size_t				rest;
    const uchar_t *			p;
    int					error_number;
    int					cmd;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) ||  (rest < (4 + 4 + 4 + 4 + 4))) {
	data_s->status = IIIMP_DATA_INVALID;
	return NULL;
    }

    d = (IIIMP_file_operation_result *)
	malloc(sizeof (IIIMP_file_operation_result));
    if (NULL == d) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }


    GET32(d->type, rest, p, data_s->byte_swap);
    GET32(d->value_nbyte, rest, p, data_s->byte_swap);
    GET32(d->value.fcntl.return_value, rest, p, data_s->byte_swap);
    GET32(error_number, rest, p, data_s->byte_swap);
    d->value.fcntl.error_number =
	iiimp_file_operation_error_number_i2l(error_number);
    GET32(cmd, rest, p, data_s->byte_swap);
    d->value.fcntl.cmd = iiimp_file_operation_fcntl_cmd_l2i(cmd);


    switch (d->value.fcntl.cmd) {
    case F_SETFD:
	d->value.fcntl.arg =
	    iiimp_file_operation_data_fcntl_setfd_unpack(data_s, &rest,
							 &p, rest);
	break;
    case F_SETFL:
	d->value.fcntl.arg =
	    iiimp_file_operation_data_fcntl_setfl_unpack(data_s, &rest,
							 &p, rest);
	break;
    case F_GETLK:
    case F_SETLK:
    case F_SETLKW:
	d->value.fcntl.arg =
	    iiimp_file_operation_data_fcntl_flock_unpack(data_s, &rest,
							 &p, rest);
	break;
    default:
	d->value.fcntl.arg = NULL;
	break;
    }

    *nbyte = rest;
    *ptr = p;

    return d;
}


IIIMP_file_operation_result *
iiimp_file_operation_result_unpack(
    IIIMP_data_s *	data_s,
    size_t *		nbyte,
    const uchar_t **	ptr,
    size_t		nbyte_max)
{
    IIIMP_file_operation_result *	d;
    size_t				rest;
    const uchar_t *			p;
    int					type;

    rest = nbyte_max;
    p = *ptr;

    GET32(type, rest, p, data_s->byte_swap);

    switch(type) {
    case IIIMP_FILE_OPERATION_CLOSE:
    case IIIMP_FILE_OPERATION_CREAT:
    case IIIMP_FILE_OPERATION_FTRUNCATE:
    case IIIMP_FILE_OPERATION_LSEEK:
    case IIIMP_FILE_OPERATION_OPEN:
	d = iiimp_file_operation_result_simple_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_READ:
	d = iiimp_file_operation_result_read_unpack(data_s, nbyte,
						    ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_TRUNCATE:
    case IIIMP_FILE_OPERATION_WRITE:
    case IIIMP_FILE_OPERATION_LINK:
	d = iiimp_file_operation_result_simple_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_READLINK:
	d = iiimp_file_operation_result_readlink_unpack(data_s, nbyte,
							ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_RENAME:
    case IIIMP_FILE_OPERATION_SYMLINK:
    case IIIMP_FILE_OPERATION_UNLINK:
    case IIIMP_FILE_OPERATION_CHMOD:
    case IIIMP_FILE_OPERATION_CHOWN:
    case IIIMP_FILE_OPERATION_FCHMOD:
    case IIIMP_FILE_OPERATION_FCHOWN:
    case IIIMP_FILE_OPERATION_LCHOWN:
    case IIIMP_FILE_OPERATION_UTIME:
    case IIIMP_FILE_OPERATION_UTIMES:
	d = iiimp_file_operation_result_simple_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_FSTAT:
    case IIIMP_FILE_OPERATION_LSTAT:
    case IIIMP_FILE_OPERATION_STAT:
	d = iiimp_file_operation_result_stat_common_unpack(data_s, nbyte,
							   ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_ACCESS:
    case IIIMP_FILE_OPERATION_CLOSEDIR:
    case IIIMP_FILE_OPERATION_OPENDIR:
	d = iiimp_file_operation_result_simple_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_READDIR:
	d = iiimp_file_operation_result_readdir_unpack(data_s, nbyte,
						       ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_REWINDDIR:
    case IIIMP_FILE_OPERATION_SEEKDIR:
	d = iiimp_file_operation_result_void_unpack(data_s, nbyte,
						    ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_TELLDIR:
    case IIIMP_FILE_OPERATION_MKDIR:
    case IIIMP_FILE_OPERATION_RMDIR:
	d = iiimp_file_operation_result_simple_unpack(data_s, nbyte,
						      ptr, nbyte_max);
	break;

    case IIIMP_FILE_OPERATION_FCNTL:
	d = iiimp_file_operation_result_fcntl_unpack(data_s, nbyte,
						     ptr, nbyte_max);
	break;

    default:
	d = NULL;
	break;
    }
    return d;
}


static const char *
fcntl_cmd_setfd_flags_string(
    int	flags)
{
    switch (flags) {
    case FD_CLOEXEC:
	return "FD_CLOEXEC";
    default:
	return "UNKNOWN";
    }
}


static void
iiimp_file_operation_data_fcntl_setfd_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_fcntl *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "%s",
		  fcntl_cmd_setfd_flags_string(m->value.file_descriptor_flags));
}


static void
iiimp_file_operation_data_fcntl_setfl_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_fcntl *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "%d",
		  m->value.file_status_flags);
}


static const char *
fcntl_cmd_flock_type_string(
    int	type)
{
    switch (type) {
    case F_RDLCK:
	return "F_RDLCK";
    case F_WRLCK:
	return "F_WRLCK";
    case F_UNLCK:
	return "F_UNLCK";
    case F_UNLKSYS:
	return "F_UNLKSYS";
    default:
	return "UNKNOWN";
    }
}


static const char *
whence_string(
    int	whence)
{
    switch (whence) {
    case SEEK_SET:
	return "SEEK_SET";
    case SEEK_CUR:
	return "SEEK_CUR";
    case SEEK_END:
	return "SEEK_END";
    default:
	return "UNKNOWN";
    }
}

static void
iiimp_file_operation_data_fcntl_flock_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_fcntl *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "[%s, %s, %d, %d]",
		  fcntl_cmd_flock_type_string(m->value.flock.type),
		  whence_string(m->value.flock.whence),
		  m->value.flock.start, m->value.flock.len);
}


static void
iiimp_file_operation_data_stat_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_stat *	m)
{
    if (NULL == m) return;
}


static void
iiimp_file_operation_data_dir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_data_dir *	m)
{
    if (NULL == m) return;
}


static void
iiimp_file_operation_request_close_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tclose(%d)\n",
		  m->value.close.file_descriptor);
}


static void
iiimp_file_operation_request_creat_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tcreat(");
    iiimp_string_print(data_s, m->value.creat.path);
    (void)fprintf(data_s->print_fp, ", 0%03o)\n", m->value.creat.mode);
}


static void
iiimp_file_operation_request_ftruncate_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tftruncate(%d, %d)\n",
		  m->value.ftruncate.file_descriptor,
		  m->value.ftruncate.length);
}


static void
iiimp_file_operation_request_lseek_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tlseek(%d, %d, %s(%d)\n",
		  m->value.lseek.file_descriptor,
		  m->value.lseek.offset,
		  whence_string(m->value.lseek.whence),
		  m->value.lseek.whence);
}


static void
iiimp_file_operation_request_open_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    int		pre;
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\topen(");
    iiimp_string_print(data_s, m->value.open.path);
    (void)fprintf(data_s->print_fp, ", ");
    pre = 0;
    if (O_CREAT & m->value.open.oflag) {
        (void)fprintf(data_s->print_fp, "O_CREAT");
	pre = 1;
    }
    if (O_RDONLY & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_RDONLY");
	pre = 1;
    }
    if (O_WRONLY & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_WRONLY");
	pre = 1;
    }
    if (O_RDWR & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_RDWR");
	pre = 1;
    }
    if (O_APPEND & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_APPEND");
	pre = 1;
    }
#if defined(O_DSYNC)
    if (O_DSYNC & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_DSYNC");
	pre = 1;
    }
#endif /* O_DSYNC */
#if defined(O_EXCL)
    if (O_EXCL & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_EXCL");
	pre = 1;
    }
#endif /* O_EXCL */
#if defined(O_LARGEFILE)
    if (O_LARGEFILE & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_LARGEFILE");
	pre = 1;
    }
#endif /* O_LARGEFILE */
#if defined(O_NOCTTY)
    if (O_NOCTTY & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_NOCTTY");
	pre = 1;
    }
#endif /* O_NOCTTY */
#if defined(O_NOFOLLOW)
    if (O_NOFOLLOW & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_NOFOLLOW");
	pre = 1;
    }
#endif /* O_NOFOLLOW */
#if defined(O_NOLINKS)
    if (O_NOLINKS & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_NOLINKS");
	pre = 1;
    }
#endif /* O_NOLINKS */
#if defined(O_NONBLOCK)
    if (O_NONBLOCK & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_NONBLOCK");
	pre = 1;
    }
#endif /* O_NONBLOCK */
#if defined(O_RSYNC)
    if (O_RSYNC & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_RSYNC");
	pre = 1;
    }
#endif /* O_RSYNC */
#if defined(O_SYNC)
    if (O_SYNC & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_SYNC");
	pre = 1;
    }
#endif /* O_SYNC */
#if defined(O_TRUNC)
    if (O_TRUNC & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_TRUNC");
	pre = 1;
    }
#endif /* O_TRUNC */
#if defined(O_XATTR)
    if (O_XATTR & m->value.open.oflag) {
	if (1 == pre) (void)fprintf(data_s->print_fp, "|");
        (void)fprintf(data_s->print_fp, "O_XATTR");
	pre = 1;
    }
#endif /* O_XATTR */
    (void)fprintf(data_s->print_fp, ", 0%3o)\n", m->value.open.mode);
}


static void
iiimp_file_operation_request_read_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tread(%d, buf, %d)\n",
		  m->value.read.file_descriptor, m->value.read.nbyte);
}


static void
iiimp_file_operation_request_truncate_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\ttruncate(");
    iiimp_string_print(data_s, m->value.truncate.path);
    (void)fprintf(data_s->print_fp, ", %d)\n", m->value.truncate.length);
}


static void
iiimp_file_operation_request_write_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    int	i;
    int	max;
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\twrite(%d, buf, %d)\n",
		  m->value.write.file_descriptor, m->value.write.value.nbyte);
    if (0 < m->value.write.value.nbyte) {
	(void)fputc('\t', data_s->print_fp);
	max = m->value.write.value.nbyte;
	max = ((256 < max) ? 256 : max);
	for (i = 0; i < max; i++) {
	    (void)fprintf(data_s->print_fp, "%02x ",
			  *(m->value.write.value.ptr + i));
	}
	(void)fputc('\n', data_s->print_fp);
    }
}


static void
iiimp_file_operation_request_link_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tlink(");
    iiimp_string_print(data_s, m->value.link.existing_path);
    (void)fprintf(data_s->print_fp, ", ");
    iiimp_string_print(data_s, m->value.link.new_path);
    (void)fprintf(data_s->print_fp, ")\n");
}


static void
iiimp_file_operation_request_readlink_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\treadlink(");
    iiimp_string_print(data_s, m->value.readlink.path);
    (void)fprintf(data_s->print_fp, ", buf, bufsiz)\n");
}


static void
iiimp_file_operation_request_rename_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\trename(");
    iiimp_string_print(data_s, m->value.rename.old_path);
    (void)fprintf(data_s->print_fp, ",");
    iiimp_string_print(data_s, m->value.rename.new_path);
    (void)fprintf(data_s->print_fp, ")\n");
}


static void
iiimp_file_operation_request_symlink_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tsymlink(");
    iiimp_string_print(data_s, m->value.symlink.existing_path);
    (void)fprintf(data_s->print_fp, ",");
    iiimp_string_print(data_s, m->value.symlink.new_path);
    (void)fprintf(data_s->print_fp, ")\n");
}


static void
iiimp_file_operation_request_unlink_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tunlink(");
    iiimp_string_print(data_s, m->value.unlink.path);
    (void)fprintf(data_s->print_fp, ")\n");
}


static void
iiimp_file_operation_request_chmod_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tchmod(");
    iiimp_string_print(data_s, m->value.chmod.path);
    (void)fprintf(data_s->print_fp, ", 0%3o)\n", m->value.chmod.mode);
}


static void
iiimp_file_operation_request_chown_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tchown(");
    iiimp_string_print(data_s, m->value.chown.path);
    (void)fprintf(data_s->print_fp, ", %d (", m->value.chown.user_id);
    iiimp_string_print(data_s, m->value.chown.user_name);
    (void)fprintf(data_s->print_fp, "), %d (", m->value.chown.group_id);
    iiimp_string_print(data_s, m->value.chown.group_name);
    (void)fprintf(data_s->print_fp, "))\n");
}


static void
iiimp_file_operation_request_fchmod_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tfchmod(%d, 0%3o)\n",
		  m->value.fchmod.file_descriptor, m->value.fchmod.mode);
}


static void
iiimp_file_operation_request_fchown_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tfchown(%d, %d (",
		  m->value.fchown.file_descriptor,
		  m->value.fchown.user_id);
    iiimp_string_print(data_s, m->value.fchown.user_name);
    (void)fprintf(data_s->print_fp, "), %d (", m->value.fchown.group_id);
    iiimp_string_print(data_s, m->value.fchown.group_name);
    (void)fprintf(data_s->print_fp, "))\n");
}


static void
iiimp_file_operation_request_lchown_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tlchown(");
    iiimp_string_print(data_s, m->value.lchown.path);
    (void)fprintf(data_s->print_fp, ", ");
    (void)fprintf(data_s->print_fp, ", %d (", m->value.lchown.user_id);
    iiimp_string_print(data_s, m->value.lchown.user_name);
    (void)fprintf(data_s->print_fp, "), %d (", m->value.lchown.group_id);
    iiimp_string_print(data_s, m->value.lchown.group_name);
    (void)fprintf(data_s->print_fp, "))\n");
}


static void
iiimp_file_operation_request_utime_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tutime(");
    iiimp_string_print(data_s, m->value.utime.path);
    (void)fprintf(data_s->print_fp, ", [%d, %d])\n",
		  m->value.utime.utimbuf_data.actime,
		  m->value.utime.utimbuf_data.modtime);
}


static void
iiimp_file_operation_request_utimes_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tutimes(");
    iiimp_string_print(data_s, m->value.utime.path);
    (void)fprintf(data_s->print_fp, ", [[%d, %d], [%d, %d]])\n",
		  m->value.utimes.timeval_data.tv_sec0,
		  m->value.utimes.timeval_data.tv_usec0,
		  m->value.utimes.timeval_data.tv_sec1,
		  m->value.utimes.timeval_data.tv_usec1);
}


static void
iiimp_file_operation_request_fstat_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tfstat(%d, buf)\n",
		  m->value.fstat.file_descriptor);
}


static void
iiimp_file_operation_request_lstat_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tlstat(");
    iiimp_string_print(data_s, m->value.lstat.path);
    (void)fprintf(data_s->print_fp, ", buf)\n");
}


static void
iiimp_file_operation_request_stat_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tstat(");
    iiimp_string_print(data_s, m->value.stat.path);
    (void)fprintf(data_s->print_fp, ", buf)\n");
}


static const char *
access_amode_string(
    int	amode)
{
    switch (amode) {
    case F_OK:
	return "F_OK";
    case X_OK:
	return "X_OK";
    case W_OK:
	return "W_OK";
    case R_OK:
	return "R_OK";
    default:
	return "UNKNOWN";
    }
}


static void
iiimp_file_operation_request_access_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\taccess(");
    iiimp_string_print(data_s, m->value.access.path);
    (void)fprintf(data_s->print_fp, ", %s(%d))\n",
		  access_amode_string(m->value.access.amode),
		  m->value.access.amode);
}


static void
iiimp_file_operation_request_closedir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tclosedir(0x%8x)\n",
		  m->value.closedir.directory_handle);
}


static void
iiimp_file_operation_request_opendir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\topendir(");
    iiimp_string_print(data_s, m->value.opendir.dirname);
    (void)fprintf(data_s->print_fp, ")\n");
}


static void
iiimp_file_operation_request_readdir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\treaddir(0x%8x)\n",
		  m->value.readdir.directory_handle);
}


static void
iiimp_file_operation_request_rewinddir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\trewinddir(0x%8x)\n",
		  m->value.rewinddir.directory_handle);
}


static void
iiimp_file_operation_request_seekdir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tseekdir(0x%8x, %d)\n",
		  m->value.seekdir.directory_handle, m->value.seekdir.loc);
}


static void
iiimp_file_operation_request_telldir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\ttelldir(0x%8x)\n",
		  m->value.rewinddir.directory_handle);
}


static void
iiimp_file_operation_request_mkdir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tmkdir(");
    iiimp_string_print(data_s, m->value.mkdir.dirname);
    (void)fprintf(data_s->print_fp, ", 0%03o)\n", m->value.mkdir.mode);
}


static void
iiimp_file_operation_request_rmdir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\trmdir(");
    iiimp_string_print(data_s, m->value.rmdir.dirname);
    (void)fprintf(data_s->print_fp, ")\n");
}


static const char *
fcntl_cmd_string(
    int	cmd)
{
    switch (cmd) {
    case F_SETFD:
	return ("F_SETFD");
    case F_SETFL:
	return ("F_SETFL");
    case F_GETLK:
	return ("F_GETLK");
    case F_SETLK:
	return ("F_SETLK");
    case F_SETLKW:
	return ("F_SETLKW");
    default:
	return ("UNKNOWN");
    }
}


static void
iiimp_file_operation_request_fcntl_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tfcntl(%d, %s, ",
		  m->value.fcntl.file_descriptor,
		  fcntl_cmd_string(m->value.fcntl.cmd));

    switch (m->value.fcntl.cmd) {
    case F_SETFD:
	iiimp_file_operation_data_fcntl_setfd_print(data_s, m->value.fcntl.arg);
	break;
    case F_SETFL:
	iiimp_file_operation_data_fcntl_setfl_print(data_s, m->value.fcntl.arg);
	break;
    case F_GETLK:
    case F_SETLK:
    case F_SETLKW:
	iiimp_file_operation_data_fcntl_flock_print(data_s, m->value.fcntl.arg);
	break;
    default:
	(void)fprintf(data_s->print_fp, "UNKNOWN");
	break;
    }
    (void)fprintf(data_s->print_fp, ")\n");
}


void
iiimp_file_operation_request_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_request *	data)
{
    if (NULL == data) return;

    switch(data->type) {
    case IIIMP_FILE_OPERATION_CLOSE:
	iiimp_file_operation_request_close_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_CREAT:
	iiimp_file_operation_request_creat_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_FTRUNCATE:
	iiimp_file_operation_request_ftruncate_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_LSEEK:
	iiimp_file_operation_request_lseek_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_OPEN:
	iiimp_file_operation_request_open_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_READ:
	iiimp_file_operation_request_read_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_TRUNCATE:
	iiimp_file_operation_request_truncate_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_WRITE:
	iiimp_file_operation_request_write_print(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_LINK:
	iiimp_file_operation_request_link_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_READLINK:
	iiimp_file_operation_request_readlink_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_RENAME:
	iiimp_file_operation_request_rename_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_SYMLINK:
	iiimp_file_operation_request_symlink_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_UNLINK:
	iiimp_file_operation_request_unlink_print(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_CHMOD:
	iiimp_file_operation_request_chmod_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_CHOWN:
	iiimp_file_operation_request_chown_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_FCHMOD:
	iiimp_file_operation_request_fchmod_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_FCHOWN:
	iiimp_file_operation_request_fchown_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_LCHOWN:
	iiimp_file_operation_request_lchown_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_UTIME:
	iiimp_file_operation_request_utime_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_UTIMES:
	iiimp_file_operation_request_utimes_print(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_FSTAT:
	iiimp_file_operation_request_fstat_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_LSTAT:
	iiimp_file_operation_request_lstat_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_STAT:
	iiimp_file_operation_request_stat_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_ACCESS:
	iiimp_file_operation_request_access_print(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_CLOSEDIR:
	iiimp_file_operation_request_closedir_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_OPENDIR:
	iiimp_file_operation_request_opendir_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_READDIR:
	iiimp_file_operation_request_readdir_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_REWINDDIR:
	iiimp_file_operation_request_rewinddir_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_SEEKDIR:
	iiimp_file_operation_request_seekdir_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_TELLDIR:
	iiimp_file_operation_request_telldir_print(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_MKDIR:
	iiimp_file_operation_request_mkdir_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_RMDIR:
	iiimp_file_operation_request_rmdir_print(data_s, data);
	break;

    case IIIMP_FILE_OPERATION_FCNTL:
	iiimp_file_operation_request_fcntl_print(data_s, data);
	break;

    default:
	break;
    }
    return;
}


static const char *
error_number_string(
    int	error_number)
{
    switch (error_number) {
    case 0:
	return "NO_ERROR";
#if defined(EPERM)
    case EPERM:
	return "EPERM";
#endif /* EPERM */
#if defined(ENOENT)
    case ENOENT:
	return "ENOENT";
#endif /* ENOENT */
#if defined(EINTR)
    case EINTR:
	return "EINTR";
#endif /* EINTR */
#if defined(EIO)
    case EIO:
	return "EIO";
#endif /* EIO */
#if defined(ENXIO)
    case ENXIO:
	return "ENXIO";
#endif /* ENXIO */
#if defined(EBADF)
    case EBADF:
	return "EBADF";
#endif /* EBADF */
#if defined(EAGAIN)
    case EAGAIN:
	return "EAGAIN";
#endif /* EAGAIN */
#if defined(ENOMEM)
    case ENOMEM:
	return "ENOMEM";
#endif /* ENOMEM */
#if defined(EACCES)
    case EACCES:
	return "EACCES";
#endif /* EACCES */
#if defined(EFAULT)
    case EFAULT:
	return "EFAULT";
#endif /* EFAULT */
#if defined(EBUSY)
    case EBUSY:
	return "EBUSY";
#endif /* EBUSY */
#if defined(EEXIST)
    case EEXIST:
	return "EEXIST";
#endif /* EEXIST */
#if defined(EXDEV)
    case EXDEV:
	return "EXDEV";
#endif /* EXDEV */
#if defined(ENOTDIR)
    case ENOTDIR:
	return "ENOTDIR";
#endif /* ENOTDIR */
#if defined(EISDIR)
    case EISDIR:
	return "EISDIR";
#endif /* EISDIR */
    case EINVAL:
	return "EINVAL";
#if defined(ENFILE)
    case ENFILE:
	return "ENFILE";
#endif /* ENFILE */
#if defined(EMFILE)
    case EMFILE:
	return "EMFILE";
#endif /* EMFILE */
#if defined(ETXTBSY)
    case ETXTBSY:
	return "ETXTBSY";
#endif /* ETXTBSY */
#if defined(EFBIG)
    case EFBIG:
	return "EFBIG";
#endif /* EFBIG */
#if defined(ENOSPC)
    case ENOSPC:
	return "ENOSPC";
#endif /* ENOSPC */
#if defined(EROFS)
    case EROFS:
	return "EROFS";
#endif /* EROFS */
#if defined(EMLINK)
    case EMLINK:
	return "EMLINK";
#endif /* EMLINK */
#if defined(EDEADLK)
    case EDEADLK:
	return "EDEADLK";
#endif /* EDEADLK */
#if defined(ENOLCK)
    case ENOLCK:
	return "ENOLCK";
#endif /* ENOLCK */
#if defined(EDQUOT)
    case EDQUOT:
	return "EDQUOT";
#endif /* EDQUOT */
#if defined(ENAMETOOLONG)
    case ENAMETOOLONG:
	return "ENAMETOOLONG";
#endif /* ENAMETOOLONG */
#if defined(EOVERFLOW)
    case EOVERFLOW:
	return "EOVERFLOW";
#endif /* EOVERFLOW */
#if defined(ENOSYS)
    case ENOSYS:
	return "ENOSYS";
#endif /* ENOSYS */
#if defined(ELOOP)
    case ELOOP:
	return "ELOOP";
#endif /* ELOOP */
    default:
	return "UNKNOWN_ERROR";
    }
}


static void
iiimp_file_operation_result_simple_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    const char *			name)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\t%s() = %d (%s(%d))\n", name,
		  m->value.simple.return_value,
		  error_number_string(m->value.simple.error_number),
		  m->value.simple.error_number);
}


static void
iiimp_file_operation_result_void_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    const char *			name)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\t%s() = (no value)\n", name);
}


/*
 * iiimp_file_operation_result_close_print()
 */

/*
 * iiimp_file_operation_result_creat_print()
 */

/*
 * iiimp_file_operation_result_ftruncate_print()
 */

/*
 * iiimp_file_operation_result_lseek_print()
 */

/*
 * iiimp_file_operation_result_open_print()
 */


static void
iiimp_file_operation_result_read_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m)
{
    int	i;
    int	max;
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tread() = %d (%s(%d))\n",
		  m->value.read.value.nbyte,
		  error_number_string(m->value.read.error_number),
		  m->value.read.error_number);
    if (0 < m->value.read.value.nbyte) {
	(void)fputc('\t', data_s->print_fp);
	max = m->value.read.value.nbyte;
	max = ((256 < max) ? 256 : max);
	for (i = 0; i < max; i++) {
	    (void)fprintf(data_s->print_fp, "%02x ",
			  *(m->value.read.value.ptr + i));
	}
	(void)fputc('\n', data_s->print_fp);
    }
}


/*
 * iiimp_file_operation_result_truncate_print()
 */

/*
 * iiimp_file_operation_result_write_print()
 */

/*
 * iiimp_file_operation_result_link_print()
 */


static void
iiimp_file_operation_result_readlink_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\treadlink(");
    iiimp_string_print(data_s, m->value.readlink.path);
    (void)fprintf(data_s->print_fp, ") = %d (%s(%d))\n",
		  m->value.readlink.return_value,
		  error_number_string(m->value.readlink.error_number),
		  m->value.readlink.error_number);
}


/*
 * iiimp_file_operation_result_rename_print()
 */

/*
 * iiimp_file_operation_result_symlink_print()
 */

/*
 * iiimp_file_operation_result_unlink_print()
 */

/*
 * iiimp_file_operation_result_chmod_print()
 */

/*
 * iiimp_file_operation_result_chown_print()
 */

/*
 * iiimp_file_operation_result_fchmod_print()
 */

/*
 * iiimp_file_operation_result_fchown_print()
 */

/*
 * iiimp_file_operation_result_lchown_print()
 */

/*
 * iiimp_file_operation_result_utime_print()
 */

/*
 * iiimp_file_operation_result_utimes_print()
 */


static void
iiimp_file_operation_result_stat_common_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m,
    const char *			name)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\t%s() = %d (%s(%d))\n", name,
		  m->value.stat.return_value,
		  error_number_string(m->value.stat.error_number),
		  m->value.stat.error_number);

    (void)fputc('\t', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "mode=0%3o ", m->value.stat.stat->mode);
    (void)fprintf(data_s->print_fp, "ino=%d ", m->value.stat.stat->ino);
    (void)fprintf(data_s->print_fp, "dev=%04x ", m->value.stat.stat->dev);
    (void)fprintf(data_s->print_fp, "ndev=%04x ", m->value.stat.stat->rdev);
    (void)fputc('\n', data_s->print_fp);

    (void)fputc('\t', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "user_name= ");
    iiimp_string_print(data_s, m->value.stat.stat->user_name);
    (void)fprintf(data_s->print_fp, "(%d) ", m->value.stat.stat->uid);
    iiimp_string_print(data_s, m->value.stat.stat->group_name);
    (void)fprintf(data_s->print_fp, "(%d) ", m->value.stat.stat->gid);
    (void)fputc('\n', data_s->print_fp);

    (void)fputc('\t', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "size=%d ", m->value.stat.stat->size);
    (void)fputc('\n', data_s->print_fp);

    (void)fputc('\t', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "atime.tv_sec=%d ",
		  m->value.stat.stat->atime.tv_sec);
    (void)fprintf(data_s->print_fp, "atime.tv_nsec=%d ",
		  m->value.stat.stat->atime.tv_nsec);
    (void)fputc('\n', data_s->print_fp);

    (void)fputc('\t', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "mtime.tv_sec=%d ",
		  m->value.stat.stat->mtime.tv_sec);
    (void)fprintf(data_s->print_fp, "mtime.tv_nsec=%d ",
		  m->value.stat.stat->mtime.tv_nsec);
    (void)fputc('\n', data_s->print_fp);

    (void)fputc('\t', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "ctime.tv_sec=%d ",
		  m->value.stat.stat->ctime.tv_sec);
    (void)fprintf(data_s->print_fp, "ctime.tv_nsec=%d ",
		  m->value.stat.stat->ctime.tv_nsec);
    (void)fputc('\n', data_s->print_fp);

    (void)fputc('\t', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "blksize=%d ", m->value.stat.stat->blksize);
    (void)fprintf(data_s->print_fp, "blocks=%d", m->value.stat.stat->blocks);
    (void)fputc('\n', data_s->print_fp);
}


/*
 * iiimp_file_operation_result_fstat_print()
 */

/*
 * iiimp_file_operation_result_lstat_print()
 */

/*
 * iiimp_file_operation_result_stat_print()
 */

/*
 * iiimp_file_operation_result_access_print()
 */

/*
 * iiimp_file_operation_result_closedir_print()
 */


static void
iiimp_file_operation_result_opendir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\topendir() = 0x%08x (%s(%d))\n",
		  m->value.opendir.return_value,
		  error_number_string(m->value.opendir.error_number),
		  m->value.opendir.error_number);
}


static void
iiimp_file_operation_result_readdir_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\treaddir() = ");
    iiimp_string_print(data_s, m->value.readdir.dir->name);
    (void)fprintf(data_s->print_fp, " (%s(%d))\n",
		  error_number_string(m->value.readdir.error_number),
		  m->value.readdir.error_number);
}


/*
 * iiimp_file_operation_result_rewinddir_print()
 */

/*
 * iiimp_file_operation_result_seekdir_print()
 */

/*
 * iiimp_file_operation_result_telldir_print()
 */

/*
 * iiimp_file_operation_result_mkdir_print()
 */

/*
 * iiimp_file_operation_result_rmdir_print()
 */


static void
iiimp_file_operation_result_fcntl_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	m)
{
    if (NULL == m) return;
    (void)fprintf(data_s->print_fp, "\tfcntl(*, %s, ",
		  fcntl_cmd_string(m->value.fcntl.cmd));

    switch (m->value.fcntl.cmd) {
    case F_SETFD:
	iiimp_file_operation_data_fcntl_setfd_print(data_s, m->value.fcntl.arg);
	break;
    case F_SETFL:
	iiimp_file_operation_data_fcntl_setfl_print(data_s, m->value.fcntl.arg);
	break;
    case F_GETLK:
    case F_SETLK:
    case F_SETLKW:
	iiimp_file_operation_data_fcntl_flock_print(data_s, m->value.fcntl.arg);
	break;
    default:
	(void)fprintf(data_s->print_fp, "UNKNOWN");
	break;
    }

    (void)fprintf(data_s->print_fp, ") = %d (%s(%d))\n",
		  m->value.fcntl.return_value,
		  error_number_string(m->value.fcntl.error_number),
		  m->value.fcntl.error_number);
}


void
iiimp_file_operation_result_print(
    IIIMP_data_s *			data_s,
    IIIMP_file_operation_result *	data)
{
    if (NULL == data) return;

    switch(data->type) {
    case IIIMP_FILE_OPERATION_CLOSE:
	iiimp_file_operation_result_simple_print(data_s, data, "close");
	break;
    case IIIMP_FILE_OPERATION_CREAT:
	iiimp_file_operation_result_simple_print(data_s, data, "creat");
	break;
    case IIIMP_FILE_OPERATION_FTRUNCATE:
	iiimp_file_operation_result_simple_print(data_s, data, "ftruncate");
	break;
    case IIIMP_FILE_OPERATION_LSEEK:
	iiimp_file_operation_result_simple_print(data_s, data, "lseek");
	break;
    case IIIMP_FILE_OPERATION_OPEN:
	iiimp_file_operation_result_simple_print(data_s, data, "open");
	break;
    case IIIMP_FILE_OPERATION_READ:
	iiimp_file_operation_result_read_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_TRUNCATE:
	iiimp_file_operation_result_simple_print(data_s, data, "truncate");
	break;
    case IIIMP_FILE_OPERATION_WRITE:
	iiimp_file_operation_result_simple_print(data_s, data, "write");
	break;

    case IIIMP_FILE_OPERATION_LINK:
	iiimp_file_operation_result_simple_print(data_s, data, "link");
	break;
    case IIIMP_FILE_OPERATION_READLINK:
	iiimp_file_operation_result_readlink_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_RENAME:
	iiimp_file_operation_result_simple_print(data_s, data, "rename");
	break;
    case IIIMP_FILE_OPERATION_SYMLINK:
	iiimp_file_operation_result_simple_print(data_s, data, "symlink");
	break;
    case IIIMP_FILE_OPERATION_UNLINK:
	iiimp_file_operation_result_simple_print(data_s, data, "unlink");
	break;

    case IIIMP_FILE_OPERATION_CHMOD:
	iiimp_file_operation_result_simple_print(data_s, data, "chmod");
	break;
    case IIIMP_FILE_OPERATION_CHOWN:
	iiimp_file_operation_result_simple_print(data_s, data, "chown");
	break;
    case IIIMP_FILE_OPERATION_FCHMOD:
	iiimp_file_operation_result_simple_print(data_s, data, "fchmod");
	break;
    case IIIMP_FILE_OPERATION_FCHOWN:
	iiimp_file_operation_result_simple_print(data_s, data, "fchown");
	break;
    case IIIMP_FILE_OPERATION_LCHOWN:
	iiimp_file_operation_result_simple_print(data_s, data, "lchown");
	break;
    case IIIMP_FILE_OPERATION_UTIME:
	iiimp_file_operation_result_simple_print(data_s, data, "utime");
	break;
    case IIIMP_FILE_OPERATION_UTIMES:
	iiimp_file_operation_result_simple_print(data_s, data, "utimes");
	break;

    case IIIMP_FILE_OPERATION_FSTAT:
	iiimp_file_operation_result_stat_common_print(data_s, data, "fstat");
	break;
    case IIIMP_FILE_OPERATION_LSTAT:
	iiimp_file_operation_result_stat_common_print(data_s, data, "lstat");
	break;
    case IIIMP_FILE_OPERATION_STAT:
	iiimp_file_operation_result_stat_common_print(data_s, data, "stat");
	break;
    case IIIMP_FILE_OPERATION_ACCESS:
	iiimp_file_operation_result_simple_print(data_s, data, "access");
	break;

    case IIIMP_FILE_OPERATION_CLOSEDIR:
	iiimp_file_operation_result_simple_print(data_s, data, "closedir");
	break;
    case IIIMP_FILE_OPERATION_OPENDIR:
	iiimp_file_operation_result_opendir_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_READDIR:
	iiimp_file_operation_result_readdir_print(data_s, data);
	break;
    case IIIMP_FILE_OPERATION_REWINDDIR:
	iiimp_file_operation_result_void_print(data_s, data, "rewinddir");
	break;
    case IIIMP_FILE_OPERATION_SEEKDIR:
	iiimp_file_operation_result_void_print(data_s, data, "seekdir");
	break;
    case IIIMP_FILE_OPERATION_TELLDIR:
	iiimp_file_operation_result_simple_print(data_s, data, "telldir");
	break;

    case IIIMP_FILE_OPERATION_MKDIR:
	iiimp_file_operation_result_simple_print(data_s, data, "mkdir");
	break;
    case IIIMP_FILE_OPERATION_RMDIR:
	iiimp_file_operation_result_simple_print(data_s, data, "rmdir");
	break;

    case IIIMP_FILE_OPERATION_FCNTL:
	iiimp_file_operation_result_fcntl_print(data_s, data);
	break;

    default:
	break;
    }
    return;
}


/*
 * prototype version - do not use
 */
IIIMP_open *
iiimp_open_new(
    IIIMP_data_s *	        data_s,
    IIIMP_card32                ns_id,
    IIIMP_card32                flags,
    IIIMP_utf8string *	        filename)
{
    IIIMP_open *	data;
    size_t 		nbyte = 0;

    data = (IIIMP_open *) malloc(sizeof(IIIMP_open));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    nbyte = 4 + 4 + 4 + filename->nbyte;

    data->nbyte = nbyte;
    data->ns_id = ns_id;
    data->flags = flags;
    data->filename = filename;

    return data;
}

IIIMP_read *
iiimp_read_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        size)
{
    IIIMP_read *        data;
 
    data = (IIIMP_read *) malloc(sizeof(IIIMP_read));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->size = size;

    return data;
}

IIIMP_write *
iiimp_write_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        size,
    void *              object)
{
    IIIMP_write *	data;

    data = (IIIMP_write *)malloc(sizeof (IIIMP_write));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    data->nbyte = 4 + 4 + 4 + size + PAD(size);
    data->ns_id = ns_id;
    data->size = size;
    data->object = object;

    return data;
}

IIIMP_close *
iiimp_close_new(
    IIIMP_data_s *	data_s,
    IIIMP_card32        ns_id)
{
    IIIMP_close *	data;

    data = (IIIMP_close *)malloc(sizeof (IIIMP_close));
    if (NULL == data) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }
    data->nbyte = 4 + 4;
    data->ns_id = ns_id;

    return data;
}

IIIMP_seek *
iiimp_seek_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        offset,
    IIIMP_card32        whence)
{
    IIIMP_seek *     data;

    data = (IIIMP_seek *)malloc(sizeof (IIIMP_seek));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->offset = offset;
    data->whence = whence;

    return data;
}

IIIMP_opendir *
iiimp_opendir_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  dirname)
{
    IIIMP_opendir *     data;
    data = (IIIMP_opendir *)malloc(sizeof (IIIMP_opendir));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + dirname->nbyte;
    data->ns_id = ns_id;
    data->dirname = dirname;

    return data;
}

IIIMP_closedir *
iiimp_closedir_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id)
{
    IIIMP_closedir *     data;
    data = (IIIMP_closedir *)malloc(sizeof (IIIMP_closedir));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4;
    data->ns_id = ns_id;

    return data;
}

IIIMP_mkdir *
iiimp_mkdir_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        mode,
    IIIMP_utf8string *  path)
{
    IIIMP_mkdir *     data;

    data = (IIIMP_mkdir *)malloc(sizeof (IIIMP_mkdir));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->mode = mode;
    data->path = path;

    return data;
}

IIIMP_rmdir *
iiimp_rmdir_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path)
{
    IIIMP_rmdir *     data;

    data = (IIIMP_rmdir *)malloc(sizeof (IIIMP_rmdir));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;

    return data;
}

IIIMP_symlink *
iiimp_symlink_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  dest,
    IIIMP_utf8string *  src)
{
    IIIMP_symlink *     data;

    data = (IIIMP_symlink *)malloc(sizeof (IIIMP_symlink));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + dest->nbyte + src->nbyte;
    data->ns_id = ns_id;
    data->dest = dest;
    data->src = src;

    return data;
}

IIIMP_stat *
iiimp_stat_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path)
{
    IIIMP_stat *     data;

    data = (IIIMP_stat *)malloc(sizeof (IIIMP_stat));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;

    return data;
}

IIIMP_lstat *
iiimp_lstat_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path)
{
    IIIMP_lstat *     data;

    data = (IIIMP_lstat *)malloc(sizeof (IIIMP_lstat));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;

    return data;
}

IIIMP_fstat *
iiimp_fstat_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id)
{
    IIIMP_fstat *     data;

    data = (IIIMP_fstat *)malloc(sizeof (IIIMP_fstat));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4;
    data->ns_id = ns_id;

    return data;
}

IIIMP_creat *
iiimp_creat_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        mode,
    IIIMP_utf8string *  path)
{
    IIIMP_creat *     data;

    data = (IIIMP_creat *)malloc(sizeof (IIIMP_creat));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->mode = mode;
    data->path = path;

    return data;
}

IIIMP_lseek *
iiimp_lseek_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        offset,
    IIIMP_card32        whence)
{
    IIIMP_lseek *     data;

    data = (IIIMP_lseek *)malloc(sizeof (IIIMP_lseek));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->offset = offset;
    data->whence = whence;

    return data;
}

IIIMP_unlink *
iiimp_unlink_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path)
{
    IIIMP_unlink *     data;

    data = (IIIMP_unlink *)malloc(sizeof (IIIMP_unlink));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;

    return data;
}

IIIMP_rename *
iiimp_rename_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  old_name,
    IIIMP_utf8string *  new_name)
{
    IIIMP_rename *     data;

    data = (IIIMP_rename *)malloc(sizeof (IIIMP_rename));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + old_name->nbyte + new_name->nbyte;
    data->ns_id = ns_id;
    data->old_name = old_name;
    data->new_name = new_name;

    return data;
}

IIIMP_fcntl *
iiimp_fcntl_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        cmd,
    IIIMP_card32        arg)
{
    IIIMP_fcntl *     data;

    data = (IIIMP_fcntl *)malloc(sizeof (IIIMP_fcntl));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + 4;
    data->ns_id = ns_id;
    data->cmd = cmd;
    data->arg = arg;

    return data;
}

IIIMP_truncate *
iiimp_truncate_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_utf8string *  path,
    IIIMP_card32        length)
{
    IIIMP_truncate *     data;

    data = (IIIMP_truncate *)malloc(sizeof (IIIMP_truncate));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4 + path->nbyte;
    data->ns_id = ns_id;
    data->path = path;
    data->length = length;

    return data;
}

IIIMP_ftruncate *
iiimp_ftruncate_new(
    IIIMP_data_s *      data_s,
    IIIMP_card32        ns_id,
    IIIMP_card32        length)
{
    IIIMP_ftruncate *     data;
    data = (IIIMP_ftruncate *)malloc(sizeof (IIIMP_ftruncate));
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    data->nbyte = 4 + 4 + 4;
    data->ns_id = ns_id;
    data->length = length;

    return data;
}

void
iiimp_open_delete(IIIMP_data_s * data_s, IIIMP_open *m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->filename);
    free(m);
    return;
}

void
iiimp_read_delete(IIIMP_data_s * data_s, IIIMP_read * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_write_delete(IIIMP_data_s * data_s, IIIMP_write * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_close_delete(IIIMP_data_s * data_s, IIIMP_close * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_seek_delete(IIIMP_data_s * data_s, IIIMP_seek * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_opendir_delete(IIIMP_data_s * data_s, IIIMP_opendir * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->dirname);
    free(m);
    return;
}

void
iiimp_closedir_delete(IIIMP_data_s * data_s, IIIMP_closedir * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_mkdir_delete(IIIMP_data_s * data_s, IIIMP_mkdir * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_rmdir_delete(IIIMP_data_s * data_s, IIIMP_rmdir * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_symlink_delete(IIIMP_data_s * data_s, IIIMP_symlink * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->dest);
    iiimp_utf8string_delete(data_s, m->src);
    free(m);
    return;
}

void
iiimp_stat_delete(IIIMP_data_s * data_s, IIIMP_stat * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_fstat_delete(IIIMP_data_s * data_s, IIIMP_fstat * m)
{
    if (NULL == m) return;
    free(m);
    return;
}


void
iiimp_lstat_delete(IIIMP_data_s * data_s, IIIMP_lstat * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_creat_delete(IIIMP_data_s * data_s, IIIMP_creat * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_lseek_delete(IIIMP_data_s * data_s, IIIMP_lseek * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_unlink_delete(IIIMP_data_s * data_s, IIIMP_unlink * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_rename_delete(IIIMP_data_s * data_s, IIIMP_rename * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->old_name);
    iiimp_utf8string_delete(data_s, m->new_name);
    free(m);
    return;
}

void
iiimp_fcntl_delete(IIIMP_data_s * data_s, IIIMP_fcntl * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_truncate_delete(IIIMP_data_s * data_s, IIIMP_truncate * m)
{
    if (NULL == m) return;
    iiimp_utf8string_delete(data_s, m->path);
    free(m);
    return;
}

void
iiimp_ftruncate_delete(IIIMP_data_s * data_s, IIIMP_ftruncate * m)
{
    if (NULL == m) return;
    free(m);
    return;
}

void
iiimp_open_pack(
    IIIMP_data_s *	data_s,
    IIIMP_open *        m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->flags, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->filename, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_read_pack(
    IIIMP_data_s *	data_s,
    IIIMP_read *        m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->size, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_write_pack(
    IIIMP_data_s *	data_s,
    IIIMP_write *       m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;
    int                 i, count, object_size;

    object_size = m->size + PAD((m->size) % 4);
    count = (object_size / 4);

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->size, rest, p, data_s->byte_swap);

    for (i=0; i<count; i++) {
	PUTU32(*((IIIMP_card32 *)m->object + i), rest, p, data_s->byte_swap);
    }

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_close_pack(
    IIIMP_data_s *	data_s,
    IIIMP_close *       m,
    size_t *            nbyte,
    uchar_t **          ptr)
{
    size_t		rest;
    uchar_t *		p;

    rest = *nbyte;
    p = *ptr;

    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_seek_pack(
    IIIMP_data_s *      data_s,
    IIIMP_seek *        m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->offset, rest, p, data_s->byte_swap);
    PUTU32(m->whence, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_opendir_pack(
    IIIMP_data_s *      data_s,
    IIIMP_opendir *     m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->dirname, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_closedir_pack(
    IIIMP_data_s *      data_s,
    IIIMP_closedir *    m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_mkdir_pack(
    IIIMP_data_s *      data_s,
    IIIMP_mkdir *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->mode, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_rmdir_pack(
    IIIMP_data_s *      data_s,
    IIIMP_rmdir *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_symlink_pack(
    IIIMP_data_s *      data_s,
    IIIMP_symlink *     m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->dest, &rest, &p);
    iiimp_utf8string_pack(data_s, m->src, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_stat_pack(
    IIIMP_data_s *      data_s,
    IIIMP_stat *        m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_lstat_pack(
    IIIMP_data_s *      data_s,
    IIIMP_lstat *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_fstat_pack(
    IIIMP_data_s *      data_s,
    IIIMP_fstat *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_creat_pack(
    IIIMP_data_s *      data_s,
    IIIMP_creat *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->mode, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_lseek_pack(
    IIIMP_data_s *      data_s,
    IIIMP_lseek *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->offset, rest, p, data_s->byte_swap);
    PUTU32(m->whence, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_unlink_pack(
    IIIMP_data_s *      data_s,
    IIIMP_unlink *      m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_rename_pack(
    IIIMP_data_s *      data_s,
    IIIMP_rename *      m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->old_name, &rest, &p);
    iiimp_utf8string_pack(data_s, m->new_name, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_fcntl_pack(
    IIIMP_data_s *      data_s,
    IIIMP_fcntl *       m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->cmd, rest, p, data_s->byte_swap);
    PUTU32(m->arg, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_truncate_pack(
    IIIMP_data_s *      data_s,
    IIIMP_truncate *    m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->length, rest, p, data_s->byte_swap);
    iiimp_utf8string_pack(data_s, m->path, &rest, &p);

    *nbyte = rest;
    *ptr = p;

    return;
}

void
iiimp_ftruncate_pack(
    IIIMP_data_s *      data_s,
    IIIMP_ftruncate *   m,
    size_t *            nbyte,
    uchar_t **          ptr)   
{
    size_t              rest;
    uchar_t *           p;

    rest = *nbyte;
    p = *ptr;
    
    PUTU32(m->nbyte, rest, p, data_s->byte_swap);
    PUTU32(m->ns_id, rest, p, data_s->byte_swap);
    PUTU32(m->length, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return;
}

IIIMP_open *
iiimp_open_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_open *        data;
    size_t              rest;
    const uchar_t *     p;
  
    rest = nbyte_max;
    p = *ptr;
    
    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }
    
    data = (IIIMP_open *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->filename = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->flags, rest, p, data_s->byte_swap);
    data->filename = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_read *
iiimp_read_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_read *        data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_read *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->size, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_write *
iiimp_write_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_write *       data;
    IIIMP_card32 *      buf;
    int                 i, count, object_size;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    object_size = (rest - 4 - 4 - 4);
    count = (object_size / 4);

    data = (IIIMP_write *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->object = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->size, rest, p, data_s->byte_swap);

    buf = (IIIMP_card32 *)malloc(data->size);
    if (NULL == buf) {
	data_s->status = IIIMP_DATA_MALLOC_ERROR;
	return NULL;
    }

    for (i=0; i<count; i++) {
	GETU32(*(buf + i), rest, p, data_s->byte_swap);
    }
    data->object = (void *)buf;

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_close *
iiimp_close_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_close *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < ( 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_close *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_seek *
iiimp_seek_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_seek *        data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_seek *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->offset, rest, p, data_s->byte_swap);
    GETU32(data->whence, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_opendir *
iiimp_opendir_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_opendir *     data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_opendir *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->dirname = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->dirname = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_closedir *
iiimp_closedir_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_closedir *    data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_closedir *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_mkdir *
iiimp_mkdir_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_mkdir *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < ( 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_mkdir *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->mode, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_rmdir *
iiimp_rmdir_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_rmdir *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_rmdir *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_symlink *
iiimp_symlink_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_symlink *     data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_symlink *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->dest = NULL;
    data->src = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->dest = iiimp_utf8string_unpack(data_s, &rest, &p, rest);
    data->src = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_stat *
iiimp_stat_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_stat *        data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_stat *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_lstat *
iiimp_lstat_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_lstat *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_lstat *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_fstat *
iiimp_fstat_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_fstat *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_fstat *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_creat *
iiimp_creat_unpack(
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{
    IIIMP_creat *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;
    }

    data = (IIIMP_creat *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->mode, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_lseek *
iiimp_lseek_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_lseek *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_lseek *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->offset, rest, p, data_s->byte_swap);
    GETU32(data->whence, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_unlink *
iiimp_unlink_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_unlink *      data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_unlink *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_rename *
iiimp_rename_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_rename *      data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_rename *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->old_name = NULL;
    data->new_name = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    data->old_name = iiimp_utf8string_unpack(data_s, &rest, &p, rest);
    data->new_name = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_fcntl *
iiimp_fcntl_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_fcntl *       data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_fcntl *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->cmd, rest, p, data_s->byte_swap);
    GETU32(data->arg, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_truncate *
iiimp_truncate_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_truncate *    data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_truncate *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;
    data->path = NULL;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->length, rest, p, data_s->byte_swap);
    data->path = iiimp_utf8string_unpack(data_s, &rest, &p, rest);

    *nbyte = rest;
    *ptr = p;

    return data;
}

IIIMP_ftruncate *
iiimp_ftruncate_unpack( 
    IIIMP_data_s *      data_s,
    size_t *            nbyte,
    const uchar_t **    ptr,
    size_t              nbyte_max)
{   
    IIIMP_ftruncate *        data;
    size_t              rest;
    const uchar_t *     p;

    rest = nbyte_max;
    p = *ptr;

    if ((*nbyte < rest) || (rest < (4 + 4 + 4))) {
        data_s->status = IIIMP_DATA_INVALID;
        return NULL;    
    }
    
    data = (IIIMP_ftruncate *)malloc(nbyte_max);
    if (NULL == data) {
        data_s->status = IIIMP_DATA_MALLOC_ERROR;
        return NULL;
    }
    data->nbyte = nbyte_max;

    GETU32(data->nbyte, rest, p, data_s->byte_swap);
    GETU32(data->ns_id, rest, p, data_s->byte_swap);
    GETU32(data->length, rest, p, data_s->byte_swap);

    *nbyte = rest;
    *ptr = p;

    return data;
}

void
iiimp_open_print(
    IIIMP_data_s *   data_s,
    IIIMP_open *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tflags = %d\n", m->flags);
    (void)fprintf(data_s->print_fp, "\tfilename = ");
    iiimp_utf8string_print(data_s, m->filename);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_read_print(
    IIIMP_data_s *   data_s,
    IIIMP_read *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tsize = %d\n", m->size);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_write_print(
    IIIMP_data_s *    data_s,
    IIIMP_write *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tsize = %d\n", m->size);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_close_print(
    IIIMP_data_s *    data_s,
    IIIMP_close *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_seek_print(
    IIIMP_data_s *    data_s,
    IIIMP_seek *      m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\toffset = %d\n", m->offset);
    (void)fprintf(data_s->print_fp, "\twhence = %d\n", m->whence);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_opendir_print(
    IIIMP_data_s *      data_s,
    IIIMP_opendir *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->dirname);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_closedir_print(
    IIIMP_data_s *      data_s,
    IIIMP_closedir *    m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_mkdir_print(
    IIIMP_data_s *      data_s,
    IIIMP_mkdir *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tmode = %d\n", m->mode);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_rmdir_print(
    IIIMP_data_s *    data_s,
    IIIMP_rmdir *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_symlink_print(
    IIIMP_data_s *      data_s,
    IIIMP_symlink *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fputc('\n', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "\tdest = ");
    iiimp_utf8string_print(data_s, m->dest);
    (void)fputc('\n', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "\tsrc = ");
    iiimp_utf8string_print(data_s, m->src);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_stat_print(
    IIIMP_data_s *   data_s,
    IIIMP_stat *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_lstat_print(
    IIIMP_data_s *    data_s,
    IIIMP_lstat *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_fstat_print(
    IIIMP_data_s *    data_s,
    IIIMP_fstat *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_creat_print(
    IIIMP_data_s *    data_s,
    IIIMP_creat *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_lseek_print(
    IIIMP_data_s *    data_s,
    IIIMP_lseek *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\toffset = %d\n", m->offset);
    (void)fprintf(data_s->print_fp, "\twhence = %d\n", m->whence);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_unlink_print(
    IIIMP_data_s *     data_s,
    IIIMP_unlink *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_rename_print(
    IIIMP_data_s *     data_s,
    IIIMP_rename *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\told name = ");
    iiimp_utf8string_print(data_s, m->old_name);
    (void)fputc('\n', data_s->print_fp);
    (void)fprintf(data_s->print_fp, "\tnew name = ");
    iiimp_utf8string_print(data_s, m->new_name);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_fcntl_print(
    IIIMP_data_s *    data_s,
    IIIMP_fcntl *     m)
{   
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tcmd = %d\n", m->cmd);
    (void)fprintf(data_s->print_fp, "\targ = %d\n", m->arg);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_truncate_print(
    IIIMP_data_s *       data_s,
    IIIMP_truncate *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tlength = %d\n", m->length);
    (void)fprintf(data_s->print_fp, "\tpath = ");
    iiimp_utf8string_print(data_s, m->path);
    (void)fputc('\n', data_s->print_fp);
}

void
iiimp_ftruncate_print(
    IIIMP_data_s *        data_s,
    IIIMP_ftruncate *     m)
{
    if (NULL == m) return;

    (void)fprintf(data_s->print_fp, "\tns_id = %d\n", m->ns_id);
    (void)fprintf(data_s->print_fp, "\tlength = %d\n", m->length);
    (void)fputc('\n', data_s->print_fp);
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
