/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icalculatorparser.h"


//
//  Templates
//
#include "iarraytemplate.h"


//
//  iCalculatorParser class
//
iCalculatorParser::iCalculatorParser(bool blank)
{
	mMaxPriority = 0;
	mErrorPosition = -1;

	mExpression = "$undefined$";  //  to check for null expressions

	if(!blank)
	{
		//
		//  Mathematical operators
		//
		this->AddOperator("^","pow",7,true);
		this->AddOperator("*","mul",5,true);
		this->AddOperator("/","div",5,true);
		this->AddOperator("+","add",4,true);
		this->AddOperator("-","sub",4,true);
		this->AddOperator("-","neg",6,false);

		//
		//  Logical operators
		//
		this->AddOperator("<","ltn",3,true);
		this->AddOperator(">","gtn",3,true);
		this->AddOperator("<=","leq",3,true);
		this->AddOperator(">=","geq",3,true);
		this->AddOperator("==","eql",2,true);
		this->AddOperator("!=","neq",2,true);
		this->AddOperator("&&","and",1,true);
		this->AddOperator("||","lor",1,true);
		this->AddOperator("!","not",6,false);

		//
		//  Vector operators
		//
		this->AddOperator(".","dot",5,true);
		this->AddOperator("%","xrs",5,true);
	}
}


iCalculatorParser::~iCalculatorParser()
{
}


bool iCalculatorParser::AddOperator(const iString& symbol, const iString& code, int priority, bool binary)
{
	if(priority <= 0) return false;

    int i;
	for(i=0; i<mOperators.Size(); i++)
	{
		if(mOperators[i].S==symbol && mOperators[i].C==code) return false;
	}

	OperatorInfo s;
	s.S = symbol;
	s.C = code;
	s.P = priority;
	s.B = binary;
	mOperators.Add(s);
	mIsVerified = false;

	return true;
}


bool iCalculatorParser::VerifyIntegrity()
{
	mMaxPriority = 0;

	if(!this->VerifyIntegrityHelper(this->TokenBeg())) return false;
	if(!this->VerifyIntegrityHelper(this->TokenEnd())) return false;
	if(!this->VerifyIntegrityHelper(this->TokenFun())) return false;
	if(!this->VerifyIntegrityHelper(this->TokenNum())) return false;
	if(!this->VerifyIntegrityHelper(this->TokenVar())) return false;

	int i;
	for(i=0; i<mOperators.Size(); i++)
	{
		if(mOperators[i].P > mMaxPriority) mMaxPriority = mOperators[i].P;
	}

	return true;
}


bool iCalculatorParser::VerifyIntegrityHelper(const iString &token)
{
	int i;
	for(i=0; i<mOperators.Size(); i++)
	{
		if(mOperators[i].S.Find(token) != -1) return false;
	}

	if(token == this->SymbolOpeningParenthesis()) return false;
	if(token == this->SymbolClosingParenthesis()) return false;
	if(token == this->SymbolSeparator()) return false;
	if(token == this->SymbolOpeningBraket()) return false;
	if(token == this->SymbolClosingBraket()) return false;

	return true;
}


//
//  Punctuation symbols
//
inline const iString& iCalculatorParser::SymbolOpeningParenthesis() const
{
	static const iString self("(");
	return self;
}


inline const iString& iCalculatorParser::SymbolClosingParenthesis() const
{
	static const iString self(")");
	return self;
}


inline const iString& iCalculatorParser::SymbolSeparator() const
{
	static const iString self(",");
	return self;
}


inline const iString& iCalculatorParser::SymbolOpeningBraket() const
{
	static const iString self("[");
	return self;
}


inline const iString& iCalculatorParser::SymbolClosingBraket() const
{
	static const iString self("]");
	return self;
}


//
//  Pseudo-code tokens
//
inline const iString& iCalculatorParser::TokenBeg() const
{
	static const iString self("{");
	return self;
}


inline const iString& iCalculatorParser::TokenEnd() const
{
	static const iString self("}");
	return self;
}


inline const iString& iCalculatorParser::TokenFun() const
{
	static const iString self("$");
	return self;
}


inline const iString& iCalculatorParser::TokenNum() const
{
	static const iString self("#");
	return self;
}


inline const iString& iCalculatorParser::TokenVar() const
{
	static const iString self("@");
	return self;
}


//
//  Other functions
//
void iCalculatorParser::SetErrorExtra(const iString &, int)
{
	//
	//  Nothing to do here
	//
}


void iCalculatorParser::SetError(const iString &message, const iString &context, int pos)
{
	mErrorMessage = "Syntax error: " + message;
	iString str(context);

	//
	//  Reverse engineer the context
	//
	int i, j;
	iString w;
	while(mLog.Size() > 0)
	{
		//
		//  Find the last match, since we parse from left to right always
		//
		w = mLog.Last().New;
		i = -1;
		while((j=str.Find(w,i+1)) != -1) i = j;
		if(i > -1)
		{
			if(i < pos) pos += (mLog.Last().Old.Length()-w.Length());
			str.Replace(i,w.Length(),mLog.Last().Old);
		}
		mLog.RemoveLast().Clear();
	}

#ifdef I_DEBUG
	w = str.Part(pos);
#endif

	mErrorPosition = mExpression.FindIgnoringWhiteSpace(str);
	if(mErrorPosition > -1)
	{
		//
		//  Count out the correct location
		//
		int i, j=mErrorPosition;
		for(i=0; i<pos; i++)
		{
			while(mExpression.IsWhiteSpace(j)) j++;
			j++;
		}
		mErrorPosition = j;
	}

	this->SetErrorExtra(message,mErrorPosition);

	mCode.Clear();
	mExpression.Clear();
}


//
//  Text manipulation helpers. These helpers are fast and include no bound checking.
//
int iCalculatorParser::FindVariableToLeft(const iString &str, int index) const
{
	if(index == 0) return 0;

	int in = index - 1;
	while(in>0 && this->IsVariableLetter(str[in])) in--;  // keep going along word letters
	while(in>=0 && in<index && !this->IsVariableFirstLetter(str[in])) in++;  // go back if we went too far
	return in;
}


int iCalculatorParser::FindVariableToRight(const iString &str, int index) const
{
	int in = index + 1;
	int len = str.Length();
	if(this->IsVariableFirstLetter(str[in]))  // the first character after the white space must be a beginning of a word
	{
		while(in<len && this->IsVariableLetter(str[in])) in++;
	}
	return --in;
}


int iCalculatorParser::FindLiteralToLeft(const iString &str, int index) const
{
    int i = 0;
	while(i<index && !this->IsRecognizedLiteral(str,i,index-i)) i++;
	return i;
}


int iCalculatorParser::FindLiteralToRight(const iString &str, int index) const
{
	index++;
	int len = str.Length() - index;
	while(len>0 && !this->IsRecognizedLiteral(str,index,len)) len--;
	return len + index - 1;
}


int iCalculatorParser::FindOperatorMatch(const iString &str, const iArray<OperatorInfo> &ops, int index, int &kop) const
{
	int k, j, js, jc = str.Length();
	kop = -1;
	for(k=0; k<ops.Size(); k++)
	{
		js = index;
		while(js < jc)
		{
			j = str.Find(ops[k].S,js);
			if(j>-1 && (j<jc || (j==jc && kop>-1 && ops[k].S.Length()>ops[kop].S.Length()))) // if we found a longer match at the same position, use it
			{
				//
				//  Is this inside a decorated token?
				//
				if(this->IsInsideAToken(str,j))
				{
					js = j + 1;
				}
				else
				{
					js = jc = j;
					kop = k;
				}
			}
			else js = jc;
		}
	}
	return jc;
}


int iCalculatorParser::FindCompleteTokenToLeft(const iString &str, iString &token, int index) const
{
	//
	//  Count opening and closing tokens until we get a match
	//
	int j = this->TokenEnd().Length(), stack = 1;
	if(str.Find(this->TokenEnd(),index-j) != index-j) return index;
	for(j=index-2*j; stack>0 && j>=0; j--)
	{
		if(str.Find(this->TokenBeg(),j) == j) stack--;
		if(str.Find(this->TokenEnd(),j) == j) stack++;
	}

	if(stack == 0)
	{
		j++;
		token = str.Part(j,index-j);
		return j;
	}
	else return index;
}


int iCalculatorParser::FindCompleteTokenToRight(const iString &str, iString &token, int index) const
{
	int iBeg, iEnd;

	index++;
	str.FindTokenMatch(this->TokenBeg(),iBeg,this->TokenEnd(),iEnd,index,true);
	if(iBeg!=index || iEnd==-1) return index - 1; else
	{
		token = str.Part(iBeg,iEnd-iBeg+1);
		return iEnd;
	}
}


bool iCalculatorParser::IsRecognizedLiteral(const iString &str, int index, int length) const
{
	bool ok;
	str.Part(index,length).ToDouble(ok);
	return ok;
}


bool iCalculatorParser::IsInsideAToken(const iString &str, int index) const
{
	int i2 = str.Find(this->TokenEnd(),index);
	if(i2 == -1) return false;
	int i1 = str.Find(this->TokenBeg(),index);
	return (i1==-1 || i1>i2);
}


bool iCalculatorParser::IsFullyParsed(const iString &str) const
{
	if(str.IsEmpty()) return true;
	if(str.Find(this->TokenBeg()) != 0) return false;
	
	iString w;
	return (this->FindCompleteTokenToRight(str,w,-1)+1 == str.Length());
}


void iCalculatorParser::DecorateToken(iString &code, const iString &op, const iString &token)
{
	code = this->TokenBeg() + token + op + code + this->TokenEnd();
}

 
#ifdef I_CHECK1
bool iCalculatorParser::ValidatePseudoCode(const iString &code) const
{
	if(!this->IsFullyParsed(code)) return false;

	//
	//  Check the outer token
	//
	int l = this->TokenBeg().Length();
	if(code.Find(this->TokenFun(),l) == l)
	{
		//
		//  This a function
		//
		int is = this->TokenFun().Length() + l - 1;
		int ie = this->FindVariableToRight(code,is);
		if(is == ie) return false;
		int ind = ie;
		iString token;
		while(ind < code.Length()-this->TokenEnd().Length()-1)
		{
			if((ie=this->FindCompleteTokenToRight(code,token,ind)) == ind) return false;
			if(!this->ValidatePseudoCode(token)) return false;
			ind = ie;
		}
		return true;
	}

	if(code.Find(this->TokenVar(),l) == l)
	{
		//
		//  This a variable
		//
		int is = this->TokenVar().Length() + l - 1;
		int ie = this->FindVariableToRight(code,is);
		if(is==ie || ie+1<code.Length()-this->TokenEnd().Length()) return false;
		return true;
	}

	if(code.Find(this->TokenNum(),l) == l)
	{
		//
		//  This a recognized literal
		//
		int ind = l + this->TokenNum().Length();
		if(!this->IsRecognizedLiteral(code,ind,code.Length()-this->TokenEnd().Length()-ind)) return false;
		return true;
	}

	return false;
}
#endif


//
//  Main worker
//
bool iCalculatorParser::Parse(const iString &str)
{
	mErrorMessage.Clear();
	mErrorPosition = -1;

	if(!mIsVerified)
	{
		if(!this->VerifyIntegrity())
		{
			this->SetError("Internal error: iCalculatorParser is configured incorrectly.","",-1);
			return false;
		}
		mIsVerified = true;
	}

	if(str == mExpression) return true; // Everything has been done already.
	
	if(str.IsEmpty())
	{
		this->SetError("A null expression is not allowed.",str,0);
		return false;
	}

	//
	//  Construct the pseudo-code
	//
    mExpression = str;
	mCode.Clear();
	iString code = str;
	code.RemoveWhiteSpace();

    //
	//  Find all parenthesis-enclosed pieces
	//
	iString w, name;
	int io, ic, in;
	while((ic=code.Find(this->SymbolClosingParenthesis())) > -1)
	{
		w = code.Part(0,ic+1);
		io = -1;
		while((in=w.Find(this->SymbolOpeningParenthesis(),io+1)) > -1) io = in;
		if(io == -1)
		{
			this->SetError("A closing parenthesis is not preceeded by an opening parenthesis.",code,in);
			return false;
		}

		in = this->FindVariableToLeft(code,io);
		//
		//  If there is a word in front, it is a function call; in both cases we treat it identically
		//
		name = w.Part(in,io-in);
		w = code.Part(in,ic-in+1);
		if(!this->ParseSegment(w,name)) return false;
		code.Replace(in,ic-in+1,w);
	}

	//
	//  Are there non-closed opening parentheses?
	//
	in = code.Find(this->SymbolOpeningParenthesis());
	if(in >= 0)
	{
		this->SetError("Unmatched opening parenthesis.",code,in);
		return false;
	}

	//
	//  No parenthesis left, parse what's left
	//
	in = code.Find(this->SymbolSeparator());
	if(in >= 0)
	{
		this->SetError("Orphan separator.",code,in);
		return false;
	}

	if(!this->ParseComponent(code)) return false;

#ifdef I_CHECK1
	if(!this->ValidatePseudoCode(code))
	{
		this->SetError("Internal error: the pseudo-code is parsed incorrectly.","",-1);
		return false;
	}
#endif

	//
	//  Clear the log, it can take a lot of memory
	//
	while(mLog.Size() > 0) mLog.RemoveLast().Clear();

	mCode = code;

	return true;
}


bool iCalculatorParser::ParseSegment(iString &code, const iString &fun)
{
	//
	//  Select the piece between the parenthesis
	//
	Logger log(this,code);
	iString tmp;
	code = code.Part(fun.Length()+this->SymbolOpeningParenthesis().Length(),code.Length()-this->SymbolClosingParenthesis().Length()-fun.Length()-this->SymbolOpeningParenthesis().Length());

	//
	//  Split by commas, there are no other parentheses inside
	//
	iString dec(fun);
	if(code.Contains(this->SymbolSeparator()) == 0)
	{
		//
		//  A single component
		//
		if(!this->ParseComponent(code)) return false;
		tmp = code;
	}
	else
	{
		//
		//  Multiple components present. Parse the individual components.
		//
		int len = this->SymbolSeparator().Length();
		int i2 = -len, i1;
		iString wold, w;
		while(i2 < code.Length())
		{
			i1 = i2;
			i2 = code.Find(this->SymbolSeparator(),i1+len); if(i2 == -1) i2 = code.Length();
			w = code.Part(i1+len,i2-i1-len);
			if(!this->ParseComponent(w)) return false;
			code.Replace(i1+len,i2-i1-len,w);
			i2 = i1 + len + w.Length();
			tmp += w; // this is just like code, but without separators
		}
		//
		//  Is this a vector construct?
		//
		if(dec.IsEmpty())
		{
			//
			//  A vector construct (a,b,c). Is there one already parsed?
			//
			dec = "vec";
			if((i1=code.Find(this->TokenFun()+dec)) != -1)
			{
				this->SetError("Nested vector constructs (a,(c,d)) are not permitted.",code,i1);
				return false;
			}
		}
	}

	if(!dec.IsEmpty())
	{
		this->DecorateToken(tmp,dec,this->TokenFun()); // decorate if it is a fun, do nothing if it is just some thing like (a+b) since a+b will be already decorated.
	}
	code = tmp;

	return true;
}


bool iCalculatorParser::ParseComponent(iString &code)
{
	//
	//  If this component is already decorated, skip it.
	//
	if(this->IsFullyParsed(code)) return true;
	Logger log(this,code);

	//
	//  Replace vector components with function calls
	//
	iString wold, w, name;
	int jo, jc, jn;
	while((jc=code.Find(this->SymbolClosingBraket())) > -1)
	{
		w = code.Part(0,jc+1);
		jo = -1;
		while((jn=w.Find(this->SymbolOpeningBraket(),jo+1)) > -1) jo = jn;
		if(jo == -1)
		{
			this->SetError("A closing bracket is not preceeded by an opening bracket.",code,jc);
			return false;
		}

		jn = this->FindVariableToLeft(w,jo);
		if(jn == jo)
		{
			//
			//  There is no word in front, it is a syntax error
			//
			this->SetError("Brackets should only be used for selecting individual vector elements.",code,jo);
			return false;
		}

		//
		//  Remove brackets
		//
		name = code.Part(jn,jo-jn);
		w = code.Part(jo+this->SymbolOpeningBraket().Length(),jc-jo+1-this->SymbolOpeningBraket().Length()-this->SymbolClosingBraket().Length());
		if(!this->ParseExpressions(w)) return false;
		wold = name + this->SymbolOpeningBraket() + w + this->SymbolClosingBraket();
		this->DecorateToken(name,"",this->TokenVar());
		w = name + w;
		this->DecorateToken(w,"arg",this->TokenFun());
		code.Replace(jn,jc-jn+1,w);
		log.LogChange(wold,w);
	}

	//
	//  Parse the rest
	//
	if(!this->ParseExpressions(code)) return false;

	return true;
}


bool iCalculatorParser::ParseExpressions(iString &code)
{
	//
	//  If this expression is already decorated, skip it.
	//
	if(this->IsFullyParsed(code)) return true;
	Logger log(this,code);

	//
	//  Go over the expression and classify tokens as literals, variables, and unknown.
	//
	iString wold, w;
	int i, k, kop, iop1 = -1, iop2;
	while(iop1 < code.Length()-1)
	{
		//
		//  Find the next operator
		//
        iop2 = this->FindOperatorMatch(code,mOperators,iop1+1,kop);

		//
		//  If this an operator right after another one, it may be a unary one or may be a part of a literal. Check it.
		//
		if(iop2 == iop1 + 1)
		{
			i = this->FindLiteralToRight(code,iop1);
			if(i>iop1 && i+1==this->FindOperatorMatch(code,mOperators,i+1,k))
			{
				//
				//  Hey, that a literal!
				//
				kop = k;
				wold = w = code.Part(iop1+1,i-iop1);
				this->DecorateToken(w,"",this->TokenNum());
				code.Replace(iop1+1,i-iop1,w);
				log.LogChange(wold,w);
				iop2 = iop1 + 1 + w.Length();
			}
		}
		else
		{
			//
			//  Identify the operand
			//
			i = this->FindCompleteTokenToRight(code,w,iop1);
			if(i > iop1)
			{
				//
				//  We found a token, is it a single one?
				//
				if(i+1 == iop2)
				{
					//
					//  Yes, just keep going
					//
					iop2 = iop1 + 1 + w.Length();
				}
				else
				{
					//
					//  No, that means a missing operator
					//
					this->SetError("Missing binary operator",code,i+1);
					return false;
				}
			}
			else
			{
				i = this->FindLiteralToRight(code,iop1);
				if(i+1 >= iop2) // the literal search may incorporate this operator char and move beyond it.
				{
					//
					//  This is a literal
					//
					wold = w = code.Part(iop1+1,i-iop1);
					this->DecorateToken(w,"",this->TokenNum());
					code.Replace(iop1+1,i-iop1,w);
					log.LogChange(wold,w);
					iop2 = iop1 + 1 + w.Length();
				}
				else
				{
					i = this->FindVariableToRight(code,iop1);
					if(i+1 == iop2)
					{
						//
						//  This is a variable
						//
						wold = w = code.Part(iop1+1,i-iop1);
						this->DecorateToken(w,"",this->TokenVar());
						code.Replace(iop1+1,i-iop1,w);
						log.LogChange(wold,w);
						iop2 = iop1 + 1 + w.Length();
					}
					else
					{
						//
						//  There is no literal or variable here, take it as an unknown token
						//
						this->SetError("Unrecognizable token",code,iop1+1);
						return false;
					}
				}
			}
		}
		if(kop == -1)
		{
			iop1 = code.Length() - 1;
		}
		else
		{
			iop1 = iop2 + mOperators[kop].S.Length() - 1;
		}
	}

	//
	//  Start searching for math operators in order of priority.
	//  Since we have no parentheses in this component, function calls have already been decorated.
	//
	iArray<OperatorInfo> ops;

	for(k=mMaxPriority; k>0; k--)
	{
		ops.Clear();
		for(i=0; i<mOperators.Size(); i++) if(mOperators[i].P == k) ops.Add(mOperators[i]);
		if(!this->ParseSomeOperators(code,ops)) return false;
		if(this->IsFullyParsed(code))
		{
			//
			//  We are done: report success
			//
			return true;
		}
	}

	//
	//  This expression is still nor parsed, this is an error
	//
	this->SetError("Internal error: an expression is not parsed completely.",code,0);
	return false;
}


bool iCalculatorParser::ParseSomeOperators(iString &code, const iArray<OperatorInfo> &ops)
{
	//
	//  If this expression is already decorated, skip it.
	//
	if(this->IsFullyParsed(code)) return true;
	Logger log(this,code);

	iString w1, w2, wold, w;
	int k, jo = -1, jc, jn1, jn2;
	while((jo=this->FindOperatorMatch(code,ops,jo+1,k)) < code.Length())
	{
		w1.Clear();
		w2.Clear();

		if(!ops[k].B)
		{
			//
			//  If this a unary operator, the token on the left must be a binary operator of equal or lower priority
			//
			if(jo > 0)
			{
				int i;
				bool found = false;
				w = code.Part(0,jo);
				for(i=0; i<mOperators.Size(); i++) if(mOperators[i].B && mOperators[i].P<=ops[k].P)
				{
					if(w.EndsWith(mOperators[i].S)) found = true;
				}
				//
				//  This is not a unary operator, postpone processing
				//
				if(!found) continue;
			}
			jn1 = jo;
		}
		else
		{
			jn1 = this->FindCompleteTokenToLeft(code,w1,jo);
			if(jn1 == jo)
			{
				this->SetError("Missing token on the left.",code,jo);
				return false;
			}
		}

		jc = jo + ops[k].S.Length() - 1;
		jn2 = this->FindCompleteTokenToRight(code,w2,jc);
		if(jn2 == jc)
		{
			this->SetError("Missing token.",code,jc+1);
			return false;
		}

		//
		//  Now decorate everything
		//
		w = w1 + w2;
		this->DecorateToken(w,ops[k].C,this->TokenFun());
		wold = code.Part(jn1,jn2-jn1+1);
		code.Replace(jn1,jn2-jn1+1,w);
		log.LogChange(wold,w);
		jo = jn1;
	}

	return true;
}


iCalculatorParser::Logger::Logger(iCalculatorParser *owner, const iString &code) : mSavedCode(code), mCode(code), mOwner(owner)
{
	IERROR_ASSERT(mOwner);
	mSavedLogLocation = mOwner->mLog.Size();
}


iCalculatorParser::Logger::~Logger()
{
	if(mOwner->GetErrorMessage().IsEmpty())
	{
		while(mOwner->mLog.Size() > mSavedLogLocation) mOwner->mLog.RemoveLast().Clear();
		this->LogChange(mSavedCode,mCode);
	}
}


void iCalculatorParser::Logger::LogChange(const iString &o, const iString &n)
{
	if(o != n)
	{
		Change c;
		c.Old = o;
		c.New = n;
		mOwner->mLog.Add(c);
	}
}


