/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "imergedatafilter.h"


#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "imath.h"
#include "iviewmodule.h"

#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkStreamingDemandDrivenPipeline.h>

//
//  Templates
//
#include "igenericfiltertemplate.h"


iMergeDataFilter::iMergeDataFilter(iViewSubject *vo) : iGenericFilter<vtkDataSetToDataSetFilter,vtkDataSet,vtkDataSet>(vo,1,false,false)
{
	mDataRank = 0;
	mDataInput = 0;
}


void iMergeDataFilter::SetDataInput(int rank, vtkDataSet *input)
{
	if(rank>=0 && rank<3 && input!=0)
	{
		mDataRank = rank;
		mDataInput = input;
		this->SetNumberOfInputPorts(2);
	    this->SetInputConnection(1,input->GetProducerPort());
	}
}


bool iMergeDataFilter::HasData(int rank)
{
	if(rank>=0 && rank<3)
	{
		this->Update();
		switch(rank)
		{
		case 0:	return (this->GetOutput()->GetPointData()->GetScalars()!=0 && this->GetOutput()->GetPointData()->GetScalars()->GetNumberOfTuples()>0);
		case 1: return (this->GetOutput()->GetPointData()->GetVectors()!=0 && this->GetOutput()->GetPointData()->GetVectors()->GetNumberOfTuples()>0);
		case 2: return (this->GetOutput()->GetPointData()->GetTensors()!=0 && this->GetOutput()->GetPointData()->GetTensors()->GetNumberOfTuples()>0);
		}
	}
	return false;
}


void iMergeDataFilter::InitExecution()
{
	vtkDataSet *input = this->GetInput();
	vtkDataSet *output = this->GetOutput();

	output->SetWholeExtent(input->GetWholeExtent());  // God only knows why these two are needed,
	output->SetUpdateExtent(input->GetUpdateExtent()); // but it took me a lot of debugging to find out they must be here
	if(mDataInput != 0) mDataInput->SetUpdateExtentToWholeExtent();

    if(mSavedOutputVector != 0)
	{
		vtkInformation* outInfo = mSavedOutputVector->GetInformationObject(0);
		outInfo->Set(vtkStreamingDemandDrivenPipeline::UPDATE_NUMBER_OF_PIECES(),1);
		outInfo->Set(vtkStreamingDemandDrivenPipeline::UPDATE_PIECE_NUMBER(),0);
	}
}


void iMergeDataFilter::ProduceOutput()
{
	vtkDataSet *input = this->GetInput();
	vtkDataSet *output = this->GetOutput();

	vtkPointData *pd = input->GetPointData();
	if(pd == 0)
	{
		output->Initialize();
		return;
	}

	output->ShallowCopy(input);

	switch(mDataRank)
	{
	case 0:
		{
			if(this->AreDataSetsCompatible(input,this->GetDataInput()) && this->GetDataInput()->GetPointData()->GetScalars()!=0)
			{
				output->GetPointData()->SetScalars(this->GetDataInput()->GetPointData()->GetScalars());
			}
			break;
		}
	case 1:
		{
			if(this->AreDataSetsCompatible(input,this->GetDataInput()) && this->GetDataInput()->GetPointData()->GetVectors()!=0)
			{
				output->GetPointData()->SetVectors(this->GetDataInput()->GetPointData()->GetVectors());
			}
			break;
		}
	case 2:
		{
			if(this->AreDataSetsCompatible(input,this->GetDataInput()) && this->GetDataInput()->GetPointData()->GetTensors()!=0)
			{
				output->GetPointData()->SetTensors(this->GetDataInput()->GetPointData()->GetTensors());
			}
			break;
		}
	}
}


bool iMergeDataFilter::AreDataSetsCompatible(vtkDataSet *ds1, vtkDataSet *ds2) const
{
	int i, numArrays;

	if(ds1==0 || ds2==0) return false;

	if(ds1->GetNumberOfPoints() != ds2->GetNumberOfPoints()) return false;
	vtkIdType numTuples, numPts = ds1->GetNumberOfPoints();

	numArrays = ds1->GetPointData()->GetNumberOfArrays();
	for(i=0; i<numArrays; i++)
	{
		numTuples = ds1->GetPointData()->GetArray(i)->GetNumberOfTuples();
		if(numTuples>0 && numTuples!=numPts) return false;
	}

	numArrays = ds2->GetPointData()->GetNumberOfArrays();
	for(i=0; i<numArrays; i++)
	{
		numTuples = ds2->GetPointData()->GetArray(i)->GetNumberOfTuples();
		if(numTuples>0 && numTuples!=numPts) return false;
	}

	//
	//  If ImageData, check that the dimensions are identical
	//
	vtkImageData *id1 = vtkImageData::SafeDownCast(ds1);
	vtkImageData *id2 = vtkImageData::SafeDownCast(ds2);
	
	if((id1==0 && id2!=0) || (id1!=0 && id2==0)) return false;
	
	if(id1!=0 && id2!=0)
	{
		int dim1[3], dim2[3];
		id1->GetDimensions(dim1);
		id2->GetDimensions(dim2);
		return (dim1[0]==dim2[0] && dim1[1]==dim2[1] && dim1[2]==dim2[2]);
	}

	//
	//  Other data types can be added here
	//
	return true;
}

