/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "imarkerfamily.h"


#include "icommoneventobservers.h"
#include "ilegend.h"
#include "imarker.h"
#include "iparameter.h"
#include "ipointglyph.h"
#include "irendertool.h"
#include "iviewmodule.h"

#include <vtkPointWidget.h>
#include <vtkProperty.h>

//
//  Templates
//
#include "iarraytemplate.h"
#include "iviewfamilytemplate.h"


iMarkerFamily* iMarkerFamily::New(iViewModule *vm)
{
	return new iMarkerFamily(vm);
}


iMarkerFamily::iMarkerFamily(iViewModule *vm) : iViewFamily<iMarker,iViewModule>(vm)
{
	mLegendPosition = 0;

	mHandles = vtkPointWidget::New(); IERROR_ASSERT(mHandles);
	mHandles->XShadowsOn();
	mHandles->YShadowsOn();
	mHandles->ZShadowsOn();
	mHandles->OutlineOff();
	mHandles->TranslationModeOff();
	mHandles->SetInteractor(mParent->GetRenderTool()->GetInteractor());
	mHandles->GetProperty()->SetColor(0.0,0.0,0.0);
	mHandles->GetProperty()->SetLineWidth(4.0);
//	mHandles->GetSelectedProperty()->SetColor(1.0,0.0,0.0);
	mHandles->GetSelectedProperty()->SetLineWidth(2.0);
	mHandles->PlaceWidget(-1.0,1.0,-1.0,1.0,-1.0,1.0);

	mMotionObserver = iMarkerEventObserver::New(mParent); IERROR_ASSERT(mMotionObserver);

	mHandles->AddObserver(vtkCommand::InteractionEvent,mMotionObserver);
	mHandles->AddObserver(vtkCommand::EndInteractionEvent,mMotionObserver);

	//
	//  vtkLegendBoxActor is defined under earlier VTK versions, but
	//  markers cannot be positioned without the vtkPointWidgets, so the legend
	//  is useless then too.
	//
	mLegend = iLegend::New(mParent->GetRenderTool()); IERROR_ASSERT(mLegend);
	this->SetLegendPosition(mLegendPosition);
	mLegend->VisibilityOff();
	mParent->GetRenderTool()->AddObject(mLegend);
}


iMarkerFamily::~iMarkerFamily()
{
	//
	//  Legend needs to be deleted before markers - otherwise, markers get stuck somewhere and GarbareCollector crashes.
	//
	mParent->GetRenderTool()->RemoveObject(mLegend);
	mLegend->Delete();

	mMotionObserver->Delete();

	mHandles->Off();
	mHandles->SetInteractor(0);
	mHandles->Delete();
}


int iMarkerFamily::CreateMember()
{
	int ret = this->iViewFamily<iMarker,iViewModule>::CreateMember();
	if(ret > -1)
	{
		this->BuildLegend();
		this->SetCurrentMemberIndex(ret);
		this->GetCurrentMember()->SetPosition(iParameter::_LocationFocalPoint);
	}
	return ret;
}


bool iMarkerFamily::DeleteMember(int i)
{
	bool c = (i == this->GetCurrentMemberIndex());
	bool ret = this->iViewFamily<iMarker,iViewModule>::DeleteMember(i);
	if(ret)
	{
		this->BuildLegend();
		if(c) this->ShowHandles(false);
	}
	return ret;
}


void iMarkerFamily::ShowLegend(bool s)
{
	if(s) 
	{
		this->BuildLegend();
		mLegend->VisibilityOn(); 
	}
	else mLegend->VisibilityOff();
}


bool iMarkerFamily::IsLegendVisible()
{
	return (mLegend->GetVisibility() != 0);
}


void iMarkerFamily::SetLegendPosition(int p)
{
	if(p>=0 && p<=1)
	{
		mLegendPosition = p;
		switch (p)
		{
		case 0:
			{
				mLegend->SetPosition(0.73,0.02);
				break;
			}
		case 1:
			{
				mLegend->SetPosition(0.02,0.02);
				break;
			}
		}
	}
}


void iMarkerFamily::SetHandlesColor(const iColor &c)
{
	mHandles->GetProperty()->SetColor(c.ToVTK());
}


void iMarkerFamily::ShowHandles(bool s)
{
	if(s)
	{
		mHandles->SetPosition(this->GetCurrentMember()->GetPosition()[0],this->GetCurrentMember()->GetPosition()[1],this->GetCurrentMember()->GetPosition()[2]);
		mHandles->On();
	}
	else
	{
		mHandles->Off();
	}
}


bool iMarkerFamily::IsHandlesVisible()
{
	return (mHandles->GetEnabled() != 0);
}



void iMarkerFamily::UpdateLegend()
{
	this->BuildLegend(-1);
}


void iMarkerFamily::BuildLegend(int n)
{
	int i;
	if(n == -1) n = this->GetCurrentMemberIndex();
	if(n<0 || n>this->GetMaxMemberIndex())
	{
		mLegend->SetNumberOfEntries(1+this->GetMaxMemberIndex());
		for(i=0; i<=this->GetMaxMemberIndex(); i++)
		{
			this->UpdateEntry(i);
		}
	}
	else this->UpdateEntry(n);
}


void iMarkerFamily::UpdateEntry(int n)
{
	//
	//  Fix a bug in vtkLegendBoxActor
	//
	iMarker *m = this->GetMember(n);
	if(m!=0 && m->GetCaptionText().Length()>0)
	{
		mLegend->SetEntry(n,m->GetMarkerObject()->GetOutput(),m->GetCaptionText().ToCharPointer(),m->GetColor(0).ToVTK());
	}
	else
	{
		mLegend->SetEntry(n,m->GetMarkerObject()->GetOutput()," ",m->GetColor(0).ToVTK());
	}
}



