/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ipicker.h"


#include "iactor.h"
#include "icommoneventobservers.h"
#include "idata.h"
#include "idataformatter.h"
#include "idatahelper.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "ierror.h"
#include "iobjectfactory.h"
#include "iprobefilter.h"
#include "iviewmodule.h"
#include "iviewsubjectobserver.h"
#include "ivtk.h"

#include <vtkCellArray.h>
#include <vtkCellPicker.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPointLocator.h>
#include <vtkPointPicker.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkProp.h>
#include <vtkProperty.h>
#include <vtkPropPicker.h>
#include <vtkSphereSource.h>
#include <vtkStructuredPoints.h>

//
//  templates
//
#include "iarraytemplate.h"


//
//  Picker must be an object so that it can be created by the ObjectFactory
//
IOBJECT_DEFINE_TYPE(iPicker,Picker,pi,iObjectType::_Helper);  // helper type

IOBJECT_DEFINE_KEY(iPicker,Accuracy,a,Float,1);
IOBJECT_DEFINE_KEY(iPicker,PickMethod,m,Int,1);
IOBJECT_DEFINE_KEY(iPicker,PointSize,ps,Float,1);


namespace iPicker_Private
{
	const int NumPickMethods = 3;

	class CellPicker : public vtkCellPicker
	{

	public:

		static CellPicker* New()
		{ 
			return new CellPicker;
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}
	};


	class PointPicker : public vtkPointPicker
	{

	public:

		static PointPicker* New()
		{ 
			return new PointPicker;
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}
	};


	class ObjectPicker : public vtkPropPicker
	{

	public:

		static ObjectPicker* New()
		{ 
			return new ObjectPicker;
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}
	};


	class PickHandler : public vtkPicker
	{

	public:

		static PickHandler* New(iPicker *parent)
		{
			return new PickHandler(parent);
		}

		virtual vtkAssemblyPath* GetPath()
		{
			return 0; // we do not highlight the picked prop
		}

		virtual int Pick(double x, double y, double z, vtkRenderer *ren)
		{
			int pm = mParent->GetPickMethod();

			if(pm>=0 && pm<NumPickMethods)
			{
				vtkPicker *p = vtkPicker::SafeDownCast(mDevices[pm]);
				if(p != 0) p->SetTolerance(this->GetTolerance());

				this->InvokeEvent(vtkCommand::StartPickEvent,0);
				int ret = mDevices[pm]->Pick(x,y,z,ren);
				mDevices[pm]->GetPickPosition(this->PickPosition);
				this->SetPath(mDevices[pm]->vtkAbstractPropPicker::GetPath());
				mParent->UpdateReport();
				this->InvokeEvent(vtkCommand::EndPickEvent,0);

				return ret;
			}
			else
			{
				vtkErrorMacro("Invalid pick method.");
				return 0;
			}
		}

		virtual void Modified()
		{
			int i;
			for(i=0; i<NumPickMethods; i++) mDevices[i]->Modified();
			this->vtkAbstractPropPicker::Modified();
		}

	private:

		PickHandler(iPicker *parent)
		{
			mParent = parent; IERROR_ASSERT(parent);

			//
			//  Pickers
			//
			mDevices[0] = CellPicker::New(); IERROR_ASSERT(mDevices[0]);
			mDevices[1] = PointPicker::New(); IERROR_ASSERT(mDevices[1]);
			mDevices[2] = ObjectPicker::New(); IERROR_ASSERT(mDevices[2]);

			//
			//  Observer
			//
			mObserver = iPickEventObserver::New(mParent->GetViewModule()); IERROR_ASSERT(mObserver);
			this->AddObserver(vtkCommand::StartPickEvent,mObserver);
			this->AddObserver(vtkCommand::EndPickEvent,mObserver);
			int i;
			for(i=0; i<NumPickMethods; i++)
			{
				mDevices[i]->AddObserver(vtkCommand::AbortCheckEvent,mParent->GetViewModule()->GetAbortRenderEventObserver());
			}
		}

		~PickHandler()
		{
			int i;
			for(i=0; i<NumPickMethods; i++) mDevices[i]->Delete();
			mObserver->Delete();
		}

		iPicker *mParent;
		iPickEventObserver *mObserver;
		vtkAbstractPropPicker *mDevices[NumPickMethods];
	};
};


using namespace iParameter;
using namespace iPicker_Private;

	
iPicker* iPicker::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	iPicker *tmp = new iPicker(vm); IERROR_ASSERT(tmp);
	iObjectFactory::InstallExtensions(tmp);
	return tmp;
}


iPicker::iPicker(iViewModule *vm) : iExtendableObject("Picker"), mViewModule(vm), mDataFormatter(vm), mPos(vm)
{
	mObjName = "";
	mObjType = _ObjectTypeUndefined;
	mDataTypePointer = 0;
	mPickMethod = _PickMethodObject;

	//
	//  Handler
	//
	mHandler = PickHandler::New(this); IERROR_ASSERT(mHandler);

	this->SetAccuracy(0.02f);

	//
	//  Graphical representation
	//
	mPointActor = iActor::New(); IERROR_ASSERT(mPointActor);
	mPointSource = vtkSphereSource::New(); IERROR_ASSERT(mPointSource);

	mPointSize = 0.1;
	this->SetPointSize(mPointSize);
	mPointSource->SetRadius(0.5);
	mPointActor->GetProperty()->SetColor(0.9,0.9,0.9);
	mPointActor->VisibilityOff();
	mPointActor->PickableOff();

	this->GetViewModule()->AddObject(mPointActor);

	//
	//  PolyData inout for a probe filter
	//
	mProbeInput = vtkPolyData::New(); IERROR_ASSERT(mProbeInput);
	vtkCellArray *v = vtkCellArray::New(); IERROR_ASSERT(v);
	v->InsertNextCell(1);
	v->InsertCellPoint(0);
	mProbeInput->SetVerts(v);
	v->Delete();
	vtkPoints *p = vtkPoints::New(VTK_DOUBLE); IERROR_ASSERT(p);
	p->SetNumberOfPoints(1);
	p->SetPoint(0,0.0,0.0,0.0);
	mProbeInput->SetPoints(p);
	p->Delete();
}


iPicker::~iPicker()
{
	mHandler->Delete();
	this->GetViewModule()->RemoveObject(mPointActor);
	mPointActor->Delete();
	mPointSource->Delete();
	mProbeInput->Delete();
}


void iPicker::Modified()
{
	mHandler->Modified();
	this->ClearCache();
}


void iPicker::UpdateReport()
{
	mObjName = "Unknown";
	mPos = mHandler->GetPickPosition();

	this->SetPointSize(mPointSize);
	mPointActor->SetPosition(mPos);

	mDataFormatter->ClearReport();
	this->UpdateReportBody(); // this can replace mPointObject via its extensions

	mPointActor->SetInput(mPointSource->GetOutput());
	mPointActor->SetScaled(true);
	mPointActor->SetPosition(mPos);

	//
	//  Check if extensions want to modify the point actor
	//
	if(mDataTypePointer != 0)
	{
		int i;
		for(i=0; i<mExtensions.Size(); i++) if(iRequiredCast<iPickerExtension>(INFO,mExtensions[i])->IsUsingData(*mDataTypePointer))
		{
			iRequiredCast<iPickerExtension>(INFO,mExtensions[i])->ModifyActor(*mDataTypePointer,mPointActor); // the last installed extension prevails
		}
	}

	this->ShowPickedPoint(mDataTypePointer != 0);
}


void iPicker::UpdateReportBody()
{
	int j;
	iString s1, s2;

	//
	//  Get the object info
	//
#ifdef IVTK_4
	vtkProp *obj = mHandler->GetProp();
#else
	vtkProp *obj = mHandler->GetViewProp();
#endif
	if(obj == 0) 
	{
		mObjName = "";
		mPointActor->VisibilityOff(); 
		return;
	}
	iViewSubjectObserver *obs = iRequiredCast<iViewSubjectObserver>(INFO,obj->GetCommand(1));

	if(obs != 0)
	{
		mObjType = obs->GetObjectType();
		mObjName = obs->GetObjectName();
		mDataTypePointer = &obs->GetDataType();
	}
	else
	{
		mObjType = _ObjectTypeUndefined;
		mObjName = "Unknown";
		mDataTypePointer = 0;
	}

	//
	//  Get the data info
	//
	if(mDataTypePointer == 0) return;

	iDataSubject *subject = this->GetViewModule()->GetReader()->GetSubject(*mDataTypePointer);
	iDataLimits *lim = subject->GetLimits();

	//
	//  Use the ProbeFilter to find attribute values
	//
	mProbeInput->GetPoints()->SetPoint(0,mPos);
	iProbeFilter *probe = subject->CreateProbeFilter(obs->GetSubject());
	probe->SetInput(mProbeInput);
	probe->SetSource(subject->GetData());
	probe->Update();
	vtkPolyData *out = probe->GetPolyDataOutput();

	if(out==0 || out->GetPointData()==0)
	{
		IERROR_LOW("Unable to probe data for the picked object");
		return;
	}

	//
	//  Ask extensions to provide extra information
	//
	int i;
	for(i=0; i<mExtensions.Size(); i++) if(iRequiredCast<iPickerExtension>(INFO,mExtensions[i])->IsUsingData(*mDataTypePointer))
	{
		iRequiredCast<iPickerExtension>(INFO,mExtensions[i])->AddInfo(*mDataTypePointer);
	}

	iDataHelper h(out);
	vtkDataArray *d = 0;
	int nvar = 0;
	if(h.IsThereScalarData())
	{
		d = out->GetPointData()->GetScalars();
		nvar = lim->GetNumVars();
		if(d==0 || nvar<1 || d->GetNumberOfComponents()!=nvar || !d->IsA("vtkFloatArray"))
		{
			IERROR_LOW("Unable to probe data for the picked object");
			return;
		}
		float *v = (float *)d->GetVoidPointer(0);
		//
		//  Make the correction for the surface
		//
		if(mObjType == _ObjectTypeSurface)
		{
			j = obs->GetIsoSurfaceData().Var;
			if(j>=0 && j<nvar) v[j] = obs->GetIsoSurfaceData().Level;
		}
		mDataFormatter->FormatScalarData(lim,nvar,v);
	}

	if(h.IsThereVectorData())
	{
		d = out->GetPointData()->GetVectors();
		nvar = 3;
		if(d==0 || nvar<1 || d->GetNumberOfComponents()!=nvar || !d->IsA("vtkFloatArray"))
		{
			IERROR_LOW("Unable to probe data for the picked object");
			return;
		}
		float *v = (float *)d->GetVoidPointer(0);
		mDataFormatter->FormatVectorData(lim,-1,v);
	}

	if(h.IsThereTensorData())
	{
		d = out->GetPointData()->GetTensors();
		nvar = 9;
		if(d==0 || nvar<1 || d->GetNumberOfComponents()!=nvar || !d->IsA("vtkFloatArray"))
		{
			IERROR_LOW("Unable to probe data for the picked object");
			return;
		}
		float *v = (float *)d->GetVoidPointer(0);
		mDataFormatter->FormatTensorData(lim,-1,v);
	}

	probe->Delete();
}


void iPicker::SetPointSize(float s)
{
	if(s > 0.0)
	{
		mPointSize = s;
		mPointActor->SetScaled(true);
		mPointActor->SetBasicScale(mPointSize);
		this->Modified();
	}
}


void iPicker::ShowPickedPoint(bool s)
{
	if(!mObjName.IsEmpty() && s)
	{
		mPointActor->VisibilityOn();
	}
	if(!s)
	{
		mPointActor->VisibilityOff();
	}
	this->GetViewModule()->Render();
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iPicker::ExtendableObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyPickMethod(),mPickMethod);
	this->PackValue(s,KeyPointSize(),mPointSize);
	this->PackValue(s,KeyAccuracy(),float(mHandler->GetTolerance()));
}


void iPicker::ExtendableObjectUnPackStateBody(const iString &s)
{
	int i; float f;

	if(this->UnPackValue(s,KeyPickMethod(),i)) this->SetPickMethod(i);
	if(this->UnPackValue(s,KeyPointSize(),f)) this->SetPointSize(f);
	if(this->UnPackValue(s,KeyAccuracy(),f)) this->SetAccuracy(f);
}


void iPicker::SetPickMethod(int s)
{
	if(s>=0 && s<NumPickMethods)
	{
		mPickMethod = s;
		this->ClearCache();
	}
}


void iPicker::SetAccuracy(float s)
{
	if(s > 0.0f)
	{
		mHandler->SetTolerance(s);
		this->ClearCache();
	}
}


//
//  Extension class
//
IOBJECTEXTENSION_DEFINE_ABSTRACT(iPicker);

void iPickerExtension::Define()
{
}

