/*
 *  Copyright (C) 2010 Tuomo Penttinen, all rights reserved.
 *
 *  Author: Tuomo Penttinen <tp@herqq.org>
 *
 *  This file is part of Herqq UPnP (HUPnP) library.
 *
 *  Herqq UPnP is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Herqq UPnP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Herqq UPnP. If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef HCLIENTDEVICE_ADAPTER_H_
#define HCLIENTDEVICE_ADAPTER_H_

#include <HUpnpCore/HUpnp>

#include <QtCore/QObject>

namespace Herqq
{

namespace Upnp
{

class HClientDeviceAdapterPrivate;

/*!
 * This is an abstract base class for HUPnP A/V convenience device classes.
 *
 * \headerfile hav_device.h HClientDeviceAdapter
 *
 * \ingroup hupnp_devicemodel
 *
 * \remarks This class is not thread-safe.
 */
class H_UPNP_CORE_EXPORT HClientDeviceAdapter :
    public QObject
{
Q_OBJECT
H_DISABLE_COPY(HClientDeviceAdapter)

protected:

    HClientDeviceAdapterPrivate* h_ptr;

    /*!
     * Creates a new instance.
     *
     * \param parent specifies the parent \c QObject.
     */
    HClientDeviceAdapter(const HResourceType& deviceType, QObject* parent = 0);

    //
    // \internal
    //
    HClientDeviceAdapter(
        const HResourceType& deviceType,
        HClientDeviceAdapterPrivate& dd, QObject* parent = 0);

    /*!
     * Prepares the instance to be associated with the specified HClientDevice.
     *
     * \param device specifies the device to be associated.
     *
     * \return \e true if the instance was successfully prepared to be used
     * with the specified HClientDevice. If \e false is returned the specified HClientDevice
     * will not be associated with this instance.
     */
    virtual bool prepareDevice(HClientDevice* device);

    /*!
     *
     */
    void setDeviceModelInfoProvider(const HDeviceModelInfoProvider&);

    /*!
     *
     */
    const HDeviceModelInfoProvider* deviceModelInfoProvider() const;

public:

    /*!
     * Destroys the instance.
     *
     * Destroys the instance.
     */
    virtual ~HClientDeviceAdapter() = 0;

    /*!
     *
     */
    enum ValidationType
    {
        /*!
         *
         */
        MinimalValidation,

        /*!
         *
         */
        FullValidation
    };

    /*!
     * Associates the specified \c HClientDevice with this instance.
     *
     * \param device specifies the HClientDevice to be associated with this instance.
     *
     * \return \e true in case the \c device was successfully associated with
     * this instance.
     *
     * \sa device()
     */
    bool setDevice(HClientDevice* device, ValidationType vt = FullValidation);

    /*!
     * Returns the \c HClientDevice associated with this instance.
     *
     * \return the \c HClientDevice associated with this instance, if any.
     * A null pointer is returned in case setDevice() has not been called
     * successfully.
     *
     * \sa setDevice()
     */
    HClientDevice* device() const;

    /*!
     * Indicates if the object is ready to be used.
     *
     * \return \e true if the object is ready to be used.
     */
    bool isReady() const;

    /*!
     * Returns the device type this adapter abstracts.
     *
     * \return the device type this adapter abstracts.
     */
    const HResourceType& deviceType() const;
};

}
}

#endif /* HCLIENTDEVICE_ADAPTER_H_ */
