/*
  gyrus-report.c

  GYRUS -- GNOME Cyrus Administrator.

  Copyright (C) 2005 Alejandro Valds <avaldes@utalca.cl>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/

#include <config.h>

#include <glade/glade.h>
#include "gyrus-report.h"
#include "glib/gi18n.h"
#include <glib/gprintf.h>
#include "gyrus-admin-mailbox.h"
#include "gyrus-common.h"
#include <string.h>

#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprintui/gnome-print-dialog.h>
#include <libgnomeprintui/gnome-print-job-preview.h>

#define MAX_USER_PAGE 40	/* maximun user for page */
#define	XINI 25			/* initial value X  */
#define	YINI 740		/* initial value Y */
#define	XLIN 550		/* width of line */

typedef enum {
	COLUMN_FOLDER = 0,
	COLUMN_MAILBOX,
	COLUMN_PERCENTAGE,
	COLUMN_QUOTA_LIMIT,
	COLUMN_QUOTA_USED,
	NUM_COLUMN
} GyrusReportColumn;

/* read glade file, connect signal and show window */
void
gyrus_report_show_report (GyrusAdmin *admin)
{
	GladeXML *xml;
       	gchar *file_glade;
	gchar *title;
	
	GtkWidget *button_cancel;
	GtkWidget *button_preview;
	GtkWidget *button_update;
	
	GyrusReportData *report;
	report = g_new0 (GyrusReportData,1);

	report->mailbox_temp = NULL;
	report->admin = admin;
	
	file_glade = g_strdup_printf ("%s%c%s", GYRUS_GLADE_DIR,
				      G_DIR_SEPARATOR, "report.glade");
	xml = glade_xml_new (file_glade, NULL, NULL);

	report->window_report = glade_xml_get_widget (xml, "window-report");
	button_cancel = glade_xml_get_widget (xml, "button-cancel");
	button_preview = glade_xml_get_widget (xml, "button-preview");
	button_update = glade_xml_get_widget (xml, "button-update");
	report->spin_report = glade_xml_get_widget (xml, "spinbutton-percentage");
	report->treeview_report = GTK_TREE_VIEW (glade_xml_get_widget (xml,"treeview-report"));
	report->button_print = glade_xml_get_widget (xml, "button-print");

	gtk_widget_set_sensitive (report->button_print, FALSE);

	/* set title */
	title = g_strdup_printf (_("Report: %s"), 
				 gyrus_admin_get_current_session_name (admin));
	gtk_window_set_title (GTK_WINDOW(report->window_report),title);
	
	/* initialize treeview_report */
	report->treeview_report = gyrus_report_initialize_tree_view (report->treeview_report);

	/* signal */
	g_signal_connect (G_OBJECT (report->window_report), "delete_event",
                          G_CALLBACK (gyrus_report_on_delete_event_window_report),
                          report);
	
	g_signal_connect (G_OBJECT (button_cancel), "clicked",
                          G_CALLBACK (gyrus_report_on_button_cancel_clicked),
                          report);

	g_signal_connect (G_OBJECT (button_update), "clicked",
                          G_CALLBACK (gyrus_report_on_button_update_clicked),
                          report);
	
	g_signal_connect (G_OBJECT (report->button_print), "clicked",
                          G_CALLBACK (gyrus_report_on_button_print_clicked),
                          report);
	gtk_widget_show_all (report->window_report);
	g_object_unref (xml);
	
	g_free (title);
	g_free (file_glade);
}


void
gyrus_report_cell_data_func (GtkTreeViewColumn *tree_column,
			     GtkCellRenderer *renderer,
			     GtkTreeModel *tree_model,
			     GtkTreeIter *iter,
			     gpointer data)
{
	gfloat  perc;
	gchar   buf[20];
	
	gtk_tree_model_get(tree_model, iter, 
			   COLUMN_PERCENTAGE, &perc,
			   -1);
	
	g_snprintf(buf, sizeof(buf), 
		   "%.1f", perc);
	
	g_object_set(renderer, "text", 
		     buf, NULL);
}

/* initialize treeview report*/
GtkTreeView *
gyrus_report_initialize_tree_view (GtkTreeView *treeview_report)
{
        GtkTreeViewColumn* column;
        GtkCellRenderer *renderer;
        GtkTreeModel *model;

	gtk_tree_view_set_reorderable  (treeview_report, FALSE);

	/* new column user and cell with pixbuf */
        column = gtk_tree_view_column_new ();
        gtk_tree_view_column_set_title (column, _("Users"));

        renderer = gtk_cell_renderer_pixbuf_new ();
        gtk_tree_view_column_pack_start (column, renderer, FALSE);
	g_object_set (G_OBJECT (renderer),
	              "stock-size", GTK_ICON_SIZE_LARGE_TOOLBAR,
	              NULL);
        gtk_tree_view_column_set_attributes (column, renderer,
					     "stock-id", COLUMN_FOLDER,
					     NULL);
	
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_column_pack_start (column, renderer, FALSE);
        gtk_tree_view_column_set_attributes (column, renderer,
					     "text", COLUMN_MAILBOX,
					     NULL);
	
	gtk_tree_view_append_column (treeview_report, column);

	/* new column quota */
        column = gtk_tree_view_column_new ();
        gtk_tree_view_column_set_title (column, _("Quota (%)"));
        
	renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func(column, renderer, 
						gyrus_report_cell_data_func, NULL, NULL);
        gtk_tree_view_column_set_attributes (column, renderer,
					     "text", COLUMN_PERCENTAGE,
					     NULL);
	gtk_tree_view_append_column (treeview_report, column);

	/* new column assigned */
        column = gtk_tree_view_column_new ();
        gtk_tree_view_column_set_title (column, _("Assigned (KB)"));
        
	renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_column_pack_start (column, renderer, FALSE);
        gtk_tree_view_column_set_attributes (column, renderer,
					     "text", COLUMN_QUOTA_LIMIT,
					     NULL);
	gtk_tree_view_append_column (treeview_report, column);
	
	/* new column used */
        column = gtk_tree_view_column_new ();
        gtk_tree_view_column_set_title (column, _("Used (KB)"));
        
	renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_column_pack_start (column, renderer, FALSE);
        gtk_tree_view_column_set_attributes (column, renderer,
					     "text", COLUMN_QUOTA_USED,
					     NULL);
	gtk_tree_view_append_column (treeview_report, column);
	
	/* create model */
	model = GTK_TREE_MODEL (gtk_list_store_new (NUM_COLUMN,
						    G_TYPE_STRING,
						    G_TYPE_STRING,
						    G_TYPE_INT,
						    G_TYPE_INT,
						    G_TYPE_INT));
	
	gtk_tree_view_set_model (treeview_report, model); 

	return treeview_report;
}

/* close window */
void
gyrus_report_on_delete_event_window_report (GtkWidget *widget, 
					    GdkEvent *event, 
					    GyrusReportData *report)
{
	gyrus_report_on_button_cancel_clicked (NULL,report);
}

/* close window */
void
gyrus_report_on_button_cancel_clicked (GtkWidget *widget, 
				       GyrusReportData *report)
{
	gtk_widget_destroy (report->window_report);
	g_free (report);
}

/* compare each mailbox */
void
gyrus_report_on_button_update_clicked (GtkWidget *widget, 
				       GyrusReportData *report)
{
	GtkListStore *store;
        GtkTreeViewColumn *column;
	GtkTreeView *treeview;
        gint n_users;
        gchar *msg;
	GyrusAdmin *admin;

	admin = report->admin;

	/* get value */
	report->n_percen = gtk_spin_button_get_value (GTK_SPIN_BUTTON(report->spin_report));
	
	/* clear store */
	store = GTK_LIST_STORE (gtk_tree_view_get_model (report->treeview_report));
	gtk_list_store_clear (store);
		
	/* get model of treeview users */
	treeview = gyrus_admin_get_users_treeview (admin);
	GtkTreeModel *model = gtk_tree_view_get_model (treeview);

	/* for each user */
	gtk_tree_model_foreach (model, gyrus_report_evaluate_quota, report);

        /* get the number of users in the tree */
        n_users = gtk_tree_model_iter_n_children (GTK_TREE_MODEL(store), NULL);
        column = gtk_tree_view_get_column 
		(GTK_TREE_VIEW(report->treeview_report), 0);

        msg = g_strdup_printf (_("Users (%d)"), n_users);
        gtk_tree_view_column_set_title (column, msg);
	
	/* enable/disable button print */
	if (n_users > 0){
		gtk_widget_set_sensitive (report->button_print, TRUE);
	}else{
		gtk_widget_set_sensitive (report->button_print, FALSE);
	}
        
	g_free (msg);
	g_object_unref (treeview);
}

/* evaluate percentage */
gboolean  
gyrus_report_evaluate_quota (GtkTreeModel *model, 
			     GtkTreePath *path, 
			     GtkTreeIter *iter, 
			     gpointer data)
{
	GyrusReportData *report = data;
	GyrusAdmin *admin = report->admin;	
	gchar *mailbox;
	gint quota_limit, quota_used;
	gint per_user;
	gchar *msg;
	gfloat temp_per;
	gchar **v_mailbox;
	
	GtkListStore *store;
	GtkTreeIter iter_new;	

	gint per = report->n_percen;	gboolean passed = TRUE;

	store = GTK_LIST_STORE (gtk_tree_view_get_model (report->treeview_report));

	/* get mailbox */
        gtk_tree_model_get (model, iter, 
			    COL_MAILBOX_NAME, &mailbox, 
			    -1);
	
	/* get quota of mailbox */
	if (gyrus_admin_mailbox_get_quota (admin, mailbox, &quota_limit, 
					   &quota_used, &msg) && quota_limit > 0)
	{
#define ALEJANDRO_FILTER
#ifdef ALEJANDRO_FILTER

		/* filter for main folder */
		v_mailbox = g_strsplit (mailbox, gyrus_admin_get_separator_char (admin), 
					-1);
	
		/* first mailbox */
		if (report->mailbox_temp==NULL){
			report->mailbox_temp = v_mailbox[1];
			passed = TRUE;
		}else{
			/* same mailbox? */
			if (strcmp(report->mailbox_temp,v_mailbox[1])==0){
				passed = FALSE;
			}else{
				/* other mailbox */
				report->mailbox_temp = v_mailbox[1];
				passed = TRUE;
			}
		}
		if (passed){
			temp_per = ((quota_used * 100 )/ quota_limit);
			per_user = (gint) temp_per;
			
			if (per_user >= per){
				/* add user to treeview */
				gtk_list_store_append (store, &iter_new);
				gtk_list_store_set (store, &iter_new,
						    COLUMN_FOLDER, GTK_STOCK_DIRECTORY,
						    COLUMN_MAILBOX, v_mailbox[1],
						    COLUMN_PERCENTAGE, per_user,
						    COLUMN_QUOTA_LIMIT, quota_limit,
						    COLUMN_QUOTA_USED, quota_used,
						    -1);
			}
		}
		g_strfreev (v_mailbox);
#else
		temp_per = ((quota_used * 100 )/ quota_limit);
		per_user = (gint) temp_per;

		if (per_user >= per){
			/* add user to treeview */
			gtk_list_store_append (store, &iter_new);
			gtk_list_store_set (store, &iter_new,
					    COLUMN_FOLDER, GTK_STOCK_DIRECTORY,
					    COLUMN_MAILBOX, mailbox,
					    COLUMN_PERCENTAGE, per_user,
					    COLUMN_QUOTA_LIMIT, quota_limit,
					    COLUMN_QUOTA_USED, quota_used,
					    -1);
		}
#endif

	} else {
		g_free (msg);
	}
	g_free (mailbox);

	return FALSE;
}

/* show print dialog */
void
gyrus_report_on_button_print_clicked(GtkWidget *widget, GyrusReportData *report)
{
	GnomePrintJob *job;
	GtkWidget *dialog;
	gint response;

	/* Create the objects */
	job = gnome_print_job_new (NULL);
	dialog = gnome_print_dialog_new (job, (guchar *)_("Print Report"), 0);

	/* Run the dialog */
        response = gnome_print_dialog_run (GNOME_PRINT_DIALOG (dialog));
        switch (response) {
        	case GNOME_PRINT_DIALOG_RESPONSE_PRINT:
			gyrus_report_create_print (job, FALSE, report);
			gtk_widget_destroy (dialog);
			break;
		case GNOME_PRINT_DIALOG_RESPONSE_PREVIEW:
		        gyrus_report_create_print (job, TRUE, report);
			gtk_widget_destroy (dialog);
		        break;
		case GNOME_PRINT_DIALOG_RESPONSE_CANCEL:
			gnome_print_job_close (job);
			gtk_widget_destroy (dialog);
		        break;
		default:
		        return;
	}

 
	///gnome_print_job_close (job);
        g_object_unref (G_OBJECT (job));
}


/* create job to print */
void
gyrus_report_create_print (GnomePrintJob *job, gboolean preview, GyrusReportData *report)
{
	report->gpc = gnome_print_job_get_context (job);

	/* generate report to print */
	gyrus_report_create_data_page (report);

	gnome_print_job_close (job);

        if (!preview) {
        	///my_status_bar_print ("Printing ...");
		gnome_print_job_print (job);
	} else {
		///my_status_bar_print ("Print previewing ...");
		gtk_widget_show (gnome_print_job_preview_new (job, (guchar *)_("Preview")));
        }

        g_object_unref (G_OBJECT (report->gpc));
}

/* print top data */
void
gyrus_report_draw_begin (GyrusReportData *report)
{
        guchar *page_name;
	
	report->xini = XINI;
	report->yini = YINI;
	report->xlin = XLIN;
       
	/* print page numbre */
	page_name = (guchar *) g_strdup_printf ("%d", report->page);
        gnome_print_beginpage (report->gpc, page_name);
        g_free (page_name);
	
	/* draw line top */
        gnome_print_moveto (report->gpc, report->xini, report->yini);
        gnome_print_lineto (report->gpc, report->xlin, report->yini);
        gnome_print_stroke (report->gpc);

	/* titles */
	gnome_print_moveto  (report->gpc, report->xini + 10, report->yini - 20);
	gnome_print_show    (report->gpc, (guchar *) _("User"));
        
	gnome_print_moveto  (report->gpc, report->xini + 120, report->yini - 20);
	gnome_print_show    (report->gpc, (guchar *) _("Quota (%)"));

	gnome_print_moveto  (report->gpc, report->xini + 240, report->yini - 20);
	gnome_print_show    (report->gpc, (guchar *) _("Assigned (KB)"));

	gnome_print_moveto  (report->gpc, report->xini + 360, report->yini - 20);
	gnome_print_show    (report->gpc, (guchar *) _("Used (KB)"));
	
	/* other line */
	gnome_print_moveto (report->gpc, report->xini, report->yini - 30);
        gnome_print_lineto (report->gpc, report->xlin, report->yini - 30);
        gnome_print_stroke (report->gpc);

	report->yini = report->yini - 50;
}


/* sent to print each user, initialize number of page, number of line. */
void
gyrus_report_create_data_page (GyrusReportData *report)
{
	const guchar font[24] = "Sans Regular";
	/*  count number of user printed */
	report->nlin = 1;

	/* number of pages */
	report->page = 1;
	
	/* define font and size */
        report->font = gnome_font_find_closest (font, 8);
        g_assert (report->font);
	
	gyrus_report_draw_begin (report);

	/* get model of treeview users */
	GtkTreeModel *model = gtk_tree_view_get_model (report->treeview_report);
	gtk_tree_model_foreach (model, gyrus_report_print_quota, report);
	
	gnome_print_showpage (report->gpc);
}

/* print quota and other data  */
gboolean
gyrus_report_print_quota (GtkTreeModel *model, GtkTreePath *path, 
			  GtkTreeIter *iter, gpointer data)
{
	GyrusReportData *report = data;
	guchar *mailbox;
	gint quota;
	gint assigned;
	gint used;

	guchar *str_tmp;
	
	gtk_tree_model_get (model, iter, 
			    COLUMN_MAILBOX, &mailbox,
			    COLUMN_PERCENTAGE, &quota, 
			    COLUMN_QUOTA_LIMIT, &assigned,
			    COLUMN_QUOTA_USED, &used, 
			    -1);

	/* print mailbox */
	gnome_print_moveto  (report->gpc, report->xini + 5, report->yini);
	gnome_print_show    (report->gpc, mailbox);
        g_free (mailbox);
	
	/* print quota */
	str_tmp = (guchar *) g_strdup_printf ("%d", quota);
	gnome_print_moveto  (report->gpc, report->xini + 130, report->yini);
	gnome_print_show    (report->gpc, str_tmp);
	g_free (str_tmp);
	
	/* print asigned */
	str_tmp = (guchar *) g_strdup_printf ("%d", assigned);
	gnome_print_moveto  (report->gpc, report->xini + 250, report->yini);
	gnome_print_show    (report->gpc, str_tmp);
	g_free (str_tmp);

	/* print used */
	str_tmp = (guchar *) g_strdup_printf ("%d", used);
	gnome_print_moveto  (report->gpc, report->xini + 370, report->yini);
	gnome_print_show    (report->gpc, str_tmp);
	g_free (str_tmp);
	
	report->yini = report->yini - 15;

	/* evaluate number of line printed */
	if (report->nlin == MAX_USER_PAGE){
		gnome_print_showpage (report->gpc);
		report->nlin = 1;
		report->page = report->page + 1;

		/* new page */
		gyrus_report_draw_begin (report);
	}else{
		report->nlin = report->nlin + 1;
	}

	///g_print ("quota-> %d\n",quota);

	return FALSE;
}
