// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         The table widget (central widget of the application)
// ****************************************************************************

// Copyright 2008, 2009, 2010 Guy Voncken
//
// This file is part of guymager.
//
// guymager is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// guymager is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with guymager. If not, see <http://www.gnu.org/licenses/>.

#include <QtGui>

#include "toolconstants.h"
#include "common.h"
#include "config.h"
#include "threadread.h"
#include "threadwrite.h"
#include "threadhash.h"
#include "threadcompress.h"
#include "dlgmessage.h"
#include "dlgacquire.h"
#include "itemdelegate.h"
#include "table.h"
#include "mainwindow.h"
#include "compileinfo.h"
#include "dlgabort.h"
#include "file.h"


class t_TableLocal
{
   public:
      t_pDeviceList  pDeviceList;
      t_MainWindow  *pMainWindow;
      QAction       *pActionAcquire;
      QAction       *pActionClone;
      QAction       *pActionAbort;
      QAction       *pActionDeviceInfo;
      QAction       *pActionRemoveSpecialDevice;
      bool            SlowDownAcquisitions;
};

t_Table::t_Table ()
{
   CHK_EXIT (ERROR_TABLE_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 not initialised

t_Table::t_Table (QWidget *pParent, t_MainWindow *pMainWindow, t_pDeviceList pDeviceList)
   :QTableView (pParent)
{
//   setStyleSheet("QTableView {background-color: yellow;}");

   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_INVALID_ACTION                ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADREAD_ALREADY_RUNNING    ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADHASH_ALREADY_RUNNING    ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADWRITE_ALREADY_RUNNING   ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADCOMPRESS_ALREADY_RUNNING))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADREAD_DOESNT_EXIT        ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_FIFO_EXISTS                   ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_CONSTRUCTOR_NOT_SUPPORTED     ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_INVALID_FORMAT                ))

   setItemDelegate      (new t_ItemDelegate (this));
   setSelectionBehavior (QAbstractItemView::SelectRows     );
   setSelectionMode     (QAbstractItemView::SingleSelection);

   pOwn = new t_TableLocal;
   pOwn->pDeviceList          = pDeviceList;
   pOwn->pMainWindow          = pMainWindow;
   pOwn->SlowDownAcquisitions = false;

   verticalHeader  ()->hide                  ();
   horizontalHeader()->setClickable          (true);
   horizontalHeader()->setSortIndicatorShown (true);
   setHorizontalScrollMode (QAbstractItemView::ScrollPerPixel);
   setVerticalScrollMode   (QAbstractItemView::ScrollPerPixel);

   pOwn->pActionAcquire             = new QAction(tr("Acquire image"        , "Context menu"), this);  addAction (pOwn->pActionAcquire            );
   pOwn->pActionClone               = new QAction(tr("Clone device"         , "Context menu"), this);  addAction (pOwn->pActionClone              );
   pOwn->pActionAbort               = new QAction(tr("Abort"                , "Context menu"), this);  addAction (pOwn->pActionAbort              );
   pOwn->pActionDeviceInfo          = new QAction(tr("Info"                 , "Context menu"), this);  addAction (pOwn->pActionDeviceInfo         );
   pOwn->pActionRemoveSpecialDevice = new QAction(tr("Remove special device", "Context menu"), this);  addAction (pOwn->pActionRemoveSpecialDevice);

   CHK_QT_EXIT (connect (this, SIGNAL(clicked (const QModelIndex &)), this, SLOT(SlotMouseClick(const QModelIndex &))))


//   pOwn->pTable->setContextMenuPolicy (Qt::ActionsContextMenu);
}

t_Table::~t_Table ()
{
   delete pOwn;
}

APIRET t_Table::GetDeviceUnderCursor (t_pDevice &pDevice)
{
   QModelIndex Index;
   int         Row;
   int         Count;

   pDevice = NULL;
   QModelIndexList IndexList = selectionModel()->selectedRows();
   Count = IndexList.count();
   switch (Count)
   {
      case 0:  break;
      case 1:  Index = IndexList.first();
               Row = model()->data(Index, t_ItemDelegate::RowNrRole).toInt();
               pDevice = pOwn->pDeviceList->at(Row);
               break;
      default: LOG_ERROR ("Strange, several rows seem to be selected.")
   }

   return NO_ERROR;
}

void t_Table::contextMenuEvent (QContextMenuEvent *pEvent)
{
   QAction     *pAction;
   QAction       LocalIndicator (tr("Local Device - cannot be acquired"           ), this);
   QAction       CloneIndicator (tr("Device used for cloning - cannot be acquired"), this);
   QMenu         Menu;
   QPoint        Pos;
   t_pDevice    pDevice;
   int           CorrectedY;
   int           Row;
   QModelIndex   Index;
   bool          Running;

   Pos = mapFromGlobal (pEvent->globalPos());           // The event coordinates are absolute screen coordinates, map them withb this widget
   CorrectedY = Pos.y() - horizontalHeader()->height(); // QTableView::rowAt seems to have a bug: It doesn't include the header height in the calculations. So, we correct it here.
   Pos.setY (CorrectedY);

   Index = indexAt (Pos);
   selectRow (Index.row());
   Row = model()->data(Index, t_ItemDelegate::RowNrRole).toInt();

   if ((Row < 0) || (Row >= pOwn->pDeviceList->count())) // User clicked in the empty area
      return;

   pDevice = pOwn->pDeviceList->at(Row);
//   LOG_INFO ("Context global %d -- widget %d -- corrected %d -- index %d", pEvent->globalPos().y(), Pos.y(), CorrectedY, RowIndex);

   Running = pDevice->pThreadRead || pDevice->pThreadWrite;

   pOwn->pActionAcquire->setEnabled (!Running);
   pOwn->pActionClone  ->setEnabled (!Running);
   pOwn->pActionAbort  ->setEnabled ( Running);

   if (pDevice->Local)
   {
      LocalIndicator.setEnabled (false);
      Menu.addAction (&LocalIndicator);
      Menu.addSeparator ();
   }
   else if (pOwn->pDeviceList->UsedAsCloneDestination(pDevice))
   {
      CloneIndicator.setEnabled (false);
      Menu.addAction (&CloneIndicator);
      Menu.addSeparator ();
   }
   else
   {
      Menu.addAction (pOwn->pActionAcquire);
      Menu.addAction (pOwn->pActionClone  );
      Menu.addAction (pOwn->pActionAbort  );
   }
   Menu.addAction (pOwn->pActionDeviceInfo);

   if (pDevice->SpecialDevice)
   {
      pOwn->pActionRemoveSpecialDevice->setEnabled (!Running);
      Menu.addAction (pOwn->pActionRemoveSpecialDevice);
   }

   pAction = Menu.exec (pEvent->globalPos());

//   pAction = QMenu::exec (actions(), pEvent->globalPos());
   if      (pAction == pOwn->pActionAcquire            ) CHK_EXIT (StartAcquisition    (pDevice, false))
   else if (pAction == pOwn->pActionClone              ) CHK_EXIT (StartAcquisition    (pDevice, true ))
   else if (pAction == pOwn->pActionAbort              ) CHK_EXIT (AbortAcquisition    (pDevice))
   else if (pAction == pOwn->pActionDeviceInfo         ) CHK_EXIT (ShowDeviceInfo      (pDevice))
   else if (pAction == pOwn->pActionRemoveSpecialDevice) CHK_EXIT (pOwn->pMainWindow->RemoveSpecialDevice(pDevice))
   else
   {
//      CHK_EXIT (ERROR_TABLE_INVALID_ACTION)
   }
}

void t_Table::SlotMouseClick (const QModelIndex & Index)
{
   int Row;

   selectRow (Index.row());
   Row = model()->data (Index, t_ItemDelegate::RowNrRole).toInt();
   emit (SignalDeviceSelected (pOwn->pDeviceList->at(Row)));
}


APIRET t_Table::ShowDeviceInfo (t_pcDevice pDevice)
{
   QString Info;

   LOG_INFO ("Info for %s requested", QSTR_TO_PSZ(pDevice->LinuxDevice))
   pOwn->SlowDownAcquisitions = true;
   CHK (t_Info::GetDeviceInfo (pDevice, true, Info))
   pOwn->SlowDownAcquisitions = false;
   CHK (t_DlgMessage::Show (tr("Device info", "Dialog title"), Info, true))

   return NO_ERROR;
}


// ----------------------------------------------------------------------------------------------------------------
//
//                                                     Acquistion start
//
// ----------------------------------------------------------------------------------------------------------------

APIRET t_Table::InfoAcquisitionStart (t_pDevice pDevice)
{
   const char *pLibGuyToolsVersionInstalled;
   const char *pLibEwfVersionInstalled;
   QString      FormatDescription;
   QString      FormatExtension;
   QString      Str;

   CHK (pDevice->Info.Create())
   CHK (pDevice->Info.Title   (tr("GUYMAGER ACQUISITION INFO FILE", "Info file")))

   t_Log::GetLibGuyToolsVersion (&pLibGuyToolsVersionInstalled);
   pLibEwfVersionInstalled = (const char *) libewf_get_version ();

   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.Title   (tr("Guymager", "Info file")))
   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.AddRow (tr("Version:: %1"              , "Info file") .arg(pCompileInfoVersion  )))
   CHK (pDevice->Info.AddRow (tr("Compilation timestamp:: %1", "Info file") .arg(pCompileInfoTimestamp)))
   CHK (pDevice->Info.AddRow (tr("Compiled with:: %1 %2"     , "Info file") .arg("gcc") .arg(__VERSION__)))
   CHK (pDevice->Info.AddRow (tr("libewf version:: %1"       , "Info file") .arg(pLibEwfVersionInstalled     )))
   CHK (pDevice->Info.AddRow (tr("libguytools version:: %1"  , "Info file") .arg(pLibGuyToolsVersionInstalled)))
   CHK (pDevice->Info.WriteTable ());

   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.Title   (tr("Device information", "Info file")))
   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.WriteDeviceInfo ())


   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.Title   (tr("Acquisition", "Info file")))
   CHK (pDevice->Info.WriteLn ())
   CHK (t_File::GetFormatDescription (pDevice,        FormatDescription))
   CHK (t_File::GetFormatExtension   (pDevice, NULL, &FormatExtension))
   CHK (pDevice->Info.AddRow (tr("Linux device:: %1"    , "Info file").arg(pDevice->LinuxDevice)))
   CHK (pDevice->Info.AddRow (tr("Device size:: %1 (%2)", "Info file").arg(pDevice->Size) .arg(t_Device::GetSizeHuman(pDevice).toString())))
   Str = tr("Format:: %1", "Info file") .arg(FormatDescription);
   if (!FormatExtension.isEmpty())
      Str += tr(" - file extension is %1", "Info file") .arg(FormatExtension);
   CHK (pDevice->Info.AddRow (Str))
   if (pDevice->Acquisition.Format != t_File::DD)
   {
      CHK (pDevice->Info.AddRow (tr("Image meta data", "Info file")))
      CHK (pDevice->Info.AddRow ("   " + t_DlgAcquire::tr("EwfCaseNumber"    ) + QString(":: %1") .arg(pDevice->Acquisition.CaseNumber    )))
      CHK (pDevice->Info.AddRow ("   " + t_DlgAcquire::tr("EwfEvidenceNumber") + QString(":: %1") .arg(pDevice->Acquisition.EvidenceNumber)))
      CHK (pDevice->Info.AddRow ("   " + t_DlgAcquire::tr("EwfExaminer"      ) + QString(":: %1") .arg(pDevice->Acquisition.Examiner      )))
      CHK (pDevice->Info.AddRow ("   " + t_DlgAcquire::tr("EwfDescription"   ) + QString(":: %1") .arg(pDevice->Acquisition.Description   )))
      CHK (pDevice->Info.AddRow ("   " + t_DlgAcquire::tr("EwfNotes"         ) + QString(":: %1") .arg(pDevice->Acquisition.Notes         )))

   }
   CHK (pDevice->Info.AddRow (tr("Image path and file name:: %1", "Info file").arg(pDevice->Acquisition.ImagePath + pDevice->Acquisition.ImageFilename + FormatExtension)))
   CHK (pDevice->Info.AddRow (tr("Info  path and file name:: %1", "Info file").arg(pDevice->Acquisition.InfoPath  + pDevice->Acquisition.InfoFilename  + t_File::pExtensionInfo)))

   if     ((pDevice->Acquisition.CalcMD5) &&
           (pDevice->Acquisition.CalcSHA256)) CHK (pDevice->Info.AddRow (tr("Hash calculation:: MD5 and SHA-256", "Info file")))
   else if (pDevice->Acquisition.CalcMD5)     CHK (pDevice->Info.AddRow (tr("Hash calculation:: MD5"            , "Info file")))
   else if (pDevice->Acquisition.CalcSHA256)  CHK (pDevice->Info.AddRow (tr("Hash calculation:: SHA-256"        , "Info file")))
   else                                       CHK (pDevice->Info.AddRow (tr("Hash calculation:: off"            , "Info file")))

   if (pDevice->Acquisition.VerifySrc)
        CHK (pDevice->Info.AddRow (tr("Source verification:: on" , "Info file")))
   else CHK (pDevice->Info.AddRow (tr("Source verification:: off", "Info file")))

   if (pDevice->Acquisition.VerifyDst)
        CHK (pDevice->Info.AddRow (tr("Image verification:: on" , "Info file")))
   else CHK (pDevice->Info.AddRow (tr("Image verification:: off", "Info file")))

   CHK (pDevice->Info.WriteTable ());

   return NO_ERROR;
}

APIRET TableFifoCalcSizeAndAlloc (t_pDevice pDevice, unsigned int Fifos, unsigned int *pMaxBlocksInUse)
{
   const unsigned int PageSize = getpagesize();
   unsigned int       MaxBlocksInUse;

   // Set the data block size
   // -----------------------
   switch (pDevice->Acquisition.Format)
   {
      case t_File::DD : pDevice->FifoBlockSize = CONFIG(FifoBlockSizeDD );   break;
      case t_File::AFF: pDevice->FifoBlockSize = CONFIG(FifoBlockSizeAFF);   break;
      case t_File::EWF:
         if (pDevice->HasCompressionThreads())
              pDevice->FifoBlockSize = EWF_MULTITHREADED_COMPRESSION_CHUNK_SIZE;
         else pDevice->FifoBlockSize = CONFIG(FifoBlockSizeEWF);
         if (pDevice->FifoBlockSize != (unsigned int) CONFIG(FifoBlockSizeEWF))
            LOG_INFO ("Running with FifoBlockSize of %d (no other size supported when running with multithreaded EWF compression)", pDevice->FifoBlockSize)
         break;
      default:  CHK (ERROR_TABLE_INVALID_FORMAT)
   }

   // Calc the real amount of memory used for every block
   // ---------------------------------------------------
   pDevice->FifoAllocBlockSize = pDevice->FifoBlockSize;    // Start with the standard data size

   if (pDevice->HasCompressionThreads())                    // Add what might be necessary for compressed  data
      pDevice->FifoAllocBlockSize += (int)(pDevice->FifoBlockSize * 0.001) + 12;  // Reserve enough room so that compressed data its in the block.

   pDevice->FifoAllocBlockSize += sizeof (t_FifoBlock);     // Add the structure overhead

//   if (!CONFIG (FifoMemoryManager))                       // Go to the next boundary when working with C-Lib memory management
//   {
      if ((pDevice->FifoAllocBlockSize % PageSize) != 0)
         pDevice->FifoAllocBlockSize += PageSize - (pDevice->FifoAllocBlockSize % PageSize);
//   }

   // Calc the number of entries for each FIFO
   // ----------------------------------------
   pDevice->FifoMaxBlocks = (int)((((long long)CONFIG(FifoMaxMem))*BYTES_PER_MEGABYTE) / ((long long)pDevice->FifoAllocBlockSize * Fifos));
   if (pDevice->FifoMaxBlocks == 0)
      pDevice->FifoMaxBlocks = 1;

   // Calc the amount of memory for the blocks and allocate it
   // --------------------------------------------------------
   MaxBlocksInUse = pDevice->FifoMaxBlocks * Fifos;  // Data in the FIFOs
   MaxBlocksInUse += 1;                              // ThreadRead might allocate an additional block that can't yet be written to the full FIFO
   if ((pDevice->Acquisition.Format == t_File::AFF) && !pDevice->HasCompressionThreads())
      MaxBlocksInUse += 1;                           // ThreadWrite might use an additional buffer if it does the AFF preprocessing
   if (pDevice->HasCompressionThreads())
      MaxBlocksInUse += CONFIG(CompressionThreads);  // Each compression thread might have one additional block for writing the compressed data
   MaxBlocksInUse += Fifos;
   *pMaxBlocksInUse = MaxBlocksInUse;

   if (CONFIG (FifoMemoryManager))
      CHK (FifoMemoryAlloc (&pDevice->pFifoMemory, MaxBlocksInUse, pDevice->FifoAllocBlockSize))

   return NO_ERROR;
}

APIRET t_Table::StartAcquisition (t_pDevice pDevice, bool Clone)
{
   t_DlgAcquire Dlg (pDevice, Clone, pOwn->pDeviceList, this);
   QString      Format;
   int          Fifos = 0;

   if (Dlg.exec() != QDialog::Accepted)
      return NO_ERROR;

   pOwn->SlowDownAcquisitions = true;
   LOG_INFO ("Starting acquisition (%s) for %s", Clone ? "clone":"image", QSTR_TO_PSZ(pDevice->LinuxDevice))
   CHK (Dlg.GetParameters (pDevice->Acquisition))
   CHK (t_File::GetFormatExtension  (pDevice->Acquisition.Format, pDevice->Acquisition.Clone, NULL, &Format))
   if (Clone)
      Format = " (clone)";
   LOG_INFO ("Image %s%s%s", QSTR_TO_PSZ(pDevice->Acquisition.ImagePath), QSTR_TO_PSZ(pDevice->Acquisition.ImageFilename), QSTR_TO_PSZ(Format))
   LOG_INFO ("Info  %s%s%s", QSTR_TO_PSZ(pDevice->Acquisition.InfoPath ), QSTR_TO_PSZ(pDevice->Acquisition.InfoFilename ), t_File::pExtensionInfo)

   CHK (pDevice->SetMessage (QString()))
   pDevice->State               = t_Device::Acquire;
   pDevice->AbortReason         = t_Device::None;
   pDevice->AbortRequest        = false;
   pDevice->AbortCount          = 0;
   pDevice->DeleteAfterAbort    = false;
   pDevice->StartTimestampVerify= QDateTime();
   pDevice->StopTimestamp       = QDateTime();
   pDevice->PrevTimestamp       = QTime();
   pDevice->PrevSpeed           = 0.0;
   pDevice->PrevPos             = 0;
   pDevice->SetCurrentWritePos    (0LL);
   pDevice->SetCurrentVerifyPosSrc(0LL);
   pDevice->SetCurrentVerifyPosDst(0LL);
   CHK (pDevice->ClearBadSectors ())

   memset (&pDevice->MD5Digest            , 0, sizeof(pDevice->MD5Digest            ));
   memset (&pDevice->MD5DigestVerifySrc   , 0, sizeof(pDevice->MD5DigestVerifySrc   ));
   memset (&pDevice->MD5DigestVerifyDst   , 0, sizeof(pDevice->MD5DigestVerifyDst   ));
   memset (&pDevice->SHA256Digest         , 0, sizeof(pDevice->SHA256Digest         ));
   memset (&pDevice->SHA256DigestVerifySrc, 0, sizeof(pDevice->SHA256DigestVerifySrc));
   memset (&pDevice->SHA256DigestVerifyDst, 0, sizeof(pDevice->SHA256DigestVerifyDst));
   pDevice->StartTimestamp = QDateTime::currentDateTime();
   CHK (InfoAcquisitionStart (pDevice))

   if (pDevice->pThreadRead  != NULL) CHK_EXIT (ERROR_TABLE_THREADREAD_ALREADY_RUNNING )
   if (pDevice->pThreadWrite != NULL) CHK_EXIT (ERROR_TABLE_THREADWRITE_ALREADY_RUNNING)
   if (pDevice->pThreadHash  != NULL) CHK_EXIT (ERROR_TABLE_THREADHASH_ALREADY_RUNNING )
   if (!pDevice->ThreadCompressList.isEmpty()) CHK_EXIT (ERROR_TABLE_THREADCOMPRESS_ALREADY_RUNNING)
   if ((pDevice->pFifoRead        != NULL) ||
       (pDevice->pFifoHashIn      != NULL) ||
       (pDevice->pFifoHashOut     != NULL) ||
       (pDevice->pFifoCompressIn  != NULL) ||
       (pDevice->pFifoCompressOut != NULL))
      CHK_EXIT (ERROR_TABLE_FIFO_EXISTS)

   // Create threads
   // --------------
   pDevice->pThreadRead  = new t_ThreadRead  (pDevice, &pOwn->SlowDownAcquisitions);
   pDevice->pThreadWrite = new t_ThreadWrite (pDevice, &pOwn->SlowDownAcquisitions);
   CHK_QT_EXIT (connect (pDevice->pThreadRead , SIGNAL(SignalEnded       (t_pDevice)), this, SLOT(SlotThreadReadFinished    (t_pDevice))))
   CHK_QT_EXIT (connect (pDevice->pThreadWrite, SIGNAL(SignalEnded       (t_pDevice)), this, SLOT(SlotThreadWriteFinished   (t_pDevice))))
   CHK_QT_EXIT (connect (pDevice->pThreadWrite, SIGNAL(SignalFreeMyHandle(t_pDevice)), this, SLOT(SlotThreadWriteWakeThreads(t_pDevice))))

   if (pDevice->HasHashThread())
   {
      pDevice->pThreadHash = new t_ThreadHash (pDevice);
      CHK_QT_EXIT (connect (pDevice->pThreadHash, SIGNAL(SignalEnded(t_pDevice)), this, SLOT(SlotThreadHashFinished (t_pDevice))))
   }

   if (pDevice->HasCompressionThreads())
   {
      for (int i=0; i < CONFIG(CompressionThreads); i++)
      {
         t_pThreadCompress pThread;

         pThread = new t_ThreadCompress (pDevice, i);
         pDevice->ThreadCompressList.append (pThread);
         CHK_QT_EXIT (connect (pThread, SIGNAL(SignalEnded(t_pDevice,int)), this, SLOT(SlotThreadCompressFinished (t_pDevice,int))))
      }
   }

   // Create fifos and assign threads
   // -------------------------------
   unsigned int MaxBlocksInUse;

   if      (!pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
   {
      Fifos = 1;
      CHK (TableFifoCalcSizeAndAlloc (pDevice, Fifos, &MaxBlocksInUse))
      pDevice->pFifoRead  = new t_FifoStd (pDevice->pFifoMemory, pDevice->FifoMaxBlocks);
      pDevice->pFifoWrite = pDevice->pFifoRead;
   }
   else if ( pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
   {
      Fifos = 2;
      CHK (TableFifoCalcSizeAndAlloc (pDevice, Fifos, &MaxBlocksInUse))
      pDevice->pFifoRead    = new t_FifoStd (pDevice->pFifoMemory, pDevice->FifoMaxBlocks);
      pDevice->pFifoHashIn  = pDevice->pFifoRead;
      pDevice->pFifoHashOut = new t_FifoStd (pDevice->pFifoMemory, pDevice->FifoMaxBlocks);
      pDevice->pFifoWrite   = pDevice->pFifoHashOut;
   }
   else if (!pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
   {
      Fifos = 2 * CONFIG(CompressionThreads);
      CHK (TableFifoCalcSizeAndAlloc (pDevice, Fifos, &MaxBlocksInUse))
      pDevice->pFifoCompressIn  = new t_FifoCompressIn  (pDevice->pFifoMemory, CONFIG(CompressionThreads), pDevice->FifoMaxBlocks);
      pDevice->pFifoRead        = pDevice->pFifoCompressIn;
      pDevice->pFifoCompressOut = new t_FifoCompressOut (pDevice->pFifoMemory, CONFIG(CompressionThreads), pDevice->FifoMaxBlocks);
      pDevice->pFifoWrite       = pDevice->pFifoCompressOut;
   }
   else if ( pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
   {
      Fifos = 1 + 2 * CONFIG(CompressionThreads);
      CHK (TableFifoCalcSizeAndAlloc (pDevice, Fifos, &MaxBlocksInUse))
      pDevice->pFifoRead        = new t_FifoStd (pDevice->pFifoMemory, pDevice->FifoMaxBlocks);
      pDevice->pFifoHashIn      = pDevice->pFifoRead;
      pDevice->pFifoCompressIn  = new t_FifoCompressIn  (pDevice->pFifoMemory, CONFIG(CompressionThreads), pDevice->FifoMaxBlocks);
      pDevice->pFifoHashOut     = pDevice->pFifoCompressIn;
      pDevice->pFifoCompressOut = new t_FifoCompressOut (pDevice->pFifoMemory, CONFIG(CompressionThreads), pDevice->FifoMaxBlocks);
      pDevice->pFifoWrite       = pDevice->pFifoCompressOut;
   }

   LOG_INFO ("Acquisition runs with %d fifos, each one with space for %d blocks of %d bytes (%0.1f MB FIFO space)",
             Fifos, pDevice->FifoMaxBlocks, pDevice->FifoBlockSize,
             ((double)Fifos * pDevice->FifoMaxBlocks * pDevice->FifoBlockSize) / BYTES_PER_MEGABYTE)
   LOG_INFO ("Including the overhead, %u blocks of %d bytes have been allocated (%0.1f MB)",
             MaxBlocksInUse, pDevice->FifoAllocBlockSize,
             ((double)MaxBlocksInUse * pDevice->FifoAllocBlockSize) / BYTES_PER_MEGABYTE)

   // Start threads
   // -------------
   pDevice->pThreadRead ->start(QThread::LowPriority);
   if (pDevice->HasHashThread())
      pDevice->pThreadHash->start(QThread::LowPriority);
   for (int i=0; i < pDevice->ThreadCompressList.count(); i++)
      pDevice->ThreadCompressList[i]->start(QThread::LowPriority);
   pDevice->pThreadWrite->start(QThread::LowPriority);

   pOwn->SlowDownAcquisitions = false;

   return NO_ERROR;
}


// ----------------------------------------------------------------------------------------------------------------
//
//                                                     Acquistion end
//
// ----------------------------------------------------------------------------------------------------------------

APIRET t_Table::InfoAcquisitionBadSectors (t_pDevice pDevice, bool Verify)
{
   QList<quint64> BadSectors;
   quint64        Count, i;
   quint64        From, To, Next;
   int            LineEntries    =  0;
   const int      MaxLineEntries = 10;
   bool           First          = true;

   CHK (pDevice->GetBadSectors (BadSectors, Verify))
   Count = BadSectors.count();
   if (Count)
   {
      if (Verify)
           LOG_INFO ("During verification, %lld bad sectors have been encountered", Count)
      else LOG_INFO ("During acquisition, %lld bad sectors have been encountered", Count)

      if (Verify)
           CHK (pDevice->Info.WriteLn (tr("During verification, %1 bad sectors have been encountered. The sector numbers are:", "Info file") .arg(Count)))
      else CHK (pDevice->Info.WriteLn (tr("During acquisition, %1 bad sectors have been encountered. They have been replaced by zeroed sectors. The sector numbers are:", "Info file") .arg(Count)))
      CHK (pDevice->Info.WriteLn ("   "))
      i = 0;
      while (i<Count)
      {
         From = BadSectors.at(i++);
         To   = From;
         while (i < Count)
         {
            Next = BadSectors.at(i);
            if (Next != To+1)
               break;
            To = Next;
            i++;
         }
         if (!First)
         {
            if (LineEntries >= MaxLineEntries)
            {
               CHK (pDevice->Info.Write   (","))
               CHK (pDevice->Info.WriteLn ("   "))
               LineEntries = 0;
            }
            else
            {
               CHK (pDevice->Info.Write (", "))
            }
         }
         First = false;
         if (From == To)
         {
            CHK (pDevice->Info.Write ("%Lu", From))
            LineEntries++;
         }
         else
         {
            CHK (pDevice->Info.Write ("%Lu-%Lu", From, To))
            LineEntries += 2;
         }
      }
   }
   else
   {
      if (Verify)
      {
         LOG_INFO ("No bad sectors encountered during verification.")
         CHK (pDevice->Info.WriteLn (tr("No bad sectors encountered during verification.", "Info file")))
      }
      else
      {
         LOG_INFO ("No bad sectors encountered during acquisition.")
         CHK (pDevice->Info.WriteLn (tr("No bad sectors encountered during acquisition.", "Info file")))
      }
   }

   return NO_ERROR;
}

static APIRET TableTimestampToISO (const QDateTime &Timestamp, QString &Str)
{
   Str = Timestamp.toString (Qt::ISODate);
   Str.replace ("T", " ");
   return NO_ERROR;
}

APIRET t_Table::InfoAcquisitionEnd (t_pDevice pDevice)
{
   QString    MD5;
   QString    MD5VerifySrc;
   QString    MD5VerifyDst;
   QString    SHA256;
   QString    SHA256VerifySrc;
   QString    SHA256VerifyDst;
   QString    StateStr;
   quint64    BadSectors;
   QString    StartStr, StopStr, VerifyStr;
   int        Hours, Minutes, Seconds;
   double     Speed;
   bool       MatchSrc = true;
   bool       MatchDst = true;

   CHK (pDevice->Info.WriteLn ())
   CHK (InfoAcquisitionBadSectors (pDevice, false))
   if (pDevice->Acquisition.VerifySrc)
      CHK (InfoAcquisitionBadSectors (pDevice, true))
   MD5 = tr("--", "Info file");
   MD5VerifySrc    = MD5;
   MD5VerifyDst    = MD5;
   SHA256          = MD5;
   SHA256VerifySrc = MD5;
   SHA256VerifyDst = MD5;

   switch (pDevice->State)
   {
      case t_Device::Finished:
         StateStr = tr("Finished successfully", "Info file");
         BadSectors = t_Device::GetBadSectorCount(pDevice).toULongLong();
         if (BadSectors)
            StateStr += " " + tr("(with %1 bad sectors)", "Info file, may be appended to 'finished successfully' message") .arg(BadSectors);
         if (pDevice->Acquisition.CalcMD5   ) CHK (HashMD5DigestStr    (&pDevice->MD5Digest   , MD5   ))
         if (pDevice->Acquisition.CalcSHA256) CHK (HashSHA256DigestStr (&pDevice->SHA256Digest, SHA256))
         if (pDevice->Acquisition.VerifySrc)
         {
            if (pDevice->Acquisition.CalcMD5   ) CHK (HashMD5DigestStr    (&pDevice->MD5DigestVerifySrc   , MD5VerifySrc   ))
            if (pDevice->Acquisition.CalcSHA256) CHK (HashSHA256DigestStr (&pDevice->SHA256DigestVerifySrc, SHA256VerifySrc))
         }
         if (pDevice->Acquisition.VerifyDst)
         {
            if (pDevice->Acquisition.CalcMD5   ) CHK (HashMD5DigestStr    (&pDevice->MD5DigestVerifyDst   , MD5VerifyDst   ))
            if (pDevice->Acquisition.CalcSHA256) CHK (HashSHA256DigestStr (&pDevice->SHA256DigestVerifyDst, SHA256VerifyDst))
         }
         break;

      case t_Device::Aborted:
         switch (pDevice->AbortReason)
         {
            case t_Device::UserRequest:
               StateStr = tr("Aborted by user", "Info file") + " ";
               if (pDevice->StartTimestampVerify.isNull())
               {
                  StateStr += tr("(during acquisition)"     , "Info file");
               }
               else
               {
                  StateStr += tr("(during source verification)", "Info file");
                  if (pDevice->Acquisition.CalcMD5   ) CHK (HashMD5DigestStr    (&pDevice->MD5Digest   , MD5   )) // If the user interrupted during verification (i.e. after
                  if (pDevice->Acquisition.CalcSHA256) CHK (HashSHA256DigestStr (&pDevice->SHA256Digest, SHA256)) // acquisition), we can calculate the acquisition hash.
               }
               break;
            case t_Device::ThreadWriteWriteError : StateStr = tr("Aborted because of image write error"                   , "Info file"); break;
            case t_Device::ThreadWriteVerifyError: StateStr = tr("Aborted because of image read error during verification", "Info file"); break;
            default:                               StateStr = tr("Aborted, strange reason (%1)"                           , "Info file") .arg(pDevice->AbortReason); break;
         }
         break;

      default:
         StateStr = tr("Strange state (%1)", "Info file") .arg(pDevice->State);
   }
   CHK (pDevice->Info.WriteLn (tr("State: %1", "Info file") .arg(StateStr)))
   CHK (pDevice->Info.WriteLn ())

   CHK (pDevice->Info.AddRow (tr("MD5 hash:: %1"                   , "Info file") .arg(MD5            )))
   CHK (pDevice->Info.AddRow (tr("MD5 hash verified source:: %1"   , "Info file") .arg(MD5VerifySrc   )))
   CHK (pDevice->Info.AddRow (tr("MD5 hash verified image:: %1"    , "Info file") .arg(MD5VerifyDst   )))
   CHK (pDevice->Info.AddRow (tr("SHA256 hash:: %1"                , "Info file") .arg(SHA256         )))
   CHK (pDevice->Info.AddRow (tr("SHA256 hash verified source:: %1", "Info file") .arg(SHA256VerifySrc)))
   CHK (pDevice->Info.AddRow (tr("SHA256 hash verified image:: %1" , "Info file") .arg(SHA256VerifyDst)))
   CHK (pDevice->Info.WriteTable ());

   if ((pDevice->State == t_Device::Finished) && pDevice->Acquisition.VerifySrc)
   {
      if (pDevice->Acquisition.CalcMD5)    MatchSrc =            HashMD5Match   (&pDevice->MD5Digest   , &pDevice->MD5DigestVerifySrc   );
      if (pDevice->Acquisition.CalcSHA256) MatchSrc = MatchSrc | HashSHA256Match(&pDevice->SHA256Digest, &pDevice->SHA256DigestVerifySrc);

      if (MatchSrc) CHK (pDevice->Info.WriteLn (tr ("Source verification OK. The device delivered the same data during acquisition and verification.")))
      else          CHK (pDevice->Info.WriteLn (tr ("Source verification FAILED. The device didn't deliver the same data during acquisition and verification. "
                                                    "Check if the defect sector list was the same during acquisition and verification (see above).")))
   }
   if ((pDevice->State == t_Device::Finished) && pDevice->Acquisition.VerifyDst)
   {
      if (pDevice->Acquisition.CalcMD5)    MatchDst =            HashMD5Match   (&pDevice->MD5Digest   , &pDevice->MD5DigestVerifyDst   );
      if (pDevice->Acquisition.CalcSHA256) MatchDst = MatchDst | HashSHA256Match(&pDevice->SHA256Digest, &pDevice->SHA256DigestVerifyDst);

      if (MatchDst) CHK (pDevice->Info.WriteLn (tr ("Image verification OK. The image contains exactely the data that was written.")))
      else          CHK (pDevice->Info.WriteLn (tr ("Image verification FAILED. The data in the image is different from what was written.")))
   }
   if (!MatchSrc || !MatchDst)
      CHK (pDevice->Info.WriteLn (tr ("Maybe you try to acquire the device again.")))

   CHK (pDevice->Info.WriteLn ())

   // Performance
   // -----------
   CHK (TableTimestampToISO (pDevice->StartTimestamp, StartStr))
   CHK (TableTimestampToISO (pDevice->StopTimestamp , StopStr ))
   StartStr += " " + tr("(ISO format YYYY-MM-DD HH:MM:SS)");
   Seconds  = pDevice->StartTimestamp.secsTo (pDevice->StopTimestamp);
   Hours    = Seconds / SECONDS_PER_HOUR  ; Seconds -= Hours   * SECONDS_PER_HOUR;
   Minutes  = Seconds / SECONDS_PER_MINUTE; Seconds -= Minutes * SECONDS_PER_MINUTE;

   if (pDevice->Acquisition.VerifySrc || pDevice->Acquisition.VerifyDst)
   {
      if (!pDevice->StartTimestampVerify.isNull())
           CHK (TableTimestampToISO (pDevice->StartTimestampVerify , VerifyStr))
      else VerifyStr = tr ("Acquisition aborted before start of verification");

      CHK (pDevice->Info.AddRow (tr("Acquisition started:: %1" , "Info file") .arg(StartStr )))
      CHK (pDevice->Info.AddRow (tr("Verification started:: %1", "Info file") .arg(VerifyStr)))
      CHK (pDevice->Info.AddRow (tr("Ended:: %1 (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(StopStr ) .arg(Hours) .arg(Minutes) .arg(Seconds)))

      if (pDevice->StartTimestampVerify.isNull())
           Seconds = GETMAX (pDevice->StartTimestamp.secsTo (pDevice->StopTimestamp)       , 1);
      else Seconds = GETMAX (pDevice->StartTimestamp.secsTo (pDevice->StartTimestampVerify), 1);
      Speed = (double) pDevice->CurrentReadPos / ((double)BYTES_PER_MEGABYTE * (double)Seconds);
      Hours   = Seconds / SECONDS_PER_HOUR  ; Seconds -= Hours   * SECONDS_PER_HOUR;
      Minutes = Seconds / SECONDS_PER_MINUTE; Seconds -= Minutes * SECONDS_PER_MINUTE;
      CHK (pDevice->Info.AddRow (tr("Acquisition speed:: %1 MByte/s (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(Speed, 0, 'f', 2) .arg(Hours) .arg(Minutes) .arg(Seconds)))

      if (!pDevice->StartTimestampVerify.isNull())
      {
         Seconds = GETMAX (pDevice->StartTimestampVerify.secsTo (pDevice->StopTimestamp), 1);
         Speed = (double) pDevice->CurrentReadPos / ((double)BYTES_PER_MEGABYTE * (double)Seconds);
         Hours   = Seconds / SECONDS_PER_HOUR  ; Seconds -= Hours   * SECONDS_PER_HOUR;
         Minutes = Seconds / SECONDS_PER_MINUTE; Seconds -= Minutes * SECONDS_PER_MINUTE;
         CHK (pDevice->Info.AddRow (tr("Verification speed:: %1 MByte/s (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(Speed, 0, 'f', 2) .arg(Hours) .arg(Minutes) .arg(Seconds)))
      }
      CHK (pDevice->Info.WriteTable ());
   }
   else
   {
      CHK (pDevice->Info.AddRow (tr("Acquisition started:: %1"                        , "Info file").arg(StartStr)))
      CHK (pDevice->Info.AddRow (tr("Ended:: %1 (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(StopStr ) .arg(Hours) .arg(Minutes) .arg(Seconds)))

      Seconds = GETMAX (pDevice->StartTimestamp.secsTo (pDevice->StopTimestamp), 1);
      Speed = (double) pDevice->CurrentReadPos / ((double)BYTES_PER_MEGABYTE * (double)Seconds);
      Hours   = Seconds / SECONDS_PER_HOUR  ; Seconds -= Hours   * SECONDS_PER_HOUR;
      Minutes = Seconds / SECONDS_PER_MINUTE; Seconds -= Minutes * SECONDS_PER_MINUTE;
      CHK (pDevice->Info.AddRow (tr("Acquisition speed:: %1 MByte/s (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(Speed, 0, 'f', 2) .arg(Hours) .arg(Minutes) .arg(Seconds)))
      CHK (pDevice->Info.WriteTable ());
   }
   CHK (pDevice->Info.WriteLn ());
   CHK (pDevice->Info.WriteLn ());

   return NO_ERROR;
}

APIRET t_Table::FinaliseThreadStructs (t_pDevice pDevice)
{
   if ((pDevice->pThreadRead  == NULL) &&        // t_Table::FinaliseThreadStructs is called upon every thread's end,
       (pDevice->pThreadWrite == NULL) &&        // but its core only is executed after the last thread ended (as
       (pDevice->pThreadHash  == NULL) &&        // we must not kill vital structures as long as any threads are
       (pDevice->ThreadCompressList.isEmpty()))  // still running).
   {
      pDevice->StopTimestamp = QDateTime::currentDateTime();

      if      (!pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         delete pDevice->pFifoRead;
      }
      else if ( pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         delete pDevice->pFifoRead;
         delete pDevice->pFifoHashOut;
      }
      else if (!pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
      {
         delete pDevice->pFifoCompressIn;
         delete pDevice->pFifoCompressOut;
      }
      else if ( pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
      {
         delete pDevice->pFifoRead;
         delete pDevice->pFifoCompressIn;
         delete pDevice->pFifoCompressOut;
      }

      pDevice->pFifoRead        = NULL;
      pDevice->pFifoHashIn      = NULL;
      pDevice->pFifoHashOut     = NULL;
      pDevice->pFifoWrite       = NULL;
      pDevice->pFifoCompressIn  = NULL;
      pDevice->pFifoCompressOut = NULL;


      if (CONFIG (FifoMemoryManager))
         CHK (FifoMemoryFree (&pDevice->pFifoMemory))

//      LibEwfGetMemStats (&LibEwfAllocs, &LibEwfFrees);
//      LOG_INFO ("LIBEWF mem  statistics: %d allocated - %d freed = %d remaining", LibEwfAllocs, LibEwfFrees, LibEwfAllocs - LibEwfFrees)

      LOG_INFO ("Acquisition of %s: All structures cleaned.", QSTR_TO_PSZ (pDevice->LinuxDevice))

      if (pDevice->AbortRequest)
           pDevice->State = t_Device::Aborted;
      else pDevice->State = t_Device::Finished;

      if (!pDevice->DeleteAfterAbort)
         CHK_EXIT (InfoAcquisitionEnd (pDevice))

      // Check if all acqusitions have completed now (for AutoExit)
      // ----------------------------------------------------------
      t_Device::t_State State;
      int     i;
      bool    AllCompleted = true;

      for (i=0; i<pOwn->pDeviceList->count() && AllCompleted; i++)
      {
         State = pOwn->pDeviceList->at(i)->State;
         AllCompleted =  (State == t_Device::Idle    ) ||
                         (State == t_Device::Finished) ||
                        ((State == t_Device::Aborted ) && (pOwn->pDeviceList->at(i)->AbortReason == t_Device::UserRequest));
      }
      if (AllCompleted)
         emit (SignalAllAcquisitionsEnded());
   }
   
   return NO_ERROR;
}

void t_Table::SlotThreadReadFinished (t_pDevice pDevice)
{
   LOG_INFO ("Acquisition of %s: Read thread finished", QSTR_TO_PSZ (pDevice->LinuxDevice))
   pDevice->pThreadRead->deleteLater();
   pDevice->pThreadRead = NULL;

   if (pDevice->AbortRequest)
        CHK_EXIT (WakeWaitingThreads(pDevice))
   else CHK_EXIT (pDevice->pFifoRead->InsertDummy ())
   CHK_EXIT (FinaliseThreadStructs (pDevice))
}

void t_Table::SlotThreadHashFinished (t_pDevice pDevice)
{
   LOG_INFO ("Acquisition of %s: Hash thread finished", QSTR_TO_PSZ (pDevice->LinuxDevice))
   pDevice->pThreadHash->deleteLater();
   pDevice->pThreadHash = NULL;

   if (pDevice->AbortRequest)
        CHK_EXIT (WakeWaitingThreads(pDevice))
   else CHK_EXIT (pDevice->pFifoHashOut->InsertDummy ())
   CHK_EXIT (FinaliseThreadStructs (pDevice))
}

void t_Table::SlotThreadCompressFinished (t_pDevice pDevice, int ThreadNr)
{
   bool NoMoreThreads = true;

   LOG_INFO ("Acquisition of %s: Compression thread #%d finished", QSTR_TO_PSZ (pDevice->LinuxDevice), ThreadNr)

   pDevice->ThreadCompressList[ThreadNr]->deleteLater();
   pDevice->ThreadCompressList[ThreadNr] = NULL;

   for (int i=0;
        (i < pDevice->ThreadCompressList.count()) && NoMoreThreads;
        i++)
      NoMoreThreads = (pDevice->ThreadCompressList[i] == NULL);

   if (NoMoreThreads)
   {
       LOG_INFO ("All %d compression threads finished", pDevice->ThreadCompressList.count())
       pDevice->ThreadCompressList.clear();
   }

   if (pDevice->AbortRequest)
        CHK_EXIT (WakeWaitingThreads(pDevice))
   else CHK_EXIT (pDevice->pFifoCompressOut->InsertDummy (ThreadNr))

   CHK_EXIT (FinaliseThreadStructs (pDevice))
}

void t_Table::SlotThreadWriteFinished (t_pDevice pDevice)
{
   LOG_INFO ("Acquisition of %s: Write thread finished", QSTR_TO_PSZ (pDevice->LinuxDevice))

   pDevice->pThreadWrite->deleteLater();
   pDevice->pThreadWrite = NULL;

   if (pDevice->AbortRequest)
      CHK_EXIT (WakeWaitingThreads(pDevice))

   CHK_EXIT (FinaliseThreadStructs (pDevice))
}

void t_Table::SlotThreadWriteWakeThreads (t_pDevice pDevice)
{
   LOG_INFO ("Acquisition of %s: Write thread asks other threads for releasing its handle", QSTR_TO_PSZ (pDevice->LinuxDevice))

   if (pDevice->AbortRequest)
   {
      LOG_INFO ("Acquisition of %s: Calling WakeWaitingThreads", QSTR_TO_PSZ (pDevice->LinuxDevice))
      CHK_EXIT (WakeWaitingThreads(pDevice))
   }
}

APIRET t_Table::AbortAcquisition0 (t_pDevice pDevice)
{
   if (pDevice->AbortRequest)
   {
      LOG_INFO ("User pressed abort, but abort flag for acquisition on %s is already set.", QSTR_TO_PSZ(pDevice->LinuxDevice))
      return NO_ERROR;
   }

   LOG_INFO ("User aborts acquisition on %s", QSTR_TO_PSZ(pDevice->LinuxDevice))
   pDevice->AbortReason  = t_Device::UserRequest;
   pDevice->AbortRequest = true;

   CHK (WakeWaitingThreads (pDevice))

   return NO_ERROR;
}

APIRET t_Table::AbortAcquisition (t_pDevice pDevice)
{
   bool Abort;

   if (pDevice->AbortRequest)
        Abort = true;
   else CHK (t_DlgAbort::Show (pDevice, Abort, pDevice->DeleteAfterAbort))

   if (Abort)
   {
      pDevice->AbortCount++;
      CHK (AbortAcquisition0 (pDevice))
   }
   return NO_ERROR;
}

APIRET t_Table::AbortAllAcquisitions (void)
{
   t_pDevice pDevice;
   int        i;

   for (i=0; i<pOwn->pDeviceList->count(); i++)
   {
      pDevice = pOwn->pDeviceList->at(i);
      if ((pDevice->State == t_Device::Acquire)       ||
          (pDevice->State == t_Device::AcquirePaused) ||
          (pDevice->State == t_Device::Verify)        ||
          (pDevice->State == t_Device::VerifyPaused))
      {
         CHK (AbortAcquisition0 (pDevice))
      }
   }

   return NO_ERROR;
}

APIRET t_Table::WakeWaitingThreads (t_pDevice pDevice)
{
   if      (!pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
   {
      CHK (pDevice->pFifoRead->WakeWaitingThreads())
   }
   else if ( pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
   {
      CHK (pDevice->pFifoRead   ->WakeWaitingThreads())
      CHK (pDevice->pFifoHashOut->WakeWaitingThreads())
   }
   else if (!pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
   {
      CHK (pDevice->pFifoCompressIn ->WakeWaitingThreads())
      CHK (pDevice->pFifoCompressOut->WakeWaitingThreads())
   }
   else if ( pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
   {
      CHK (pDevice->pFifoRead       ->WakeWaitingThreads())
      CHK (pDevice->pFifoCompressIn ->WakeWaitingThreads())
      CHK (pDevice->pFifoCompressOut->WakeWaitingThreads())
   }

   return NO_ERROR;
}

