#ifndef SCANBAG_H
#define SCANBAG_H

#include <map>
#include <list>
#include <set>
#include <vector>
#include <string>

#include <wibble/sys/mutex.h>
#include "scanner/scan.h"

struct ScanBagListener
{
	virtual ~ScanBagListener() {}
	virtual void candidatesChanged() = 0;
};

class ScanBag : protected std::map< std::string, std::map<std::string, bool> >
{
	wibble::sys::Mutex mutex;

	std::list<scanner::Scan*> scans;
	std::string defaultProfile;

	bool any_success;

	ScanBagListener* listener;

	ScanBag() : listener(0), any_success(false) {}
	// TODO: deallocate scans
	~ScanBag();

public:
	// These functions are unsafe to call after the scans start
	void setListener(ScanBagListener* listener)
	{
		this->listener = listener;
	}

	/// Add a new test to the candidate profiles
	void add(scanner::Scan* scan);

	/// Get the list of available tests
	const std::list<scanner::Scan*>& getScans() const { return scans; }

	/// Remove all items from this ScanBag
	void clear()
	{
		std::map<std::string, std::map<std::string, bool> >::clear();
		scans.clear();
		defaultProfile.clear();
		any_success = false;
	}

	// These functions are safe to call after the scans start

	/// Notify the success of a test, returning the list of candidate profiles
	void notifySuccess(scanner::Scan* scan);

	/// Return true if at least one scanner has notified success so far
	bool anySuccess();

	/// Get the less specific profile among the available ones
	std::string getLessSpecific();

	/// Get the default profile
	std::string getDefault();

	/**
	 * If some success has been reported and only one candidate is left, return
	 * it.  Else, return the empty string
	 */
	std::string getFinal();

	/// Get the remaning profiles
	std::vector<std::string> getRemaining();

	/// Debug method: dump the contents to stdout
	void dump();

	/// Singleton access method
	static ScanBag& get();
};

// vim:set ts=3 sw=3:
#endif
