/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; coding: utf-8 -*- */
/**
 * This file contains tests that validates that the ImageViewDrawer
 * class works correctly.
 **/
#include <assert.h>
#include <src/gtkimageview.h>

/**
 * test_only_scale_op_on_new_identical_pixbuf:
 *
 * The objective of this test is to verify that the cached pixbuf that
 * the ImageViewDrawer holds is not discarded when it receives a new
 * pixbuf with the exact same bit depth and colorspace.
 **/
static void
test_only_scale_op_on_new_identical_pixbuf ()
{
    printf ("test_only_scale_op_on_new_identical_pixbuf\n");

    GdkPixbuf *pixbuf1 = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 9, 9);
    GdkPixbuf *pixbuf2 = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 9, 9);
    GdkPixbuf *last_pb = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 50, 50);
    
    int interp = GDK_INTERP_BILINEAR;
    GdkRectangle area = {0, 0, 10, 10};
    
    DrawSettings parms1 = {1, area, 0, 0, interp, pixbuf1, 0, 0};
    DrawSettings parms2 = {1, area, 0, 0, interp, pixbuf2, 0, 0};

    DrawMethod meth = draw_settings_get_method (&parms1, &parms2, last_pb);
    assert (meth == DRAW_METHOD_SCALE);

    g_object_unref (pixbuf1);
    g_object_unref (pixbuf2);
    g_object_unref (last_pb);
}

/**
 * test_cache_is_used_on_equal_settings:
 *
 * The objective of this test is to verify that the cache is used when
 * the new draw settings are identical to the old ones if the area is
 * smaller than the cached pixbuf. If so, the DrawMethod should be
 * %DRAW_METHOD_CONTAINS.
 **/
static void
test_cache_is_used_on_equal_settings ()
{
    printf ("test_cache_is_used_on_equal_settings\n");
    int interp = GDK_INTERP_BILINEAR;
    GdkRectangle area = {0, 0, 10, 10};
    GdkPixbuf *pb = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 30, 30);
    DrawSettings parms1 = {1, area, 0, 0, interp, pb, 0, 0};
    DrawSettings parms2 = {1, area, 0, 0, interp, pb, 0, 0};

    DrawMethod meth = draw_settings_get_method (&parms1, &parms2, pb);
    assert (meth == DRAW_METHOD_CONTAINS);

    g_object_unref (pb);
}

/**
 * test_contains_even_if_cache_to_small:
 *
 * The objective of this test is to verfy that the method is
 * %DRAW_METHOD_CONTAINS even if the cache is smaller than the
 * requsted draw area.
 **/
static void
test_contains_even_if_cache_to_small ()
{
    printf ("test_contains_even_if_cache_to_small\n");
    int interp = GDK_INTERP_BILINEAR;
    GdkRectangle area = {0, 0, 20, 20};
    
    GdkPixbuf *draw_pb = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 30, 30);
    DrawSettings parms1 = {1, area, 0, 0, interp, draw_pb, 0, 0};
    DrawSettings parms2 = {1, area, 0, 0, interp, draw_pb, 0, 0};

    GdkPixbuf *last_pb = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 10, 10);
    DrawMethod meth = draw_settings_get_method (&parms1, &parms2, last_pb);
    assert (meth == DRAW_METHOD_CONTAINS);
    g_object_unref (draw_pb);
    g_object_unref (last_pb);
}


/**
 * test_scroll_needed_if_rect_size_not_equal:
 *
 * The objective of this test is to verify that a scroll operation is
 * needed if the size of the rectangle changes.
 **/
static void
test_scroll_needed_if_rect_size_not_equal ()
{
    printf ("test_scroll_needed_if_rect_size_not_equal\n");
    int interp = GDK_INTERP_BILINEAR;
    GdkPixbuf *pb = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 9, 9);
    DrawSettings parms1 = {1, (GdkRectangle){0, 0, 20, 20},
                           0, 0, interp, pb, 0, 0};
    DrawSettings parms2 = {1, (GdkRectangle){20, 0, 10, 10},
                           0, 0, interp, pb, 0, 0};

    DrawMethod meth = draw_settings_get_method (&parms1, &parms2, pb);
    assert (meth == DRAW_METHOD_SCROLL);

    g_object_unref (pb);
}

/**
 * test_default_draw_settings:
 *
 * The objective of this test is to verify that the default draw
 * settings in the image drawer results in the correct draw flags.
 **/
static void
test_default_draw_settings ()
{
    printf ("test_default_draw_settings\n");
    ImageViewDrawer *drawer = image_view_drawer_new ();
    int interp = GDK_INTERP_BILINEAR;
    GdkPixbuf *pb = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 9, 9);
    DrawSettings parms = {1, (GdkRectangle){0, 0, 20, 20},
                          0, 0, interp, pb, 0, 0};
    DrawMethod meth = draw_settings_get_method (&drawer->old, &parms,
                                                drawer->last_pixbuf);
    assert (meth == DRAW_METHOD_SCALE);
    
    image_view_drawer_free (drawer);
    g_object_unref (pb);
}

/**
 * test_force_scale_operation:
 *
 * The objective of this test is to verify that forcing the
 * ImageViewDrawer to apply a scale operation works as expected.
 **/
static void
test_force_scale_operation ()
{
    printf ("test_force_scale_operation\n");
    ImageViewDrawer *drawer = image_view_drawer_new ();
    
    GdkInterpType interp = GDK_INTERP_BILINEAR;
    GdkPixbuf *pb = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, 9, 9);
    DrawSettings parms = {1, (GdkRectangle){0, 0, 5, 5},
                          0, 0, interp, pb, 0, 0};
    drawer->old = parms;


    /* DrawSettings are identical, so no allocation or scale necessary. */
    assert (draw_settings_get_method (&drawer->old, &parms, pb)
            == DRAW_METHOD_CONTAINS);

    /* Force a scale operation in the next draw. */
    image_view_drawer_force_scale (drawer);

    DrawMethod meth = draw_settings_get_method (&drawer->old, &parms, pb);
    assert (meth == DRAW_METHOD_SCALE);
    
    image_view_drawer_free (drawer);
    g_object_unref (pb);
}

int
main(int argc, char *argv[])
{
    gtk_init (&argc, &argv);
    test_only_scale_op_on_new_identical_pixbuf ();
    test_cache_is_used_on_equal_settings ();
    test_contains_even_if_cache_to_small ();
    test_scroll_needed_if_rect_size_not_equal ();
    test_default_draw_settings ();
    test_force_scale_operation ();
    printf ("6 tests passed.\n");
}
