/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4; coding: utf-8 -*- 
 *
 * Copyright © 2007 Björn Lindqvist <bjourne@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

/**
 * SECTION:gtkiimagetool
 * @see_also: <ulink url =
 * "http://developer.gnome.org/doc/API/2.0/gobject/gtype-non-instantiable-classed.html">Non-instantiable
 * classed types: Interfaces</ulink>
 * @short_description: Interface for objects capable of being used as
 * tools by #GtkImageView
 *
 * <para>
 *   GtkIImageTool is an interface that defines how GtkImageView
 *   interacts with objects that acts as tools. GtkImageView delegates
 *   many of its most important tasks (such as drawing) to its tool which
 *   carries out all the hard work. The GtkImageView package comes with
 *   two tools; #GtkImageToolDragger and #GtkImageToolSelector, but by
 *   implementing your own tool it is possible to extend GtkImageView to
 *   do stuff its author (thats me) didn't imagine.
 * </para>
 * <para>
 *   GtkImageView uses #GtkImageToolDragger by default, as that tool is
 *   he most generally useful one. However, it is trivial to make it use
 *   another tool.
 *   <informalexample>
 *     <programlisting>
 *       GtkImageView *view = GTK_IMAGE_VIEW (gtk_image_view_new ());
 *       GtkIImageTool *tool = gtk_image_tool_selector_new (view);
 *       gtk_image_view_set_tool (view, tool);
 *     </programlisting>  
 *   </informalexample>
 *   Using the above code makes the view use the selector tool instead of
 *   the default dragger tool.
 * </para>
 **/
#include "gtkiimagetool.h"

/*************************************************************/
/***** Stuff that deals with the type ************************/
/*************************************************************/
static void
gtk_iimage_tool_base_init (gpointer g_class)
{
    static gboolean initialized = FALSE;
    if (!initialized)
        initialized = TRUE;
}

GType
gtk_iimage_tool_get_type (void)
{
    static GType type = 0;
    if (type)
        return type;
    static const GTypeInfo info = {
        sizeof (GtkIImageToolClass),
        gtk_iimage_tool_base_init,
        NULL,
        NULL,
        NULL,
        NULL,
        0,
        0,
        NULL
    };
    type = g_type_register_static (G_TYPE_INTERFACE,
                                   "GtkIImageTool",
                                   &info,
                                   0);
    return type;
}

/*************************************************************/
/***** Pseudo signal handlers ********************************/
/*************************************************************/
gboolean
gtk_iimage_tool_button_press (GtkIImageTool  *tool,
                              GdkEventButton *ev)
{
    return GTK_IIMAGE_TOOL_GET_CLASS (tool)->button_press (tool, ev);
}

gboolean
gtk_iimage_tool_button_release (GtkIImageTool  *tool,
                                GdkEventButton *ev)
{
    return GTK_IIMAGE_TOOL_GET_CLASS (tool)->button_release (tool, ev);
}

gboolean
gtk_iimage_tool_motion_notify (GtkIImageTool  *tool,
                               GdkEventMotion *ev)
{
    return GTK_IIMAGE_TOOL_GET_CLASS (tool)->motion_notify (tool, ev);
}

/**
 * gtk_iimage_tool_pixbuf_changed:
 * @tool: the tool
 * @reset_fit: whether the view is resetting its fit mode or not
 *
 * This method is called by the view whenever its pixbuf or its tool
 * changes. That is, when any of the methods
 * gtk_image_view_set_pixbuf() or gtk_image_view_set_tool() is
 * invoked. If the @reset_fit parameter is %TRUE, then the tool is
 * free to treat the pixbuf as completely new. See also
 * GtkImageView::pixbuf-changed.
 **/
void
gtk_iimage_tool_pixbuf_changed (GtkIImageTool *tool,
                                gboolean       reset_fit)
{
    return GTK_IIMAGE_TOOL_GET_CLASS (tool)->pixbuf_changed (tool, reset_fit);
}

/**
 * gtk_iimage_tool_paint_image:
 * @tool: The tool
 * @ds: A set of draw settings to use for this draw.
 * @drawable: A drawable to draw the image data on.
 *
 * Called whenever the image view decides that any part of the image
 * it shows needs to be redrawn.
 **/
void
gtk_iimage_tool_paint_image (GtkIImageTool *tool,
                             DrawSettings  *ds,
                             GdkDrawable   *drawable)
{
    return GTK_IIMAGE_TOOL_GET_CLASS (tool)->paint_image (tool, ds, drawable);
}

/*************************************************************/
/***** Read-only properties **********************************/
/*************************************************************/
/**
 * gtk_iimage_tool_cursor_at_point:
 * @x: the mouse pointers X-coordinate
 * @y: the mouse pointers Y-coordinate
 * @returns: The appropriate cursor
 *
 * Ask the tool what cursor it wants displayed.
 **/
GdkCursor*
gtk_iimage_tool_cursor_at_point (GtkIImageTool *tool,
                                 int            x,
                                 int            y)
{
    return GTK_IIMAGE_TOOL_GET_CLASS (tool)->cursor_at_point (tool, x, y);
}

