/* gtd_disp.c - Handle messages which need to be shown
 *
 * Copyright (C) 1995, 1996 Eric M. Ludlam
 * Copyright (C) 1997 Free Software Foundation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * 
 * Description:
 *   This provides a DISP_message routine which will allow linking
 * against the ETL library which will use this as a hook for
 * displaying critical error messages, or other such things.
 * 
 * $Log: gtd_disp.c,v $
 * Revision 1.3  1997/12/14 19:18:56  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.2  1996/03/02 03:31:20  zappo
 * Fixed some warnings
 *
 * Revision 1.1  1995/12/10  03:53:38  zappo
 * Initial revision
 *
 * History:
 * zappo   12/5/95    Created
 *
 * Tokens: ::Header:: gtalkd.h
 */

#include "gtalklib.h"
#include "gtalkd.h"

static int syslogdebug = FALSE;
static int initialized = FALSE;


/*
 * Function: DISP_init
 *
 *   Initializes the message writing display.  If in daemon mode, we
 * start up our connection to the syslog daemon.  If we are in
 * tty-mode, with verbosity on, then we can write to stdout.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   12/5/95    Created
 */
void DISP_init(Ctxt, sldb)
     struct DaemonContext *Ctxt;
     int sldb;
{
  if(!verbose)
    {
      syslogdebug = sldb;

      /* do that log thing.  Check number or parameters it needs dependant
       * on the operating system
       */
      openlog("gtalkd", LOG_PID
#if NUM_OPENLOG_PARAMETERS == 3
	      , LOG_DAEMON
#endif
	      );
      
      if(syslogdebug)
	syslog(LOG_DEBUG, "Syslog debug flag is set");
    }
  initialized = TRUE;
}


/*
 * Function: DISP_message
 *
 *   Displays a message, either with syslog (when in daemon mode) or
 * onto the tty if started that way, and verbosity specified.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt     - Context
 *              msg      - Pointer toCharacter of message
 *              severity - Number of severity
 * History:
 * zappo   12/5/95    Created
 */
void DISP_message(Ctxt, msg, severity)
     struct DaemonContext *Ctxt;
     char *msg;
     int severity;
{
  if(verbose || !initialized)
    {
      if(severity == LOG_CRIT)
	perror(msg);
      else
	{
	  printf(msg);
	  printf("\n");
	}
    }
  else
    {
      if(syslogdebug > 0)
	{
	  if(severity == LOG_CRIT)
	    {
	      severity = LOG_ERR;
	      /* I'd like to do syslog version of perror, but I don't
	       * think there is one
	       */
	    }
	  syslog(severity, msg);
	}
    }
}
