/* GStreamer multi process transform
 * Copyright (C) 2007 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:gstmultiproctrans
 * @short_description: Base class for multiple process transformers
 *
 * <refsect2>
 * <para>
 * This is a base class for elements that process and/or transform data
 * by having another program/process do the actual processing
 * for each buffer.  That is, for each incoming individual piece of data,
 * a new process is spawned and presented the data on its stdin,
 * and the element will send out what it receives from the process' stdout
 * as a single buffer.
 *
 * It provides for most of the details such as setting up sink and src pads,
 * state changes, etc.  An inheriting element need only concern about
 * providing for the proper pad templates and (optionally but recommended)
 * implementing a custom set_caps, which is called during (each)
 * caps negotiation.  This typically gives the inheritor a chance to provide
 * the right arguments to the program for the circumstances and specifics of
 * data at hand.
 * Also note that these arguments for the program may vary across each
 * invocation.
 * </para>
 * </refsect2>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <sys/select.h>
#include <fcntl.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef _MSC_VER
#include <io.h>
#endif
#include <errno.h>
#include <string.h>

#include "gstmultiproctrans.h"

#define PARENT_READ(trans)  trans->readpipe[0]
#define CHILD_WRITE(trans)  trans->readpipe[1]
#define CHILD_READ(trans)   trans->writepipe[0]
#define PARENT_WRITE(trans) trans->writepipe[1]

GST_DEBUG_CATEGORY_STATIC (multi_proc_trans_debug);
#define GST_CAT_DEFAULT multi_proc_trans_debug

/* pad templates are determined by inheriting element */

/* properties */
enum
{
  PROP_0,
  PROP_CMD,
  PROP_BLOCKSIZE
};

#define DEFAULT_BLOCKSIZE  16384

static void gst_multi_proc_trans_finalize (GObject * object);
static gboolean gst_multi_proc_trans_setcaps (GstPad * pad, GstCaps * caps);
static GstFlowReturn gst_multi_proc_trans_chain (GstPad * pad,
    GstBuffer * buffer);
static GstStateChangeReturn gst_multi_proc_trans_change_state (GstElement *
    element, GstStateChange transition);

/* properties */
static void gst_multi_proc_trans_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_multi_proc_trans_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstMultiProcTrans, gst_multi_proc_trans, GstElement,
    GST_TYPE_ELEMENT);

static void
gst_multi_proc_trans_base_init (gpointer klass)
{

}

static void
gst_multi_proc_trans_class_init (GstMultiProcTransClass * klass)
{
  GstElementClass *element_class;
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);

  GST_DEBUG_CATEGORY_INIT (multi_proc_trans_debug, "multiproctrans", 0,
      "Process Transform");

  gobject_class->finalize = GST_DEBUG_FUNCPTR (gst_multi_proc_trans_finalize);

  gobject_class->set_property = gst_multi_proc_trans_set_property;
  gobject_class->get_property = gst_multi_proc_trans_get_property;

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_CMD,
      g_param_spec_string ("command", "command",
          "Shell Command into which to send output", NULL, G_PARAM_READWRITE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_BLOCKSIZE,
      g_param_spec_uint ("blocksize", "Block size",
          "Size in bytes to read from process per buffer",
          1, G_MAXUINT, DEFAULT_BLOCKSIZE, G_PARAM_READWRITE));

  element_class->change_state =
      GST_DEBUG_FUNCPTR (gst_multi_proc_trans_change_state);
}

static void
gst_multi_proc_trans_init (GstMultiProcTrans * mptrans,
    GstMultiProcTransClass * klass)
{
  GstElement *element = GST_ELEMENT (mptrans);
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  mptrans->sinkpad =
      gst_pad_new_from_template (gst_element_class_get_pad_template
      (element_class, GST_MULTI_PROC_TRANS_SINK_NAME), "sink");
  gst_pad_set_setcaps_function (mptrans->sinkpad,
      GST_DEBUG_FUNCPTR (gst_multi_proc_trans_setcaps));
  gst_pad_set_chain_function (mptrans->sinkpad,
      GST_DEBUG_FUNCPTR (gst_multi_proc_trans_chain));
  gst_element_add_pad (element, mptrans->sinkpad);

  mptrans->srcpad =
      gst_pad_new_from_template (gst_element_class_get_pad_template
      (element_class, GST_MULTI_PROC_TRANS_SRC_NAME), "src");
  gst_pad_use_fixed_caps (mptrans->srcpad);
  gst_element_add_pad (element, mptrans->srcpad);

  mptrans->adapter = gst_adapter_new ();

  /* init properties. */
  mptrans->cmd = NULL;
  mptrans->args = g_array_sized_new (TRUE, TRUE, sizeof (gchar *), 10);
  mptrans->blocksize = DEFAULT_BLOCKSIZE;

  PARENT_READ (mptrans) = -1;
  CHILD_WRITE (mptrans) = -1;
  CHILD_READ (mptrans) = -1;
  CHILD_WRITE (mptrans) = -1;
}

static void
gst_multi_proc_trans_reset_args (GstMultiProcTrans * mptrans)
{
  guint i;

  if (mptrans->args) {
    for (i = 0; i < mptrans->args->len; ++i) {
      g_free (g_array_index (mptrans->args, gchar *, i));
      g_array_remove_index_fast (mptrans->args, i);
    }
  }
}

static void
gst_multi_proc_trans_finalize (GObject * object)
{
  GstMultiProcTrans *mptrans = GST_MULTI_PROC_TRANS (object);

  g_free (mptrans->cmd);
  gst_multi_proc_trans_reset_args (mptrans);
  g_array_free (mptrans->args, TRUE);
  gst_object_unref (mptrans->adapter);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
gst_multi_proc_trans_setcaps (GstPad * pad, GstCaps * caps)
{
  GstMultiProcTrans *mptrans = GST_MULTI_PROC_TRANS (GST_PAD_PARENT (pad));
  GstMultiProcTransClass *klass = GST_MULTI_PROC_TRANS_GET_CLASS (mptrans);
  gboolean ret = TRUE;
  GstCaps *outcaps = NULL;

  mptrans->did_caps = TRUE;

  if (!klass->set_caps) {
    GST_WARNING_OBJECT (mptrans, "no set_caps function set");
    return FALSE;
  }

  if (!(ret = klass->set_caps (mptrans, caps, &outcaps)))
    goto refuse_caps;

  /* only set caps if we have a specific idea, otherwise leave blank */
  if (outcaps) {
    ret = gst_pad_set_caps (mptrans->srcpad, outcaps);
    gst_caps_unref (outcaps);
  }
  if (!ret)
    goto refuse_caps;

  return TRUE;

refuse_caps:
  {
    GST_WARNING_OBJECT (mptrans, "refused caps %" GST_PTR_FORMAT, caps);
    return FALSE;
  }
}


static gboolean
gst_multi_proc_trans_setup (GstMultiProcTrans * mptrans)
{
  GError *error = NULL;
  gchar *name;

  if (!mptrans->cmd)
    goto setup;

  /* first some variable abuse to announce some stuff */
  name = g_strjoinv (" ", (gchar **) mptrans->args->data);
  GST_INFO_OBJECT (mptrans, "executing %s with args %s", mptrans->cmd, name);
  g_free (name);

  /* first argument should be the file name of the process being exeucuted */
  name = g_strdup (mptrans->cmd);
  g_array_prepend_val (mptrans->args, name);

  /* this will also close all of the child's descriptors
   * except stdin/stdout/stderr; which is really needed,
   * otherwise if the main process closes a descriptor,
   * the child still pretends it is open, which may stall pipes */
  g_spawn_async_with_pipes (NULL, (gchar **) mptrans->args->data, NULL,
      G_SPAWN_SEARCH_PATH, NULL, NULL,
      &mptrans->pid, &PARENT_WRITE (mptrans), &PARENT_READ (mptrans),
      NULL, &error);

  /* remove the process from the args */
  g_free (name);
  g_array_remove_index (mptrans->args, 0);

  if (error)
    goto fork;

  /* should not block in communication with process */
  fcntl (PARENT_WRITE (mptrans), F_SETFL, O_NONBLOCK);
  fcntl (PARENT_READ (mptrans), F_SETFL, O_NONBLOCK);

  return TRUE;

  /* special cases */
fork:
  {
    GST_ELEMENT_ERROR (mptrans, RESOURCE, FAILED,
        ("glib error %s", error->message ? error->message : ""),
        GST_ERROR_SYSTEM);
    return FALSE;
  }
setup:
  {
    GST_ERROR_OBJECT (mptrans, "no command has been specified");
    return FALSE;
  }
}

static GstFlowReturn
gst_multi_proc_trans_chain (GstPad * pad, GstBuffer * buf)
{
  GstMultiProcTrans *mptrans = GST_MULTI_PROC_TRANS (GST_PAD_PARENT (pad));
  GstFlowReturn result = GST_FLOW_OK;
  fd_set readfds;
  fd_set writefds;
  gint res;
  guint8 *idata;
  GstBuffer *outbuf = NULL;
  guint isize;

  /* the preceding element may not have had a specific idea on caps */
  if (!mptrans->did_caps) {
    if (!gst_multi_proc_trans_setcaps (mptrans->sinkpad, NULL))
      goto not_negotiated;
  }

  /* cater for possible controllable properties */
  gst_object_sync_values (G_OBJECT (mptrans), GST_BUFFER_TIMESTAMP (buf));

  /* new process is started for each buffer */
  if (!gst_multi_proc_trans_setup (mptrans))
    return GST_FLOW_ERROR;

  idata = GST_BUFFER_DATA (buf);
  isize = GST_BUFFER_SIZE (buf);
  outbuf = NULL;

#ifndef HAVE_WIN32
  while (TRUE) {
    FD_ZERO (&readfds);
    FD_SET (PARENT_READ (mptrans), &readfds);

    FD_ZERO (&writefds);
    if (isize)
      FD_SET (PARENT_WRITE (mptrans), &writefds);

    do {
      GST_DEBUG_OBJECT (mptrans, "going into select, have %d bytes to write",
          isize);
      res = select (FD_SETSIZE, &readfds, &writefds, NULL, NULL);
    } while ((res == -1 && errno == EINTR));

    GST_DEBUG_OBJECT (mptrans, "leaving select, read %d, write %d",
        FD_ISSET (PARENT_READ (mptrans), &readfds),
        isize ? FD_ISSET (PARENT_WRITE (mptrans), &writefds) : 0);
    if (res == -1)
      goto select_error;
#endif

    /* try to get data to the process */
#ifndef HAVE_WIN32
    if (isize > 0 && FD_ISSET (PARENT_WRITE (mptrans), &writefds)) {
#else
    if (isize > 0) {
#endif
      while (TRUE) {
        res = write (PARENT_WRITE (mptrans), idata, isize);
        if (res < 0) {
          if (errno != EAGAIN && errno != EPIPE)
            goto write_error;
          else
            break;
        }
        GST_DEBUG_OBJECT (mptrans, "wrote %d bytes", res);
        isize -= res;
        idata += res;
        /* make process feel no more to consume */
        if (!isize) {
          close (PARENT_WRITE (mptrans));
          PARENT_WRITE (mptrans) = -1;
          break;
        }
      }
    }

    /* try to get data from the process */
#ifndef HAVE_WIN32
    if (FD_ISSET (PARENT_READ (mptrans), &readfds)) {
#else
    if (TRUE) {
#endif
      while (TRUE) {
        if (!outbuf)
          outbuf = gst_buffer_new_and_alloc (mptrans->blocksize);
        res = read (PARENT_READ (mptrans), GST_BUFFER_DATA (outbuf),
            GST_BUFFER_SIZE (outbuf));
        if (res < 0) {
          if (errno != EAGAIN)
            goto read_error;
          else
            break;
        }
        GST_DEBUG_OBJECT (mptrans, "read %d bytes", res);
        if (res == 0)
          goto eos;
        if (res < mptrans->blocksize)
          gst_buffer_replace (&outbuf, gst_buffer_create_sub (outbuf, 0, res));
        gst_adapter_push (mptrans->adapter, outbuf);
        outbuf = NULL;
      }
    }
  }

eos:
  /* prevent complaints if no bytes were gathered;
   * could be if the process died on us */
  if (gst_adapter_available (mptrans->adapter) > 0) {
    /* get all we collected into the adapter */
    outbuf = gst_buffer_new ();
    GST_BUFFER_SIZE (outbuf) = gst_adapter_available (mptrans->adapter);
    GST_BUFFER_DATA (outbuf) = gst_adapter_take (mptrans->adapter,
        gst_adapter_available (mptrans->adapter));
    /* decorate */
    gst_buffer_copy_metadata (outbuf, buf, GST_BUFFER_COPY_TIMESTAMPS);
    gst_buffer_set_caps (outbuf, GST_PAD_CAPS (mptrans->srcpad));

    result = gst_pad_push (mptrans->srcpad, outbuf);
    outbuf = NULL;
  }

clean_and_exit:
  /* should clean up link to and the child process itself */
  if (PARENT_WRITE (mptrans) > 0)
    close (PARENT_WRITE (mptrans));
  close (PARENT_READ (mptrans));
  g_spawn_close_pid (mptrans->pid);
  PARENT_WRITE (mptrans) = -1;
  PARENT_READ (mptrans) = -1;
  mptrans->pid = (GPid) 0;

exit:
  /* clean up internal stuff */
  if (outbuf)
    gst_buffer_unref (outbuf);
  gst_adapter_clear (mptrans->adapter);
  gst_buffer_unref (buf);

  return result;

  /* special cases */
not_negotiated:
  {
    /* could be serious,
     * but could be ok if preceded by another extra-process element or so
     * that can not do any caps setting */
    GST_ELEMENT_WARNING (mptrans, CORE, NEGOTIATION, (NULL),
        ("format wasn't negotiated before chain function"));

    result = GST_FLOW_NOT_NEGOTIATED;
    goto exit;
  }
write_error:
  {
    GST_ELEMENT_ERROR (mptrans, RESOURCE, WRITE, (NULL), GST_ERROR_SYSTEM);

    result = GST_FLOW_ERROR;
    goto clean_and_exit;
  }
read_error:
  {
    GST_ELEMENT_ERROR (mptrans, RESOURCE, READ, (NULL), GST_ERROR_SYSTEM);

    result = GST_FLOW_ERROR;
    goto clean_and_exit;
  }
#ifndef HAVE_WIN32
select_error:
  {
    GST_ELEMENT_ERROR (mptrans, RESOURCE, READ, (NULL),
        ("select on file descriptor: %s.", g_strerror (errno)));

    result = GST_FLOW_ERROR;
    goto clean_and_exit;
  }
#endif
}

static void
gst_multi_proc_trans_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstMultiProcTrans *mptrans;

  g_return_if_fail (GST_IS_MULTI_PROC_TRANS (object));

  mptrans = GST_MULTI_PROC_TRANS (object);

  switch (prop_id) {
    case PROP_CMD:
      g_free (mptrans->cmd);
      mptrans->cmd = g_value_dup_string (value);
      break;
    case PROP_BLOCKSIZE:
      mptrans->blocksize = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_multi_proc_trans_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{
  GstMultiProcTrans *mptrans;

  g_return_if_fail (GST_IS_MULTI_PROC_TRANS (object));

  mptrans = GST_MULTI_PROC_TRANS (object);

  switch (prop_id) {
    case PROP_CMD:
      g_value_take_string (value, g_strdup (mptrans->cmd));
      break;
    case PROP_BLOCKSIZE:
      g_value_set_uint (value, mptrans->blocksize);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstStateChangeReturn
gst_multi_proc_trans_change_state (GstElement * element,
    GstStateChange transition)
{
  GstMultiProcTrans *mptrans = GST_MULTI_PROC_TRANS (element);
  GstStateChangeReturn ret;

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto done;

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      gst_multi_proc_trans_reset_args (mptrans);
      mptrans->did_caps = FALSE;
      break;
    default:
      break;
  }

done:
  return ret;
}
