/* GStreamer mp2enc (mjpegtools) runtime wrapper
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * mp2enc.c: gstreamer mp2enc runtime wrapper
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

 /**
 * SECTION:element-mp2enc
 *
 * <refsect2>
 * <para>
 * Performs MPEG-1 layer I/II encoding by means of the
 * mp2enc tool from the
 * <ulink url="http://mjpeg.sourceforge.net/">mjpegtools</ulink> suite
 * (used as a runtime child process).
 *
 * Note that mp2enc needs input in WAV format,
 * so you will typicall have to precede this element by
 * <link linkend="GstWavEnc">wavenc</link>.
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>

#include "gstproctrans.h"

#define GST_TYPE_MP2ENC \
  gst_mp2enc_get_type ()
#define GST_MP2ENC(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), GST_TYPE_MP2ENC, GstMp2enc))
#define GST_MP2ENC_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), GST_TYPE_MP2ENC, GstMp2encClass))
#define GST_IS_MP2ENC(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), GST_TYPE_MP2ENC))
#define GST_IS_MP2ENC_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), GST_TYPE_MP2ENC))

typedef struct _GstMp2enc GstMp2enc;
typedef struct _GstMp2encClass GstMp2encClass;

struct _GstMp2enc
{
  GstProcTrans parent;

  /* a hack */
  GstPadEventFunction proc_event;

  /* mp2enc parameters */
  guint bitrate;
  guint layer;
  gboolean crc;
  gboolean vcd;
};

struct _GstMp2encClass
{
  GstProcTransClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (mp2enc_debug);
#define GST_CAT_DEFAULT mp2enc_debug

static GstStaticPadTemplate sink_template =
GST_STATIC_PAD_TEMPLATE (GST_PROC_TRANS_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-wav, "
        "rate = (int) { 32000, 44100, 48000 }, " "channels = (int) [ 1, 2 ] ")
    );

static GstStaticPadTemplate src_template =
GST_STATIC_PAD_TEMPLATE (GST_PROC_TRANS_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/mpeg, "
        "mpegversion = (int) 1, "
        "layer = (int) [ 1, 2], " "bitrate = (int) [ 0, MAX ] ")
    );

/* properties */
enum
{
  PROP_0,
  PROP_BITRATE,
  PROP_LAYER,
  PROP_CRC,
  PROP_VCD
};

#define DEFAULT_BITRATE  224
#define DEFAULT_LAYER      2
#define DEFAULT_CRC     FALSE
#define DEFAULT_VCD     FALSE

static gboolean gst_mp2enc_set_caps (GstProcTrans * ptrans, GstCaps * incaps,
    GstCaps ** outcaps);
static gboolean gst_mp2enc_sink_event (GstPad * pad, GstEvent * event);
static GstStateChangeReturn gst_mp2enc_change_state (GstElement * element,
    GstStateChange transition);

/* properties */
static void gst_mp2enc_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_mp2enc_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstMp2enc, gst_mp2enc, GstProcTrans, GST_TYPE_PROC_TRANS);

static void
gst_mp2enc_base_init (gpointer klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details_simple (element_class, "mp2enc audio encoder",
      "Codec/Encoder/Audio", "MPEG-1/2 audio encoder",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>, Andrew Stevens");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&sink_template));
}

static void
gst_mp2enc_class_init (GstMp2encClass * klass)
{
  GstElementClass *element_class;
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);

  GST_DEBUG_CATEGORY_INIT (mp2enc_debug, "mp2enc", 0, "MPEG1/2 encoder");

  gobject_class->set_property = gst_mp2enc_set_property;
  gobject_class->get_property = gst_mp2enc_get_property;

  g_object_class_install_property (gobject_class, PROP_BITRATE,
      g_param_spec_uint ("bitrate", "Bitrate",
          "Compressed audio bitrate (kbps)", 0, 10 * 1024, DEFAULT_BITRATE,
          G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_LAYER,
      g_param_spec_uint ("layer", "Layer", "MPEG-1 layer",
          1, 2, DEFAULT_LAYER, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_CRC,
      g_param_spec_boolean ("crc", "CRC", "Use CRC error protection",
          DEFAULT_CRC, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_VCD,
      g_param_spec_boolean ("vcd", "VCD", "Force VCD compatible output; "
          "restricts the possibilities for bitrate and sample rate must be 44.1 kbps",
          DEFAULT_VCD, G_PARAM_READWRITE));

  element_class->change_state = GST_DEBUG_FUNCPTR (gst_mp2enc_change_state);
}

static void
gst_mp2enc_init (GstMp2enc * enc, GstMp2encClass * klass)
{
  GstProcTrans *proctrans = GST_PROC_TRANS (enc);
  GstProcTransClass *proctransclass = GST_PROC_TRANS_CLASS (klass);

  proctransclass->set_caps = GST_DEBUG_FUNCPTR (gst_mp2enc_set_caps);

  /* most properties are initialized by base element */
  proctrans->cmd = g_strdup ("mp2enc");

  /* HACK to override event function so we can drop some */
  if (proctrans->sinkpad) {
    enc->proc_event = GST_PAD_EVENTFUNC (proctrans->sinkpad);
    gst_pad_set_event_function (proctrans->sinkpad,
        GST_DEBUG_FUNCPTR (gst_mp2enc_sink_event));
  }

  enc->bitrate = DEFAULT_BITRATE;
  enc->layer = DEFAULT_LAYER;
  enc->crc = DEFAULT_CRC;
  enc->vcd = DEFAULT_VCD;
}

static gboolean
gst_mp2enc_set_caps (GstProcTrans * ptrans, GstCaps * incaps,
    GstCaps ** outcaps)
{
  GstMp2enc *enc = GST_MP2ENC (ptrans);
  gchar *opt;

  /* bitrate */
  opt = g_strdup ("-b");
  g_array_append_val (ptrans->args, opt);
  opt = g_strdup_printf ("%u", enc->bitrate);
  g_array_append_val (ptrans->args, opt);
  /* layer */
  opt = g_strdup ("-l");
  g_array_append_val (ptrans->args, opt);
  opt = g_strdup_printf ("%u", enc->layer);
  g_array_append_val (ptrans->args, opt);
  /* crc */
  if (enc->crc) {
    opt = g_strdup ("-e");
    g_array_append_val (ptrans->args, opt);
  }
  /* vcd */
  if (enc->vcd) {
    opt = g_strdup ("-V");
    g_array_append_val (ptrans->args, opt);
  }
  /* align verbosity level with the debug category */
  opt = g_strdup ("-v");
  g_array_append_val (ptrans->args, opt);
  switch (gst_debug_category_get_threshold (mp2enc_debug)) {
    case GST_LEVEL_LOG:
      opt = g_strdup ("2");
      break;
    case GST_LEVEL_DEBUG:
    case GST_LEVEL_INFO:
      opt = g_strdup ("1");
      break;
    default:
      opt = g_strdup ("0");
      break;
  }
  g_array_append_val (ptrans->args, opt);

  /* output to standard out */
  opt = g_strdup ("-o");
  g_array_append_val (ptrans->args, opt);
  opt = g_strdup ("-");
  g_array_append_val (ptrans->args, opt);

  /* set the outgoing caps */
  *outcaps = gst_caps_new_simple ("audio/mpeg",
      "mpegversion", G_TYPE_INT, 1,
      "layer", G_TYPE_INT, enc->layer,
      "bitrate", G_TYPE_INT, enc->bitrate * 1000, NULL);

  return TRUE;
}

/* mp2enc needs to be preceded by wavenc, which typically sends new_segment
 * events expecting filesink to perform a seek to rewrite the header.
 * These need to be dropped here to prevent a seek from happening in the mp2
 * stream produced by mp2enc.
 * We'll accept for now that the wav header ends up encoded as audio at eos. */
static gboolean
gst_mp2enc_sink_event (GstPad * pad, GstEvent * event)
{
  GstMp2enc *enc;

  enc = GST_MP2ENC (GST_PAD_PARENT (pad));

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_NEWSEGMENT:
    {
      GstFormat format;

      gst_event_parse_new_segment (event, NULL, NULL, &format, NULL, NULL,
          NULL);
      /* eat this event if it is a byte-based one, so a seek request */
      if (format == GST_FORMAT_BYTES)
        return TRUE;
      break;
    }
    default:
      break;
  }

  /* chain up to the parent */
  return enc->proc_event (pad, event);
}

static void
gst_mp2enc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstMp2enc *enc;

  g_return_if_fail (GST_IS_MP2ENC (object));

  enc = GST_MP2ENC (object);

  switch (prop_id) {
    case PROP_BITRATE:
      enc->bitrate = g_value_get_uint (value);
      break;
    case PROP_LAYER:
      enc->layer = g_value_get_uint (value);
      break;
    case PROP_CRC:
      enc->crc = g_value_get_boolean (value);
      break;
    case PROP_VCD:
      enc->vcd = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_mp2enc_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstMp2enc *enc;

  g_return_if_fail (GST_IS_MP2ENC (object));

  enc = GST_MP2ENC (object);

  switch (prop_id) {
    case PROP_BITRATE:
      g_value_set_uint (value, enc->bitrate);
      break;
    case PROP_LAYER:
      g_value_set_uint (value, enc->layer);
      break;
    case PROP_CRC:
      g_value_set_boolean (value, enc->crc);
      break;
    case PROP_VCD:
      g_value_set_boolean (value, enc->vcd);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstStateChangeReturn
gst_mp2enc_change_state (GstElement * element, GstStateChange transition)
{
  GstStateChangeReturn ret;

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto done;

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      break;
    default:
      break;
  }

done:
  return ret;
}
