/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * VirtualDub filter:
 * Copyright (C) 1999-2000 Donald A. Graft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-xsharpen
 *
 * <refsect2>
 * <para>
 * This filter performs a subtle but useful sharpening effect. The result
 * is a sharpening effect that not only avoids amplifying noise, but also
 * tends to reduce it. A welcome side effect is that files processed with
 * this filter tend to compress to smaller files.
 * <itemizedlist>
 * <listitem>
 * <para>
 * When <link linkend="GstMsharpen--strength">strength</link>
 * is maximal, mapped pixels are not blended with the original pixel values,
 * so a full-strength effect is achieved.  As the value is reduced, each
 * mapped pixel is blended with more of the original pixel. At a value of 0,
 * the original pixels are passed through and there is no sharpening effect.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * <link linkend="GstMsharpen--threshold">threshold</link> indicates how
 * close a pixel must be to the brightest or dimmest pixel to be mapped.
 * If a pixel is more than threshold away from the brightest or dimmest pixel,
 * it is not mapped.  Thus, as the threshold is reduced, pixels in the mid
 * range start to be spared.
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * VirtualDub xsharpen filter [Donald A. Graft]
 * </listitem>
 * <listitem>
 * Also available in transcode (xsharpen filter)
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-vd.h"

#include <string.h>


#define GST_TYPE_XSHARPEN \
  (gst_xsharpen_get_type())
#define GST_XSHARPEN(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_XSHARPEN,GstXsharpen))
#define GST_XSHARPEN_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_XSHARPEN,GstXsharpenClass))
#define GST_IS_XSHARPEN(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_XSHARPEN))
#define GST_IS_XSHARPEN_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_XSHARPEN))


typedef struct _GstXsharpen GstXsharpen;
typedef struct _GstXsharpenClass GstXsharpenClass;

struct _GstXsharpen
{
  GstVideoFilter videofilter;

  gint width, height;

  guint threshold, strength;
};


struct _GstXsharpenClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (xsharpen_debug);
#define GST_CAT_DEFAULT xsharpen_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_THRESHOLD,
  PROP_STRENGTH
      /* FILL ME */
};

#define DEFAULT_THRESHOLD  256
#define DEFAULT_STRENGTH   200

static GstStaticPadTemplate gst_xsharpen_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_xsharpen_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstFlowReturn gst_xsharpen_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_xsharpen_start (GstBaseTransform * btrans);
static gboolean gst_xsharpen_stop (GstBaseTransform * btrans);

static void gst_xsharpen_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_xsharpen_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstXsharpen, gst_xsharpen, GstVideoFilter,
    GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE (GstXsharpen, gst_xsharpen);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_xsharpen);

static void
gst_xsharpen_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Deinter",
      "Filter/Effect/Video",
      "Sharpen by mapping pixels to the closest of window max or min",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Donald A. Graft");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_xsharpen_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_xsharpen_src_template));
}

static void
gst_xsharpen_class_init (GstXsharpenClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (xsharpen_debug, "xsharpen", 0, "xsharpen");

  gobject_class->set_property = gst_xsharpen_set_property;
  gobject_class->get_property = gst_xsharpen_get_property;

  g_object_class_install_property (gobject_class, PROP_THRESHOLD,
      g_param_spec_uint ("threshold", "Threshold",
          "Threshold for pixel to be mapped",
          0, 256, DEFAULT_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_STRENGTH,
      g_param_spec_uint ("strength", "Strength",
          "Strength/weight of modification of a mapped pixel",
          0, 256, DEFAULT_STRENGTH,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_xsharpen_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_xsharpen_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_xsharpen_transform);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_xsharpen_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_xsharpen_stop);
}

static void
gst_xsharpen_init (GstXsharpen * filter, GstXsharpenClass * g_class)
{
  filter->threshold = DEFAULT_THRESHOLD;
  filter->strength = DEFAULT_STRENGTH;
}

static GstFlowReturn
gst_xsharpen_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstXsharpen *filter;
  guint8 *src, *dest;
  GstFlowReturn ret = GST_FLOW_OK;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_XSHARPEN (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);
  dest = (guint8 *) GST_BUFFER_DATA (out);

  guint width = filter->width;
  guint height = filter->height;

  guint strength = filter->strength;
  guint invstrength = 256 - strength;
  guint threshold = filter->threshold;

  guint stride = GST_VIDEO_I420_Y_ROWSTRIDE (width);

  /* first copy everything so that chroma info is transferred
   * also need to pass-through border luma lines as well */
  memcpy (dest, src, GST_BUFFER_SIZE (in));

  /* run the 3x3 rank-order sharpening kernel over the pixels */
  src += stride;
  dest += stride;

  guint x, y, luma, maxdiff, mindiff;

  for (y = 1; y < height - 1; y++) {

    for (x = 1; x < width - 1; x++) {
      /* find the brightest and dimmest pixels in the 3x3 window
       * surrounding the current pixel */
      guint lumamax = -1000;
      guint lumamin = 1000;
      guint lumac;

      luma = (src - stride)[x - 1];
      if (luma > lumamax)
        lumamax = luma;
      if (luma < lumamin)
        lumamin = luma;

      luma = (src - stride)[x];
      if (luma > lumamax)
        lumamax = luma;
      if (luma < lumamin)
        lumamin = luma;

      luma = (src - stride)[x + 1];
      if (luma > lumamax)
        lumamax = luma;
      if (luma < lumamin)
        lumamin = luma;

      luma = src[x - 1];
      if (luma > lumamax)
        lumamax = luma;
      if (luma < lumamin)
        lumamin = luma;

      luma = src[x];
      lumac = luma;
      if (luma > lumamax)
        lumamax = luma;
      if (luma < lumamin)
        lumamin = luma;

      luma = src[x + 1];
      if (luma > lumamax)
        lumamax = luma;
      if (luma < lumamin)
        lumamin = luma;

      luma = (src + stride)[x - 1];
      if (luma > lumamax)
        lumamax = luma;
      if (luma < lumamin)
        lumamin = luma;

      luma = (src + stride)[x];
      if (luma > lumamax)
        lumamax = luma;
      if (luma < lumamin)
        lumamin = luma;

      luma = (src + stride)[x + 1];
      if (luma > lumamax)
        lumamax = luma;
      if (luma < lumamin)
        lumamin = luma;

      /* Determine whether the current pixel is closer to the
       * brightest or the dimmest pixel. Then compare the current
       * pixel to that closest pixel. If the difference is within
       * threshold, map the current pixel to the closest pixel;
       * otherwise pass it through. */

      guint p = 256;
      if (strength != 0) {
        mindiff = lumac - lumamin;
        maxdiff = lumamax - lumac;
        if (mindiff > maxdiff) {
          if (maxdiff < threshold)
            p = lumamax & 0xff;
        } else {
          if (mindiff < threshold)
            p = lumamin & 0xff;
        }
      }
      if (p == 256) {
        dest[x] = src[x];
      } else {
        int t;
        lumac = src[x] & 0xff;
        t = ((strength * p + invstrength * lumac) >> 8) & 0xff;
        if (t > 240)
          t = 240;
        if (t < 16)
          t = 16;
        dest[x] = t & 0xff;
      }
    }
    src += stride;
    dest += stride;
  }

  return ret;
}


static gboolean
gst_xsharpen_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_xsharpen_stop (GstBaseTransform * btrans)
{
  return TRUE;
}

static void
gst_xsharpen_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstXsharpen *src;

  g_return_if_fail (GST_IS_XSHARPEN (object));
  src = GST_XSHARPEN (object);

  switch (prop_id) {
    case PROP_THRESHOLD:
      src->threshold = g_value_get_uint (value);
      break;
    case PROP_STRENGTH:
      src->strength = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_xsharpen_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstXsharpen *src;

  g_return_if_fail (GST_IS_XSHARPEN (object));
  src = GST_XSHARPEN (object);

  switch (prop_id) {
    case PROP_THRESHOLD:
      g_value_set_uint (value, src->threshold);
      break;
    case PROP_STRENGTH:
      g_value_set_uint (value, src->strength);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
