/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * Avisynth filter plugin:
 * Copyright (C) 2003 Donald A. Graft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-kerneldeint
 *
 * <refsect2>
 * <para>
 * [based on Avisynth filter documentation]
 * </para>
 * <para>
 * This filter deinterlaces using a kernel approach.
 * It gives greatly improved vertical resolution in deinterlaced areas
 * compared to simple field discarding.
 * </para>
 * <para>
 * If you set the <link linkend="GstKernelDeint--threshold">threshold</link>
 * to 0, you can get totally artifact free results (due to lack of thresholding)
 * but with much less loss of vertical resolution than simple field discarding.
 * For optimal results, however, set a motion threshold that allows static
 * areas of the picture to be passed through. In this mode, the kernel-based
 * deinterlacing of the moving areas preserves their vertical resolution compared
 * to simple interpolation.
 * </para>
 * <para>
 * This filter assumes that top field is first.  If not, use a filter (e.g. fields)
 * to swap them.  Note that whether the top field is indeed first can be verified
 * by separating fields (using e.g. fields) and verifying that motion is progressive
 * (and not jumping back and forth).
 * </para>
 * <para>
 * The <link linkend="GstKernelDeint--threshold">threshold</link> parameter
 * defines the "motion" thresold. Moving areas are kernel-deinterlaced while
 * non-moving areas are passed through.
 * Use the <link linkend="GstKernelDeint--map">map</link> parameter to tweak
 * the threshold parameter so that just the combed areas of the frame are
 * deinterlaced.
 * </para>
 * <para>
 * <link linkend="GstKernelDeint--sharp">sharp</link>, when set to true,
 * selects a kernel that provides better vertical resolution and performs
 * some sharpening of the video. For less sharpening but also less vertical
 * resolution, set this parameter to false.
 * </para>
 * <para>
 * <link linkend="GstKernelDeint--two-way">two-way</link>, when set to true,
 * selects a kernel that includes both the previous and the following fields
 * for deinterlacing. When set to false, the kernel includes only the previous
 * field. The latter one-way kernel is faster, crisper, and gives less blending
 * (this last advantage makes the filter perform better on anime).
 * </para>
 * <para>
 * <link linkend="GstKernelDeint--map">map</link>, when set to true, shows
 * the areas that are "moving" as determined by the
 * <link linkend="GstKernelDeint--threshold">threshold</link> parameter
 * and which will be kernel-deinterlaced.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * Avisynth kerndeint filter [Donald A. Graft]
 * </listitem>
 * <listitem>
 * Also available in mplayer (kerndeint filter)
 * </listitem>
 * <listitem>
 * Also available in avidemux (kernel deinterlacer)
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-vd.h"

#include <string.h>
#include <stdlib.h>


#define GST_TYPE_KERNEL_DEINT \
  (gst_kernel_deint_get_type())
#define GST_KERNEL_DEINT(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_KERNEL_DEINT,GstKernelDeint))
#define GST_KERNEL_DEINT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_KERNEL_DEINT,GstKernelDeintClass))
#define GST_IS_KERNEL_DEINT(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_KERNEL_DEINT))
#define GST_IS_KERNEL_DEINT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_KERNEL_DEINT))


typedef struct _GstKernelDeint GstKernelDeint;
typedef struct _GstKernelDeintClass GstKernelDeintClass;

struct _GstKernelDeint
{
  GstVideoFilter videofilter;

  gint width, height;

  /* properties */
  guint threshold;
  gboolean sharp, two_way, map;

  /* image type */
  guint img;

  /* previous buffer */
  GstBuffer *prev;
};


struct _GstKernelDeintClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (kernel_deint_debug);
#define GST_CAT_DEFAULT kernel_deint_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_THRESHOLD,
  PROP_SHARP,
  PROP_TWO_WAY,
  PROP_MAP
      /* FILL ME */
};

#define MIN_THRESHOLD       0
#define MAX_THRESHOLD      100
#define DEFAULT_THRESHOLD   10
#define DEFAULT_SHARP      FALSE
#define DEFAULT_TWO_WAY    FALSE
#define DEFAULT_MAP        FALSE

enum
{
  IMGFMT_YUV,
  IMGFMT_YUY2,
  IMGFMT_RGB
};

static GstStaticPadTemplate gst_kernel_deint_src_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV
        ("{ IYUV, I420, YV12, YUY2, YUYV, YVYU }") " ; " GST_VIDEO_CAPS_RGBx
        " ; " GST_VIDEO_CAPS_BGRx " ; " GST_VIDEO_CAPS_xRGB " ; "
        GST_VIDEO_CAPS_xBGR)
    );

static GstStaticPadTemplate gst_kernel_deint_sink_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV
        ("{ IYUV, I420, YV12, YUY2, YUYV, YVYU }") " ; " GST_VIDEO_CAPS_RGBx
        " ; " GST_VIDEO_CAPS_BGRx " ; " GST_VIDEO_CAPS_xRGB " ; "
        GST_VIDEO_CAPS_xBGR)
    );

static gboolean gst_kernel_deint_hook_caps (GstKernelDeint * filter,
    GstCaps * incaps, GstCaps * outcaps);
static GstFlowReturn gst_kernel_deint_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_kernel_deint_start (GstBaseTransform * btrans);
static gboolean gst_kernel_deint_stop (GstBaseTransform * btrans);

static void gst_kernel_deint_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_kernel_deint_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstKernelDeint, gst_kernel_deint, GstVideoFilter,
    GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL (GstKernelDeint, gst_kernel_deint,
    gst_kernel_deint_hook_caps);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_kernel_deint);

static void
gst_kernel_deint_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "KernelDeint",
      "Filter/Effect/Video", "Adaptive kernel deinterlacer",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Donald A. Graft");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_kernel_deint_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_kernel_deint_src_template));
}

static void
gst_kernel_deint_class_init (GstKernelDeintClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (kernel_deint_debug, "kerneldeint", 0, "kerneldeint");

  gobject_class->set_property = gst_kernel_deint_set_property;
  gobject_class->get_property = gst_kernel_deint_get_property;

  g_object_class_install_property (gobject_class, PROP_THRESHOLD,
      g_param_spec_uint ("threshold", "Threshold",
          "Threshold",
          MIN_THRESHOLD, MAX_THRESHOLD, DEFAULT_THRESHOLD,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_SHARP,
      g_param_spec_boolean ("sharp", "Sharp",
          "Enable/disable additional sharping",
          DEFAULT_SHARP, G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_TWO_WAY,
      g_param_spec_boolean ("two-way", "Two-Way",
          "Enable/disable two-way sharpening",
          DEFAULT_TWO_WAY, G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_MAP,
      g_param_spec_boolean ("map", "Map",
          "Paint/ignore pixels exceeding threshold",
          DEFAULT_MAP, G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_kernel_deint_set_caps);
  trans_class->get_unit_size =
      GST_DEBUG_FUNCPTR (gst_kernel_deint_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_kernel_deint_transform);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_kernel_deint_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_kernel_deint_stop);
}

static void
gst_kernel_deint_init (GstKernelDeint * filter, GstKernelDeintClass * g_class)
{
  filter->threshold = DEFAULT_THRESHOLD;
  filter->sharp = DEFAULT_SHARP;
  filter->two_way = DEFAULT_TWO_WAY;
  filter->map = DEFAULT_MAP;
}

static void
gst_kernel_deint_free (GstKernelDeint * filter)
{
  if (filter->prev)
    gst_buffer_unref (filter->prev);
  filter->prev = NULL;
}

static gboolean
gst_kernel_deint_hook_caps (GstKernelDeint * filter, GstCaps * incaps,
    GstCaps * outcaps)
{
  GstStructure *structure;
  gboolean ret = FALSE;

  structure = gst_caps_get_structure (incaps, 0);

  gst_kernel_deint_free (filter);
  if (gst_structure_has_name (structure, "video/x-raw-rgb")) {
    ret = TRUE;
    filter->img = IMGFMT_RGB;
  } else {
    guint32 fourcc;

    if ((ret = gst_structure_get_fourcc (structure, "format", &fourcc)))
      switch (fourcc) {
        case GST_MAKE_FOURCC ('Y', 'U', 'Y', '2'):
        case GST_MAKE_FOURCC ('Y', 'Y', 'Y', 'V'):
        case GST_MAKE_FOURCC ('Y', 'V', 'Y', 'U'):
          filter->img = IMGFMT_YUY2;
          break;
        case GST_MAKE_FOURCC ('I', 'Y', 'U', 'V'):
        case GST_MAKE_FOURCC ('I', '4', '2', '0'):
        case GST_MAKE_FOURCC ('Y', 'V', '1', '2'):
        default:
          filter->img = IMGFMT_YUV;
      }
  }

  return ret;
}


#define PROGRESSIVE  0x00000001

static void
gst_kernel_deint_do_deint (GstKernelDeint * filter, guint8 * src, guint8 * dest,
    guint8 * prev)
{
  const guint8 *srcp, *prvp, *prvp_saved, *prvpp, *prvpn, *prvppp, *prvpnn,
      *prvp4p, *prvp4n;
  const guint8 *srcp_saved;
  const guint8 *srcpp, *srcppp, *srcpn, *srcpnn, *srcp3p, *srcp3n, *srcp4p,
      *srcp4n;
  guint8 *dstp;
  guint8 *dstp_saved;

  int src_pitch;
  int dst_pitch;
  int w;
  int h;
  int x, y, z;
  int val, hi, lo;
  double valf;

  guint32 pitch, height, width;
  guint32 offset = 0;

  guint32 order, threshold;
  guint8 sharp, twoway, map;

  height = filter->height;
  width = filter->width;

  /* always considered top-field first */
  order = 1;
  threshold = filter->threshold;
  sharp = filter->sharp;
  twoway = filter->two_way;
  map = filter->map;

  for (z = 0; z < (filter->img == IMGFMT_YUV ? 3 : 1); z++) {

    switch (z) {
      case 0:
        offset = 0;
        switch (filter->img) {
          case IMGFMT_YUV:
            pitch = GST_VIDEO_I420_Y_ROWSTRIDE (width);
            break;
          case IMGFMT_RGB:
            pitch = 4 * width;
            break;
          default:
            pitch = 2 * width;
        }
        break;
      case 1:
        offset = GST_VIDEO_I420_U_OFFSET (width, height);
        pitch = GST_VIDEO_I420_U_ROWSTRIDE (width);
        break;
      case 2:
        offset = GST_VIDEO_I420_V_OFFSET (width, height);
        pitch = GST_VIDEO_I420_V_ROWSTRIDE (width);
        break;
      default:
        g_assert_not_reached ();
        break;
    }
    srcp = srcp_saved = src + offset;
    dstp = dstp_saved = dest + offset;
    prvp_saved = prvp = prev + offset;

    src_pitch = pitch;
    dst_pitch = pitch;

    w = pitch;
    h = height;
    if (z) {
      h >>= 1;
    }

    srcp = srcp_saved + (1 - order) * src_pitch;
    dstp = dstp_saved + (1 - order) * dst_pitch;
    for (y = 0; y < h; y += 2) {
      oil_memcpy (dstp, srcp, w);
      srcp += 2 * src_pitch;
      dstp += 2 * dst_pitch;
    }

    /* Copy through the lines that will be missed below. */
    oil_memcpy (dstp_saved + order * dst_pitch,
        srcp_saved + (1 - order) * src_pitch, w);
    oil_memcpy (dstp_saved + (2 + order) * dst_pitch,
        srcp_saved + (3 - order) * src_pitch, w);
    oil_memcpy (dstp_saved + (h - 2 + order) * dst_pitch,
        srcp_saved + (h - 1 - order) * src_pitch, w);
    oil_memcpy (dstp_saved + (h - 4 + order) * dst_pitch,
        srcp_saved + (h - 3 - order) * src_pitch, w);

    /* For the other field choose adaptively between using the previous field
     * or the interpolant from the current field. */
    prvp = prvp_saved + 5 * src_pitch - (1 - order) * src_pitch;
    prvpp = prvp - src_pitch;
    prvppp = prvp - 2 * src_pitch;
    prvp4p = prvp - 4 * src_pitch;
    prvpn = prvp + src_pitch;
    prvpnn = prvp + 2 * src_pitch;
    prvp4n = prvp + 4 * src_pitch;
    srcp = srcp_saved + 5 * src_pitch - (1 - order) * src_pitch;
    srcpp = srcp - src_pitch;
    srcppp = srcp - 2 * src_pitch;
    srcp3p = srcp - 3 * src_pitch;
    srcp4p = srcp - 4 * src_pitch;
    srcpn = srcp + src_pitch;
    srcpnn = srcp + 2 * src_pitch;
    srcp3n = srcp + 3 * src_pitch;
    srcp4n = srcp + 4 * src_pitch;
    dstp = dstp_saved + 5 * dst_pitch - (1 - order) * dst_pitch;
    for (y = 5 - (1 - order); y <= h - 5 - (1 - order); y += 2) {
      for (x = 0; x < w; x++) {
        if ((threshold == 0) ||
            (abs ((int) prvp[x] - (int) srcp[x]) > threshold) ||
            (abs ((int) prvpp[x] - (int) srcpp[x]) > threshold) ||
            (abs ((int) prvpn[x] - (int) srcpn[x]) > threshold)) {
          if (map == TRUE) {
            switch (filter->img) {
              case IMGFMT_YUV:
                if (z == 0)
                  dstp[x] = 235;
                else
                  dstp[x] = 128;
                break;
              case IMGFMT_RGB:{
                int g = x & ~3;

                dstp[g++] = 255;
                dstp[g++] = 255;
                dstp[g++] = 255;
                dstp[g] = 255;
                x = g;
                break;
              }
              default:{
                int g = x & ~3;

                dstp[g++] = 235;
                dstp[g++] = 128;
                dstp[g++] = 235;
                dstp[g] = 128;
                x = g;
                break;
              }
            }
          } else {

            switch (filter->img) {
              case IMGFMT_YUV:
                hi = (z == 0) ? 235 : 240;
                lo = 16;
                break;
              case IMGFMT_RGB:
                hi = 255;
                lo = 0;
                break;
              default:
                hi = (x & 1) ? 240 : 235;
                lo = 16;
                break;
            }

            if (sharp == TRUE) {
              if (twoway == TRUE)
                valf = +0.526 * ((int) srcpp[x] + (int) srcpn[x])
                    + 0.170 * ((int) srcp[x] + (int) prvp[x])
                    - 0.116 * ((int) srcppp[x] +
                    (int) srcpnn[x] + (int) prvppp[x] + (int) prvpnn[x])
                    - 0.026 * ((int) srcp3p[x] + (int) srcp3n[x])
                    + 0.031 * ((int) srcp4p[x] +
                    (int) srcp4n[x] + (int) prvp4p[x] + (int) prvp4n[x]);
              else
                valf = +0.526 * ((int) srcpp[x] + (int) srcpn[x])
                    + 0.170 * ((int) prvp[x])
                    - 0.116 * ((int) prvppp[x] + (int) prvpnn[x])
                    - 0.026 * ((int) srcp3p[x] + (int) srcp3n[x])
                    + 0.031 * ((int) prvp4p[x] + (int) prvp4p[x]);
              if (valf > hi)
                valf = hi;
              else if (valf < lo)
                valf = lo;
              dstp[x] = (int) valf;
            } else {
              if (twoway == TRUE)
                val = (8 * ((int) srcpp[x] + (int) srcpn[x]) +
                    2 * ((int) srcp[x] + (int) prvp[x]) -
                    (int) (srcppp[x]) -
                    (int) (srcpnn[x]) -
                    (int) (prvppp[x]) - (int) (prvpnn[x])) >> 4;
              else
                val = (8 * ((int) srcpp[x] + (int) srcpn[x]) +
                    2 * ((int) prvp[x]) -
                    (int) (prvppp[x]) - (int) (prvpnn[x])) >> 4;
              if (val > hi)
                val = hi;
              else if (val < lo)
                val = lo;
              dstp[x] = (int) val;
            }
          }
        } else {
          dstp[x] = srcp[x];
        }
      }
      prvp += 2 * src_pitch;
      prvpp += 2 * src_pitch;
      prvppp += 2 * src_pitch;
      prvpn += 2 * src_pitch;
      prvpnn += 2 * src_pitch;
      prvp4p += 2 * src_pitch;
      prvp4n += 2 * src_pitch;
      srcp += 2 * src_pitch;
      srcpp += 2 * src_pitch;
      srcppp += 2 * src_pitch;
      srcp3p += 2 * src_pitch;
      srcp4p += 2 * src_pitch;
      srcpn += 2 * src_pitch;
      srcpnn += 2 * src_pitch;
      srcp3n += 2 * src_pitch;
      srcp4n += 2 * src_pitch;
      dstp += 2 * dst_pitch;
    }
  }
}

static GstFlowReturn
gst_kernel_deint_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstKernelDeint *filter = GST_KERNEL_DEINT (btrans);

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  if (filter->prev)
    gst_kernel_deint_do_deint (filter, GST_BUFFER_DATA (in),
        GST_BUFFER_DATA (out), GST_BUFFER_DATA (filter->prev));

  /* get rid of old */
  if (filter->prev)
    gst_buffer_unref (filter->prev);
  /* and make sure the current one sticks around */
  gst_buffer_ref (in);
  filter->prev = in;

  return GST_FLOW_OK;
}


static gboolean
gst_kernel_deint_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_kernel_deint_stop (GstBaseTransform * btrans)
{
  GstKernelDeint *filter;

  filter = GST_KERNEL_DEINT (btrans);

  gst_kernel_deint_free (filter);

  return TRUE;
}

static void
gst_kernel_deint_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstKernelDeint *src;

  g_return_if_fail (GST_IS_KERNEL_DEINT (object));
  src = GST_KERNEL_DEINT (object);

  switch (prop_id) {
    case PROP_THRESHOLD:
      src->threshold = g_value_get_uint (value);
      break;
    case PROP_SHARP:
      src->sharp = g_value_get_boolean (value);
      break;
    case PROP_TWO_WAY:
      src->two_way = g_value_get_boolean (value);
      break;
    case PROP_MAP:
      src->map = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_kernel_deint_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstKernelDeint *src;

  g_return_if_fail (GST_IS_KERNEL_DEINT (object));
  src = GST_KERNEL_DEINT (object);

  switch (prop_id) {
    case PROP_THRESHOLD:
      g_value_set_uint (value, src->threshold);
      break;
    case PROP_SHARP:
      g_value_set_boolean (value, src->sharp);
      break;
    case PROP_TWO_WAY:
      g_value_set_boolean (value, src->two_way);
      break;
    case PROP_MAP:
      g_value_set_boolean (value, src->map);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
