/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * transcode ivtc filter
 * Copyright (C) Thanassis Tsiodras - August 2002
 * Based on the excellent work of Donald Graft in Decomb.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-ivtc
 * @see_also: decimate
 *
 * <refsect2>
 * <para>
 * This filter performs inverse telecine adaptively.
 * It tries to reconstruct original frames by re-assembling these
 * from best matching top and bottom frames in a telecined stream.
 * After this operation, it remains to alter the framerate by dropping
 * some frames, which can be done by <link linkend="GstDecimate">decimate</link>.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * transcode ivtc filter [Thanassis Tsiodras],
 * based on avisynth Decomb work [Donald Graft]
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <stdlib.h>


#define GST_TYPE_IVTC \
  (gst_ivtc_get_type())
#define GST_IVTC(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_IVTC,GstIvtc))
#define GST_IVTC_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_IVTC,GstIvtcClass))
#define GST_IS_IVTC(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_IVTC))
#define GST_IS_IVTC_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_IVTC))


#define DEFAULT_REPLACE_TOP     TRUE
#define DEFAULT_PERFORM_MAGIC  FALSE


typedef struct _GstIvtc GstIvtc;
typedef struct _GstIvtcClass GstIvtcClass;

struct _GstIvtc
{
  GstBaseTransform videotransform;

  gint width, height;

  /* properties */
  gboolean top, magic;

  /* holds frames */
  GQueue *queue;
};


struct _GstIvtcClass
{
  GstBaseTransformClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (ivtc_debug);
#define GST_CAT_DEFAULT ivtc_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_REPLACE_TOP,
  PROP_PERFORM_MAGIC
      /* FILL ME */
};

static GstStaticPadTemplate gst_ivtc_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_ivtc_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static gboolean gst_ivtc_hook_caps (GstIvtc * filter,
    GstCaps * incaps, GstCaps * outcaps);
static void gst_ivtc_flush (GstIvtc * filter, gboolean send);
static gboolean gst_ivtc_sink_event (GstBaseTransform * btrans,
    GstEvent * event);
static GstFlowReturn gst_ivtc_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_ivtc_start (GstBaseTransform * btrans);
static gboolean gst_ivtc_stop (GstBaseTransform * btrans);

static void gst_ivtc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_ivtc_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstIvtc, gst_ivtc, GstBaseTransform, GST_TYPE_BASE_TRANSFORM);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL (GstIvtc, gst_ivtc,
    gst_ivtc_hook_caps);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_ivtc);

static void
gst_ivtc_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Ivtc",
      "Filter/Effect/Video", "Inverse Telecine",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>, Thanassis Tsiodras");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_ivtc_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_ivtc_src_template));
}

static void
gst_ivtc_class_init (GstIvtcClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (ivtc_debug, "ivtc", 0, "ivtc");

  gobject_class->set_property = gst_ivtc_set_property;
  gobject_class->get_property = gst_ivtc_get_property;

  g_object_class_install_property (gobject_class, PROP_REPLACE_TOP,
      g_param_spec_boolean ("replace-top", "Replace Top Field",
          "Replace top field", DEFAULT_REPLACE_TOP, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_PERFORM_MAGIC,
      g_param_spec_boolean ("perform-magic", "Perform Magic",
          "Perform some magic", DEFAULT_PERFORM_MAGIC, G_PARAM_READWRITE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_ivtc_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_ivtc_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_ivtc_transform);
  trans_class->event = GST_DEBUG_FUNCPTR (gst_ivtc_sink_event);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_ivtc_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_ivtc_stop);
}

static void
gst_ivtc_init (GstIvtc * filter, GstIvtcClass * g_class)
{
  filter->top = DEFAULT_REPLACE_TOP;
  filter->magic = DEFAULT_PERFORM_MAGIC;
}


static gboolean
gst_ivtc_hook_caps (GstIvtc * filter, GstCaps * incaps, GstCaps * outcaps)
{
  gst_ivtc_flush (filter, TRUE);

  return TRUE;
}


static void
gst_ivtc_flush (GstIvtc * filter, gboolean send)
{
  GstBuffer *buf;
  GstBaseTransform *btrans;

  btrans = GST_BASE_TRANSFORM (filter);

  while (!g_queue_is_empty (filter->queue)) {
    buf = g_queue_pop_head (filter->queue);
    if (send) {
      gst_buffer_set_caps (buf, GST_PAD_CAPS (btrans->srcpad));
      gst_pad_push (btrans->srcpad, buf);
    } else
      gst_buffer_unref (buf);
  }
}

static gboolean
gst_ivtc_sink_event (GstBaseTransform * btrans, GstEvent * event)
{
  GstIvtc *filter;

  filter = GST_IVTC (btrans);

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_EOS:
      gst_ivtc_flush (filter, TRUE);
      break;
    default:
      break;
  }

  return GST_BASE_TRANSFORM_CLASS (parent_class)->event (btrans, event);
}

// TODO consider making this a library function (used also in fields)
static inline void
gst_ivtc_copy_field (guint8 * in, guint8 * out, guint stride, guint height)
{
  guint increment;

  increment = stride << 1;
  height >>= 1;

  while (height--) {
    oil_memcpy (out, in, stride);
    out += increment;
    in += increment;
  }
}

/* is called when queue holds exactly 3 frames,
 * and returns the best matching frame in given buffer */
static void
gst_ivtc_replace (GstIvtc * filter, GstBuffer * out)
{
  guint8 *curr, *pprev, *pnext, *cprev, *cnext, *nprev, *nnext, *dstp;
  guint8 *cdata, *pdata, *ndata;
  gint height, width;
  gint y, p, c, n, lowest, chosen;

  /* let's make sure */
  g_return_if_fail (g_queue_get_length (filter->queue) == 3);

  height = filter->height;
  width = GST_VIDEO_I420_Y_ROWSTRIDE (filter->width);

  pdata = GST_BUFFER_DATA ((GstBuffer *) g_queue_peek_nth (filter->queue, 0));
  cdata = GST_BUFFER_DATA ((GstBuffer *) g_queue_peek_nth (filter->queue, 1));
  ndata = GST_BUFFER_DATA ((GstBuffer *) g_queue_peek_nth (filter->queue, 2));

  y = (filter->top ? 1 : 2) * width;

  /* comments apply to default top field replacement */

  /* bottom field of current */
  curr = &cdata[y];
  /* top field of previous */
  pprev = &pdata[y - width];
  /* top field of previous - 2nd scanline */
  pnext = &pdata[y + width];
  /* top field of current */
  cprev = &cdata[y - width];
  /* top field of current - 2nd scanline */
  cnext = &cdata[y + width];
  /* top field of next */
  nprev = &ndata[y - width];
  /* top field of next - 2nd scanline */
  nnext = &ndata[y + width];

  p = c = n = 0;

  /* Try to match the bottom field of the current frame to the
   * top fields of the previous, current, and next frames.
   * Output the assembled frame that matches up best.
   * For matching, subsample the frames for speed. */
#define T 100
  for (y = 0; y < height - 2; y += 4) {
    gint comb, c, x;

    for (x = 0; x < width;) {
      c = curr[x];
      /* This combing metric is based on an original idea of Gunnar Thalin. */
      comb = ((long) pprev[x] - c) * ((long) pnext[x] - c);
      if (comb > T)
        p++;

      comb = ((long) cprev[x] - c) * ((long) cnext[x] - c);
      if (comb > T)
        c++;

      comb = ((long) nprev[x] - c) * ((long) nnext[x] - c);
      if (comb > T)
        n++;

      if (!(++x & 3))
        x += 12;
    }

    curr += width * 4;
    pprev += width * 4;
    pnext += width * 4;
    cprev += width * 4;
    cnext += width * 4;
    nprev += width * 4;
    nnext += width * 4;
  }

  lowest = c;
  chosen = 1;
  if (p < lowest) {
    lowest = p;
    chosen = 0;
  }
  if (n < lowest) {
    lowest = n;
    chosen = 2;
  }

  if (filter->magic && c < 50 && abs (lowest - c) < 10 && (p + c + n) > 1000) {
    lowest = c;
    chosen = 1;
  }

  GST_INFO_OBJECT (filter, "Telecide: p=%u  c=%u  n=%u [using %d]\n",
      p, c, n, chosen);

  /* top field comes from the chosen */
  if (chosen == 0)
    curr = pdata;
  else if (chosen == 1)
    curr = cdata;
  else
    curr = ndata;


  dstp = GST_BUFFER_DATA (out);

  /* first the replaced field; top field by default */
  /* luma */
  y = (filter->top ? 0 : 1) * GST_VIDEO_I420_Y_ROWSTRIDE (filter->width);
  gst_ivtc_copy_field (curr + y, dstp + y, width, height);
  /* both (!) chroma planes in one go */
  y = (filter->top ? 0 : 1) * GST_VIDEO_I420_U_ROWSTRIDE (filter->width);
  gst_ivtc_copy_field (curr + y + GST_VIDEO_I420_U_OFFSET (filter->width,
          height), dstp + y + GST_VIDEO_I420_U_OFFSET (filter->width, height),
      GST_VIDEO_I420_U_ROWSTRIDE (filter->width), height);

  /* then the other field, which is taken from 'current' frame */
  curr = cdata;
  y = (filter->top ? 1 : 0) * GST_VIDEO_I420_Y_ROWSTRIDE (filter->width);
  gst_ivtc_copy_field (curr + y, dstp + y, width, height);
  /* both (!) chroma planes in one go */
  y = (filter->top ? 1 : 0) * GST_VIDEO_I420_U_ROWSTRIDE (filter->width);
  gst_ivtc_copy_field (curr + y + GST_VIDEO_I420_U_OFFSET (filter->width,
          height), dstp + y + GST_VIDEO_I420_U_OFFSET (filter->width, height),
      GST_VIDEO_I420_U_ROWSTRIDE (filter->width), height);
}

static GstFlowReturn
gst_ivtc_transform (GstBaseTransform * btrans, GstBuffer * in, GstBuffer * out)
{
  GstIvtc *filter = GST_IVTC (btrans);

  g_queue_push_tail (filter->queue, in);
  /* keep a ref as it will be dropped in any case, as input buffer */
  gst_buffer_ref (in);

  /* delay a few frames during start-up */
  if (G_UNLIKELY (g_queue_get_length (filter->queue) < 3)) {
    return GST_BASE_TRANSFORM_FLOW_DROPPED;
  }

  /* enough buffers if we make it here, a re-constructed frame can be produced */
  gst_ivtc_replace (filter, out);
  /* oldest frame no longer needed */
  in = g_queue_pop_head (filter->queue);
  gst_buffer_unref (in);

  return GST_FLOW_OK;
}


static gboolean
gst_ivtc_start (GstBaseTransform * btrans)
{
  GstIvtc *filter = GST_IVTC (btrans);

  filter->queue = g_queue_new ();

  return TRUE;
}

static gboolean
gst_ivtc_stop (GstBaseTransform * btrans)
{
  GstIvtc *filter = GST_IVTC (btrans);

  gst_ivtc_flush (filter, FALSE);
  g_queue_free (filter->queue);

  return TRUE;
}

static void
gst_ivtc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstIvtc *src;

  g_return_if_fail (GST_IS_IVTC (object));
  src = GST_IVTC (object);

  switch (prop_id) {
    case PROP_REPLACE_TOP:
      src->top = g_value_get_boolean (value);
      break;
    case PROP_PERFORM_MAGIC:
      src->magic = g_value_get_boolean (value);
      break;
  }
}

static void
gst_ivtc_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstIvtc *src;

  g_return_if_fail (GST_IS_IVTC (object));
  src = GST_IVTC (object);

  switch (prop_id) {
    case PROP_REPLACE_TOP:
      g_value_set_boolean (value, src->top);
      break;
    case PROP_PERFORM_MAGIC:
      g_value_set_boolean (value, src->magic);
      break;
  }
}
