/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * mplayer filter:
 * Copyright (C) 2002 Jindrich Makovicka <makovick@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-delogo
 *
 * <refsect2>
 * <para>
 * Suppresses  a TV station logo by a simple interpolation of the surrounding
 * pixels.  Just set a rectangle covering the logo and watch it disappear
 * (and sometimes something even uglier appear - your mileage may vary).
 * </para>
 * <para>
 * If <link linkend="GstDelogo--border">border</link> is set to a negative
 * value, a green border is drawn to simplify finding the right location
 * parameters.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * mplayer delogo filter [Jindrich Makovicka]
 * </listitem>
 * <listitem>
 * Also available in avidemux (MPlayer delogo)
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-mencoder.h"

#include <string.h>


#define GST_TYPE_DELOGO \
  (gst_delogo_get_type())
#define GST_DELOGO(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_DELOGO,GstDelogo))
#define GST_DELOGO_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_DELOGO,GstDelogoClass))
#define GST_IS_DELOGO(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_DELOGO))
#define GST_IS_DELOGO_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_DELOGO))


typedef struct _GstDelogo GstDelogo;
typedef struct _GstDelogoClass GstDelogoClass;

struct _GstDelogo
{
  GstVideoFilter videofilter;

  gint width, height;

  /* properties; external view */
  guint logo_x, logo_y, logo_w, logo_h;
  gint border;

  /* properties; processed */
  guint xoff, yoff, lw, lh;
  gint band, show;
};


struct _GstDelogoClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (delogo_debug);
#define GST_CAT_DEFAULT delogo_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_LOGO_X,
  PROP_LOGO_Y,
  PROP_LOGO_W,
  PROP_LOGO_H,
  PROP_BORDER
      /* FILL ME */
};

#define DEFAULT_LOGO_X       0
#define DEFAULT_LOGO_Y       0
#define DEFAULT_LOGO_W      10
#define DEFAULT_LOGO_H      10
#define DEFAULT_BORDER       0


static GstStaticPadTemplate gst_delogo_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_delogo_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static gboolean gst_delogo_set_caps (GstBaseTransform * btrans,
    GstCaps * incaps, GstCaps * outcaps);
static gboolean gst_delogo_get_unit_size (GstBaseTransform * btrans,
    GstCaps * caps, guint * size);
static GstFlowReturn gst_delogo_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_delogo_start (GstBaseTransform * btrans);
static gboolean gst_delogo_stop (GstBaseTransform * btrans);

static void gst_delogo_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_delogo_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstDelogo, gst_delogo, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE (GstDelogo, gst_delogo);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_delogo);

static void
gst_delogo_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Delogo",
      "Filter/Effect/Video", "Simple tv station logo remover",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Jindrich Makovicka");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_delogo_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_delogo_src_template));
}

static void
gst_delogo_class_init (GstDelogoClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (delogo_debug, "delogo", 0, "delogo");

  gobject_class->set_property = gst_delogo_set_property;
  gobject_class->get_property = gst_delogo_get_property;

  g_object_class_install_property (gobject_class, PROP_LOGO_X,
      g_param_spec_uint ("logo-x", "Logo X",
          "X-coordinate of logo top left corner",
          0, G_MAXUINT, DEFAULT_LOGO_X,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_LOGO_Y,
      g_param_spec_uint ("logo-y", "Logo Y",
          "Y-coordinate of logo top left corner",
          0, G_MAXUINT, DEFAULT_LOGO_Y,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_LOGO_W,
      g_param_spec_uint ("logo-w", "Logo Width",
          "Width of logo",
          0, G_MAXUINT, DEFAULT_LOGO_W,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_LOGO_H,
      g_param_spec_uint ("logo-h", "Logo Width",
          "Height of logo",
          0, G_MAXUINT, DEFAULT_LOGO_H,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_BORDER,
      g_param_spec_int ("border", "Border",
          "Thickness of fuzzy edge of rectangle (<0 = draw border)",
          -1, G_MAXINT, DEFAULT_BORDER,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_delogo_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_delogo_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_delogo_transform);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_delogo_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_delogo_stop);
}

static void
gst_delogo_init (GstDelogo * filter, GstDelogoClass * g_class)
{
  filter->logo_x = DEFAULT_LOGO_X;
  filter->logo_y = DEFAULT_LOGO_Y;
  filter->logo_w = DEFAULT_LOGO_W;
  filter->logo_h = DEFAULT_LOGO_H;
  filter->border = DEFAULT_BORDER;
}

static void
gst_delogo_delogo (guint8 * dst, guint8 * src, gint dstStride, gint srcStride,
    gint width, gint height, gint logo_x, gint logo_y, gint logo_w, gint logo_h,
    gint band, gint show, gint direct)
{
  gint y, x;
  gint interp, dist;
  guint8 *xdst, *xsrc;

  guint8 *topleft, *botleft, *topright;
  gint xclipl, xclipr, yclipt, yclipb;
  gint logo_x1, logo_x2, logo_y1, logo_y2;

  xclipl = MAX (-logo_x, 0);
  xclipr = MAX (logo_x + logo_w - width, 0);
  yclipt = MAX (-logo_y, 0);
  yclipb = MAX (logo_y + logo_h - height, 0);

  logo_x1 = logo_x + xclipl;
  logo_x2 = logo_x + logo_w - xclipr;
  logo_y1 = logo_y + yclipt;
  logo_y2 = logo_y + logo_h - yclipb;

  topleft = src + logo_y1 * srcStride + logo_x1;
  topright = src + logo_y1 * srcStride + logo_x2 - 1;
  botleft = src + (logo_y2 - 1) * srcStride + logo_x1;

  if (!direct)
    oil_memcpy (dst, src, width * height);

  dst += (logo_y1 + 1) * dstStride;
  src += (logo_y1 + 1) * srcStride;

  for (y = logo_y1 + 1; y < logo_y2 - 1; y++) {
    for (x = logo_x1 + 1, xdst = dst + logo_x1 + 1, xsrc = src + logo_x1 + 1;
        x < logo_x2 - 1; x++, xdst++, xsrc++) {
      interp = ((topleft[srcStride * (y - logo_y - yclipt)]
              + topleft[srcStride * (y - logo_y - 1 - yclipt)]
              + topleft[srcStride * (y - logo_y + 1 - yclipt)])
          * (logo_w - (x - logo_x)) / logo_w
          + (topright[srcStride * (y - logo_y - yclipt)]
              + topright[srcStride * (y - logo_y - 1 - yclipt)]
              + topright[srcStride * (y - logo_y + 1 - yclipt)])
          * (x - logo_x) / logo_w + (topleft[x - logo_x - xclipl]
              + topleft[x - logo_x - 1 - xclipl]
              + topleft[x - logo_x + 1 - xclipl])
          * (logo_h - (y - logo_y)) / logo_h + (botleft[x - logo_x - xclipl]
              + botleft[x - logo_x - 1 - xclipl]
              + botleft[x - logo_x + 1 - xclipl])
          * (y - logo_y) / logo_h) / 6;
/*              interp = (topleft[srcStride*(y-logo_y)]*(logo_w-(x-logo_x))/logo_w
      + topright[srcStride*(y-logo_y)]*(x-logo_x)/logo_w
      + topleft[x-logo_x]*(logo_h-(y-logo_y))/logo_h
      + botleft[x-logo_x]*(y-logo_y)/logo_h
                          )/2;*/
      if (y >= logo_y + band && y < logo_y + logo_h - band &&
          x >= logo_x + band && x < logo_x + logo_w - band) {
        *xdst = interp;
      } else {
        dist = 0;
        if (x < logo_x + band)
          dist = MAX (dist, logo_x - x + band);
        else if (x >= logo_x + logo_w - band)
          dist = MAX (dist, x - (logo_x + logo_w - 1 - band));
        if (y < logo_y + band)
          dist = MAX (dist, logo_y - y + band);
        else if (y >= logo_y + logo_h - band)
          dist = MAX (dist, y - (logo_y + logo_h - 1 - band));
        *xdst = (*xsrc * dist + interp * (band - dist)) / band;

        if (show && (dist == band - 1))
          *xdst = 0;
      }
    }

    dst += dstStride;
    src += srcStride;
  }
}

static void
gst_delogo_set_props (GstDelogo * filter)
{
  filter->show = 0;
  filter->band = filter->border;

  if (filter->band < 0) {
    filter->band = 4;
    filter->show = 1;
  }

  filter->lw = filter->logo_w + filter->band * 2;
  filter->lh = filter->logo_h + filter->band * 2;
  filter->xoff = filter->logo_x - filter->band;
  filter->yoff = filter->logo_y - filter->band;
}

static GstFlowReturn
gst_delogo_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstDelogo *filter;
  guint8 *src, *dest;
  guint width, height, stride;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_DELOGO (btrans);

  width = filter->width;
  height = filter->height;

  /* luma */
  src = (guint8 *) GST_BUFFER_DATA (in);
  dest = (guint8 *) GST_BUFFER_DATA (out);
  stride = GST_VIDEO_I420_Y_ROWSTRIDE (width);

  gst_delogo_delogo (dest, src, stride, stride, stride, height,
      filter->xoff, filter->yoff, filter->lw, filter->lh,
      filter->band, filter->show, FALSE);

  /* chroma */
  src += GST_VIDEO_I420_U_OFFSET (width, height);
  dest += GST_VIDEO_I420_U_OFFSET (width, height);
  stride = GST_VIDEO_I420_U_ROWSTRIDE (width);

  gst_delogo_delogo (dest, src, stride, stride, stride, height / 2,
      filter->xoff / 2, filter->yoff / 2, filter->lw / 2, filter->lh / 2,
      filter->band / 2, filter->show, FALSE);

  src =
      (guint8 *) GST_BUFFER_DATA (in) + GST_VIDEO_I420_V_OFFSET (width, height);
  dest =
      (guint8 *) GST_BUFFER_DATA (out) + GST_VIDEO_I420_V_OFFSET (width,
      height);

  gst_delogo_delogo (dest, src, stride, stride, stride, height / 2,
      filter->xoff / 2, filter->yoff / 2, filter->lw / 2, filter->lh / 2,
      filter->band / 2, filter->show, FALSE);

  return GST_FLOW_OK;
}


static gboolean
gst_delogo_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_delogo_stop (GstBaseTransform * btrans)
{
  return TRUE;
}

static void
gst_delogo_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstDelogo *src;

  g_return_if_fail (GST_IS_DELOGO (object));
  src = GST_DELOGO (object);

  switch (prop_id) {
    case PROP_LOGO_X:
      src->logo_x = g_value_get_uint (value);
      break;
    case PROP_LOGO_Y:
      src->logo_y = g_value_get_uint (value);
      break;
    case PROP_LOGO_W:
      src->logo_w = g_value_get_uint (value);
      break;
    case PROP_LOGO_H:
      src->logo_h = g_value_get_uint (value);
      break;
    case PROP_BORDER:
      src->border = g_value_get_int (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  gst_delogo_set_props (src);
}

static void
gst_delogo_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstDelogo *src;

  g_return_if_fail (GST_IS_DELOGO (object));
  src = GST_DELOGO (object);

  switch (prop_id) {
    case PROP_LOGO_X:
      g_value_set_uint (value, src->logo_x);
      break;
    case PROP_LOGO_Y:
      g_value_set_uint (value, src->logo_y);
      break;
    case PROP_LOGO_W:
      g_value_set_uint (value, src->logo_w);
      break;
    case PROP_LOGO_H:
      g_value_set_uint (value, src->logo_h);
      break;
    case PROP_BORDER:
      g_value_set_int (value, src->border);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
