/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-dam
 *
 * <refsect2>
 * <para>
 * Special-purpose element that enables a "cutting" muxing pipeline.
 * This is used for transcoding purposes where it is not desired to mux
 * the entire original stream into a new stream, but only selected parts of it.
 * The desired parts in such a scenario are selected depending on
 * <link linkend="GstDam--segment-mode">segment-mode</link> property:
 * <itemizedlist>
 * <listitem>
 * <para>
 * using regular seek-events (segment-mode or seek-mode);
 * an initial flushing seek followed by a number of segmenting seeks and
 * finally concluded with a normal seek.
 * The element then drops buffers that are out of the selected segment
 * range, to compensate for some elements not doing so (yes, there are such ...).
 * The usual end-of-segment messages (or final EOS) allow the 'application'
 * to pass from one segment to the next.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * cut-mode: allowing only a certain range of buffers to pass where the range is
 * specified by gst time (starting from 0).  When using
 * <link linkend="GstDam--use-count">counts</link>, such times are then
 * converted to either a frame count or a sample count and compared with
 * the corresponding count of the incoming data (using the format info
 * supplied by the caps) (so buffer timestamps are disregarded).
 * Alternatively, buffer timestamps can be used to decide on clipping.
 * </para>
 * </listitem>
 * </itemizedlist>
 * In addition, audio data can optionally be clipped to sample
 * <link linkend="GstDam--precision">precision</link> when
 * deciding based on buffer timestamps (it is always done so
 * when clipping based on counts).
 * </para>
 * <para>
 * The element can also be requested to send an
 * <link linkend="GstDam--force-eos">eos</link> event, so that streaming
 * can be cleanly completed, and e.g. muxers can complete EOS-actions.
 * </para>
 * <title>Usage</title>
 * <para>
 * This element is (likely) essentially in a muxing pipeline
 * whenever only a part of the input is selected, even when using segments.
 * The functionality to drop out-of-segment data is typically present in (base)sinks,
 * so this usually happens transparently in a playing pipeline.
 * However, in a muxing pipeline, a filesink can and should clearly have no impact
 * on this; so it is up to the rest of pipeline to perform this dropping.
 * Failing to do so would not only result in too much and unwanted data,
 * but will probably even block the muxing pipeline (in a collectpads instance),
 * due to (typically) an imbalance (in volume and timestamps)
 * in video and audio data.
 * </para>
 * <para>
 * In case of a demuxer element involved in providing input, it should be separated
 * from it by means of queue (or other thread boundary).
 * </para>
 * <para>
 * The sequence of events/operation is roughly as follows.
 * <itemizedlist>
 * <listitem>
 * <para>
 * The element starts up and announces its existence by posting a message on the bus.
 * This should be detected (synchronously) by the application using it,
 * which should block pads to prevent data flowing through prior to setup
 * being completed.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * This setup (typically happening when all relevant pads have blocked) consists
 * of setting the proper "mode", and providing section info in case of cut-mode.
 * After this, streaming can be continued by (optionally) performing seek and
 * (typically) unblocking pads.  The element then performs as described above,
 * depending on mode, and either the last normal seek will give rise to EOS,
 * or the element generates EOS when having passed all data for the last section.
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-entrans.h"

#define GST_TYPE_DAM \
  (gst_dam_get_type())
#define GST_DAM(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_DAM,GstDam))
#define GST_DAM_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_DAM,GstDamClass))
#define GST_IS_DAM(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_DAM))
#define GST_IS_DAM_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_DAM))


typedef struct _SectionInfo SectionInfo;
typedef struct _GstDam GstDam;
typedef struct _GstDamClass GstDamClass;

struct _SectionInfo
{
  guint64 begin_count, end_count;
  GstClockTime begin_time, end_time;
};

struct _GstDam
{
  GstBaseTransform basetransform;

  /* what mode we operate in */
  gboolean segment_mode;
  /* if cutting, use count or time */
  gboolean use_count;
  /* if audio and time-based, do precision slicing */
  gboolean precision;
  /* what we are actually doing in the drop area */
  gboolean dropping;
  /* count based dropping/cutting */
  /* eos has been requested for clean-up of stream */
  gboolean eos;
  /* do we respond to position query or pass along */
  gboolean handle_query;
  /* section management */
  gint section;
  GList *sections;
  GMemChunk *memchunk;
  SectionInfo *current;
  /* type of stream */
  gboolean type;
  /* (frame) position in stream */
  glong cut_position;
  /* audio position = byte position in stream */
  guint64 audio_count;
  /* stream audio info */
  gint sample, rate;
  /* stream video info */
  gint fps_num, fps_denom;
  /* segment info for current section */
  GstSegment segment;
  /* newsegment event we should sent for section */
  GstEvent *newsegment;
  /* segment-event sent for current section ? */
  gboolean sent_segment;
  /* last timestamp seen */
  GstClockTime last_stamp;

  /* the parent's setcaps function */
  GstPadSetCapsFunction btrans_setcaps;

};

enum
{
  TYPE_VIDEO,
  TYPE_AUDIO,
  TYPE_OTHER
};

struct _GstDamClass
{
  GstBaseTransformClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (dam_debug);
#define GST_CAT_DEFAULT dam_debug

/* GstDam signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_SEGMENT_MODE,
  PROP_USE_COUNT,
  PROP_PRECISION,
  PROP_QUERY,
  PROP_SECTION,
  PROP_BEGIN_COUNT,
  PROP_END_COUNT,
  PROP_BEGIN_TIME,
  PROP_END_TIME,
  PROP_SAVE,
  PROP_EOS,
  PROP_SAMPLERATE,
  PROP_SAMPLEWIDTH,
  PROP_FRAMERATE
      /* FILL ME */
};

#define  DEFAULT_SEGMENT_MODE   TRUE
#define  DEFAULT_QUERY          FALSE
#define  DEFAULT_USE_COUNT      TRUE
#define  DEFAULT_PRECISION      FALSE

static GstStaticPadTemplate gst_dam_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static GstStaticPadTemplate gst_dam_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS_ANY);

static gboolean gst_dam_start (GstBaseTransform * trans);
static gboolean gst_dam_stop (GstBaseTransform * trans);
static gboolean gst_dam_setcaps (GstPad * pad, GstCaps * caps);
static gboolean gst_dam_src_query (GstPad * pad, GstQuery * query);
static gboolean gst_dam_event (GstBaseTransform * trans, GstEvent * event);
static GstFlowReturn gst_dam_chain (GstPad * pad, GstBuffer * in);

static void gst_dam_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_dam_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstDam, gst_dam, GstBaseTransform, GST_TYPE_BASE_TRANSFORM);

static void
gst_dam_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Dam",
      "Generic", "Block and/or filter stream",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_dam_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_dam_src_template));
}

static void
gst_dam_class_init (GstDamClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (dam_debug, "dam", 0, "dam");

  gobject_class->set_property = gst_dam_set_property;
  gobject_class->get_property = gst_dam_get_property;

  g_object_class_install_property (gobject_class, PROP_SEGMENT_MODE,
      g_param_spec_boolean ("segment-mode", "Segment-Based Dam",
          "Control and filter flow based on segments and seeks",
          DEFAULT_SEGMENT_MODE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_USE_COUNT,
      g_param_spec_boolean ("use-count", "Count-Based Filter",
          "Filter flow based on byte or frame count",
          DEFAULT_USE_COUNT, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_PRECISION,
      g_param_spec_boolean ("precision", "Segment-Based Dam",
          "Precision filter, slicing (audio) buffers if needed",
          DEFAULT_PRECISION, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_QUERY,
      g_param_spec_boolean ("handle-query", "Handle Query",
          "Respond to position query", DEFAULT_QUERY, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_SECTION,
      g_param_spec_int ("section", "Section",
          "Current filtered section", -1, G_MAXINT, -1, G_PARAM_READABLE));

  g_object_class_install_property (gobject_class, PROP_BEGIN_COUNT,
      g_param_spec_long ("begin-count", "Begin Section",
          "Begin of section in frames", -1, G_MAXLONG, -1, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_END_COUNT,
      g_param_spec_long ("end-count", "End Section",
          "End of section in frames", -1, G_MAXLONG, -1, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_BEGIN_TIME,
      g_param_spec_uint64 ("begin-time", "Begin Section",
          "Begin of section in time", 0, G_MAXUINT64, 0, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_END_TIME,
      g_param_spec_uint64 ("end-time", "End Section",
          "End of section in time", 0, G_MAXUINT64, 0, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_SAVE,
      g_param_spec_boolean ("save-section", "Save Section",
          "Commit current section info for processing",
          TRUE, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_EOS,
      g_param_spec_boolean ("force-eos", "Force EOS",
          "Force End-Of-Stream", TRUE, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_SAMPLERATE,
      g_param_spec_int ("samplerate", "samplerate",
          "Samplerate discovered in stream and used for cutting and stamping",
          0, G_MAXINT, 0, G_PARAM_READABLE));

  g_object_class_install_property (gobject_class, PROP_SAMPLEWIDTH,
      g_param_spec_int ("samplewidth", "samplewidth",
          "Width of a sample as deduced from stream and used for cutting and stamping",
          0, G_MAXINT, 0, G_PARAM_READABLE));

  g_object_class_install_property (gobject_class, PROP_FRAMERATE,
      g_param_spec_string ("framerate", "framerate",
          "Framerate discovered in stream and used for cutting and stamping",
          "1/1", G_PARAM_READABLE));

  trans_class->start = GST_DEBUG_FUNCPTR (gst_dam_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_dam_stop);
  trans_class->event = GST_DEBUG_FUNCPTR (gst_dam_event);
}

static void
gst_dam_init (GstDam * dam, GstDamClass * g_class)
{
  GstBaseTransform *trans = GST_BASE_TRANSFORM (dam);

  gst_base_transform_set_passthrough (trans, TRUE);

  /* HACK to override chain function so we can drop buffers */
  if (trans->sinkpad)
    gst_pad_set_chain_function (trans->sinkpad,
        GST_DEBUG_FUNCPTR (gst_dam_chain));

  /* HACK need some caps info */
  if (trans->sinkpad) {
    dam->btrans_setcaps = GST_PAD_SETCAPSFUNC (trans->sinkpad);
    gst_pad_set_setcaps_function (trans->sinkpad,
        GST_DEBUG_FUNCPTR (gst_dam_setcaps));
  }

  /* may have to handle query */
  if (trans->srcpad)
    gst_pad_set_query_function (trans->srcpad,
        GST_DEBUG_FUNCPTR (gst_dam_src_query));

  dam->section = -1;
  dam->segment_mode = DEFAULT_SEGMENT_MODE;
  dam->use_count = DEFAULT_USE_COUNT;
  dam->precision = DEFAULT_PRECISION;
  dam->handle_query = DEFAULT_QUERY;
  /* no div by zero anywhere */
  dam->fps_denom = 1;
}

static GstMessage *
gst_dam_new_message (GstDam * dam, const gchar * msg)
{
  GstStructure *s;

  s = gst_structure_new ("dam", msg, G_TYPE_BOOLEAN, TRUE, NULL);
  return gst_message_new_element (GST_OBJECT (dam), s);
}

static void
gst_dam_calc_counts (GstDam * dam)
{
  GList *node;

  node = g_list_first (dam->sections);
  while (node) {
    SectionInfo *info = (SectionInfo *) node->data;

    node = g_list_next (node);

    if (dam->type == TYPE_AUDIO) {
      info->begin_count =
          gst_util_uint64_scale (info->begin_time, dam->rate, GST_SECOND)
          * dam->sample;
      info->end_count =
          gst_util_uint64_scale (info->end_time, dam->rate, GST_SECOND)
          * dam->sample;
    } else if (dam->type == TYPE_VIDEO) {
      info->begin_count =
          gst_util_uint64_scale (info->begin_time, dam->fps_num,
          GST_SECOND * dam->fps_denom);
      info->end_count =
          gst_util_uint64_scale (info->end_time, dam->fps_num,
          GST_SECOND * dam->fps_denom);
    }

    if (!GST_CLOCK_TIME_IS_VALID (info->end_time))
      info->end_count = G_MAXUINT64;
  }
}

static gboolean
gst_dam_setcaps (GstPad * pad, GstCaps * caps)
{
  GstDam *dam = GST_DAM (GST_PAD_PARENT (pad));
  GstStructure *structure;
  const gchar *mime;
  gint width, channels;
  gboolean ret = TRUE;

  structure = gst_caps_get_structure (caps, 0);
  mime = gst_structure_get_name (structure);
  if (g_strrstr (mime, "audio")) {
    dam->type = TYPE_AUDIO;
    if (!gst_structure_get_int (structure, "rate", &dam->rate))
      goto not_found;
    if (!gst_structure_get_int (structure, "width", &width))
      goto not_found;
    if (!gst_structure_get_int (structure, "channels", &channels))
      goto not_found;
    dam->sample = channels * width / 8;
    g_object_freeze_notify (G_OBJECT (dam));
    g_object_notify (G_OBJECT (dam), "samplerate");
    g_object_notify (G_OBJECT (dam), "samplewidth");
    g_object_thaw_notify (G_OBJECT (dam));
  } else if (g_strrstr (mime, "video")) {
    const GValue *fps;

    dam->type = TYPE_VIDEO;
    fps = gst_structure_get_value (structure, "framerate");
    if (fps) {
      g_return_val_if_fail (GST_VALUE_HOLDS_FRACTION (fps), FALSE);
      dam->fps_num = gst_value_get_fraction_numerator (fps);
      dam->fps_denom = gst_value_get_fraction_denominator (fps);
    }
    g_object_notify (G_OBJECT (dam), "framerate");
  } else {                      /* some other type, subtitle, etc */
    dam->type = TYPE_OTHER;
    /* force time based, non precision cutting */
    if (!dam->segment_mode) {
      dam->use_count = FALSE;
      dam->precision = FALSE;
    }
  }
  gst_dam_calc_counts (dam);

exit:
  {
    if (ret && dam->btrans_setcaps)
      ret = dam->btrans_setcaps (pad, caps);
    return ret;
  }

not_found:
  {
    if ((dam->segment_mode || !dam->use_count) && !dam->precision)
      ret = TRUE;
    else
      ret = FALSE;
    goto exit;
  }
}

static gboolean
gst_dam_src_query (GstPad * pad, GstQuery * query)
{
  GstDam *dam = GST_DAM (GST_PAD_PARENT (pad));
  GstBaseTransform *trans = GST_BASE_TRANSFORM (dam);
  GstPad *peer;

  switch (GST_QUERY_TYPE (query)) {
    case GST_QUERY_POSITION:
      if (dam->handle_query) {
        gst_query_set_position (query, GST_FORMAT_TIME, dam->last_stamp);
        return TRUE;
      }
      break;
    default:
      break;
  }

  if (trans->sinkpad) {
    peer = GST_PAD_PEER (trans->sinkpad);
    if (peer)
      return gst_pad_query (peer, query);
  }

  return FALSE;
}

static gboolean
gst_dam_event (GstBaseTransform * trans, GstEvent * event)
{
  GstDam *dam;
  gboolean ret = TRUE;

  dam = GST_DAM (trans);

  GST_BASE_TRANSFORM_CLASS (parent_class)->event (trans, event);

  if (!dam->segment_mode && (GST_EVENT_TYPE (event) == GST_EVENT_NEWSEGMENT)) {
    /* eat up segments */
    ret = FALSE;
  }

  return ret;
}


/* performs precise cutting of an audio buffer w.r.t. begin and end of segment
   returns -1 if buffer is before the desired segment,
            0 if partially (or completely) in
            1 if buffer is beyond
  in case of 0, there may be some leftover if it was only partially
*/
static inline gint
gst_dam_cut (GstDam * dam, GstBuffer ** buf, GstBuffer ** leftover)
{
  gint rate, sample;
  GstBuffer *outbuf;
  gint start_offset, end_offset;
  guint size;
  guint64 begin_count, end_count;
  gint result = 0;

  /* more convenient writing */
  sample = dam->sample;
  rate = dam->rate;
  size = GST_BUFFER_SIZE (*buf);

  begin_count = dam->current->begin_count;
  end_count = dam->current->end_count;

  /* clean case; all passes */
  if (dam->audio_count >= begin_count && dam->audio_count + size <= end_count) {
    GST_DEBUG_OBJECT (dam, "audio buffer within segment");
    result = start_offset = 0;
    end_offset = size;
    goto stamp;
  }

  /* buffer not yet in segment */
  if (dam->audio_count + size <= begin_count) {
    GST_DEBUG_OBJECT (dam, "audio buffer before segment");
    result = -1;
    goto exit;
  }

  /* buffer beyond segment */
  if (dam->audio_count >= end_count) {
    GST_DEBUG_OBJECT (dam, "audio buffer after segment");
    result = 1;
    goto exit;
  }

  /* buffer partially in segment */
  /* find offsets */
  if (begin_count >= dam->audio_count)
    start_offset = begin_count - dam->audio_count;
  else
    start_offset = 0;
  GST_DEBUG ("%d", start_offset);
  end_offset = (MIN (dam->audio_count + size, end_count) - dam->audio_count);
  GST_DEBUG ("%d", end_offset);
  /* play safe */
  start_offset = MIN (start_offset, size);
  GST_DEBUG ("%d", start_offset);
  end_offset = MIN (MAX (end_offset, start_offset), size);
  GST_DEBUG ("%d", end_offset);

  if (size - end_offset > 0) {
    /* have some leftover */
    GST_DEBUG_OBJECT (dam, "creating leftover audio buffer, %d-%d",
        end_offset, size);
    *leftover = gst_buffer_create_sub (*buf, end_offset, size - end_offset);
    *leftover = gst_buffer_make_metadata_writable (*leftover);
    GST_BUFFER_TIMESTAMP (*leftover) =
        gst_util_uint64_scale ((dam->audio_count + end_offset) / sample,
        GST_SECOND, rate);
    GST_BUFFER_DURATION (*leftover) =
        gst_util_uint64_scale ((size - end_offset) / sample, GST_SECOND, rate);
  } else
    *leftover = NULL;

  GST_DEBUG_OBJECT (dam, "creating sub audio buffer, %d-%d",
      start_offset, end_offset);
  outbuf = gst_buffer_create_sub (*buf, start_offset,
      end_offset - start_offset);
  gst_buffer_unref (*buf);
  *buf = outbuf;

stamp:
  *buf = gst_buffer_make_metadata_writable (*buf);
  gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_TRANSFORM (dam)->srcpad));
  GST_BUFFER_TIMESTAMP (*buf) =
      gst_util_uint64_scale ((dam->audio_count + start_offset) / sample,
      GST_SECOND, rate);
  GST_BUFFER_DURATION (*buf) =
      gst_util_uint64_scale ((end_offset - start_offset) / sample,
      GST_SECOND, rate);
exit:
  dam->audio_count += size;
  return result;
}

/* returns -1 if buffer is before the desired segment,
            0 if partially (or completely) in
            1 if buffer is beyond
*/
static inline gboolean
gst_dam_segment (GstDam * dam, GstSegment * segment, GstBuffer * buf,
    GstBuffer ** leftover)
{
  GstClockTime start_time, stop_time = GST_CLOCK_TIME_NONE, duration;
  gboolean drop = FALSE;

  start_time = GST_BUFFER_TIMESTAMP (buf);
  duration = GST_BUFFER_DURATION (buf);

  /* if invalid start_time or not time based segment - we let it pass */
  if (GST_CLOCK_TIME_IS_VALID (start_time) && segment != NULL
      && segment->format == GST_FORMAT_TIME) {
    if (GST_CLOCK_TIME_IS_VALID (duration))
      stop_time = start_time + duration;
    else
      stop_time = start_time;
    GST_DEBUG_OBJECT (dam, "Checking with segment start %" GST_TIME_FORMAT ""
        ", stop %" GST_TIME_FORMAT ", duration %" GST_TIME_FORMAT,
        GST_TIME_ARGS (segment->start), GST_TIME_ARGS (segment->stop),
        GST_TIME_ARGS (segment->duration));
    if (!gst_segment_clip (segment, GST_FORMAT_TIME, start_time, stop_time,
            NULL, NULL)) {
      drop = TRUE;
      GST_DEBUG_OBJECT (dam, "Dropping buffer by segment");
    }
    /* update the segment with last seen */
    gst_segment_set_last_stop (segment, GST_FORMAT_TIME, stop_time);
  }

  if (drop) {                   /* before or after ? */
    if (start_time > (GstClockTime) dam->segment.stop)
      return 1;
    else
      return -1;
  } else {
    if (dam->type == TYPE_AUDIO && dam->precision) {
      /* we believe the buffer is right about this */
      dam->audio_count
          =
          gst_util_uint64_scale (GST_BUFFER_TIMESTAMP (buf), dam->rate,
          GST_SECOND)
          * dam->sample;
      /* should be agreement on being in segment */
      if (gst_dam_cut (dam, &buf, leftover))
        g_warning ("No precision cut for segment-passed buffer");
    }
    return 0;
  }
}

static GstFlowReturn
gst_dam_chain (GstPad * pad, GstBuffer * buf)
{
  GstDam *dam;
  GList *entry;
  GstFlowReturn result = GST_FLOW_OK;
  GstBaseTransform *trans = GST_BASE_TRANSFORM (GST_PAD_PARENT (pad));
  dam = GST_DAM (trans);
  GstBuffer *leftover = NULL;

  GST_DEBUG_OBJECT (dam,
      "Received buffer of time %" GST_TIME_FORMAT ", duration %" GST_TIME_FORMAT
      ", size %d", GST_TIME_ARGS (GST_BUFFER_TIMESTAMP (buf)),
      GST_TIME_ARGS (GST_BUFFER_DURATION (buf)), GST_BUFFER_SIZE (buf));

  /* if we are eos */
  if (dam->dropping || GST_BUFFER_FLAG_IS_SET (buf, GST_BUFFER_FLAG_GAP)) {
    goto drop;
  }

  if (dam->eos) {
    gst_pad_push_event (trans->srcpad, gst_event_new_eos ());
    result = GST_FLOW_OK;
    dam->dropping = TRUE;
    goto drop;
  }

  if (dam->segment_mode) {
    if (trans->have_newsegment) {
      /* need this for precision cutting */
      dam->current->begin_count =
          gst_util_uint64_scale (trans->segment.start, dam->rate, GST_SECOND)
          * dam->sample;
      if (GST_CLOCK_TIME_IS_VALID (trans->segment.stop))
        dam->current->end_count =
            gst_util_uint64_scale (trans->segment.stop, dam->rate, GST_SECOND)
            * dam->sample;
      else
        dam->current->end_count = G_MAXUINT64;
      trans->have_newsegment = FALSE;
    }
    if (gst_dam_segment (dam, &trans->segment, buf, &leftover))
      goto drop;
    else {
      /* never mind any leftover, it will come again if needed */
      if (leftover)
        gst_buffer_unref (leftover);
      leftover = NULL;
      goto push;
    }
  }

  /* non-segment based cutting; some ugly stuff here ... */

  /* make mark that we have started seeing data */
  if (dam->section < 0) {
    dam->section = 0;
    g_object_notify (G_OBJECT (dam), "section");
    entry = g_list_first (dam->sections);
    /* there !must! be sections in this state */
    g_return_val_if_fail (entry != NULL, GST_FLOW_ERROR);
    dam->current = entry->data;
    dam->newsegment = gst_event_new_new_segment (FALSE, 1.0, GST_FORMAT_TIME,
        dam->current->begin_time, dam->current->end_time,
        dam->current->begin_time);
    gst_segment_set_newsegment (&dam->segment, FALSE, 1.0, GST_FORMAT_TIME,
        dam->current->begin_time, dam->current->end_time,
        dam->current->begin_time);
  }

  /* do it anyway, only needed if video */
  dam->cut_position++;

check:
  if (dam->type == TYPE_VIDEO && dam->use_count) {
    /* timestamp needed either way */
    if (G_LIKELY (dam->fps_num > 0)) {
      gst_buffer_make_metadata_writable (buf);
      GST_BUFFER_TIMESTAMP (buf) =
          gst_util_uint64_scale (dam->cut_position,
          dam->fps_denom * GST_SECOND, dam->fps_num);
      GST_BUFFER_DURATION (buf) =
          gst_util_uint64_scale (GST_SECOND, dam->fps_denom, dam->fps_num);
    }
    /* decide to drop or push */
    if (dam->cut_position < dam->current->begin_count)
      goto drop;
    if (dam->cut_position < dam->current->end_count)
      goto push;
  } else {
    gint res;

    if (dam->use_count)
      /* so not video */
      res = gst_dam_cut (dam, &buf, &leftover);
    else
      res = gst_dam_segment (dam, &dam->segment, buf, &leftover);

    if (!res)
      goto push;
    if (res < 0)
      goto drop;
  }

  /* ended section, need next one */
  dam->section++;
  g_object_notify (G_OBJECT (dam), "section");
  dam->sent_segment = FALSE;
  entry = g_list_nth (dam->sections, dam->section);
  if (entry) {
    dam->current = entry->data;
    /* prepare segment stuff for new section */
    if (dam->newsegment)
      gst_event_unref (dam->newsegment);
    dam->newsegment = gst_event_new_new_segment (FALSE, 1.0, GST_FORMAT_TIME,
        dam->current->begin_time, dam->current->end_time,
        dam->current->begin_time);
    gst_segment_set_newsegment (&dam->segment, FALSE, 1.0, GST_FORMAT_TIME,
        dam->current->begin_time, dam->current->end_time,
        dam->current->begin_time);
    /* if sections are close together, buffer may already be in next section */
    goto check;
  } else {
    dam->current = NULL;
    dam->dropping = TRUE;
    gst_pad_push_event (trans->srcpad, gst_event_new_eos ());
    if (leftover)
      gst_buffer_unref (leftover);
    goto drop;
  }

  /* common code for either mode */
push:
  if (!dam->segment_mode && !dam->sent_segment) {
    /* need to generate a segment event for non-segment cutting */
    GST_DEBUG_OBJECT (dam, "sending newsegment");
    gst_pad_push_event (trans->srcpad, dam->newsegment);
    /* is no longer ours */
    dam->newsegment = NULL;
    dam->sent_segment = TRUE;
  }

  dam->last_stamp = GST_BUFFER_TIMESTAMP (buf);
  result = gst_pad_push (trans->srcpad, buf);
  if (result == GST_FLOW_OK && leftover) {
    /* have another go with the remainder */
    buf = leftover;
    leftover = NULL;
    goto check;
  } else
    return result;

drop:
  {
    GST_DEBUG_OBJECT (dam, "dropping buffer");
    dam->last_stamp = GST_BUFFER_TIMESTAMP (buf);
    gst_buffer_unref (buf);
    return result;
  }
}

static gboolean
gst_dam_start (GstBaseTransform * trans)
{
  GstDam *dam = GST_DAM (trans);

  dam->dropping = FALSE;
  dam->eos = FALSE;
  dam->section = -1;
  dam->sections = NULL;
  dam->cut_position = -1;
  dam->audio_count = 0;
  dam->sample = dam->rate = 1;
  dam->fps_num = dam->fps_denom = -1;
  gst_segment_init (&dam->segment, GST_FORMAT_TIME);
  dam->newsegment = NULL;
  dam->sent_segment = FALSE;
  dam->last_stamp = 0;
  dam->memchunk = g_mem_chunk_create (SectionInfo, 10, G_ALLOC_ONLY);
  dam->current = g_chunk_new (SectionInfo, dam->memchunk);

  /* announce existence */
  gst_element_post_message (GST_ELEMENT (trans),
      gst_dam_new_message (dam, "announce"));

  return TRUE;
}

static gboolean
gst_dam_stop (GstBaseTransform * trans)
{
  GstDam *dam = GST_DAM (trans);

  dam->section = -1;
  if (dam->sections)
    g_list_free (dam->sections);
  g_mem_chunk_destroy (dam->memchunk);
  dam->memchunk = NULL;
  dam->sections = NULL;
  dam->current = NULL;
  if (dam->newsegment)
    gst_event_unref (dam->newsegment);
  dam->newsegment = NULL;

  return TRUE;
}

static void
gst_dam_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstDam *dam;

  g_return_if_fail (GST_IS_DAM (object));
  dam = GST_DAM (object);

  if (dam->section >= 0 && prop_id != PROP_EOS) {
    g_critical ("Cannot set property on %s once streaming has begun.",
        GST_OBJECT_NAME (object));
    return;
  }

  if (!dam->current && prop_id != PROP_EOS && prop_id != PROP_QUERY
      && prop_id != PROP_USE_COUNT && prop_id != PROP_PRECISION) {
    g_critical ("Cannot set property on %s before streaming has started.",
        GST_OBJECT_NAME (object));
    return;
  }

  switch (prop_id) {
    case PROP_SEGMENT_MODE:
      dam->segment_mode = g_value_get_boolean (value);
      break;
    case PROP_USE_COUNT:
      dam->use_count = g_value_get_boolean (value);
      break;
    case PROP_PRECISION:
      dam->precision = g_value_get_boolean (value);
      break;
    case PROP_QUERY:
      dam->handle_query = g_value_get_boolean (value);
      break;
    case PROP_SECTION:
      dam->section = g_value_get_int (value);
      break;
    case PROP_BEGIN_COUNT:
      dam->current->begin_count = g_value_get_long (value);
      break;
    case PROP_END_COUNT:
      dam->current->end_count = g_value_get_long (value);
      break;
    case PROP_BEGIN_TIME:
      dam->current->begin_time = g_value_get_uint64 (value);
      break;
    case PROP_END_TIME:
      dam->current->end_time = g_value_get_uint64 (value);
      break;
    case PROP_SAVE:
      dam->sections = g_list_append (dam->sections, dam->current);
      dam->current = g_chunk_new (SectionInfo, dam->memchunk);
      break;
    case PROP_EOS:
      dam->eos = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_dam_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstDam *dam;

  g_return_if_fail (GST_IS_DAM (object));
  dam = GST_DAM (object);

  switch (prop_id) {
    case PROP_SEGMENT_MODE:
      g_value_set_boolean (value, dam->segment_mode);
      break;
    case PROP_USE_COUNT:
      g_value_set_boolean (value, dam->use_count);
      break;
    case PROP_PRECISION:
      g_value_set_boolean (value, dam->precision);
      break;
    case PROP_QUERY:
      g_value_set_boolean (value, dam->handle_query);
      break;
    case PROP_SECTION:
      g_value_set_int (value, dam->section);
      break;
    case PROP_SAMPLERATE:
      g_value_set_int (value, dam->rate);
      break;
    case PROP_SAMPLEWIDTH:
      g_value_set_int (value, dam->sample);
      break;
    case PROP_FRAMERATE:
    {
      GValue fvalue = { 0, };

      g_value_init (&fvalue, GST_TYPE_FRACTION);
      gst_value_set_fraction (&fvalue, dam->fps_num, dam->fps_denom);
      g_value_transform (&fvalue, value);
      break;
    }
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
